/*
 * Copyright (c) 2021
 * NDE Netzdesign und -entwicklung AG, Hamburg, Germany
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program (see the file LICENSE.txt for more
 * details); if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 */

package org.acplt.oncrpc.apps.jrpcgen;

/**
 * An XDR definition represents a constant definition or one of the
 * complex type definitions. A complex type is one of <em>enum</em>,
 * <em>struct</em>, <em>union</em> or <em>typedef</em>.
 * An XDR definition consists of the definition type and an identifier.
 * The definition types are defined in the enumeration {@link Type} of this
 * class.
 * 
 * <p>XDR definitions are documentable and therefore extend the class
 * {@link JrpcgenDocumentable}. As they provide an identifier, they implement
 * the interface {@link JrpcgenItem} and therefore can be used in item tables
 * based on {@link JrpcgenItemTable}.
 * 
 * <p>Although this class is not abstract, it is not intended to be used
 * directly. It will be used as a base class for concrete realisations of constant
 * definitions and complex type definitions, enabling the identification of
 * their nature at this generalized level.
 * 
 * @author Harald Wirths {@literal <hwirths@nde.ag>}
 *
 */
public class JrpcgenXdrDefinition extends JrpcgenDocumentable implements JrpcgenItem {

	/**
	 * Type definition for a table of XDR definitions. 
	 */
	public static class Table extends JrpcgenItemTable<JrpcgenXdrDefinition> {}
	
	/**
	 * Specification of the possible XDR definition types.
	 */
	public enum Type {
		/**
		 * An XDR definition of this type is expected to belong
		 * to a constant definition.
		 */
		CONST,
		
		/**
		 * An XDR definition of this type is expected to belong
		 * to an enumeration definition.
		 */
		ENUM,
		
		/**
		 * An XDR definition of this type is expected to belong
		 * to a structure definition.
		 */
		STRUCT,
		
		/**
		 * An XDR definition of this type is expected to belong
		 * to a union definition.
		 */
		UNION,
		
		/**
		 * An XDR definition of this type is expected to belong
		 * to a type definition.
		 */
		TYPEDEF;
	}
	
	/**
	 * Creates a new XDR definition instance.
	 * 
	 * @param identifier The identifier of the new XDR definition.
	 * @param type The definition type of the new XDR definition.
	 */
	public JrpcgenXdrDefinition(String identifier, Type type) {
		this.identifier = identifier;
		this.type = type;
	}
	
	/**
	 * Returns the identifier of this XDR definition.
	 * 
	 * @return The identifier of this XDR definition.
	 */
	@Override
	final public  String getIdentifier() {
		return identifier;
	}
	
	/**
	 * Returns the XDR aka definition type of this
	 * XDR definition.
	 *  
	 * @return The XDR (definition) type of this XDR definition.
	 * @see Type
	 */
	final public Type getXdrType() {
		return type;
	}
	
	/**
	 * Returns whether this XDR definition belongs to a constant
	 * definition.
	 * 
	 * @return {@code true} in case this XDR definition belongs to
	 *         a constant definition, {@code false} otherwise.
	 */
	final public boolean isConst() {
		return Type.CONST.equals(type);
	}
	
	/**
	 * Returns whether this XDR definition belongs to an
	 * enumeration definition.
	 * 
	 * @return {@code true} in case this XDR definition belongs
	 *         to an enumeration definition, {@code false} otherwise.
	 */
	final public boolean isEnum() {
		return Type.ENUM.equals(type);
	}
	
	/**
	 * Returns whether this XDR definition belongs to a
	 * stucture definition.
	 * 
	 * @return {@code true} in case this XDR definition belongs
	 *         to a structure definition, {@code false} otherwise.
	 */
	final public boolean isStruct() {
		return Type.STRUCT.equals(type);
	}
	
	/**
	 * Returns whether this XDR definition belongs to a
	 * union definition.
	 * 
	 * @return {@code true} in case this XDR definition belongs
	 *         to a union definition, {@code false} otherwise.
	 */
	final public boolean isUnion() {
		return Type.UNION.equals(type);
	}
	
	/**
	 * Returns whether this XDR definition belongs to a
	 * type definition.
	 * 
	 * @return {@code true} in case this XDR definition belongs
	 *         to a type definition, {@code false} otherwise.
	 */
	final public boolean isTypedef() {
		return Type.TYPEDEF.equals(type);
	}
	
	private final String identifier;
	private final Type type;
	
}
