// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#ifndef DOUBLE_FIXED_PRECISION_TO_CHARS_TEST_CASES_4_HPP
#define DOUBLE_FIXED_PRECISION_TO_CHARS_TEST_CASES_4_HPP

#include <charconv>

#include "test.hpp"
using namespace std;

inline constexpr DoublePrecisionToCharsTestCase double_fixed_precision_to_chars_test_cases_4[] = {
    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffffffffffp-1022, chars_format::fixed, 1074,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022250738585"
        "0720088902458687608585988765042311224095946549352480256244000922823569517877588880375915526423097809504343"
        "1208587738715835729182199302029437922422355981982750124204178896957131179108226104397197960400045489739193"
        "8079198936081525613113376149842043271751033627391549782731594143828136275113838604094249464942286316695429"
        "1050802018159266421349966065178030950759130587198464239060686371020051087232827846788436319445158661350412"
        "2347901479236958520832159762106637540161373658304419360371477835530668283453563400507407304013560296804637"
        "5918583163124224521599262546494300836851861719422417646455137135420132217031370496583210154654068035397417"
        "9060225895030235019375197730309457631732108525072993050897615825191597207572324554347709124613174935802817"
        "34466552734375"},
    {0x1.fffffffffffffp-1022, chars_format::fixed, 1074,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000044501477170"
        "1440227211481959341826395186963909270329129604685221944964444404215389103305904781627017582829831782607924"
        "2213740172877389189291055314414815641243486759976282126534658507104573762744298025962244902903779698114444"
        "6145705102663115100318287949527959668236039986479250965780342141637013812613333119898765515451440315261253"
        "8132666529513060001849177663286607555958373922409899478075565940981010216121988146052587425791790000716759"
        "9934414508608720568157791543592301891033496486942061405218289243144579760516365090360651414037721744226256"
        "1590244668525767372446430075513332450079650686719491377688478005309963967709758965844137894433796621993967"
        "3169362804570848666132067970177289160800206986794085513437288676754097207572324554347709124613174935802817"
        "34466552734375"},
    {0x1.fffffffffffffp-1021, chars_format::fixed, 1073,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000089002954340"
        "2880454422963918683652790373927818540658259209370443889928888808430778206611809563254035165659663565215848"
        "4427480345754778378582110628829631282486973519952564253069317014209147525488596051924489805807559396228889"
        "2291410205326230200636575899055919336472079972958501931560684283274027625226666239797531030902880630522507"
        "6265333059026120003698355326573215111916747844819798956151131881962020432243976292105174851583580001433519"
        "9868829017217441136315583087184603782066992973884122810436578486289159521032730180721302828075443488452512"
        "3180489337051534744892860151026664900159301373438982755376956010619927935419517931688275788867593243987934"
        "6338725609141697332264135940354578321600413973588171026874577353508194415144649108695418249226349871605634"
        "6893310546875"},
    {0x1.fffffffffffffp-1020, chars_format::fixed, 1072,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000178005908680"
        "5760908845927837367305580747855637081316518418740887779857777616861556413223619126508070331319327130431696"
        "8854960691509556757164221257659262564973947039905128506138634028418295050977192103848979611615118792457778"
        "4582820410652460401273151798111838672944159945917003863121368566548055250453332479595062061805761261045015"
        "2530666118052240007396710653146430223833495689639597912302263763924040864487952584210349703167160002867039"
        "9737658034434882272631166174369207564133985947768245620873156972578319042065460361442605656150886976905024"
        "6360978674103069489785720302053329800318602746877965510753912021239855870839035863376551577735186487975869"
        "2677451218283394664528271880709156643200827947176342053749154707016388830289298217390836498452699743211269"
        "378662109375"},
    {0x1.fffffffffffffp-1019, chars_format::fixed, 1071,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000356011817361"
        "1521817691855674734611161495711274162633036837481775559715555233723112826447238253016140662638654260863393"
        "7709921383019113514328442515318525129947894079810257012277268056836590101954384207697959223230237584915556"
        "9165640821304920802546303596223677345888319891834007726242737133096110500906664959190124123611522522090030"
        "5061332236104480014793421306292860447666991379279195824604527527848081728975905168420699406334320005734079"
        "9475316068869764545262332348738415128267971895536491241746313945156638084130920722885211312301773953810049"
        "2721957348206138979571440604106659600637205493755931021507824042479711741678071726753103155470372975951738"
        "5354902436566789329056543761418313286401655894352684107498309414032777660578596434781672996905399486422538"
        "75732421875"},
    {0x1.fffffffffffffp-1018, chars_format::fixed, 1070,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000712023634722"
        "3043635383711349469222322991422548325266073674963551119431110467446225652894476506032281325277308521726787"
        "5419842766038227028656885030637050259895788159620514024554536113673180203908768415395918446460475169831113"
        "8331281642609841605092607192447354691776639783668015452485474266192221001813329918380248247223045044180061"
        "0122664472208960029586842612585720895333982758558391649209055055696163457951810336841398812668640011468159"
        "8950632137739529090524664697476830256535943791072982483492627890313276168261841445770422624603547907620098"
        "5443914696412277959142881208213319201274410987511862043015648084959423483356143453506206310940745951903477"
        "0709804873133578658113087522836626572803311788705368214996618828065555321157192869563345993810798972845077"
        "5146484375"},
    {0x1.fffffffffffffp-1017, chars_format::fixed, 1069,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001424047269444"
        "6087270767422698938444645982845096650532147349927102238862220934892451305788953012064562650554617043453575"
        "0839685532076454057313770061274100519791576319241028049109072227346360407817536830791836892920950339662227"
        "6662563285219683210185214384894709383553279567336030904970948532384442003626659836760496494446090088360122"
        "0245328944417920059173685225171441790667965517116783298418110111392326915903620673682797625337280022936319"
        "7901264275479058181049329394953660513071887582145964966985255780626552336523682891540845249207095815240197"
        "0887829392824555918285762416426638402548821975023724086031296169918846966712286907012412621881491903806954"
        "1419609746267157316226175045673253145606623577410736429993237656131110642314385739126691987621597945690155"
        "029296875"},
    {0x1.fffffffffffffp-1016, chars_format::fixed, 1068,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002848094538889"
        "2174541534845397876889291965690193301064294699854204477724441869784902611577906024129125301109234086907150"
        "1679371064152908114627540122548201039583152638482056098218144454692720815635073661583673785841900679324455"
        "3325126570439366420370428769789418767106559134672061809941897064768884007253319673520992988892180176720244"
        "0490657888835840118347370450342883581335931034233566596836220222784653831807241347365595250674560045872639"
        "5802528550958116362098658789907321026143775164291929933970511561253104673047365783081690498414191630480394"
        "1775658785649111836571524832853276805097643950047448172062592339837693933424573814024825243762983807613908"
        "2839219492534314632452350091346506291213247154821472859986475312262221284628771478253383975243195891380310"
        "05859375"},
    {0x1.fffffffffffffp-1015, chars_format::fixed, 1067,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005696189077778"
        "4349083069690795753778583931380386602128589399708408955448883739569805223155812048258250602218468173814300"
        "3358742128305816229255080245096402079166305276964112196436288909385441631270147323167347571683801358648910"
        "6650253140878732840740857539578837534213118269344123619883794129537768014506639347041985977784360353440488"
        "0981315777671680236694740900685767162671862068467133193672440445569307663614482694731190501349120091745279"
        "1605057101916232724197317579814642052287550328583859867941023122506209346094731566163380996828383260960788"
        "3551317571298223673143049665706553610195287900094896344125184679675387866849147628049650487525967615227816"
        "5678438985068629264904700182693012582426494309642945719972950624524442569257542956506767950486391782760620"
        "1171875"},
    {0x1.fffffffffffffp-1014, chars_format::fixed, 1066,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011392378155556"
        "8698166139381591507557167862760773204257178799416817910897767479139610446311624096516501204436936347628600"
        "6717484256611632458510160490192804158332610553928224392872577818770883262540294646334695143367602717297821"
        "3300506281757465681481715079157675068426236538688247239767588259075536029013278694083971955568720706880976"
        "1962631555343360473389481801371534325343724136934266387344880891138615327228965389462381002698240183490558"
        "3210114203832465448394635159629284104575100657167719735882046245012418692189463132326761993656766521921576"
        "7102635142596447346286099331413107220390575800189792688250369359350775733698295256099300975051935230455633"
        "1356877970137258529809400365386025164852988619285891439945901249048885138515085913013535900972783565521240"
        "234375"},
    {0x1.fffffffffffffp-1013, chars_format::fixed, 1065,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000022784756311113"
        "7396332278763183015114335725521546408514357598833635821795534958279220892623248193033002408873872695257201"
        "3434968513223264917020320980385608316665221107856448785745155637541766525080589292669390286735205434595642"
        "6601012563514931362963430158315350136852473077376494479535176518151072058026557388167943911137441413761952"
        "3925263110686720946778963602743068650687448273868532774689761782277230654457930778924762005396480366981116"
        "6420228407664930896789270319258568209150201314335439471764092490024837384378926264653523987313533043843153"
        "4205270285192894692572198662826214440781151600379585376500738718701551467396590512198601950103870460911266"
        "2713755940274517059618800730772050329705977238571782879891802498097770277030171826027071801945567131042480"
        "46875"},
    {0x1.fffffffffffffp-1012, chars_format::fixed, 1064,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000045569512622227"
        "4792664557526366030228671451043092817028715197667271643591069916558441785246496386066004817747745390514402"
        "6869937026446529834040641960771216633330442215712897571490311275083533050161178585338780573470410869191285"
        "3202025127029862725926860316630700273704946154752988959070353036302144116053114776335887822274882827523904"
        "7850526221373441893557927205486137301374896547737065549379523564554461308915861557849524010792960733962233"
        "2840456815329861793578540638517136418300402628670878943528184980049674768757852529307047974627066087686306"
        "8410540570385789385144397325652428881562303200759170753001477437403102934793181024397203900207740921822532"
        "5427511880549034119237601461544100659411954477143565759783604996195540554060343652054143603891134262084960"
        "9375"},
    {0x1.fffffffffffffp-1011, chars_format::fixed, 1063,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000091139025244454"
        "9585329115052732060457342902086185634057430395334543287182139833116883570492992772132009635495490781028805"
        "3739874052893059668081283921542433266660884431425795142980622550167066100322357170677561146940821738382570"
        "6404050254059725451853720633261400547409892309505977918140706072604288232106229552671775644549765655047809"
        "5701052442746883787115854410972274602749793095474131098759047129108922617831723115699048021585921467924466"
        "5680913630659723587157081277034272836600805257341757887056369960099349537515705058614095949254132175372613"
        "6821081140771578770288794651304857763124606401518341506002954874806205869586362048794407800415481843645065"
        "0855023761098068238475202923088201318823908954287131519567209992391081108120687304108287207782268524169921"
        "875"},
    {0x1.fffffffffffffp-1010, chars_format::fixed, 1062,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000182278050488909"
        "9170658230105464120914685804172371268114860790669086574364279666233767140985985544264019270990981562057610"
        "7479748105786119336162567843084866533321768862851590285961245100334132200644714341355122293881643476765141"
        "2808100508119450903707441266522801094819784619011955836281412145208576464212459105343551289099531310095619"
        "1402104885493767574231708821944549205499586190948262197518094258217845235663446231398096043171842935848933"
        "1361827261319447174314162554068545673201610514683515774112739920198699075031410117228191898508264350745227"
        "3642162281543157540577589302609715526249212803036683012005909749612411739172724097588815600830963687290130"
        "1710047522196136476950405846176402637647817908574263039134419984782162216241374608216574415564537048339843"
        "75"},
    {0x1.fffffffffffffp-1009, chars_format::fixed, 1061,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000364556100977819"
        "8341316460210928241829371608344742536229721581338173148728559332467534281971971088528038541981963124115221"
        "4959496211572238672325135686169733066643537725703180571922490200668264401289428682710244587763286953530282"
        "5616201016238901807414882533045602189639569238023911672562824290417152928424918210687102578199062620191238"
        "2804209770987535148463417643889098410999172381896524395036188516435690471326892462796192086343685871697866"
        "2723654522638894348628325108137091346403221029367031548225479840397398150062820234456383797016528701490454"
        "7284324563086315081155178605219431052498425606073366024011819499224823478345448195177631201661927374580260"
        "3420095044392272953900811692352805275295635817148526078268839969564324432482749216433148831129074096679687"
        "5"},
    {0x1.fffffffffffffp-1008, chars_format::fixed, 1060,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000729112201955639"
        "6682632920421856483658743216689485072459443162676346297457118664935068563943942177056077083963926248230442"
        "9918992423144477344650271372339466133287075451406361143844980401336528802578857365420489175526573907060565"
        "1232402032477803614829765066091204379279138476047823345125648580834305856849836421374205156398125240382476"
        "5608419541975070296926835287778196821998344763793048790072377032871380942653784925592384172687371743395732"
        "5447309045277788697256650216274182692806442058734063096450959680794796300125640468912767594033057402980909"
        "4568649126172630162310357210438862104996851212146732048023638998449646956690896390355262403323854749160520"
        "684019008878454590780162338470561055059127163429705215653767993912864886496549843286629766225814819335937"
        "5"},
    {0x1.fffffffffffffp-1007, chars_format::fixed, 1059,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001458224403911279"
        "3365265840843712967317486433378970144918886325352692594914237329870137127887884354112154167927852496460885"
        "9837984846288954689300542744678932266574150902812722287689960802673057605157714730840978351053147814121130"
        "2464804064955607229659530132182408758558276952095646690251297161668611713699672842748410312796250480764953"
        "1216839083950140593853670575556393643996689527586097580144754065742761885307569851184768345374743486791465"
        "0894618090555577394513300432548365385612884117468126192901919361589592600251280937825535188066114805961818"
        "9137298252345260324620714420877724209993702424293464096047277996899293913381792780710524806647709498321041"
        "36803801775690918156032467694112211011825432685941043130753598782572977299309968657325953245162963867187"
        "5"},
    {0x1.fffffffffffffp-1006, chars_format::fixed, 1058,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002916448807822558"
        "6730531681687425934634972866757940289837772650705385189828474659740274255775768708224308335855704992921771"
        "9675969692577909378601085489357864533148301805625444575379921605346115210315429461681956702106295628242260"
        "4929608129911214459319060264364817517116553904191293380502594323337223427399345685496820625592500961529906"
        "2433678167900281187707341151112787287993379055172195160289508131485523770615139702369536690749486973582930"
        "1789236181111154789026600865096730771225768234936252385803838723179185200502561875651070376132229611923637"
        "8274596504690520649241428841755448419987404848586928192094555993798587826763585561421049613295418996642082"
        "73607603551381836312064935388224422023650865371882086261507197565145954598619937314651906490325927734375"},
    {0x1.fffffffffffffp-1005, chars_format::fixed, 1057,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005832897615645117"
        "3461063363374851869269945733515880579675545301410770379656949319480548511551537416448616671711409985843543"
        "9351939385155818757202170978715729066296603611250889150759843210692230420630858923363913404212591256484520"
        "9859216259822428918638120528729635034233107808382586761005188646674446854798691370993641251185001923059812"
        "4867356335800562375414682302225574575986758110344390320579016262971047541230279404739073381498973947165860"
        "3578472362222309578053201730193461542451536469872504771607677446358370401005123751302140752264459223847275"
        "6549193009381041298482857683510896839974809697173856384189111987597175653527171122842099226590837993284165"
        "4721520710276367262412987077644884404730173074376417252301439513029190919723987462930381298065185546875"},
    {0x1.fffffffffffffp-1004, chars_format::fixed, 1056,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011665795231290234"
        "6922126726749703738539891467031761159351090602821540759313898638961097023103074832897233343422819971687087"
        "8703878770311637514404341957431458132593207222501778301519686421384460841261717846727826808425182512969041"
        "9718432519644857837276241057459270068466215616765173522010377293348893709597382741987282502370003846119624"
        "9734712671601124750829364604451149151973516220688780641158032525942095082460558809478146762997947894331720"
        "7156944724444619156106403460386923084903072939745009543215354892716740802010247502604281504528918447694551"
        "3098386018762082596965715367021793679949619394347712768378223975194351307054342245684198453181675986568330"
        "944304142055273452482597415528976880946034614875283450460287902605838183944797492586076259613037109375"},
    {0x1.fffffffffffffp-1003, chars_format::fixed, 1055,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000023331590462580469"
        "3844253453499407477079782934063522318702181205643081518627797277922194046206149665794466686845639943374175"
        "7407757540623275028808683914862916265186414445003556603039372842768921682523435693455653616850365025938083"
        "9436865039289715674552482114918540136932431233530347044020754586697787419194765483974565004740007692239249"
        "9469425343202249501658729208902298303947032441377561282316065051884190164921117618956293525995895788663441"
        "4313889448889238312212806920773846169806145879490019086430709785433481604020495005208563009057836895389102"
        "6196772037524165193931430734043587359899238788695425536756447950388702614108684491368396906363351973136661"
        "88860828411054690496519483105795376189206922975056690092057580521167636788959498517215251922607421875"},
    {0x1.fffffffffffffp-1002, chars_format::fixed, 1054,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000046663180925160938"
        "7688506906998814954159565868127044637404362411286163037255594555844388092412299331588933373691279886748351"
        "4815515081246550057617367829725832530372828890007113206078745685537843365046871386911307233700730051876167"
        "8873730078579431349104964229837080273864862467060694088041509173395574838389530967949130009480015384478499"
        "8938850686404499003317458417804596607894064882755122564632130103768380329842235237912587051991791577326882"
        "8627778897778476624425613841547692339612291758980038172861419570866963208040990010417126018115673790778205"
        "2393544075048330387862861468087174719798477577390851073512895900777405228217368982736793812726703946273323"
        "7772165682210938099303896621159075237841384595011338018411516104233527357791899703443050384521484375"},
    {0x1.fffffffffffffp-1001, chars_format::fixed, 1053,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000093326361850321877"
        "5377013813997629908319131736254089274808724822572326074511189111688776184824598663177866747382559773496702"
        "9631030162493100115234735659451665060745657780014226412157491371075686730093742773822614467401460103752335"
        "7747460157158862698209928459674160547729724934121388176083018346791149676779061935898260018960030768956999"
        "7877701372808998006634916835609193215788129765510245129264260207536760659684470475825174103983583154653765"
        "7255557795556953248851227683095384679224583517960076345722839141733926416081980020834252036231347581556410"
        "4787088150096660775725722936174349439596955154781702147025791801554810456434737965473587625453407892546647"
        "554433136442187619860779324231815047568276919002267603682303220846705471558379940688610076904296875"},
    {0x1.fffffffffffffp-1000, chars_format::fixed, 1052,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000186652723700643755"
        "0754027627995259816638263472508178549617449645144652149022378223377552369649197326355733494765119546993405"
        "9262060324986200230469471318903330121491315560028452824314982742151373460187485547645228934802920207504671"
        "5494920314317725396419856919348321095459449868242776352166036693582299353558123871796520037920061537913999"
        "5755402745617996013269833671218386431576259531020490258528520415073521319368940951650348207967166309307531"
        "4511115591113906497702455366190769358449167035920152691445678283467852832163960041668504072462695163112820"
        "9574176300193321551451445872348698879193910309563404294051583603109620912869475930947175250906815785093295"
        "10886627288437523972155864846363009513655383800453520736460644169341094311675988137722015380859375"},
    {0x1.fffffffffffffp-999, chars_format::fixed, 1051,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000373305447401287510"
        "1508055255990519633276526945016357099234899290289304298044756446755104739298394652711466989530239093986811"
        "8524120649972400460938942637806660242982631120056905648629965484302746920374971095290457869605840415009343"
        "0989840628635450792839713838696642190918899736485552704332073387164598707116247743593040075840123075827999"
        "1510805491235992026539667342436772863152519062040980517057040830147042638737881903300696415934332618615062"
        "9022231182227812995404910732381538716898334071840305382891356566935705664327920083337008144925390326225641"
        "9148352600386643102902891744697397758387820619126808588103167206219241825738951861894350501813631570186590"
        "2177325457687504794431172969272601902731076760090704147292128833868218862335197627544403076171875"},
    {0x1.fffffffffffffp-998, chars_format::fixed, 1050,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000746610894802575020"
        "3016110511981039266553053890032714198469798580578608596089512893510209478596789305422933979060478187973623"
        "7048241299944800921877885275613320485965262240113811297259930968605493840749942190580915739211680830018686"
        "1979681257270901585679427677393284381837799472971105408664146774329197414232495487186080151680246151655998"
        "3021610982471984053079334684873545726305038124081961034114081660294085277475763806601392831868665237230125"
        "8044462364455625990809821464763077433796668143680610765782713133871411328655840166674016289850780652451283"
        "8296705200773286205805783489394795516775641238253617176206334412438483651477903723788701003627263140373180"
        "435465091537500958886234593854520380546215352018140829458425766773643772467039525508880615234375"},
    {0x1.fffffffffffffp-997, chars_format::fixed, 1049,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000001493221789605150040"
        "6032221023962078533106107780065428396939597161157217192179025787020418957193578610845867958120956375947247"
        "4096482599889601843755770551226640971930524480227622594519861937210987681499884381161831478423361660037372"
        "3959362514541803171358855354786568763675598945942210817328293548658394828464990974372160303360492303311996"
        "6043221964943968106158669369747091452610076248163922068228163320588170554951527613202785663737330474460251"
        "6088924728911251981619642929526154867593336287361221531565426267742822657311680333348032579701561304902567"
        "6593410401546572411611566978789591033551282476507234352412668824876967302955807447577402007254526280746360"
        "87093018307500191777246918770904076109243070403628165891685153354728754493407905101776123046875"},
    {0x1.fffffffffffffp-996, chars_format::fixed, 1048,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000002986443579210300081"
        "2064442047924157066212215560130856793879194322314434384358051574040837914387157221691735916241912751894494"
        "8192965199779203687511541102453281943861048960455245189039723874421975362999768762323662956846723320074744"
        "7918725029083606342717710709573137527351197891884421634656587097316789656929981948744320606720984606623993"
        "2086443929887936212317338739494182905220152496327844136456326641176341109903055226405571327474660948920503"
        "2177849457822503963239285859052309735186672574722443063130852535485645314623360666696065159403122609805135"
        "3186820803093144823223133957579182067102564953014468704825337649753934605911614895154804014509052561492721"
        "7418603661500038355449383754180815221848614080725633178337030670945750898681581020355224609375"},
    {0x1.fffffffffffffp-995, chars_format::fixed, 1047,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000005972887158420600162"
        "4128884095848314132424431120261713587758388644628868768716103148081675828774314443383471832483825503788989"
        "6385930399558407375023082204906563887722097920910490378079447748843950725999537524647325913693446640149489"
        "5837450058167212685435421419146275054702395783768843269313174194633579313859963897488641213441969213247986"
        "4172887859775872424634677478988365810440304992655688272912653282352682219806110452811142654949321897841006"
        "4355698915645007926478571718104619470373345149444886126261705070971290629246721333392130318806245219610270"
        "6373641606186289646446267915158364134205129906028937409650675299507869211823229790309608029018105122985443"
        "483720732300007671089876750836163044369722816145126635667406134189150179736316204071044921875"},
    {0x1.fffffffffffffp-994, chars_format::fixed, 1046,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000011945774316841200324"
        "8257768191696628264848862240523427175516777289257737537432206296163351657548628886766943664967651007577979"
        "2771860799116814750046164409813127775444195841820980756158895497687901451999075049294651827386893280298979"
        "1674900116334425370870842838292550109404791567537686538626348389267158627719927794977282426883938426495972"
        "8345775719551744849269354957976731620880609985311376545825306564705364439612220905622285309898643795682012"
        "8711397831290015852957143436209238940746690298889772252523410141942581258493442666784260637612490439220541"
        "2747283212372579292892535830316728268410259812057874819301350599015738423646459580619216058036210245970886"
        "96744146460001534217975350167232608873944563229025327133481226837830035947263240814208984375"},
    {0x1.fffffffffffffp-993, chars_format::fixed, 1045,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000023891548633682400649"
        "6515536383393256529697724481046854351033554578515475074864412592326703315097257773533887329935302015155958"
        "5543721598233629500092328819626255550888391683641961512317790995375802903998150098589303654773786560597958"
        "3349800232668850741741685676585100218809583135075373077252696778534317255439855589954564853767876852991945"
        "6691551439103489698538709915953463241761219970622753091650613129410728879224441811244570619797287591364025"
        "7422795662580031705914286872418477881493380597779544505046820283885162516986885333568521275224980878441082"
        "5494566424745158585785071660633456536820519624115749638602701198031476847292919161238432116072420491941773"
        "9348829292000306843595070033446521774788912645805065426696245367566007189452648162841796875"},
    {0x1.fffffffffffffp-992, chars_format::fixed, 1044,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000047783097267364801299"
        "3031072766786513059395448962093708702067109157030950149728825184653406630194515547067774659870604030311917"
        "1087443196467259000184657639252511101776783367283923024635581990751605807996300197178607309547573121195916"
        "6699600465337701483483371353170200437619166270150746154505393557068634510879711179909129707535753705983891"
        "3383102878206979397077419831906926483522439941245506183301226258821457758448883622489141239594575182728051"
        "4845591325160063411828573744836955762986761195559089010093640567770325033973770667137042550449961756882165"
        "0989132849490317171570143321266913073641039248231499277205402396062953694585838322476864232144840983883547"
        "869765858400061368719014006689304354957782529161013085339249073513201437890529632568359375"},
    {0x1.fffffffffffffp-991, chars_format::fixed, 1043,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000095566194534729602598"
        "6062145533573026118790897924187417404134218314061900299457650369306813260389031094135549319741208060623834"
        "2174886392934518000369315278505022203553566734567846049271163981503211615992600394357214619095146242391833"
        "3399200930675402966966742706340400875238332540301492309010787114137269021759422359818259415071507411967782"
        "6766205756413958794154839663813852967044879882491012366602452517642915516897767244978282479189150365456102"
        "9691182650320126823657147489673911525973522391118178020187281135540650067947541334274085100899923513764330"
        "1978265698980634343140286642533826147282078496462998554410804792125907389171676644953728464289681967767095"
        "73953171680012273743802801337860870991556505832202617067849814702640287578105926513671875"},
    {0x1.fffffffffffffp-990, chars_format::fixed, 1042,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000191132389069459205197"
        "2124291067146052237581795848374834808268436628123800598915300738613626520778062188271098639482416121247668"
        "4349772785869036000738630557010044407107133469135692098542327963006423231985200788714429238190292484783666"
        "6798401861350805933933485412680801750476665080602984618021574228274538043518844719636518830143014823935565"
        "3532411512827917588309679327627705934089759764982024733204905035285831033795534489956564958378300730912205"
        "9382365300640253647314294979347823051947044782236356040374562271081300135895082668548170201799847027528660"
        "3956531397961268686280573285067652294564156992925997108821609584251814778343353289907456928579363935534191"
        "4790634336002454748760560267572174198311301166440523413569962940528057515621185302734375"},
    {0x1.fffffffffffffp-989, chars_format::fixed, 1041,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000382264778138918410394"
        "4248582134292104475163591696749669616536873256247601197830601477227253041556124376542197278964832242495336"
        "8699545571738072001477261114020088814214266938271384197084655926012846463970401577428858476380584969567333"
        "3596803722701611867866970825361603500953330161205969236043148456549076087037689439273037660286029647871130"
        "7064823025655835176619358655255411868179519529964049466409810070571662067591068979913129916756601461824411"
        "8764730601280507294628589958695646103894089564472712080749124542162600271790165337096340403599694055057320"
        "7913062795922537372561146570135304589128313985851994217643219168503629556686706579814913857158727871068382"
        "958126867200490949752112053514434839662260233288104682713992588105611503124237060546875"},
    {0x1.fffffffffffffp-988, chars_format::fixed, 1040,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000764529556277836820788"
        "8497164268584208950327183393499339233073746512495202395661202954454506083112248753084394557929664484990673"
        "7399091143476144002954522228040177628428533876542768394169311852025692927940803154857716952761169939134666"
        "7193607445403223735733941650723207001906660322411938472086296913098152174075378878546075320572059295742261"
        "4129646051311670353238717310510823736359039059928098932819620141143324135182137959826259833513202923648823"
        "7529461202561014589257179917391292207788179128945424161498249084325200543580330674192680807199388110114641"
        "5826125591845074745122293140270609178256627971703988435286438337007259113373413159629827714317455742136765"
        "91625373440098189950422410702886967932452046657620936542798517621122300624847412109375"},
    {0x1.fffffffffffffp-987, chars_format::fixed, 1039,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000001529059112555673641577"
        "6994328537168417900654366786998678466147493024990404791322405908909012166224497506168789115859328969981347"
        "4798182286952288005909044456080355256857067753085536788338623704051385855881606309715433905522339878269333"
        "4387214890806447471467883301446414003813320644823876944172593826196304348150757757092150641144118591484522"
        "8259292102623340706477434621021647472718078119856197865639240282286648270364275919652519667026405847297647"
        "5058922405122029178514359834782584415576358257890848322996498168650401087160661348385361614398776220229283"
        "1652251183690149490244586280541218356513255943407976870572876674014518226746826319259655428634911484273531"
        "8325074688019637990084482140577393586490409331524187308559703524224460124969482421875"},
    {0x1.fffffffffffffp-986, chars_format::fixed, 1038,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000003058118225111347283155"
        "3988657074336835801308733573997356932294986049980809582644811817818024332448995012337578231718657939962694"
        "9596364573904576011818088912160710513714135506171073576677247408102771711763212619430867811044679756538666"
        "8774429781612894942935766602892828007626641289647753888345187652392608696301515514184301282288237182969045"
        "6518584205246681412954869242043294945436156239712395731278480564573296540728551839305039334052811694595295"
        "0117844810244058357028719669565168831152716515781696645992996337300802174321322696770723228797552440458566"
        "3304502367380298980489172561082436713026511886815953741145753348029036453493652638519310857269822968547063"
        "665014937603927598016896428115478717298081866304837461711940704844892024993896484375"},
    {0x1.fffffffffffffp-985, chars_format::fixed, 1037,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000006116236450222694566310"
        "7977314148673671602617467147994713864589972099961619165289623635636048664897990024675156463437315879925389"
        "9192729147809152023636177824321421027428271012342147153354494816205543423526425238861735622089359513077333"
        "7548859563225789885871533205785656015253282579295507776690375304785217392603031028368602564576474365938091"
        "3037168410493362825909738484086589890872312479424791462556961129146593081457103678610078668105623389190590"
        "0235689620488116714057439339130337662305433031563393291985992674601604348642645393541446457595104880917132"
        "6609004734760597960978345122164873426053023773631907482291506696058072906987305277038621714539645937094127"
        "33002987520785519603379285623095743459616373260967492342388140968978404998779296875"},
    {0x1.fffffffffffffp-984, chars_format::fixed, 1036,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000012232472900445389132621"
        "5954628297347343205234934295989427729179944199923238330579247271272097329795980049350312926874631759850779"
        "8385458295618304047272355648642842054856542024684294306708989632411086847052850477723471244178719026154667"
        "5097719126451579771743066411571312030506565158591015553380750609570434785206062056737205129152948731876182"
        "6074336820986725651819476968173179781744624958849582925113922258293186162914207357220157336211246778381180"
        "0471379240976233428114878678260675324610866063126786583971985349203208697285290787082892915190209761834265"
        "3218009469521195921956690244329746852106047547263814964583013392116145813974610554077243429079291874188254"
        "6600597504157103920675857124619148691923274652193498468477628193795680999755859375"},
    {0x1.fffffffffffffp-983, chars_format::fixed, 1035,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000024464945800890778265243"
        "1909256594694686410469868591978855458359888399846476661158494542544194659591960098700625853749263519701559"
        "6770916591236608094544711297285684109713084049368588613417979264822173694105700955446942488357438052309335"
        "0195438252903159543486132823142624061013130317182031106761501219140869570412124113474410258305897463752365"
        "2148673641973451303638953936346359563489249917699165850227844516586372325828414714440314672422493556762360"
        "0942758481952466856229757356521350649221732126253573167943970698406417394570581574165785830380419523668530"
        "6436018939042391843913380488659493704212095094527629929166026784232291627949221108154486858158583748376509"
        "320119500831420784135171424923829738384654930438699693695525638759136199951171875"},
    {0x1.fffffffffffffp-982, chars_format::fixed, 1034,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000048929891601781556530486"
        "3818513189389372820939737183957710916719776799692953322316989085088389319183920197401251707498527039403119"
        "3541833182473216189089422594571368219426168098737177226835958529644347388211401910893884976714876104618670"
        "0390876505806319086972265646285248122026260634364062213523002438281739140824248226948820516611794927504730"
        "4297347283946902607277907872692719126978499835398331700455689033172744651656829428880629344844987113524720"
        "1885516963904933712459514713042701298443464252507146335887941396812834789141163148331571660760839047337061"
        "2872037878084783687826760977318987408424190189055259858332053568464583255898442216308973716317167496753018"
        "64023900166284156827034284984765947676930986087739938739105127751827239990234375"},
    {0x1.fffffffffffffp-981, chars_format::fixed, 1033,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000097859783203563113060972"
        "7637026378778745641879474367915421833439553599385906644633978170176778638367840394802503414997054078806238"
        "7083666364946432378178845189142736438852336197474354453671917059288694776422803821787769953429752209237340"
        "0781753011612638173944531292570496244052521268728124427046004876563478281648496453897641033223589855009460"
        "8594694567893805214555815745385438253956999670796663400911378066345489303313658857761258689689974227049440"
        "3771033927809867424919029426085402596886928505014292671775882793625669578282326296663143321521678094674122"
        "5744075756169567375653521954637974816848380378110519716664107136929166511796884432617947432634334993506037"
        "2804780033256831365406856996953189535386197217547987747821025550365447998046875"},
    {0x1.fffffffffffffp-980, chars_format::fixed, 1032,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000195719566407126226121945"
        "5274052757557491283758948735830843666879107198771813289267956340353557276735680789605006829994108157612477"
        "4167332729892864756357690378285472877704672394948708907343834118577389552845607643575539906859504418474680"
        "1563506023225276347889062585140992488105042537456248854092009753126956563296992907795282066447179710018921"
        "7189389135787610429111631490770876507913999341593326801822756132690978606627317715522517379379948454098880"
        "7542067855619734849838058852170805193773857010028585343551765587251339156564652593326286643043356189348245"
        "1488151512339134751307043909275949633696760756221039433328214273858333023593768865235894865268669987012074"
        "560956006651366273081371399390637907077239443509597549564205110073089599609375"},
    {0x1.fffffffffffffp-979, chars_format::fixed, 1031,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000391439132814252452243891"
        "0548105515114982567517897471661687333758214397543626578535912680707114553471361579210013659988216315224954"
        "8334665459785729512715380756570945755409344789897417814687668237154779105691215287151079813719008836949360"
        "3127012046450552695778125170281984976210085074912497708184019506253913126593985815590564132894359420037843"
        "4378778271575220858223262981541753015827998683186653603645512265381957213254635431045034758759896908197761"
        "5084135711239469699676117704341610387547714020057170687103531174502678313129305186652573286086712378696490"
        "2976303024678269502614087818551899267393521512442078866656428547716666047187537730471789730537339974024149"
        "12191201330273254616274279878127581415447888701919509912841022014617919921875"},
    {0x1.fffffffffffffp-978, chars_format::fixed, 1030,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000782878265628504904487782"
        "1096211030229965135035794943323374667516428795087253157071825361414229106942723158420027319976432630449909"
        "6669330919571459025430761513141891510818689579794835629375336474309558211382430574302159627438017673898720"
        "6254024092901105391556250340563969952420170149824995416368039012507826253187971631181128265788718840075686"
        "8757556543150441716446525963083506031655997366373307207291024530763914426509270862090069517519793816395523"
        "0168271422478939399352235408683220775095428040114341374207062349005356626258610373305146572173424757392980"
        "5952606049356539005228175637103798534787043024884157733312857095433332094375075460943579461074679948048298"
        "2438240266054650923254855975625516283089577740383901982568204402923583984375"},
    {0x1.fffffffffffffp-977, chars_format::fixed, 1029,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000001565756531257009808975564"
        "2192422060459930270071589886646749335032857590174506314143650722828458213885446316840054639952865260899819"
        "3338661839142918050861523026283783021637379159589671258750672948619116422764861148604319254876035347797441"
        "2508048185802210783112500681127939904840340299649990832736078025015652506375943262362256531577437680151373"
        "7515113086300883432893051926167012063311994732746614414582049061527828853018541724180139035039587632791046"
        "0336542844957878798704470817366441550190856080228682748414124698010713252517220746610293144346849514785961"
        "1905212098713078010456351274207597069574086049768315466625714190866664188750150921887158922149359896096596"
        "487648053210930184650971195125103256617915548076780396513640880584716796875"},
    {0x1.fffffffffffffp-976, chars_format::fixed, 1028,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000003131513062514019617951128"
        "4384844120919860540143179773293498670065715180349012628287301445656916427770892633680109279905730521799638"
        "6677323678285836101723046052567566043274758319179342517501345897238232845529722297208638509752070695594882"
        "5016096371604421566225001362255879809680680599299981665472156050031305012751886524724513063154875360302747"
        "5030226172601766865786103852334024126623989465493228829164098123055657706037083448360278070079175265582092"
        "0673085689915757597408941634732883100381712160457365496828249396021426505034441493220586288693699029571922"
        "3810424197426156020912702548415194139148172099536630933251428381733328377500301843774317844298719792193192"
        "97529610642186036930194239025020651323583109615356079302728176116943359375"},
    {0x1.fffffffffffffp-975, chars_format::fixed, 1027,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000006263026125028039235902256"
        "8769688241839721080286359546586997340131430360698025256574602891313832855541785267360218559811461043599277"
        "3354647356571672203446092105135132086549516638358685035002691794476465691059444594417277019504141391189765"
        "0032192743208843132450002724511759619361361198599963330944312100062610025503773049449026126309750720605495"
        "0060452345203533731572207704668048253247978930986457658328196246111315412074166896720556140158350531164184"
        "1346171379831515194817883269465766200763424320914730993656498792042853010068882986441172577387398059143844"
        "7620848394852312041825405096830388278296344199073261866502856763466656755000603687548635688597439584386385"
        "9505922128437207386038847805004130264716621923071215860545635223388671875"},
    {0x1.fffffffffffffp-974, chars_format::fixed, 1026,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000012526052250056078471804513"
        "7539376483679442160572719093173994680262860721396050513149205782627665711083570534720437119622922087198554"
        "6709294713143344406892184210270264173099033276717370070005383588952931382118889188834554039008282782379530"
        "0064385486417686264900005449023519238722722397199926661888624200125220051007546098898052252619501441210990"
        "0120904690407067463144415409336096506495957861972915316656392492222630824148333793441112280316701062328368"
        "2692342759663030389635766538931532401526848641829461987312997584085706020137765972882345154774796118287689"
        "5241696789704624083650810193660776556592688398146523733005713526933313510001207375097271377194879168772771"
        "901184425687441477207769561000826052943324384614243172109127044677734375"},
    {0x1.fffffffffffffp-973, chars_format::fixed, 1025,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000025052104500112156943609027"
        "5078752967358884321145438186347989360525721442792101026298411565255331422167141069440874239245844174397109"
        "3418589426286688813784368420540528346198066553434740140010767177905862764237778377669108078016565564759060"
        "0128770972835372529800010898047038477445444794399853323777248400250440102015092197796104505239002882421980"
        "0241809380814134926288830818672193012991915723945830633312784984445261648296667586882224560633402124656736"
        "5384685519326060779271533077863064803053697283658923974625995168171412040275531945764690309549592236575379"
        "0483393579409248167301620387321553113185376796293047466011427053866627020002414750194542754389758337545543"
        "80236885137488295441553912200165210588664876922848634421825408935546875"},
    {0x1.fffffffffffffp-972, chars_format::fixed, 1024,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000050104209000224313887218055"
        "0157505934717768642290876372695978721051442885584202052596823130510662844334282138881748478491688348794218"
        "6837178852573377627568736841081056692396133106869480280021534355811725528475556755338216156033131129518120"
        "0257541945670745059600021796094076954890889588799706647554496800500880204030184395592209010478005764843960"
        "0483618761628269852577661637344386025983831447891661266625569968890523296593335173764449121266804249313473"
        "0769371038652121558543066155726129606107394567317847949251990336342824080551063891529380619099184473150758"
        "0966787158818496334603240774643106226370753592586094932022854107733254040004829500389085508779516675091087"
        "6047377027497659088310782440033042117732975384569726884365081787109375"},
    {0x1.fffffffffffffp-971, chars_format::fixed, 1023,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000100208418000448627774436110"
        "0315011869435537284581752745391957442102885771168404105193646261021325688668564277763496956983376697588437"
        "3674357705146755255137473682162113384792266213738960560043068711623451056951113510676432312066262259036240"
        "0515083891341490119200043592188153909781779177599413295108993601001760408060368791184418020956011529687920"
        "0967237523256539705155323274688772051967662895783322533251139937781046593186670347528898242533608498626946"
        "1538742077304243117086132311452259212214789134635695898503980672685648161102127783058761238198368946301516"
        "1933574317636992669206481549286212452741507185172189864045708215466508080009659000778171017559033350182175"
        "209475405499531817662156488006608423546595076913945376873016357421875"},
    {0x1.fffffffffffffp-970, chars_format::fixed, 1022,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000200416836000897255548872220"
        "0630023738871074569163505490783914884205771542336808210387292522042651377337128555526993913966753395176874"
        "7348715410293510510274947364324226769584532427477921120086137423246902113902227021352864624132524518072480"
        "1030167782682980238400087184376307819563558355198826590217987202003520816120737582368836041912023059375840"
        "1934475046513079410310646549377544103935325791566645066502279875562093186373340695057796485067216997253892"
        "3077484154608486234172264622904518424429578269271391797007961345371296322204255566117522476396737892603032"
        "3867148635273985338412963098572424905483014370344379728091416430933016160019318001556342035118066700364350"
        "41895081099906363532431297601321684709319015382789075374603271484375"},
    {0x1.fffffffffffffp-969, chars_format::fixed, 1021,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000400833672001794511097744440"
        "1260047477742149138327010981567829768411543084673616420774585044085302754674257111053987827933506790353749"
        "4697430820587021020549894728648453539169064854955842240172274846493804227804454042705729248265049036144960"
        "2060335565365960476800174368752615639127116710397653180435974404007041632241475164737672083824046118751680"
        "3868950093026158820621293098755088207870651583133290133004559751124186372746681390115592970134433994507784"
        "6154968309216972468344529245809036848859156538542783594015922690742592644408511132235044952793475785206064"
        "7734297270547970676825926197144849810966028740688759456182832861866032320038636003112684070236133400728700"
        "8379016219981272706486259520264336941863803076557815074920654296875"},
    {0x1.fffffffffffffp-968, chars_format::fixed, 1020,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000801667344003589022195488880"
        "2520094955484298276654021963135659536823086169347232841549170088170605509348514222107975655867013580707498"
        "9394861641174042041099789457296907078338129709911684480344549692987608455608908085411458496530098072289920"
        "4120671130731920953600348737505231278254233420795306360871948808014083264482950329475344167648092237503360"
        "7737900186052317641242586197510176415741303166266580266009119502248372745493362780231185940268867989015569"
        "2309936618433944936689058491618073697718313077085567188031845381485185288817022264470089905586951570412129"
        "5468594541095941353651852394289699621932057481377518912365665723732064640077272006225368140472266801457401"
        "675803243996254541297251904052867388372760615311563014984130859375"},
    {0x1.fffffffffffffp-967, chars_format::fixed, 1019,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000001603334688007178044390977760"
        "5040189910968596553308043926271319073646172338694465683098340176341211018697028444215951311734027161414997"
        "8789723282348084082199578914593814156676259419823368960689099385975216911217816170822916993060196144579840"
        "8241342261463841907200697475010462556508466841590612721743897616028166528965900658950688335296184475006721"
        "5475800372104635282485172395020352831482606332533160532018239004496745490986725560462371880537735978031138"
        "4619873236867889873378116983236147395436626154171134376063690762970370577634044528940179811173903140824259"
        "0937189082191882707303704788579399243864114962755037824731331447464129280154544012450736280944533602914803"
        "35160648799250908259450380810573477674552123062312602996826171875"},
    {0x1.fffffffffffffp-966, chars_format::fixed, 1018,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000003206669376014356088781955521"
        "0080379821937193106616087852542638147292344677388931366196680352682422037394056888431902623468054322829995"
        "7579446564696168164399157829187628313352518839646737921378198771950433822435632341645833986120392289159681"
        "6482684522927683814401394950020925113016933683181225443487795232056333057931801317901376670592368950013443"
        "0951600744209270564970344790040705662965212665066321064036478008993490981973451120924743761075471956062276"
        "9239746473735779746756233966472294790873252308342268752127381525940741155268089057880359622347806281648518"
        "1874378164383765414607409577158798487728229925510075649462662894928258560309088024901472561889067205829606"
        "7032129759850181651890076162114695534910424612462520599365234375"},
    {0x1.fffffffffffffp-965, chars_format::fixed, 1017,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000006413338752028712177563911042"
        "0160759643874386213232175705085276294584689354777862732393360705364844074788113776863805246936108645659991"
        "5158893129392336328798315658375256626705037679293475842756397543900867644871264683291667972240784578319363"
        "2965369045855367628802789900041850226033867366362450886975590464112666115863602635802753341184737900026886"
        "1903201488418541129940689580081411325930425330132642128072956017986981963946902241849487522150943912124553"
        "8479492947471559493512467932944589581746504616684537504254763051881482310536178115760719244695612563297036"
        "3748756328767530829214819154317596975456459851020151298925325789856517120618176049802945123778134411659213"
        "406425951970036330378015232422939106982084922492504119873046875"},
    {0x1.fffffffffffffp-964, chars_format::fixed, 1016,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000012826677504057424355127822084"
        "0321519287748772426464351410170552589169378709555725464786721410729688149576227553727610493872217291319983"
        "0317786258784672657596631316750513253410075358586951685512795087801735289742529366583335944481569156638726"
        "5930738091710735257605579800083700452067734732724901773951180928225332231727205271605506682369475800053772"
        "3806402976837082259881379160162822651860850660265284256145912035973963927893804483698975044301887824249107"
        "6958985894943118987024935865889179163493009233369075008509526103762964621072356231521438489391225126594072"
        "7497512657535061658429638308635193950912919702040302597850651579713034241236352099605890247556268823318426"
        "81285190394007266075603046484587821396416984498500823974609375"},
    {0x1.fffffffffffffp-963, chars_format::fixed, 1015,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000025653355008114848710255644168"
        "0643038575497544852928702820341105178338757419111450929573442821459376299152455107455220987744434582639966"
        "0635572517569345315193262633501026506820150717173903371025590175603470579485058733166671888963138313277453"
        "1861476183421470515211159600167400904135469465449803547902361856450664463454410543211013364738951600107544"
        "7612805953674164519762758320325645303721701320530568512291824071947927855787608967397950088603775648498215"
        "3917971789886237974049871731778358326986018466738150017019052207525929242144712463042876978782450253188145"
        "4995025315070123316859276617270387901825839404080605195701303159426068482472704199211780495112537646636853"
        "6257038078801453215120609296917564279283396899700164794921875"},
    {0x1.fffffffffffffp-962, chars_format::fixed, 1014,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000051306710016229697420511288336"
        "1286077150995089705857405640682210356677514838222901859146885642918752598304910214910441975488869165279932"
        "1271145035138690630386525267002053013640301434347806742051180351206941158970117466333343777926276626554906"
        "3722952366842941030422319200334801808270938930899607095804723712901328926908821086422026729477903200215089"
        "5225611907348329039525516640651290607443402641061137024583648143895855711575217934795900177207551296996430"
        "7835943579772475948099743463556716653972036933476300034038104415051858484289424926085753957564900506376290"
        "9990050630140246633718553234540775803651678808161210391402606318852136964945408398423560990225075293273707"
        "251407615760290643024121859383512855856679379940032958984375"},
    {0x1.fffffffffffffp-961, chars_format::fixed, 1013,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000102613420032459394841022576672"
        "2572154301990179411714811281364420713355029676445803718293771285837505196609820429820883950977738330559864"
        "2542290070277381260773050534004106027280602868695613484102360702413882317940234932666687555852553253109812"
        "7445904733685882060844638400669603616541877861799214191609447425802657853817642172844053458955806400430179"
        "0451223814696658079051033281302581214886805282122274049167296287791711423150435869591800354415102593992861"
        "5671887159544951896199486927113433307944073866952600068076208830103716968578849852171507915129801012752581"
        "9980101260280493267437106469081551607303357616322420782805212637704273929890816796847121980450150586547414"
        "50281523152058128604824371876702571171335875988006591796875"},
    {0x1.fffffffffffffp-960, chars_format::fixed, 1012,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000205226840064918789682045153344"
        "5144308603980358823429622562728841426710059352891607436587542571675010393219640859641767901955476661119728"
        "5084580140554762521546101068008212054561205737391226968204721404827764635880469865333375111705106506219625"
        "4891809467371764121689276801339207233083755723598428383218894851605315707635284345688106917911612800860358"
        "0902447629393316158102066562605162429773610564244548098334592575583422846300871739183600708830205187985723"
        "1343774319089903792398973854226866615888147733905200136152417660207433937157699704343015830259602025505163"
        "9960202520560986534874212938163103214606715232644841565610425275408547859781633593694243960900301173094829"
        "0056304630411625720964874375340514234267175197601318359375"},
    {0x1.fffffffffffffp-959, chars_format::fixed, 1011,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000410453680129837579364090306689"
        "0288617207960717646859245125457682853420118705783214873175085143350020786439281719283535803910953322239457"
        "0169160281109525043092202136016424109122411474782453936409442809655529271760939730666750223410213012439250"
        "9783618934743528243378553602678414466167511447196856766437789703210631415270568691376213835823225601720716"
        "1804895258786632316204133125210324859547221128489096196669185151166845692601743478367201417660410375971446"
        "2687548638179807584797947708453733231776295467810400272304835320414867874315399408686031660519204051010327"
        "9920405041121973069748425876326206429213430465289683131220850550817095719563267187388487921800602346189658"
        "011260926082325144192974875068102846853435039520263671875"},
    {0x1.fffffffffffffp-958, chars_format::fixed, 1010,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000820907360259675158728180613378"
        "0577234415921435293718490250915365706840237411566429746350170286700041572878563438567071607821906644478914"
        "0338320562219050086184404272032848218244822949564907872818885619311058543521879461333500446820426024878501"
        "9567237869487056486757107205356828932335022894393713532875579406421262830541137382752427671646451203441432"
        "3609790517573264632408266250420649719094442256978192393338370302333691385203486956734402835320820751942892"
        "5375097276359615169595895416907466463552590935620800544609670640829735748630798817372063321038408102020655"
        "9840810082243946139496851752652412858426860930579366262441701101634191439126534374776975843601204692379316"
        "02252185216465028838594975013620569370687007904052734375"},
    {0x1.fffffffffffffp-957, chars_format::fixed, 1009,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000001641814720519350317456361226756"
        "1154468831842870587436980501830731413680474823132859492700340573400083145757126877134143215643813288957828"
        "0676641124438100172368808544065696436489645899129815745637771238622117087043758922667000893640852049757003"
        "9134475738974112973514214410713657864670045788787427065751158812842525661082274765504855343292902406882864"
        "7219581035146529264816532500841299438188884513956384786676740604667382770406973913468805670641641503885785"
        "0750194552719230339191790833814932927105181871241601089219341281659471497261597634744126642076816204041311"
        "9681620164487892278993703505304825716853721861158732524883402203268382878253068749553951687202409384758632"
        "0450437043293005767718995002724113874137401580810546875"},
    {0x1.fffffffffffffp-956, chars_format::fixed, 1008,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000003283629441038700634912722453512"
        "2308937663685741174873961003661462827360949646265718985400681146800166291514253754268286431287626577915656"
        "1353282248876200344737617088131392872979291798259631491275542477244234174087517845334001787281704099514007"
        "8268951477948225947028428821427315729340091577574854131502317625685051322164549531009710686585804813765729"
        "4439162070293058529633065001682598876377769027912769573353481209334765540813947826937611341283283007771570"
        "1500389105438460678383581667629865854210363742483202178438682563318942994523195269488253284153632408082623"
        "9363240328975784557987407010609651433707443722317465049766804406536765756506137499107903374404818769517264"
        "090087408658601153543799000544822774827480316162109375"},
    {0x1.fffffffffffffp-955, chars_format::fixed, 1007,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000006567258882077401269825444907024"
        "4617875327371482349747922007322925654721899292531437970801362293600332583028507508536572862575253155831312"
        "2706564497752400689475234176262785745958583596519262982551084954488468348175035690668003574563408199028015"
        "6537902955896451894056857642854631458680183155149708263004635251370102644329099062019421373171609627531458"
        "8878324140586117059266130003365197752755538055825539146706962418669531081627895653875222682566566015543140"
        "3000778210876921356767163335259731708420727484966404356877365126637885989046390538976506568307264816165247"
        "8726480657951569115974814021219302867414887444634930099533608813073531513012274998215806748809637539034528"
        "18017481731720230708759800108964554965496063232421875"},
    {0x1.fffffffffffffp-954, chars_format::fixed, 1006,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000013134517764154802539650889814048"
        "9235750654742964699495844014645851309443798585062875941602724587200665166057015017073145725150506311662624"
        "5413128995504801378950468352525571491917167193038525965102169908976936696350071381336007149126816398056031"
        "3075805911792903788113715285709262917360366310299416526009270502740205288658198124038842746343219255062917"
        "7756648281172234118532260006730395505511076111651078293413924837339062163255791307750445365133132031086280"
        "6001556421753842713534326670519463416841454969932808713754730253275771978092781077953013136614529632330495"
        "7452961315903138231949628042438605734829774889269860199067217626147063026024549996431613497619275078069056"
        "3603496346344046141751960021792910993099212646484375"},
    {0x1.fffffffffffffp-953, chars_format::fixed, 1005,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000026269035528309605079301779628097"
        "8471501309485929398991688029291702618887597170125751883205449174401330332114030034146291450301012623325249"
        "0826257991009602757900936705051142983834334386077051930204339817953873392700142762672014298253632796112062"
        "6151611823585807576227430571418525834720732620598833052018541005480410577316396248077685492686438510125835"
        "5513296562344468237064520013460791011022152223302156586827849674678124326511582615500890730266264062172561"
        "2003112843507685427068653341038926833682909939865617427509460506551543956185562155906026273229059264660991"
        "4905922631806276463899256084877211469659549778539720398134435252294126052049099992863226995238550156138112"
        "720699269268809228350392004358582198619842529296875"},
    {0x1.fffffffffffffp-952, chars_format::fixed, 1004,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000052538071056619210158603559256195"
        "6943002618971858797983376058583405237775194340251503766410898348802660664228060068292582900602025246650498"
        "1652515982019205515801873410102285967668668772154103860408679635907746785400285525344028596507265592224125"
        "2303223647171615152454861142837051669441465241197666104037082010960821154632792496155370985372877020251671"
        "1026593124688936474129040026921582022044304446604313173655699349356248653023165231001781460532528124345122"
        "4006225687015370854137306682077853667365819879731234855018921013103087912371124311812052546458118529321982"
        "9811845263612552927798512169754422939319099557079440796268870504588252104098199985726453990477100312276225"
        "44139853853761845670078400871716439723968505859375"},
    {0x1.fffffffffffffp-951, chars_format::fixed, 1003,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000105076142113238420317207118512391"
        "3886005237943717595966752117166810475550388680503007532821796697605321328456120136585165801204050493300996"
        "3305031964038411031603746820204571935337337544308207720817359271815493570800571050688057193014531184448250"
        "4606447294343230304909722285674103338882930482395332208074164021921642309265584992310741970745754040503342"
        "2053186249377872948258080053843164044088608893208626347311398698712497306046330462003562921065056248690244"
        "8012451374030741708274613364155707334731639759462469710037842026206175824742248623624105092916237058643965"
        "9623690527225105855597024339508845878638199114158881592537741009176504208196399971452907980954200624552450"
        "8827970770752369134015680174343287944793701171875"},
    {0x1.fffffffffffffp-950, chars_format::fixed, 1002,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000210152284226476840634414237024782"
        "7772010475887435191933504234333620951100777361006015065643593395210642656912240273170331602408100986601992"
        "6610063928076822063207493640409143870674675088616415441634718543630987141601142101376114386029062368896500"
        "9212894588686460609819444571348206677765860964790664416148328043843284618531169984621483941491508081006684"
        "4106372498755745896516160107686328088177217786417252694622797397424994612092660924007125842130112497380489"
        "6024902748061483416549226728311414669463279518924939420075684052412351649484497247248210185832474117287931"
        "9247381054450211711194048679017691757276398228317763185075482018353008416392799942905815961908401249104901"
        "765594154150473826803136034868657588958740234375"},
    {0x1.fffffffffffffp-949, chars_format::fixed, 1001,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000420304568452953681268828474049565"
        "5544020951774870383867008468667241902201554722012030131287186790421285313824480546340663204816201973203985"
        "3220127856153644126414987280818287741349350177232830883269437087261974283202284202752228772058124737793001"
        "8425789177372921219638889142696413355531721929581328832296656087686569237062339969242967882983016162013368"
        "8212744997511491793032320215372656176354435572834505389245594794849989224185321848014251684260224994760979"
        "2049805496122966833098453456622829338926559037849878840151368104824703298968994494496420371664948234575863"
        "8494762108900423422388097358035383514552796456635526370150964036706016832785599885811631923816802498209803"
        "53118830830094765360627206973731517791748046875"},
    {0x1.fffffffffffffp-948, chars_format::fixed, 1000,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000840609136905907362537656948099131"
        "1088041903549740767734016937334483804403109444024060262574373580842570627648961092681326409632403946407970"
        "6440255712307288252829974561636575482698700354465661766538874174523948566404568405504457544116249475586003"
        "6851578354745842439277778285392826711063443859162657664593312175373138474124679938485935765966032324026737"
        "6425489995022983586064640430745312352708871145669010778491189589699978448370643696028503368520449989521958"
        "4099610992245933666196906913245658677853118075699757680302736209649406597937988988992840743329896469151727"
        "6989524217800846844776194716070767029105592913271052740301928073412033665571199771623263847633604996419607"
        "0623766166018953072125441394746303558349609375"},
    {0x1.fffffffffffffp-947, chars_format::fixed, 999,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000001681218273811814725075313896198262"
        "2176083807099481535468033874668967608806218888048120525148747161685141255297922185362652819264807892815941"
        "2880511424614576505659949123273150965397400708931323533077748349047897132809136811008915088232498951172007"
        "3703156709491684878555556570785653422126887718325315329186624350746276948249359876971871531932064648053475"
        "2850979990045967172129280861490624705417742291338021556982379179399956896741287392057006737040899979043916"
        "8199221984491867332393813826491317355706236151399515360605472419298813195875977977985681486659792938303455"
        "3979048435601693689552389432141534058211185826542105480603856146824067331142399543246527695267209992839214"
        "124753233203790614425088278949260711669921875"},
    {0x1.fffffffffffffp-946, chars_format::fixed, 998,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000003362436547623629450150627792396524"
        "4352167614198963070936067749337935217612437776096241050297494323370282510595844370725305638529615785631882"
        "5761022849229153011319898246546301930794801417862647066155496698095794265618273622017830176464997902344014"
        "7406313418983369757111113141571306844253775436650630658373248701492553896498719753943743063864129296106950"
        "5701959980091934344258561722981249410835484582676043113964758358799913793482574784114013474081799958087833"
        "6398443968983734664787627652982634711412472302799030721210944838597626391751955955971362973319585876606910"
        "7958096871203387379104778864283068116422371653084210961207712293648134662284799086493055390534419985678428"
        "24950646640758122885017655789852142333984375"},
    {0x1.fffffffffffffp-945, chars_format::fixed, 997,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000006724873095247258900301255584793048"
        "8704335228397926141872135498675870435224875552192482100594988646740565021191688741450611277059231571263765"
        "1522045698458306022639796493092603861589602835725294132310993396191588531236547244035660352929995804688029"
        "4812626837966739514222226283142613688507550873301261316746497402985107792997439507887486127728258592213901"
        "1403919960183868688517123445962498821670969165352086227929516717599827586965149568228026948163599916175667"
        "2796887937967469329575255305965269422824944605598061442421889677195252783503911911942725946639171753213821"
        "5916193742406774758209557728566136232844743306168421922415424587296269324569598172986110781068839971356856"
        "4990129328151624577003531157970428466796875"},
    {0x1.fffffffffffffp-944, chars_format::fixed, 996,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000013449746190494517800602511169586097"
        "7408670456795852283744270997351740870449751104384964201189977293481130042383377482901222554118463142527530"
        "3044091396916612045279592986185207723179205671450588264621986792383177062473094488071320705859991609376058"
        "9625253675933479028444452566285227377015101746602522633492994805970215585994879015774972255456517184427802"
        "2807839920367737377034246891924997643341938330704172455859033435199655173930299136456053896327199832351334"
        "5593775875934938659150510611930538845649889211196122884843779354390505567007823823885451893278343506427643"
        "1832387484813549516419115457132272465689486612336843844830849174592538649139196345972221562137679942713712"
        "998025865630324915400706231594085693359375"},
    {0x1.fffffffffffffp-943, chars_format::fixed, 995,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000026899492380989035601205022339172195"
        "4817340913591704567488541994703481740899502208769928402379954586962260084766754965802445108236926285055060"
        "6088182793833224090559185972370415446358411342901176529243973584766354124946188976142641411719983218752117"
        "9250507351866958056888905132570454754030203493205045266985989611940431171989758031549944510913034368855604"
        "5615679840735474754068493783849995286683876661408344911718066870399310347860598272912107792654399664702669"
        "1187551751869877318301021223861077691299778422392245769687558708781011134015647647770903786556687012855286"
        "3664774969627099032838230914264544931378973224673687689661698349185077298278392691944443124275359885427425"
        "99605173126064983080141246318817138671875"},
    {0x1.fffffffffffffp-942, chars_format::fixed, 994,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000053798984761978071202410044678344390"
        "9634681827183409134977083989406963481799004417539856804759909173924520169533509931604890216473852570110121"
        "2176365587666448181118371944740830892716822685802353058487947169532708249892377952285282823439966437504235"
        "8501014703733916113777810265140909508060406986410090533971979223880862343979516063099889021826068737711209"
        "1231359681470949508136987567699990573367753322816689823436133740798620695721196545824215585308799329405338"
        "2375103503739754636602042447722155382599556844784491539375117417562022268031295295541807573113374025710572"
        "7329549939254198065676461828529089862757946449347375379323396698370154596556785383888886248550719770854851"
        "9921034625212996616028249263763427734375"},
    {0x1.fffffffffffffp-941, chars_format::fixed, 993,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000107597969523956142404820089356688781"
        "9269363654366818269954167978813926963598008835079713609519818347849040339067019863209780432947705140220242"
        "4352731175332896362236743889481661785433645371604706116975894339065416499784755904570565646879932875008471"
        "7002029407467832227555620530281819016120813972820181067943958447761724687959032126199778043652137475422418"
        "2462719362941899016273975135399981146735506645633379646872267481597241391442393091648431170617598658810676"
        "4750207007479509273204084895444310765199113689568983078750234835124044536062590591083615146226748051421145"
        "4659099878508396131352923657058179725515892898694750758646793396740309193113570767777772497101439541709703"
        "984206925042599323205649852752685546875"},
    {0x1.fffffffffffffp-940, chars_format::fixed, 992,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000215195939047912284809640178713377563"
        "8538727308733636539908335957627853927196017670159427219039636695698080678134039726419560865895410280440484"
        "8705462350665792724473487778963323570867290743209412233951788678130832999569511809141131293759865750016943"
        "4004058814935664455111241060563638032241627945640362135887916895523449375918064252399556087304274950844836"
        "4925438725883798032547950270799962293471013291266759293744534963194482782884786183296862341235197317621352"
        "9500414014959018546408169790888621530398227379137966157500469670248089072125181182167230292453496102842290"
        "9318199757016792262705847314116359451031785797389501517293586793480618386227141535555544994202879083419407"
        "96841385008519864641129970550537109375"},
    {0x1.fffffffffffffp-939, chars_format::fixed, 991,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000430391878095824569619280357426755127"
        "7077454617467273079816671915255707854392035340318854438079273391396161356268079452839121731790820560880969"
        "7410924701331585448946975557926647141734581486418824467903577356261665999139023618282262587519731500033886"
        "8008117629871328910222482121127276064483255891280724271775833791046898751836128504799112174608549901689672"
        "9850877451767596065095900541599924586942026582533518587489069926388965565769572366593724682470394635242705"
        "9000828029918037092816339581777243060796454758275932315000939340496178144250362364334460584906992205684581"
        "8636399514033584525411694628232718902063571594779003034587173586961236772454283071111089988405758166838815"
        "9368277001703972928225994110107421875"},
    {0x1.fffffffffffffp-938, chars_format::fixed, 990,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000860783756191649139238560714853510255"
        "4154909234934546159633343830511415708784070680637708876158546782792322712536158905678243463581641121761939"
        "4821849402663170897893951115853294283469162972837648935807154712523331998278047236564525175039463000067773"
        "6016235259742657820444964242254552128966511782561448543551667582093797503672257009598224349217099803379345"
        "9701754903535192130191801083199849173884053165067037174978139852777931131539144733187449364940789270485411"
        "8001656059836074185632679163554486121592909516551864630001878680992356288500724728668921169813984411369163"
        "7272799028067169050823389256465437804127143189558006069174347173922473544908566142222179976811516333677631"
        "873655400340794585645198822021484375"},
    {0x1.fffffffffffffp-937, chars_format::fixed, 989,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000001721567512383298278477121429707020510"
        "8309818469869092319266687661022831417568141361275417752317093565584645425072317811356486927163282243523878"
        "9643698805326341795787902231706588566938325945675297871614309425046663996556094473129050350078926000135547"
        "2032470519485315640889928484509104257933023565122897087103335164187595007344514019196448698434199606758691"
        "9403509807070384260383602166399698347768106330134074349956279705555862263078289466374898729881578540970823"
        "6003312119672148371265358327108972243185819033103729260003757361984712577001449457337842339627968822738327"
        "4545598056134338101646778512930875608254286379116012138348694347844947089817132284444359953623032667355263"
        "74731080068158917129039764404296875"},
    {0x1.fffffffffffffp-936, chars_format::fixed, 988,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000003443135024766596556954242859414041021"
        "6619636939738184638533375322045662835136282722550835504634187131169290850144635622712973854326564487047757"
        "9287397610652683591575804463413177133876651891350595743228618850093327993112188946258100700157852000271094"
        "4064941038970631281779856969018208515866047130245794174206670328375190014689028038392897396868399213517383"
        "8807019614140768520767204332799396695536212660268148699912559411111724526156578932749797459763157081941647"
        "2006624239344296742530716654217944486371638066207458520007514723969425154002898914675684679255937645476654"
        "9091196112268676203293557025861751216508572758232024276697388695689894179634264568888719907246065334710527"
        "4946216013631783425807952880859375"},
    {0x1.fffffffffffffp-935, chars_format::fixed, 987,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000006886270049533193113908485718828082043"
        "3239273879476369277066750644091325670272565445101671009268374262338581700289271245425947708653128974095515"
        "8574795221305367183151608926826354267753303782701191486457237700186655986224377892516201400315704000542188"
        "8129882077941262563559713938036417031732094260491588348413340656750380029378056076785794793736798427034767"
        "7614039228281537041534408665598793391072425320536297399825118822223449052313157865499594919526314163883294"
        "4013248478688593485061433308435888972743276132414917040015029447938850308005797829351369358511875290953309"
        "8182392224537352406587114051723502433017145516464048553394777391379788359268529137777439814492130669421054"
        "989243202726356685161590576171875"},
    {0x1.fffffffffffffp-934, chars_format::fixed, 986,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000013772540099066386227816971437656164086"
        "6478547758952738554133501288182651340545130890203342018536748524677163400578542490851895417306257948191031"
        "7149590442610734366303217853652708535506607565402382972914475400373311972448755785032402800631408001084377"
        "6259764155882525127119427876072834063464188520983176696826681313500760058756112153571589587473596854069535"
        "5228078456563074083068817331197586782144850641072594799650237644446898104626315730999189839052628327766588"
        "8026496957377186970122866616871777945486552264829834080030058895877700616011595658702738717023750581906619"
        "6364784449074704813174228103447004866034291032928097106789554782759576718537058275554879628984261338842109"
        "97848640545271337032318115234375"},
    {0x1.fffffffffffffp-933, chars_format::fixed, 985,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000027545080198132772455633942875312328173"
        "2957095517905477108267002576365302681090261780406684037073497049354326801157084981703790834612515896382063"
        "4299180885221468732606435707305417071013215130804765945828950800746623944897511570064805601262816002168755"
        "2519528311765050254238855752145668126928377041966353393653362627001520117512224307143179174947193708139071"
        "0456156913126148166137634662395173564289701282145189599300475288893796209252631461998379678105256655533177"
        "6052993914754373940245733233743555890973104529659668160060117791755401232023191317405477434047501163813239"
        "2729568898149409626348456206894009732068582065856194213579109565519153437074116551109759257968522677684219"
        "9569728109054267406463623046875"},
    {0x1.fffffffffffffp-932, chars_format::fixed, 984,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000055090160396265544911267885750624656346"
        "5914191035810954216534005152730605362180523560813368074146994098708653602314169963407581669225031792764126"
        "8598361770442937465212871414610834142026430261609531891657901601493247889795023140129611202525632004337510"
        "5039056623530100508477711504291336253856754083932706787306725254003040235024448614286358349894387416278142"
        "0912313826252296332275269324790347128579402564290379198600950577787592418505262923996759356210513311066355"
        "2105987829508747880491466467487111781946209059319336320120235583510802464046382634810954868095002327626478"
        "5459137796298819252696912413788019464137164131712388427158219131038306874148233102219518515937045355368439"
        "913945621810853481292724609375"},
    {0x1.fffffffffffffp-931, chars_format::fixed, 983,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000110180320792531089822535771501249312693"
        "1828382071621908433068010305461210724361047121626736148293988197417307204628339926815163338450063585528253"
        "7196723540885874930425742829221668284052860523219063783315803202986495779590046280259222405051264008675021"
        "0078113247060201016955423008582672507713508167865413574613450508006080470048897228572716699788774832556284"
        "1824627652504592664550538649580694257158805128580758397201901155575184837010525847993518712421026622132710"
        "4211975659017495760982932934974223563892418118638672640240471167021604928092765269621909736190004655252957"
        "0918275592597638505393824827576038928274328263424776854316438262076613748296466204439037031874090710736879"
        "82789124362170696258544921875"},
    {0x1.fffffffffffffp-930, chars_format::fixed, 982,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000220360641585062179645071543002498625386"
        "3656764143243816866136020610922421448722094243253472296587976394834614409256679853630326676900127171056507"
        "4393447081771749860851485658443336568105721046438127566631606405972991559180092560518444810102528017350042"
        "0156226494120402033910846017165345015427016335730827149226901016012160940097794457145433399577549665112568"
        "3649255305009185329101077299161388514317610257161516794403802311150369674021051695987037424842053244265420"
        "8423951318034991521965865869948447127784836237277345280480942334043209856185530539243819472380009310505914"
        "1836551185195277010787649655152077856548656526849553708632876524153227496592932408878074063748181421473759"
        "6557824872434139251708984375"},
    {0x1.fffffffffffffp-929, chars_format::fixed, 981,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000440721283170124359290143086004997250772"
        "7313528286487633732272041221844842897444188486506944593175952789669228818513359707260653353800254342113014"
        "8786894163543499721702971316886673136211442092876255133263212811945983118360185121036889620205056034700084"
        "0312452988240804067821692034330690030854032671461654298453802032024321880195588914290866799155099330225136"
        "7298510610018370658202154598322777028635220514323033588807604622300739348042103391974074849684106488530841"
        "6847902636069983043931731739896894255569672474554690560961884668086419712371061078487638944760018621011828"
        "3673102370390554021575299310304155713097313053699107417265753048306454993185864817756148127496362842947519"
        "311564974486827850341796875"},
    {0x1.fffffffffffffp-928, chars_format::fixed, 980,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000881442566340248718580286172009994501545"
        "4627056572975267464544082443689685794888376973013889186351905579338457637026719414521306707600508684226029"
        "7573788327086999443405942633773346272422884185752510266526425623891966236720370242073779240410112069400168"
        "0624905976481608135643384068661380061708065342923308596907604064048643760391177828581733598310198660450273"
        "4597021220036741316404309196645554057270441028646067177615209244601478696084206783948149699368212977061683"
        "3695805272139966087863463479793788511139344949109381121923769336172839424742122156975277889520037242023656"
        "7346204740781108043150598620608311426194626107398214834531506096612909986371729635512296254992725685895038"
        "62312994897365570068359375"},
    {0x1.fffffffffffffp-927, chars_format::fixed, 979,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000001762885132680497437160572344019989003090"
        "9254113145950534929088164887379371589776753946027778372703811158676915274053438829042613415201017368452059"
        "5147576654173998886811885267546692544845768371505020533052851247783932473440740484147558480820224138800336"
        "1249811952963216271286768137322760123416130685846617193815208128097287520782355657163467196620397320900546"
        "9194042440073482632808618393291108114540882057292134355230418489202957392168413567896299398736425954123366"
        "7391610544279932175726926959587577022278689898218762243847538672345678849484244313950555779040074484047313"
        "4692409481562216086301197241216622852389252214796429669063012193225819972743459271024592509985451371790077"
        "2462598979473114013671875"},
    {0x1.fffffffffffffp-926, chars_format::fixed, 978,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000003525770265360994874321144688039978006181"
        "8508226291901069858176329774758743179553507892055556745407622317353830548106877658085226830402034736904119"
        "0295153308347997773623770535093385089691536743010041066105702495567864946881480968295116961640448277600672"
        "2499623905926432542573536274645520246832261371693234387630416256194575041564711314326934393240794641801093"
        "8388084880146965265617236786582216229081764114584268710460836978405914784336827135792598797472851908246733"
        "4783221088559864351453853919175154044557379796437524487695077344691357698968488627901111558080148968094626"
        "9384818963124432172602394482433245704778504429592859338126024386451639945486918542049185019970902743580154"
        "492519795894622802734375"},
    {0x1.fffffffffffffp-925, chars_format::fixed, 977,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000007051540530721989748642289376079956012363"
        "7016452583802139716352659549517486359107015784111113490815244634707661096213755316170453660804069473808238"
        "0590306616695995547247541070186770179383073486020082132211404991135729893762961936590233923280896555201344"
        "4999247811852865085147072549291040493664522743386468775260832512389150083129422628653868786481589283602187"
        "6776169760293930531234473573164432458163528229168537420921673956811829568673654271585197594945703816493466"
        "9566442177119728702907707838350308089114759592875048975390154689382715397936977255802223116160297936189253"
        "8769637926248864345204788964866491409557008859185718676252048772903279890973837084098370039941805487160308"
        "98503959178924560546875"},
    {0x1.fffffffffffffp-924, chars_format::fixed, 976,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000014103081061443979497284578752159912024727"
        "4032905167604279432705319099034972718214031568222226981630489269415322192427510632340907321608138947616476"
        "1180613233391991094495082140373540358766146972040164264422809982271459787525923873180467846561793110402688"
        "9998495623705730170294145098582080987329045486772937550521665024778300166258845257307737572963178567204375"
        "3552339520587861062468947146328864916327056458337074841843347913623659137347308543170395189891407632986933"
        "9132884354239457405815415676700616178229519185750097950780309378765430795873954511604446232320595872378507"
        "7539275852497728690409577929732982819114017718371437352504097545806559781947674168196740079883610974320617"
        "9700791835784912109375"},
    {0x1.fffffffffffffp-923, chars_format::fixed, 975,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000028206162122887958994569157504319824049454"
        "8065810335208558865410638198069945436428063136444453963260978538830644384855021264681814643216277895232952"
        "2361226466783982188990164280747080717532293944080328528845619964542919575051847746360935693123586220805377"
        "9996991247411460340588290197164161974658090973545875101043330049556600332517690514615475145926357134408750"
        "7104679041175722124937894292657729832654112916674149683686695827247318274694617086340790379782815265973867"
        "8265768708478914811630831353401232356459038371500195901560618757530861591747909023208892464641191744757015"
        "5078551704995457380819155859465965638228035436742874705008195091613119563895348336393480159767221948641235"
        "940158367156982421875"},
    {0x1.fffffffffffffp-922, chars_format::fixed, 974,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000056412324245775917989138315008639648098909"
        "6131620670417117730821276396139890872856126272888907926521957077661288769710042529363629286432555790465904"
        "4722452933567964377980328561494161435064587888160657057691239929085839150103695492721871386247172441610755"
        "9993982494822920681176580394328323949316181947091750202086660099113200665035381029230950291852714268817501"
        "4209358082351444249875788585315459665308225833348299367373391654494636549389234172681580759565630531947735"
        "6531537416957829623261662706802464712918076743000391803121237515061723183495818046417784929282383489514031"
        "0157103409990914761638311718931931276456070873485749410016390183226239127790696672786960319534443897282471"
        "88031673431396484375"},
    {0x1.fffffffffffffp-921, chars_format::fixed, 973,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000112824648491551835978276630017279296197819"
        "2263241340834235461642552792279781745712252545777815853043914155322577539420085058727258572865111580931808"
        "9444905867135928755960657122988322870129175776321314115382479858171678300207390985443742772494344883221511"
        "9987964989645841362353160788656647898632363894183500404173320198226401330070762058461900583705428537635002"
        "8418716164702888499751577170630919330616451666696598734746783308989273098778468345363161519131261063895471"
        "3063074833915659246523325413604929425836153486000783606242475030123446366991636092835569858564766979028062"
        "0314206819981829523276623437863862552912141746971498820032780366452478255581393345573920639068887794564943"
        "7606334686279296875"},
    {0x1.fffffffffffffp-920, chars_format::fixed, 972,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000225649296983103671956553260034558592395638"
        "4526482681668470923285105584559563491424505091555631706087828310645155078840170117454517145730223161863617"
        "8889811734271857511921314245976645740258351552642628230764959716343356600414781970887485544988689766443023"
        "9975929979291682724706321577313295797264727788367000808346640396452802660141524116923801167410857075270005"
        "6837432329405776999503154341261838661232903333393197469493566617978546197556936690726323038262522127790942"
        "6126149667831318493046650827209858851672306972001567212484950060246892733983272185671139717129533958056124"
        "0628413639963659046553246875727725105824283493942997640065560732904956511162786691147841278137775589129887"
        "521266937255859375"},
    {0x1.fffffffffffffp-919, chars_format::fixed, 971,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000451298593966207343913106520069117184791276"
        "9052965363336941846570211169119126982849010183111263412175656621290310157680340234909034291460446323727235"
        "7779623468543715023842628491953291480516703105285256461529919432686713200829563941774971089977379532886047"
        "9951859958583365449412643154626591594529455576734001616693280792905605320283048233847602334821714150540011"
        "3674864658811553999006308682523677322465806666786394938987133235957092395113873381452646076525044255581885"
        "2252299335662636986093301654419717703344613944003134424969900120493785467966544371342279434259067916112248"
        "1256827279927318093106493751455450211648566987885995280131121465809913022325573382295682556275551178259775"
        "04253387451171875"},
    {0x1.fffffffffffffp-918, chars_format::fixed, 970,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000902597187932414687826213040138234369582553"
        "8105930726673883693140422338238253965698020366222526824351313242580620315360680469818068582920892647454471"
        "5559246937087430047685256983906582961033406210570512923059838865373426401659127883549942179954759065772095"
        "9903719917166730898825286309253183189058911153468003233386561585811210640566096467695204669643428301080022"
        "7349729317623107998012617365047354644931613333572789877974266471914184790227746762905292153050088511163770"
        "4504598671325273972186603308839435406689227888006268849939800240987570935933088742684558868518135832224496"
        "2513654559854636186212987502910900423297133975771990560262242931619826044651146764591365112551102356519550"
        "0850677490234375"},
    {0x1.fffffffffffffp-917, chars_format::fixed, 969,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000001805194375864829375652426080276468739165107"
        "6211861453347767386280844676476507931396040732445053648702626485161240630721360939636137165841785294908943"
        "1118493874174860095370513967813165922066812421141025846119677730746852803318255767099884359909518131544191"
        "9807439834333461797650572618506366378117822306936006466773123171622421281132192935390409339286856602160045"
        "4699458635246215996025234730094709289863226667145579755948532943828369580455493525810584306100177022327540"
        "9009197342650547944373206617678870813378455776012537699879600481975141871866177485369117737036271664448992"
        "5027309119709272372425975005821800846594267951543981120524485863239652089302293529182730225102204713039100"
        "170135498046875"},
    {0x1.fffffffffffffp-916, chars_format::fixed, 968,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000003610388751729658751304852160552937478330215"
        "2423722906695534772561689352953015862792081464890107297405252970322481261442721879272274331683570589817886"
        "2236987748349720190741027935626331844133624842282051692239355461493705606636511534199768719819036263088383"
        "9614879668666923595301145237012732756235644613872012933546246343244842562264385870780818678573713204320090"
        "9398917270492431992050469460189418579726453334291159511897065887656739160910987051621168612200354044655081"
        "8018394685301095888746413235357741626756911552025075399759200963950283743732354970738235474072543328897985"
        "0054618239418544744851950011643601693188535903087962241048971726479304178604587058365460450204409426078200"
        "34027099609375"},
    {0x1.fffffffffffffp-915, chars_format::fixed, 967,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000007220777503459317502609704321105874956660430"
        "4847445813391069545123378705906031725584162929780214594810505940644962522885443758544548663367141179635772"
        "4473975496699440381482055871252663688267249684564103384478710922987411213273023068399537439638072526176767"
        "9229759337333847190602290474025465512471289227744025867092492686489685124528771741561637357147426408640181"
        "8797834540984863984100938920378837159452906668582319023794131775313478321821974103242337224400708089310163"
        "6036789370602191777492826470715483253513823104050150799518401927900567487464709941476470948145086657795970"
        "0109236478837089489703900023287203386377071806175924482097943452958608357209174116730920900408818852156400"
        "6805419921875"},
    {0x1.fffffffffffffp-914, chars_format::fixed, 966,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000014441555006918635005219408642211749913320860"
        "9694891626782139090246757411812063451168325859560429189621011881289925045770887517089097326734282359271544"
        "8947950993398880762964111742505327376534499369128206768957421845974822426546046136799074879276145052353535"
        "8459518674667694381204580948050931024942578455488051734184985372979370249057543483123274714294852817280363"
        "7595669081969727968201877840757674318905813337164638047588263550626956643643948206484674448801416178620327"
        "2073578741204383554985652941430966507027646208100301599036803855801134974929419882952941896290173315591940"
        "0218472957674178979407800046574406772754143612351848964195886905917216714418348233461841800817637704312801"
        "361083984375"},
    {0x1.fffffffffffffp-913, chars_format::fixed, 965,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000028883110013837270010438817284423499826641721"
        "9389783253564278180493514823624126902336651719120858379242023762579850091541775034178194653468564718543089"
        "7895901986797761525928223485010654753068998738256413537914843691949644853092092273598149758552290104707071"
        "6919037349335388762409161896101862049885156910976103468369970745958740498115086966246549428589705634560727"
        "5191338163939455936403755681515348637811626674329276095176527101253913287287896412969348897602832357240654"
        "4147157482408767109971305882861933014055292416200603198073607711602269949858839765905883792580346631183880"
        "0436945915348357958815600093148813545508287224703697928391773811834433428836696466923683601635275408625602"
        "72216796875"},
    {0x1.fffffffffffffp-912, chars_format::fixed, 964,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000057766220027674540020877634568846999653283443"
        "8779566507128556360987029647248253804673303438241716758484047525159700183083550068356389306937129437086179"
        "5791803973595523051856446970021309506137997476512827075829687383899289706184184547196299517104580209414143"
        "3838074698670777524818323792203724099770313821952206936739941491917480996230173932493098857179411269121455"
        "0382676327878911872807511363030697275623253348658552190353054202507826574575792825938697795205664714481308"
        "8294314964817534219942611765723866028110584832401206396147215423204539899717679531811767585160693262367760"
        "0873891830696715917631200186297627091016574449407395856783547623668866857673392933847367203270550817251205"
        "4443359375"},
    {0x1.fffffffffffffp-911, chars_format::fixed, 963,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000115532440055349080041755269137693999306566887"
        "7559133014257112721974059294496507609346606876483433516968095050319400366167100136712778613874258874172359"
        "1583607947191046103712893940042619012275994953025654151659374767798579412368369094392599034209160418828286"
        "7676149397341555049636647584407448199540627643904413873479882983834961992460347864986197714358822538242910"
        "0765352655757823745615022726061394551246506697317104380706108405015653149151585651877395590411329428962617"
        "6588629929635068439885223531447732056221169664802412792294430846409079799435359063623535170321386524735520"
        "1747783661393431835262400372595254182033148898814791713567095247337733715346785867694734406541101634502410"
        "888671875"},
    {0x1.fffffffffffffp-910, chars_format::fixed, 962,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000231064880110698160083510538275387998613133775"
        "5118266028514225443948118588993015218693213752966867033936190100638800732334200273425557227748517748344718"
        "3167215894382092207425787880085238024551989906051308303318749535597158824736738188785198068418320837656573"
        "5352298794683110099273295168814896399081255287808827746959765967669923984920695729972395428717645076485820"
        "1530705311515647491230045452122789102493013394634208761412216810031306298303171303754791180822658857925235"
        "3177259859270136879770447062895464112442339329604825584588861692818159598870718127247070340642773049471040"
        "3495567322786863670524800745190508364066297797629583427134190494675467430693571735389468813082203269004821"
        "77734375"},
    {0x1.fffffffffffffp-909, chars_format::fixed, 961,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000462129760221396320167021076550775997226267551"
        "0236532057028450887896237177986030437386427505933734067872380201277601464668400546851114455497035496689436"
        "6334431788764184414851575760170476049103979812102616606637499071194317649473476377570396136836641675313147"
        "0704597589366220198546590337629792798162510575617655493919531935339847969841391459944790857435290152971640"
        "3061410623031294982460090904245578204986026789268417522824433620062612596606342607509582361645317715850470"
        "6354519718540273759540894125790928224884678659209651169177723385636319197741436254494140681285546098942080"
        "6991134645573727341049601490381016728132595595259166854268380989350934861387143470778937626164406538009643"
        "5546875"},
    {0x1.fffffffffffffp-908, chars_format::fixed, 960,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000924259520442792640334042153101551994452535102"
        "0473064114056901775792474355972060874772855011867468135744760402555202929336801093702228910994070993378873"
        "2668863577528368829703151520340952098207959624205233213274998142388635298946952755140792273673283350626294"
        "1409195178732440397093180675259585596325021151235310987839063870679695939682782919889581714870580305943280"
        "6122821246062589964920181808491156409972053578536835045648867240125225193212685215019164723290635431700941"
        "2709039437080547519081788251581856449769357318419302338355446771272638395482872508988281362571092197884161"
        "3982269291147454682099202980762033456265191190518333708536761978701869722774286941557875252328813076019287"
        "109375"},
    {0x1.fffffffffffffp-907, chars_format::fixed, 959,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000001848519040885585280668084306203103988905070204"
        "0946128228113803551584948711944121749545710023734936271489520805110405858673602187404457821988141986757746"
        "5337727155056737659406303040681904196415919248410466426549996284777270597893905510281584547346566701252588"
        "2818390357464880794186361350519171192650042302470621975678127741359391879365565839779163429741160611886561"
        "2245642492125179929840363616982312819944107157073670091297734480250450386425370430038329446581270863401882"
        "5418078874161095038163576503163712899538714636838604676710893542545276790965745017976562725142184395768322"
        "7964538582294909364198405961524066912530382381036667417073523957403739445548573883115750504657626152038574"
        "21875"},
    {0x1.fffffffffffffp-906, chars_format::fixed, 958,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000003697038081771170561336168612406207977810140408"
        "1892256456227607103169897423888243499091420047469872542979041610220811717347204374808915643976283973515493"
        "0675454310113475318812606081363808392831838496820932853099992569554541195787811020563169094693133402505176"
        "5636780714929761588372722701038342385300084604941243951356255482718783758731131679558326859482321223773122"
        "4491284984250359859680727233964625639888214314147340182595468960500900772850740860076658893162541726803765"
        "0836157748322190076327153006327425799077429273677209353421787085090553581931490035953125450284368791536645"
        "5929077164589818728396811923048133825060764762073334834147047914807478891097147766231501009315252304077148"
        "4375"},
    {0x1.fffffffffffffp-905, chars_format::fixed, 957,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000007394076163542341122672337224812415955620280816"
        "3784512912455214206339794847776486998182840094939745085958083220441623434694408749617831287952567947030986"
        "1350908620226950637625212162727616785663676993641865706199985139109082391575622041126338189386266805010353"
        "1273561429859523176745445402076684770600169209882487902712510965437567517462263359116653718964642447546244"
        "8982569968500719719361454467929251279776428628294680365190937921001801545701481720153317786325083453607530"
        "1672315496644380152654306012654851598154858547354418706843574170181107163862980071906250900568737583073291"
        "1858154329179637456793623846096267650121529524146669668294095829614957782194295532463002018630504608154296"
        "875"},
    {0x1.fffffffffffffp-904, chars_format::fixed, 956,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000014788152327084682245344674449624831911240561632"
        "7569025824910428412679589695552973996365680189879490171916166440883246869388817499235662575905135894061972"
        "2701817240453901275250424325455233571327353987283731412399970278218164783151244082252676378772533610020706"
        "2547122859719046353490890804153369541200338419764975805425021930875135034924526718233307437929284895092489"
        "7965139937001439438722908935858502559552857256589360730381875842003603091402963440306635572650166907215060"
        "3344630993288760305308612025309703196309717094708837413687148340362214327725960143812501801137475166146582"
        "3716308658359274913587247692192535300243059048293339336588191659229915564388591064926004037261009216308593"
        "75"},
    {0x1.fffffffffffffp-903, chars_format::fixed, 955,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000029576304654169364490689348899249663822481123265"
        "5138051649820856825359179391105947992731360379758980343832332881766493738777634998471325151810271788123944"
        "5403634480907802550500848650910467142654707974567462824799940556436329566302488164505352757545067220041412"
        "5094245719438092706981781608306739082400676839529951610850043861750270069849053436466614875858569790184979"
        "5930279874002878877445817871717005119105714513178721460763751684007206182805926880613271145300333814430120"
        "6689261986577520610617224050619406392619434189417674827374296680724428655451920287625003602274950332293164"
        "7432617316718549827174495384385070600486118096586678673176383318459831128777182129852008074522018432617187"
        "5"},
    {0x1.fffffffffffffp-902, chars_format::fixed, 954,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000059152609308338728981378697798499327644962246531"
        "0276103299641713650718358782211895985462720759517960687664665763532987477555269996942650303620543576247889"
        "0807268961815605101001697301820934285309415949134925649599881112872659132604976329010705515090134440082825"
        "0188491438876185413963563216613478164801353679059903221700087723500540139698106872933229751717139580369959"
        "1860559748005757754891635743434010238211429026357442921527503368014412365611853761226542290600667628860241"
        "3378523973155041221234448101238812785238868378835349654748593361448857310903840575250007204549900664586329"
        "486523463343709965434899076877014120097223619317335734635276663691966225755436425970401614904403686523437"
        "5"},
    {0x1.fffffffffffffp-901, chars_format::fixed, 953,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000118305218616677457962757395596998655289924493062"
        "0552206599283427301436717564423791970925441519035921375329331527065974955110539993885300607241087152495778"
        "1614537923631210202003394603641868570618831898269851299199762225745318265209952658021411030180268880165650"
        "0376982877752370827927126433226956329602707358119806443400175447001080279396213745866459503434279160739918"
        "3721119496011515509783271486868020476422858052714885843055006736028824731223707522453084581201335257720482"
        "6757047946310082442468896202477625570477736757670699309497186722897714621807681150500014409099801329172658"
        "97304692668741993086979815375402824019444723863467146927055332738393245151087285194080322980880737304687"
        "5"},
    {0x1.fffffffffffffp-900, chars_format::fixed, 952,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000236610437233354915925514791193997310579848986124"
        "1104413198566854602873435128847583941850883038071842750658663054131949910221079987770601214482174304991556"
        "3229075847262420404006789207283737141237663796539702598399524451490636530419905316042822060360537760331300"
        "0753965755504741655854252866453912659205414716239612886800350894002160558792427491732919006868558321479836"
        "7442238992023031019566542973736040952845716105429771686110013472057649462447415044906169162402670515440965"
        "3514095892620164884937792404955251140955473515341398618994373445795429243615362301000028818199602658345317"
        "94609385337483986173959630750805648038889447726934293854110665476786490302174570388160645961761474609375"},
    {0x1.fffffffffffffp-899, chars_format::fixed, 951,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000473220874466709831851029582387994621159697972248"
        "2208826397133709205746870257695167883701766076143685501317326108263899820442159975541202428964348609983112"
        "6458151694524840808013578414567474282475327593079405196799048902981273060839810632085644120721075520662600"
        "1507931511009483311708505732907825318410829432479225773600701788004321117584854983465838013737116642959673"
        "4884477984046062039133085947472081905691432210859543372220026944115298924894830089812338324805341030881930"
        "7028191785240329769875584809910502281910947030682797237988746891590858487230724602000057636399205316690635"
        "8921877067496797234791926150161129607777889545386858770822133095357298060434914077632129192352294921875"},
    {0x1.fffffffffffffp-898, chars_format::fixed, 950,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000946441748933419663702059164775989242319395944496"
        "4417652794267418411493740515390335767403532152287371002634652216527799640884319951082404857928697219966225"
        "2916303389049681616027156829134948564950655186158810393598097805962546121679621264171288241442151041325200"
        "3015863022018966623417011465815650636821658864958451547201403576008642235169709966931676027474233285919346"
        "9768955968092124078266171894944163811382864421719086744440053888230597849789660179624676649610682061763861"
        "4056383570480659539751169619821004563821894061365594475977493783181716974461449204000115272798410633381271"
        "784375413499359446958385230032225921555577909077371754164426619071459612086982815526425838470458984375"},
    {0x1.fffffffffffffp-897, chars_format::fixed, 949,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000001892883497866839327404118329551978484638791888992"
        "8835305588534836822987481030780671534807064304574742005269304433055599281768639902164809715857394439932450"
        "5832606778099363232054313658269897129901310372317620787196195611925092243359242528342576482884302082650400"
        "6031726044037933246834022931631301273643317729916903094402807152017284470339419933863352054948466571838693"
        "9537911936184248156532343789888327622765728843438173488880107776461195699579320359249353299221364123527722"
        "8112767140961319079502339239642009127643788122731188951954987566363433948922898408000230545596821266762543"
        "56875082699871889391677046006445184311115581815474350832885323814291922417396563105285167694091796875"},
    {0x1.fffffffffffffp-896, chars_format::fixed, 948,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000003785766995733678654808236659103956969277583777985"
        "7670611177069673645974962061561343069614128609149484010538608866111198563537279804329619431714788879864901"
        "1665213556198726464108627316539794259802620744635241574392391223850184486718485056685152965768604165300801"
        "2063452088075866493668045863262602547286635459833806188805614304034568940678839867726704109896933143677387"
        "9075823872368496313064687579776655245531457686876346977760215552922391399158640718498706598442728247055445"
        "6225534281922638159004678479284018255287576245462377903909975132726867897845796816000461091193642533525087"
        "1375016539974377878335409201289036862223116363094870166577064762858384483479312621057033538818359375"},
    {0x1.fffffffffffffp-895, chars_format::fixed, 947,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000007571533991467357309616473318207913938555167555971"
        "5341222354139347291949924123122686139228257218298968021077217732222397127074559608659238863429577759729802"
        "3330427112397452928217254633079588519605241489270483148784782447700368973436970113370305931537208330601602"
        "4126904176151732987336091726525205094573270919667612377611228608069137881357679735453408219793866287354775"
        "8151647744736992626129375159553310491062915373752693955520431105844782798317281436997413196885456494110891"
        "2451068563845276318009356958568036510575152490924755807819950265453735795691593632000922182387285067050174"
        "275003307994875575667081840257807372444623272618974033315412952571676896695862524211406707763671875"},
    {0x1.fffffffffffffp-894, chars_format::fixed, 946,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000015143067982934714619232946636415827877110335111943"
        "0682444708278694583899848246245372278456514436597936042154435464444794254149119217318477726859155519459604"
        "6660854224794905856434509266159177039210482978540966297569564895400737946873940226740611863074416661203204"
        "8253808352303465974672183453050410189146541839335224755222457216138275762715359470906816439587732574709551"
        "6303295489473985252258750319106620982125830747505387911040862211689565596634562873994826393770912988221782"
        "4902137127690552636018713917136073021150304981849511615639900530907471591383187264001844364774570134100348"
        "55000661598975115133416368051561474488924654523794806663082590514335379339172504842281341552734375"},
    {0x1.fffffffffffffp-893, chars_format::fixed, 945,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000030286135965869429238465893272831655754220670223886"
        "1364889416557389167799696492490744556913028873195872084308870928889588508298238434636955453718311038919209"
        "3321708449589811712869018532318354078420965957081932595139129790801475893747880453481223726148833322406409"
        "6507616704606931949344366906100820378293083678670449510444914432276551525430718941813632879175465149419103"
        "2606590978947970504517500638213241964251661495010775822081724423379131193269125747989652787541825976443564"
        "9804274255381105272037427834272146042300609963699023231279801061814943182766374528003688729549140268200697"
        "1000132319795023026683273610312294897784930904758961332616518102867075867834500968456268310546875"},
    {0x1.fffffffffffffp-892, chars_format::fixed, 944,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000060572271931738858476931786545663311508441340447772"
        "2729778833114778335599392984981489113826057746391744168617741857779177016596476869273910907436622077838418"
        "6643416899179623425738037064636708156841931914163865190278259581602951787495760906962447452297666644812819"
        "3015233409213863898688733812201640756586167357340899020889828864553103050861437883627265758350930298838206"
        "5213181957895941009035001276426483928503322990021551644163448846758262386538251495979305575083651952887129"
        "9608548510762210544074855668544292084601219927398046462559602123629886365532749056007377459098280536401394"
        "200026463959004605336654722062458979556986180951792266523303620573415173566900193691253662109375"},
    {0x1.fffffffffffffp-891, chars_format::fixed, 943,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000121144543863477716953863573091326623016882680895544"
        "5459557666229556671198785969962978227652115492783488337235483715558354033192953738547821814873244155676837"
        "3286833798359246851476074129273416313683863828327730380556519163205903574991521813924894904595333289625638"
        "6030466818427727797377467624403281513172334714681798041779657729106206101722875767254531516701860597676413"
        "0426363915791882018070002552852967857006645980043103288326897693516524773076502991958611150167303905774259"
        "9217097021524421088149711337088584169202439854796092925119204247259772731065498112014754918196561072802788"
        "40005292791800921067330944412491795911397236190358453304660724114683034713380038738250732421875"},
    {0x1.fffffffffffffp-890, chars_format::fixed, 942,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000242289087726955433907727146182653246033765361791089"
        "0919115332459113342397571939925956455304230985566976674470967431116708066385907477095643629746488311353674"
        "6573667596718493702952148258546832627367727656655460761113038326411807149983043627849789809190666579251277"
        "2060933636855455594754935248806563026344669429363596083559315458212412203445751534509063033403721195352826"
        "0852727831583764036140005105705935714013291960086206576653795387033049546153005983917222300334607811548519"
        "8434194043048842176299422674177168338404879709592185850238408494519545462130996224029509836393122145605576"
        "8001058558360184213466188882498359182279447238071690660932144822936606942676007747650146484375"},
    {0x1.fffffffffffffp-889, chars_format::fixed, 941,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000484578175453910867815454292365306492067530723582178"
        "1838230664918226684795143879851912910608461971133953348941934862233416132771814954191287259492976622707349"
        "3147335193436987405904296517093665254735455313310921522226076652823614299966087255699579618381333158502554"
        "4121867273710911189509870497613126052689338858727192167118630916424824406891503069018126066807442390705652"
        "1705455663167528072280010211411871428026583920172413153307590774066099092306011967834444600669215623097039"
        "6868388086097684352598845348354336676809759419184371700476816989039090924261992448059019672786244291211153"
        "600211711672036842693237776499671836455889447614338132186428964587321388535201549530029296875"},
    {0x1.fffffffffffffp-888, chars_format::fixed, 940,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000969156350907821735630908584730612984135061447164356"
        "3676461329836453369590287759703825821216923942267906697883869724466832265543629908382574518985953245414698"
        "6294670386873974811808593034187330509470910626621843044452153305647228599932174511399159236762666317005108"
        "8243734547421822379019740995226252105378677717454384334237261832849648813783006138036252133614884781411304"
        "3410911326335056144560020422823742856053167840344826306615181548132198184612023935668889201338431246194079"
        "3736776172195368705197690696708673353619518838368743400953633978078181848523984896118039345572488582422307"
        "20042342334407368538647555299934367291177889522867626437285792917464277707040309906005859375"},
    {0x1.fffffffffffffp-887, chars_format::fixed, 939,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000001938312701815643471261817169461225968270122894328712"
        "7352922659672906739180575519407651642433847884535813395767739448933664531087259816765149037971906490829397"
        "2589340773747949623617186068374661018941821253243686088904306611294457199864349022798318473525332634010217"
        "6487469094843644758039481990452504210757355434908768668474523665699297627566012276072504267229769562822608"
        "6821822652670112289120040845647485712106335680689652613230363096264396369224047871337778402676862492388158"
        "7473552344390737410395381393417346707239037676737486801907267956156363697047969792236078691144977164844614"
        "4008468466881473707729511059986873458235577904573525287457158583492855541408061981201171875"},
    {0x1.fffffffffffffp-886, chars_format::fixed, 938,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000003876625403631286942523634338922451936540245788657425"
        "4705845319345813478361151038815303284867695769071626791535478897867329062174519633530298075943812981658794"
        "5178681547495899247234372136749322037883642506487372177808613222588914399728698045596636947050665268020435"
        "2974938189687289516078963980905008421514710869817537336949047331398595255132024552145008534459539125645217"
        "3643645305340224578240081691294971424212671361379305226460726192528792738448095742675556805353724984776317"
        "4947104688781474820790762786834693414478075353474973603814535912312727394095939584472157382289954329689228"
        "801693693376294741545902211997374691647115580914705057491431716698571108281612396240234375"},
    {0x1.fffffffffffffp-885, chars_format::fixed, 937,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000007753250807262573885047268677844903873080491577314850"
        "9411690638691626956722302077630606569735391538143253583070957795734658124349039267060596151887625963317589"
        "0357363094991798494468744273498644075767285012974744355617226445177828799457396091193273894101330536040870"
        "5949876379374579032157927961810016843029421739635074673898094662797190510264049104290017068919078251290434"
        "7287290610680449156480163382589942848425342722758610452921452385057585476896191485351113610707449969552634"
        "9894209377562949641581525573669386828956150706949947207629071824625454788191879168944314764579908659378457"
        "60338738675258948309180442399474938329423116182941011498286343339714221656322479248046875"},
    {0x1.fffffffffffffp-884, chars_format::fixed, 936,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000015506501614525147770094537355689807746160983154629701"
        "8823381277383253913444604155261213139470783076286507166141915591469316248698078534121192303775251926635178"
        "0714726189983596988937488546997288151534570025949488711234452890355657598914792182386547788202661072081741"
        "1899752758749158064315855923620033686058843479270149347796189325594381020528098208580034137838156502580869"
        "4574581221360898312960326765179885696850685445517220905842904770115170953792382970702227221414899939105269"
        "9788418755125899283163051147338773657912301413899894415258143649250909576383758337888629529159817318756915"
        "2067747735051789661836088479894987665884623236588202299657268667942844331264495849609375"},
    {0x1.fffffffffffffp-883, chars_format::fixed, 935,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000031013003229050295540189074711379615492321966309259403"
        "7646762554766507826889208310522426278941566152573014332283831182938632497396157068242384607550503853270356"
        "1429452379967193977874977093994576303069140051898977422468905780711315197829584364773095576405322144163482"
        "3799505517498316128631711847240067372117686958540298695592378651188762041056196417160068275676313005161738"
        "9149162442721796625920653530359771393701370891034441811685809540230341907584765941404454442829799878210539"
        "9576837510251798566326102294677547315824602827799788830516287298501819152767516675777259058319634637513830"
        "413549547010357932367217695978997533176924647317640459931453733588568866252899169921875"},
    {0x1.fffffffffffffp-882, chars_format::fixed, 934,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000062026006458100591080378149422759230984643932618518807"
        "5293525109533015653778416621044852557883132305146028664567662365877264994792314136484769215101007706540712"
        "2858904759934387955749954187989152606138280103797954844937811561422630395659168729546191152810644288326964"
        "7599011034996632257263423694480134744235373917080597391184757302377524082112392834320136551352626010323477"
        "8298324885443593251841307060719542787402741782068883623371619080460683815169531882808908885659599756421079"
        "9153675020503597132652204589355094631649205655599577661032574597003638305535033351554518116639269275027660"
        "82709909402071586473443539195799506635384929463528091986290746717713773250579833984375"},
    {0x1.fffffffffffffp-881, chars_format::fixed, 933,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000124052012916201182160756298845518461969287865237037615"
        "0587050219066031307556833242089705115766264610292057329135324731754529989584628272969538430202015413081424"
        "5717809519868775911499908375978305212276560207595909689875623122845260791318337459092382305621288576653929"
        "5198022069993264514526847388960269488470747834161194782369514604755048164224785668640273102705252020646955"
        "6596649770887186503682614121439085574805483564137767246743238160921367630339063765617817771319199512842159"
        "8307350041007194265304409178710189263298411311199155322065149194007276611070066703109036233278538550055321"
        "6541981880414317294688707839159901327076985892705618397258149343542754650115966796875"},
    {0x1.fffffffffffffp-880, chars_format::fixed, 932,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000248104025832402364321512597691036923938575730474075230"
        "1174100438132062615113666484179410231532529220584114658270649463509059979169256545939076860404030826162849"
        "1435619039737551822999816751956610424553120415191819379751246245690521582636674918184764611242577153307859"
        "0396044139986529029053694777920538976941495668322389564739029209510096328449571337280546205410504041293911"
        "3193299541774373007365228242878171149610967128275534493486476321842735260678127531235635542638399025684319"
        "6614700082014388530608818357420378526596822622398310644130298388014553222140133406218072466557077100110643"
        "308396376082863458937741567831980265415397178541123679451629868708550930023193359375"},
    {0x1.fffffffffffffp-879, chars_format::fixed, 931,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000496208051664804728643025195382073847877151460948150460"
        "2348200876264125230227332968358820463065058441168229316541298927018119958338513091878153720808061652325698"
        "2871238079475103645999633503913220849106240830383638759502492491381043165273349836369529222485154306615718"
        "0792088279973058058107389555841077953882991336644779129478058419020192656899142674561092410821008082587822"
        "6386599083548746014730456485756342299221934256551068986972952643685470521356255062471271085276798051368639"
        "3229400164028777061217636714840757053193645244796621288260596776029106444280266812436144933114154200221286"
        "61679275216572691787548313566396053083079435708224735890325973741710186004638671875"},
    {0x1.fffffffffffffp-878, chars_format::fixed, 930,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000992416103329609457286050390764147695754302921896300920"
        "4696401752528250460454665936717640926130116882336458633082597854036239916677026183756307441616123304651396"
        "5742476158950207291999267007826441698212481660767277519004984982762086330546699672739058444970308613231436"
        "1584176559946116116214779111682155907765982673289558258956116838040385313798285349122184821642016165175645"
        "2773198167097492029460912971512684598443868513102137973945905287370941042712510124942542170553596102737278"
        "6458800328057554122435273429681514106387290489593242576521193552058212888560533624872289866228308400442573"
        "2335855043314538357509662713279210616615887141644947178065194748342037200927734375"},
    {0x1.fffffffffffffp-877, chars_format::fixed, 929,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000001984832206659218914572100781528295391508605843792601840"
        "9392803505056500920909331873435281852260233764672917266165195708072479833354052367512614883232246609302793"
        "1484952317900414583998534015652883396424963321534555038009969965524172661093399345478116889940617226462872"
        "3168353119892232232429558223364311815531965346579116517912233676080770627596570698244369643284032330351290"
        "5546396334194984058921825943025369196887737026204275947891810574741882085425020249885084341107192205474557"
        "2917600656115108244870546859363028212774580979186485153042387104116425777121067249744579732456616800885146"
        "467171008662907671501932542655842123323177428328989435613038949668407440185546875"},
    {0x1.fffffffffffffp-876, chars_format::fixed, 928,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000003969664413318437829144201563056590783017211687585203681"
        "8785607010113001841818663746870563704520467529345834532330391416144959666708104735025229766464493218605586"
        "2969904635800829167997068031305766792849926643069110076019939931048345322186798690956233779881234452925744"
        "6336706239784464464859116446728623631063930693158233035824467352161541255193141396488739286568064660702581"
        "1092792668389968117843651886050738393775474052408551895783621149483764170850040499770168682214384410949114"
        "5835201312230216489741093718726056425549161958372970306084774208232851554242134499489159464913233601770292"
        "93434201732581534300386508531168424664635485665797887122607789933681488037109375"},
    {0x1.fffffffffffffp-875, chars_format::fixed, 927,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000007939328826636875658288403126113181566034423375170407363"
        "7571214020226003683637327493741127409040935058691669064660782832289919333416209470050459532928986437211172"
        "5939809271601658335994136062611533585699853286138220152039879862096690644373597381912467559762468905851489"
        "2673412479568928929718232893457247262127861386316466071648934704323082510386282792977478573136129321405162"
        "2185585336779936235687303772101476787550948104817103791567242298967528341700080999540337364428768821898229"
        "1670402624460432979482187437452112851098323916745940612169548416465703108484268998978318929826467203540585"
        "8686840346516306860077301706233684932927097133159577424521557986736297607421875"},
    {0x1.fffffffffffffp-874, chars_format::fixed, 926,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000015878657653273751316576806252226363132068846750340814727"
        "5142428040452007367274654987482254818081870117383338129321565664579838666832418940100919065857972874422345"
        "1879618543203316671988272125223067171399706572276440304079759724193381288747194763824935119524937811702978"
        "5346824959137857859436465786914494524255722772632932143297869408646165020772565585954957146272258642810324"
        "4371170673559872471374607544202953575101896209634207583134484597935056683400161999080674728857537643796458"
        "3340805248920865958964374874904225702196647833491881224339096832931406216968537997956637859652934407081171"
        "737368069303261372015460341246736986585419426631915484904311597347259521484375"},
    {0x1.fffffffffffffp-873, chars_format::fixed, 925,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000031757315306547502633153612504452726264137693500681629455"
        "0284856080904014734549309974964509636163740234766676258643131329159677333664837880201838131715945748844690"
        "3759237086406633343976544250446134342799413144552880608159519448386762577494389527649870239049875623405957"
        "0693649918275715718872931573828989048511445545265864286595738817292330041545131171909914292544517285620648"
        "8742341347119744942749215088405907150203792419268415166268969195870113366800323998161349457715075287592916"
        "6681610497841731917928749749808451404393295666983762448678193665862812433937075995913275719305868814162343"
        "47473613860652274403092068249347397317083885326383096980862319469451904296875"},
    {0x1.fffffffffffffp-872, chars_format::fixed, 924,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000063514630613095005266307225008905452528275387001363258910"
        "0569712161808029469098619949929019272327480469533352517286262658319354667329675760403676263431891497689380"
        "7518474172813266687953088500892268685598826289105761216319038896773525154988779055299740478099751246811914"
        "1387299836551431437745863147657978097022891090531728573191477634584660083090262343819828585089034571241297"
        "7484682694239489885498430176811814300407584838536830332537938391740226733600647996322698915430150575185833"
        "3363220995683463835857499499616902808786591333967524897356387331725624867874151991826551438611737628324686"
        "9494722772130454880618413649869479463416777065276619396172463893890380859375"},
    {0x1.fffffffffffffp-871, chars_format::fixed, 923,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000127029261226190010532614450017810905056550774002726517820"
        "1139424323616058938197239899858038544654960939066705034572525316638709334659351520807352526863782995378761"
        "5036948345626533375906177001784537371197652578211522432638077793547050309977558110599480956199502493623828"
        "2774599673102862875491726295315956194045782181063457146382955269169320166180524687639657170178069142482595"
        "4969365388478979770996860353623628600815169677073660665075876783480453467201295992645397830860301150371666"
        "6726441991366927671714998999233805617573182667935049794712774663451249735748303983653102877223475256649373"
        "898944554426090976123682729973895892683355413055323879234492778778076171875"},
    {0x1.fffffffffffffp-870, chars_format::fixed, 922,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000254058522452380021065228900035621810113101548005453035640"
        "2278848647232117876394479799716077089309921878133410069145050633277418669318703041614705053727565990757523"
        "0073896691253066751812354003569074742395305156423044865276155587094100619955116221198961912399004987247656"
        "5549199346205725750983452590631912388091564362126914292765910538338640332361049375279314340356138284965190"
        "9938730776957959541993720707247257201630339354147321330151753566960906934402591985290795661720602300743333"
        "3452883982733855343429997998467611235146365335870099589425549326902499471496607967306205754446950513298747"
        "79788910885218195224736545994779178536671082611064775846898555755615234375"},
    {0x1.fffffffffffffp-869, chars_format::fixed, 921,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000508117044904760042130457800071243620226203096010906071280"
        "4557697294464235752788959599432154178619843756266820138290101266554837338637406083229410107455131981515046"
        "0147793382506133503624708007138149484790610312846089730552311174188201239910232442397923824798009974495313"
        "1098398692411451501966905181263824776183128724253828585531821076677280664722098750558628680712276569930381"
        "9877461553915919083987441414494514403260678708294642660303507133921813868805183970581591323441204601486666"
        "6905767965467710686859995996935222470292730671740199178851098653804998942993215934612411508893901026597495"
        "5957782177043639044947309198955835707334216522212955169379711151123046875"},
    {0x1.fffffffffffffp-868, chars_format::fixed, 920,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000001016234089809520084260915600142487240452406192021812142560"
        "9115394588928471505577919198864308357239687512533640276580202533109674677274812166458820214910263963030092"
        "0295586765012267007249416014276298969581220625692179461104622348376402479820464884795847649596019948990626"
        "2196797384822903003933810362527649552366257448507657171063642153354561329444197501117257361424553139860763"
        "9754923107831838167974882828989028806521357416589285320607014267843627737610367941163182646882409202973333"
        "3811535930935421373719991993870444940585461343480398357702197307609997885986431869224823017787802053194991"
        "191556435408727808989461839791167141466843304442591033875942230224609375"},
    {0x1.fffffffffffffp-867, chars_format::fixed, 919,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000002032468179619040168521831200284974480904812384043624285121"
        "8230789177856943011155838397728616714479375025067280553160405066219349354549624332917640429820527926060184"
        "0591173530024534014498832028552597939162441251384358922209244696752804959640929769591695299192039897981252"
        "4393594769645806007867620725055299104732514897015314342127284306709122658888395002234514722849106279721527"
        "9509846215663676335949765657978057613042714833178570641214028535687255475220735882326365293764818405946666"
        "7623071861870842747439983987740889881170922686960796715404394615219995771972863738449646035575604106389982"
        "38311287081745561797892367958233428293368660888518206775188446044921875"},
    {0x1.fffffffffffffp-866, chars_format::fixed, 918,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000004064936359238080337043662400569948961809624768087248570243"
        "6461578355713886022311676795457233428958750050134561106320810132438698709099248665835280859641055852120368"
        "1182347060049068028997664057105195878324882502768717844418489393505609919281859539183390598384079795962504"
        "8787189539291612015735241450110598209465029794030628684254568613418245317776790004469029445698212559443055"
        "9019692431327352671899531315956115226085429666357141282428057071374510950441471764652730587529636811893333"
        "5246143723741685494879967975481779762341845373921593430808789230439991543945727476899292071151208212779964"
        "7662257416349112359578473591646685658673732177703641355037689208984375"},
    {0x1.fffffffffffffp-865, chars_format::fixed, 917,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000008129872718476160674087324801139897923619249536174497140487"
        "2923156711427772044623353590914466857917500100269122212641620264877397418198497331670561719282111704240736"
        "2364694120098136057995328114210391756649765005537435688836978787011219838563719078366781196768159591925009"
        "7574379078583224031470482900221196418930059588061257368509137226836490635553580008938058891396425118886111"
        "8039384862654705343799062631912230452170859332714282564856114142749021900882943529305461175059273623786667"
        "0492287447483370989759935950963559524683690747843186861617578460879983087891454953798584142302416425559929"
        "532451483269822471915694718329337131734746435540728271007537841796875"},
    {0x1.fffffffffffffp-864, chars_format::fixed, 916,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000016259745436952321348174649602279795847238499072348994280974"
        "5846313422855544089246707181828933715835000200538244425283240529754794836396994663341123438564223408481472"
        "4729388240196272115990656228420783513299530011074871377673957574022439677127438156733562393536319183850019"
        "5148758157166448062940965800442392837860119176122514737018274453672981271107160017876117782792850237772223"
        "6078769725309410687598125263824460904341718665428565129712228285498043801765887058610922350118547247573334"
        "0984574894966741979519871901927119049367381495686373723235156921759966175782909907597168284604832851119859"
        "06490296653964494383138943665867426346949287108145654201507568359375"},
    {0x1.fffffffffffffp-863, chars_format::fixed, 915,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000032519490873904642696349299204559591694476998144697988561949"
        "1692626845711088178493414363657867431670000401076488850566481059509589672793989326682246877128446816962944"
        "9458776480392544231981312456841567026599060022149742755347915148044879354254876313467124787072638367700039"
        "0297516314332896125881931600884785675720238352245029474036548907345962542214320035752235565585700475544447"
        "2157539450618821375196250527648921808683437330857130259424456570996087603531774117221844700237094495146668"
        "1969149789933483959039743803854238098734762991372747446470313843519932351565819815194336569209665702239718"
        "1298059330792898876627788733173485269389857421629130840301513671875"},
    {0x1.fffffffffffffp-862, chars_format::fixed, 914,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000065038981747809285392698598409119183388953996289395977123898"
        "3385253691422176356986828727315734863340000802152977701132962119019179345587978653364493754256893633925889"
        "8917552960785088463962624913683134053198120044299485510695830296089758708509752626934249574145276735400078"
        "0595032628665792251763863201769571351440476704490058948073097814691925084428640071504471131171400951088894"
        "4315078901237642750392501055297843617366874661714260518848913141992175207063548234443689400474188990293336"
        "3938299579866967918079487607708476197469525982745494892940627687039864703131639630388673138419331404479436"
        "259611866158579775325557746634697053877971484325826168060302734375"},
    {0x1.fffffffffffffp-861, chars_format::fixed, 913,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000130077963495618570785397196818238366777907992578791954247796"
        "6770507382844352713973657454631469726680001604305955402265924238038358691175957306728987508513787267851779"
        "7835105921570176927925249827366268106396240088598971021391660592179517417019505253868499148290553470800156"
        "1190065257331584503527726403539142702880953408980117896146195629383850168857280143008942262342801902177788"
        "8630157802475285500785002110595687234733749323428521037697826283984350414127096468887378800948377980586672"
        "7876599159733935836158975215416952394939051965490989785881255374079729406263279260777346276838662808958872"
        "51922373231715955065111549326939410775594296865165233612060546875"},
    {0x1.fffffffffffffp-860, chars_format::fixed, 912,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000260155926991237141570794393636476733555815985157583908495593"
        "3541014765688705427947314909262939453360003208611910804531848476076717382351914613457975017027574535703559"
        "5670211843140353855850499654732536212792480177197942042783321184359034834039010507736998296581106941600312"
        "2380130514663169007055452807078285405761906817960235792292391258767700337714560286017884524685603804355577"
        "7260315604950571001570004221191374469467498646857042075395652567968700828254192937774757601896755961173345"
        "5753198319467871672317950430833904789878103930981979571762510748159458812526558521554692553677325617917745"
        "0384474646343191013022309865387882155118859373033046722412109375"},
    {0x1.fffffffffffffp-859, chars_format::fixed, 911,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000520311853982474283141588787272953467111631970315167816991186"
        "7082029531377410855894629818525878906720006417223821609063696952153434764703829226915950034055149071407119"
        "1340423686280707711700999309465072425584960354395884085566642368718069668078021015473996593162213883200624"
        "4760261029326338014110905614156570811523813635920471584584782517535400675429120572035769049371207608711155"
        "4520631209901142003140008442382748938934997293714084150791305135937401656508385875549515203793511922346691"
        "1506396638935743344635900861667809579756207861963959143525021496318917625053117043109385107354651235835490"
        "076894929268638202604461973077576431023771874606609344482421875"},
    {0x1.fffffffffffffp-858, chars_format::fixed, 910,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000001040623707964948566283177574545906934223263940630335633982373"
        "4164059062754821711789259637051757813440012834447643218127393904306869529407658453831900068110298142814238"
        "2680847372561415423401998618930144851169920708791768171133284737436139336156042030947993186324427766401248"
        "9520522058652676028221811228313141623047627271840943169169565035070801350858241144071538098742415217422310"
        "9041262419802284006280016884765497877869994587428168301582610271874803313016771751099030407587023844693382"
        "3012793277871486689271801723335619159512415723927918287050042992637835250106234086218770214709302471670980"
        "15378985853727640520892394615515286204754374921321868896484375"},
    {0x1.fffffffffffffp-857, chars_format::fixed, 909,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000002081247415929897132566355149091813868446527881260671267964746"
        "8328118125509643423578519274103515626880025668895286436254787808613739058815316907663800136220596285628476"
        "5361694745122830846803997237860289702339841417583536342266569474872278672312084061895986372648855532802497"
        "9041044117305352056443622456626283246095254543681886338339130070141602701716482288143076197484830434844621"
        "8082524839604568012560033769530995755739989174856336603165220543749606626033543502198060815174047689386764"
        "6025586555742973378543603446671238319024831447855836574100085985275670500212468172437540429418604943341960"
        "3075797170745528104178478923103057240950874984264373779296875"},
    {0x1.fffffffffffffp-856, chars_format::fixed, 908,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000004162494831859794265132710298183627736893055762521342535929493"
        "6656236251019286847157038548207031253760051337790572872509575617227478117630633815327600272441192571256953"
        "0723389490245661693607994475720579404679682835167072684533138949744557344624168123791972745297711065604995"
        "8082088234610704112887244913252566492190509087363772676678260140283205403432964576286152394969660869689243"
        "6165049679209136025120067539061991511479978349712673206330441087499213252067087004396121630348095378773529"
        "2051173111485946757087206893342476638049662895711673148200171970551341000424936344875080858837209886683920"
        "615159434149105620835695784620611448190174996852874755859375"},
    {0x1.fffffffffffffp-855, chars_format::fixed, 907,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000008324989663719588530265420596367255473786111525042685071858987"
        "3312472502038573694314077096414062507520102675581145745019151234454956235261267630655200544882385142513906"
        "1446778980491323387215988951441158809359365670334145369066277899489114689248336247583945490595422131209991"
        "6164176469221408225774489826505132984381018174727545353356520280566410806865929152572304789939321739378487"
        "2330099358418272050240135078123983022959956699425346412660882174998426504134174008792243260696190757547058"
        "4102346222971893514174413786684953276099325791423346296400343941102682000849872689750161717674419773367841"
        "23031886829821124167139156924122289638034999370574951171875"},
    {0x1.fffffffffffffp-854, chars_format::fixed, 906,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000016649979327439177060530841192734510947572223050085370143717974"
        "6624945004077147388628154192828125015040205351162291490038302468909912470522535261310401089764770285027812"
        "2893557960982646774431977902882317618718731340668290738132555798978229378496672495167890981190844262419983"
        "2328352938442816451548979653010265968762036349455090706713040561132821613731858305144609579878643478756974"
        "4660198716836544100480270156247966045919913398850692825321764349996853008268348017584486521392381515094116"
        "8204692445943787028348827573369906552198651582846692592800687882205364001699745379500323435348839546735682"
        "4606377365964224833427831384824457927606999874114990234375"},
    {0x1.fffffffffffffp-853, chars_format::fixed, 905,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000033299958654878354121061682385469021895144446100170740287435949"
        "3249890008154294777256308385656250030080410702324582980076604937819824941045070522620802179529540570055624"
        "5787115921965293548863955805764635237437462681336581476265111597956458756993344990335781962381688524839966"
        "4656705876885632903097959306020531937524072698910181413426081122265643227463716610289219159757286957513948"
        "9320397433673088200960540312495932091839826797701385650643528699993706016536696035168973042784763030188233"
        "6409384891887574056697655146739813104397303165693385185601375764410728003399490759000646870697679093471364"
        "921275473192844966685566276964891585521399974822998046875"},
    {0x1.fffffffffffffp-852, chars_format::fixed, 904,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000066599917309756708242123364770938043790288892200341480574871898"
        "6499780016308589554512616771312500060160821404649165960153209875639649882090141045241604359059081140111249"
        "1574231843930587097727911611529270474874925362673162952530223195912917513986689980671563924763377049679932"
        "9313411753771265806195918612041063875048145397820362826852162244531286454927433220578438319514573915027897"
        "8640794867346176401921080624991864183679653595402771301287057399987412033073392070337946085569526060376467"
        "2818769783775148113395310293479626208794606331386770371202751528821456006798981518001293741395358186942729"
        "84255094638568993337113255392978317104279994964599609375"},
    {0x1.fffffffffffffp-851, chars_format::fixed, 903,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000133199834619513416484246729541876087580577784400682961149743797"
        "2999560032617179109025233542625000120321642809298331920306419751279299764180282090483208718118162280222498"
        "3148463687861174195455823223058540949749850725346325905060446391825835027973379961343127849526754099359865"
        "8626823507542531612391837224082127750096290795640725653704324489062572909854866441156876639029147830055795"
        "7281589734692352803842161249983728367359307190805542602574114799974824066146784140675892171139052120752934"
        "5637539567550296226790620586959252417589212662773540742405503057642912013597963036002587482790716373885459"
        "6851018927713798667422651078595663420855998992919921875"},
    {0x1.fffffffffffffp-850, chars_format::fixed, 902,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000266399669239026832968493459083752175161155568801365922299487594"
        "5999120065234358218050467085250000240643285618596663840612839502558599528360564180966417436236324560444996"
        "6296927375722348390911646446117081899499701450692651810120892783651670055946759922686255699053508198719731"
        "7253647015085063224783674448164255500192581591281451307408648978125145819709732882313753278058295660111591"
        "4563179469384705607684322499967456734718614381611085205148229599949648132293568281351784342278104241505869"
        "1275079135100592453581241173918504835178425325547081484811006115285824027195926072005174965581432747770919"
        "370203785542759733484530215719132684171199798583984375"},
    {0x1.fffffffffffffp-849, chars_format::fixed, 901,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000532799338478053665936986918167504350322311137602731844598975189"
        "1998240130468716436100934170500000481286571237193327681225679005117199056721128361932834872472649120889993"
        "2593854751444696781823292892234163798999402901385303620241785567303340111893519845372511398107016397439463"
        "4507294030170126449567348896328511000385163182562902614817297956250291639419465764627506556116591320223182"
        "9126358938769411215368644999934913469437228763222170410296459199899296264587136562703568684556208483011738"
        "2550158270201184907162482347837009670356850651094162969622012230571648054391852144010349931162865495541838"
        "74040757108551946696906043143826536834239959716796875"},
    {0x1.fffffffffffffp-848, chars_format::fixed, 900,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000001065598676956107331873973836335008700644622275205463689197950378"
        "3996480260937432872201868341000000962573142474386655362451358010234398113442256723865669744945298241779986"
        "5187709502889393563646585784468327597998805802770607240483571134606680223787039690745022796214032794878926"
        "9014588060340252899134697792657022000770326365125805229634595912500583278838931529255013112233182640446365"
        "8252717877538822430737289999869826938874457526444340820592918399798592529174273125407137369112416966023476"
        "5100316540402369814324964695674019340713701302188325939244024461143296108783704288020699862325730991083677"
        "4808151421710389339381208628765307366847991943359375"},
    {0x1.fffffffffffffp-847, chars_format::fixed, 899,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000002131197353912214663747947672670017401289244550410927378395900756"
        "7992960521874865744403736682000001925146284948773310724902716020468796226884513447731339489890596483559973"
        "0375419005778787127293171568936655195997611605541214480967142269213360447574079381490045592428065589757853"
        "8029176120680505798269395585314044001540652730251610459269191825001166557677863058510026224466365280892731"
        "6505435755077644861474579999739653877748915052888681641185836799597185058348546250814274738224833932046953"
        "0200633080804739628649929391348038681427402604376651878488048922286592217567408576041399724651461982167354"
        "961630284342077867876241725753061473369598388671875"},
    {0x1.fffffffffffffp-846, chars_format::fixed, 898,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000004262394707824429327495895345340034802578489100821854756791801513"
        "5985921043749731488807473364000003850292569897546621449805432040937592453769026895462678979781192967119946"
        "0750838011557574254586343137873310391995223211082428961934284538426720895148158762980091184856131179515707"
        "6058352241361011596538791170628088003081305460503220918538383650002333115355726117020052448932730561785463"
        "3010871510155289722949159999479307755497830105777363282371673599194370116697092501628549476449667864093906"
        "0401266161609479257299858782696077362854805208753303756976097844573184435134817152082799449302923964334709"
        "92326056868415573575248345150612294673919677734375"},
    {0x1.fffffffffffffp-845, chars_format::fixed, 897,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000008524789415648858654991790690680069605156978201643709513583603027"
        "1971842087499462977614946728000007700585139795093242899610864081875184907538053790925357959562385934239892"
        "1501676023115148509172686275746620783990446422164857923868569076853441790296317525960182369712262359031415"
        "2116704482722023193077582341256176006162610921006441837076767300004666230711452234040104897865461123570926"
        "6021743020310579445898319998958615510995660211554726564743347198388740233394185003257098952899335728187812"
        "0802532323218958514599717565392154725709610417506607513952195689146368870269634304165598898605847928669419"
        "8465211373683114715049669030122458934783935546875"},
    {0x1.fffffffffffffp-844, chars_format::fixed, 896,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000017049578831297717309983581381360139210313956403287419027167206054"
        "3943684174998925955229893456000015401170279590186485799221728163750369815076107581850715919124771868479784"
        "3003352046230297018345372551493241567980892844329715847737138153706883580592635051920364739424524718062830"
        "4233408965444046386155164682512352012325221842012883674153534600009332461422904468080209795730922247141853"
        "2043486040621158891796639997917231021991320423109453129486694396777480466788370006514197905798671456375624"
        "1605064646437917029199435130784309451419220835013215027904391378292737740539268608331197797211695857338839"
        "693042274736622943009933806024491786956787109375"},
    {0x1.fffffffffffffp-843, chars_format::fixed, 895,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000034099157662595434619967162762720278420627912806574838054334412108"
        "7887368349997851910459786912000030802340559180372971598443456327500739630152215163701431838249543736959568"
        "6006704092460594036690745102986483135961785688659431695474276307413767161185270103840729478849049436125660"
        "8466817930888092772310329365024704024650443684025767348307069200018664922845808936160419591461844494283706"
        "4086972081242317783593279995834462043982640846218906258973388793554960933576740013028395811597342912751248"
        "3210129292875834058398870261568618902838441670026430055808782756585475481078537216662395594423391714677679"
        "38608454947324588601986761204898357391357421875"},
    {0x1.fffffffffffffp-842, chars_format::fixed, 894,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000068198315325190869239934325525440556841255825613149676108668824217"
        "5774736699995703820919573824000061604681118360745943196886912655001479260304430327402863676499087473919137"
        "2013408184921188073381490205972966271923571377318863390948552614827534322370540207681458957698098872251321"
        "6933635861776185544620658730049408049300887368051534696614138400037329845691617872320839182923688988567412"
        "8173944162484635567186559991668924087965281692437812517946777587109921867153480026056791623194685825502496"
        "6420258585751668116797740523137237805676883340052860111617565513170950962157074433324791188846783429355358"
        "7721690989464917720397352240979671478271484375"},
    {0x1.fffffffffffffp-841, chars_format::fixed, 893,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000136396630650381738479868651050881113682511651226299352217337648435"
        "1549473399991407641839147648000123209362236721491886393773825310002958520608860654805727352998174947838274"
        "4026816369842376146762980411945932543847142754637726781897105229655068644741080415362917915396197744502643"
        "3867271723552371089241317460098816098601774736103069393228276800074659691383235744641678365847377977134825"
        "6347888324969271134373119983337848175930563384875625035893555174219843734306960052113583246389371651004993"
        "2840517171503336233595481046274475611353766680105720223235131026341901924314148866649582377693566858710717"
        "544338197892983544079470448195934295654296875"},
    {0x1.fffffffffffffp-840, chars_format::fixed, 892,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000272793261300763476959737302101762227365023302452598704434675296870"
        "3098946799982815283678295296000246418724473442983772787547650620005917041217721309611454705996349895676548"
        "8053632739684752293525960823891865087694285509275453563794210459310137289482160830725835830792395489005286"
        "7734543447104742178482634920197632197203549472206138786456553600149319382766471489283356731694755954269651"
        "2695776649938542268746239966675696351861126769751250071787110348439687468613920104227166492778743302009986"
        "5681034343006672467190962092548951222707533360211440446470262052683803848628297733299164755387133717421435"
        "08867639578596708815894089639186859130859375"},
    {0x1.fffffffffffffp-839, chars_format::fixed, 891,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000545586522601526953919474604203524454730046604905197408869350593740"
        "6197893599965630567356590592000492837448946885967545575095301240011834082435442619222909411992699791353097"
        "6107265479369504587051921647783730175388571018550907127588420918620274578964321661451671661584790978010573"
        "5469086894209484356965269840395264394407098944412277572913107200298638765532942978566713463389511908539302"
        "5391553299877084537492479933351392703722253539502500143574220696879374937227840208454332985557486604019973"
        "1362068686013344934381924185097902445415066720422880892940524105367607697256595466598329510774267434842870"
        "1773527915719341763178817927837371826171875"},
    {0x1.fffffffffffffp-838, chars_format::fixed, 890,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000001091173045203053907838949208407048909460093209810394817738701187481"
        "2395787199931261134713181184000985674897893771935091150190602480023668164870885238445818823985399582706195"
        "2214530958739009174103843295567460350777142037101814255176841837240549157928643322903343323169581956021147"
        "0938173788418968713930539680790528788814197888824555145826214400597277531065885957133426926779023817078605"
        "0783106599754169074984959866702785407444507079005000287148441393758749874455680416908665971114973208039946"
        "2724137372026689868763848370195804890830133440845761785881048210735215394513190933196659021548534869685740"
        "354705583143868352635763585567474365234375"},
    {0x1.fffffffffffffp-837, chars_format::fixed, 889,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000002182346090406107815677898416814097818920186419620789635477402374962"
        "4791574399862522269426362368001971349795787543870182300381204960047336329741770476891637647970799165412390"
        "4429061917478018348207686591134920701554284074203628510353683674481098315857286645806686646339163912042294"
        "1876347576837937427861079361581057577628395777649110291652428801194555062131771914266853853558047634157210"
        "1566213199508338149969919733405570814889014158010000574296882787517499748911360833817331942229946416079892"
        "5448274744053379737527696740391609781660266881691523571762096421470430789026381866393318043097069739371480"
        "70941116628773670527152717113494873046875"},
    {0x1.fffffffffffffp-836, chars_format::fixed, 888,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000004364692180812215631355796833628195637840372839241579270954804749924"
        "9583148799725044538852724736003942699591575087740364600762409920094672659483540953783275295941598330824780"
        "8858123834956036696415373182269841403108568148407257020707367348962196631714573291613373292678327824084588"
        "3752695153675874855722158723162115155256791555298220583304857602389110124263543828533707707116095268314420"
        "3132426399016676299939839466811141629778028316020001148593765575034999497822721667634663884459892832159785"
        "0896549488106759475055393480783219563320533763383047143524192842940861578052763732786636086194139478742961"
        "4188223325754734105430543422698974609375"},
    {0x1.fffffffffffffp-835, chars_format::fixed, 887,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000008729384361624431262711593667256391275680745678483158541909609499849"
        "9166297599450089077705449472007885399183150175480729201524819840189345318967081907566550591883196661649561"
        "7716247669912073392830746364539682806217136296814514041414734697924393263429146583226746585356655648169176"
        "7505390307351749711444317446324230310513583110596441166609715204778220248527087657067415414232190536628840"
        "6264852798033352599879678933622283259556056632040002297187531150069998995645443335269327768919785664319570"
        "1793098976213518950110786961566439126641067526766094287048385685881723156105527465573272172388278957485922"
        "837644665150946821086108684539794921875"},
    {0x1.fffffffffffffp-834, chars_format::fixed, 886,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000017458768723248862525423187334512782551361491356966317083819218999699"
        "8332595198900178155410898944015770798366300350961458403049639680378690637934163815133101183766393323299123"
        "5432495339824146785661492729079365612434272593629028082829469395848786526858293166453493170713311296338353"
        "5010780614703499422888634892648460621027166221192882333219430409556440497054175314134830828464381073257681"
        "2529705596066705199759357867244566519112113264080004594375062300139997991290886670538655537839571328639140"
        "3586197952427037900221573923132878253282135053532188574096771371763446312211054931146544344776557914971845"
        "67528933030189364217221736907958984375"},
    {0x1.fffffffffffffp-833, chars_format::fixed, 885,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000034917537446497725050846374669025565102722982713932634167638437999399"
        "6665190397800356310821797888031541596732600701922916806099279360757381275868327630266202367532786646598247"
        "0864990679648293571322985458158731224868545187258056165658938791697573053716586332906986341426622592676707"
        "0021561229406998845777269785296921242054332442385764666438860819112880994108350628269661656928762146515362"
        "5059411192133410399518715734489133038224226528160009188750124600279995982581773341077311075679142657278280"
        "7172395904854075800443147846265756506564270107064377148193542743526892624422109862293088689553115829943691"
        "3505786606037872843444347381591796875"},
    {0x1.fffffffffffffp-832, chars_format::fixed, 884,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000069835074892995450101692749338051130205445965427865268335276875998799"
        "3330380795600712621643595776063083193465201403845833612198558721514762551736655260532404735065573293196494"
        "1729981359296587142645970916317462449737090374516112331317877583395146107433172665813972682853245185353414"
        "0043122458813997691554539570593842484108664884771529332877721638225761988216701256539323313857524293030725"
        "0118822384266820799037431468978266076448453056320018377500249200559991965163546682154622151358285314556561"
        "4344791809708151600886295692531513013128540214128754296387085487053785248844219724586177379106231659887382"
        "701157321207574568688869476318359375"},
    {0x1.fffffffffffffp-831, chars_format::fixed, 883,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000139670149785990900203385498676102260410891930855730536670553751997598"
        "6660761591201425243287191552126166386930402807691667224397117443029525103473310521064809470131146586392988"
        "3459962718593174285291941832634924899474180749032224662635755166790292214866345331627945365706490370706828"
        "0086244917627995383109079141187684968217329769543058665755443276451523976433402513078646627715048586061450"
        "0237644768533641598074862937956532152896906112640036755000498401119983930327093364309244302716570629113122"
        "8689583619416303201772591385063026026257080428257508592774170974107570497688439449172354758212463319774765"
        "40231464241514913737773895263671875"},
    {0x1.fffffffffffffp-830, chars_format::fixed, 882,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000279340299571981800406770997352204520821783861711461073341107503995197"
        "3321523182402850486574383104252332773860805615383334448794234886059050206946621042129618940262293172785976"
        "6919925437186348570583883665269849798948361498064449325271510333580584429732690663255890731412980741413656"
        "0172489835255990766218158282375369936434659539086117331510886552903047952866805026157293255430097172122900"
        "0475289537067283196149725875913064305793812225280073510000996802239967860654186728618488605433141258226245"
        "7379167238832606403545182770126052052514160856515017185548341948215140995376878898344709516424926639549530"
        "8046292848302982747554779052734375"},
    {0x1.fffffffffffffp-829, chars_format::fixed, 881,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000558680599143963600813541994704409041643567723422922146682215007990394"
        "6643046364805700973148766208504665547721611230766668897588469772118100413893242084259237880524586345571953"
        "3839850874372697141167767330539699597896722996128898650543020667161168859465381326511781462825961482827312"
        "0344979670511981532436316564750739872869319078172234663021773105806095905733610052314586510860194344245800"
        "0950579074134566392299451751826128611587624450560147020001993604479935721308373457236977210866282516452491"
        "4758334477665212807090365540252104105028321713030034371096683896430281990753757796689419032849853279099061"
        "609258569660596549510955810546875"},
    {0x1.fffffffffffffp-828, chars_format::fixed, 880,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000001117361198287927201627083989408818083287135446845844293364430015980789"
        "3286092729611401946297532417009331095443222461533337795176939544236200827786484168518475761049172691143906"
        "7679701748745394282335534661079399195793445992257797301086041334322337718930762653023562925651922965654624"
        "0689959341023963064872633129501479745738638156344469326043546211612191811467220104629173021720388688491600"
        "1901158148269132784598903503652257223175248901120294040003987208959871442616746914473954421732565032904982"
        "9516668955330425614180731080504208210056643426060068742193367792860563981507515593378838065699706558198123"
        "21851713932119309902191162109375"},
    {0x1.fffffffffffffp-827, chars_format::fixed, 879,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000002234722396575854403254167978817636166574270893691688586728860031961578"
        "6572185459222803892595064834018662190886444923066675590353879088472401655572968337036951522098345382287813"
        "5359403497490788564671069322158798391586891984515594602172082668644675437861525306047125851303845931309248"
        "1379918682047926129745266259002959491477276312688938652087092423224383622934440209258346043440777376983200"
        "3802316296538265569197807007304514446350497802240588080007974417919742885233493828947908843465130065809965"
        "9033337910660851228361462161008416420113286852120137484386735585721127963015031186757676131399413116396246"
        "4370342786423861980438232421875"},
    {0x1.fffffffffffffp-826, chars_format::fixed, 878,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000004469444793151708806508335957635272333148541787383377173457720063923157"
        "3144370918445607785190129668037324381772889846133351180707758176944803311145936674073903044196690764575627"
        "0718806994981577129342138644317596783173783969031189204344165337289350875723050612094251702607691862618496"
        "2759837364095852259490532518005918982954552625377877304174184846448767245868880418516692086881554753966400"
        "7604632593076531138395614014609028892700995604481176160015948835839485770466987657895817686930260131619931"
        "8066675821321702456722924322016832840226573704240274968773471171442255926030062373515352262798826232792492"
        "874068557284772396087646484375"},
    {0x1.fffffffffffffp-825, chars_format::fixed, 877,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000008938889586303417613016671915270544666297083574766754346915440127846314"
        "6288741836891215570380259336074648763545779692266702361415516353889606622291873348147806088393381529151254"
        "1437613989963154258684277288635193566347567938062378408688330674578701751446101224188503405215383725236992"
        "5519674728191704518981065036011837965909105250755754608348369692897534491737760837033384173763109507932801"
        "5209265186153062276791228029218057785401991208962352320031897671678971540933975315791635373860520263239863"
        "6133351642643404913445848644033665680453147408480549937546942342884511852060124747030704525597652465584985"
        "74813711456954479217529296875"},
    {0x1.fffffffffffffp-824, chars_format::fixed, 876,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000017877779172606835226033343830541089332594167149533508693830880255692629"
        "2577483673782431140760518672149297527091559384533404722831032707779213244583746696295612176786763058302508"
        "2875227979926308517368554577270387132695135876124756817376661349157403502892202448377006810430767450473985"
        "1039349456383409037962130072023675931818210501511509216696739385795068983475521674066768347526219015865603"
        "0418530372306124553582456058436115570803982417924704640063795343357943081867950631583270747721040526479727"
        "2266703285286809826891697288067331360906294816961099875093884685769023704120249494061409051195304931169971"
        "4962742291390895843505859375"},
    {0x1.fffffffffffffp-823, chars_format::fixed, 875,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000035755558345213670452066687661082178665188334299067017387661760511385258"
        "5154967347564862281521037344298595054183118769066809445662065415558426489167493392591224353573526116605016"
        "5750455959852617034737109154540774265390271752249513634753322698314807005784404896754013620861534900947970"
        "2078698912766818075924260144047351863636421003023018433393478771590137966951043348133536695052438031731206"
        "0837060744612249107164912116872231141607964835849409280127590686715886163735901263166541495442081052959454"
        "4533406570573619653783394576134662721812589633922199750187769371538047408240498988122818102390609862339942"
        "992548458278179168701171875"},
    {0x1.fffffffffffffp-822, chars_format::fixed, 874,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000071511116690427340904133375322164357330376668598134034775323521022770517"
        "0309934695129724563042074688597190108366237538133618891324130831116852978334986785182448707147052233210033"
        "1500911919705234069474218309081548530780543504499027269506645396629614011568809793508027241723069801895940"
        "4157397825533636151848520288094703727272842006046036866786957543180275933902086696267073390104876063462412"
        "1674121489224498214329824233744462283215929671698818560255181373431772327471802526333082990884162105918908"
        "9066813141147239307566789152269325443625179267844399500375538743076094816480997976245636204781219724679885"
        "98509691655635833740234375"},
    {0x1.fffffffffffffp-821, chars_format::fixed, 873,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000143022233380854681808266750644328714660753337196268069550647042045541034"
        "0619869390259449126084149377194380216732475076267237782648261662233705956669973570364897414294104466420066"
        "3001823839410468138948436618163097061561087008998054539013290793259228023137619587016054483446139603791880"
        "8314795651067272303697040576189407454545684012092073733573915086360551867804173392534146780209752126924824"
        "3348242978448996428659648467488924566431859343397637120510362746863544654943605052666165981768324211837817"
        "8133626282294478615133578304538650887250358535688799000751077486152189632961995952491272409562439449359771"
        "9701938331127166748046875"},
    {0x1.fffffffffffffp-820, chars_format::fixed, 872,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000286044466761709363616533501288657429321506674392536139101294084091082068"
        "1239738780518898252168298754388760433464950152534475565296523324467411913339947140729794828588208932840132"
        "6003647678820936277896873236326194123122174017996109078026581586518456046275239174032108966892279207583761"
        "6629591302134544607394081152378814909091368024184147467147830172721103735608346785068293560419504253849648"
        "6696485956897992857319296934977849132863718686795274241020725493727089309887210105332331963536648423675635"
        "6267252564588957230267156609077301774500717071377598001502154972304379265923991904982544819124878898719543"
        "940387666225433349609375"},
    {0x1.fffffffffffffp-819, chars_format::fixed, 871,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000572088933523418727233067002577314858643013348785072278202588168182164136"
        "2479477561037796504336597508777520866929900305068951130593046648934823826679894281459589657176417865680265"
        "2007295357641872555793746472652388246244348035992218156053163173036912092550478348064217933784558415167523"
        "3259182604269089214788162304757629818182736048368294934295660345442207471216693570136587120839008507699297"
        "3392971913795985714638593869955698265727437373590548482041450987454178619774420210664663927073296847351271"
        "2534505129177914460534313218154603549001434142755196003004309944608758531847983809965089638249757797439087"
        "88077533245086669921875"},
    {0x1.fffffffffffffp-818, chars_format::fixed, 870,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000001144177867046837454466134005154629717286026697570144556405176336364328272"
        "4958955122075593008673195017555041733859800610137902261186093297869647653359788562919179314352835731360530"
        "4014590715283745111587492945304776492488696071984436312106326346073824185100956696128435867569116830335046"
        "6518365208538178429576324609515259636365472096736589868591320690884414942433387140273174241678017015398594"
        "6785943827591971429277187739911396531454874747181096964082901974908357239548840421329327854146593694702542"
        "5069010258355828921068626436309207098002868285510392006008619889217517063695967619930179276499515594878175"
        "7615506649017333984375"},
    {0x1.fffffffffffffp-817, chars_format::fixed, 869,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000002288355734093674908932268010309259434572053395140289112810352672728656544"
        "9917910244151186017346390035110083467719601220275804522372186595739295306719577125838358628705671462721060"
        "8029181430567490223174985890609552984977392143968872624212652692147648370201913392256871735138233660670093"
        "3036730417076356859152649219030519272730944193473179737182641381768829884866774280546348483356034030797189"
        "3571887655183942858554375479822793062909749494362193928165803949816714479097680842658655708293187389405085"
        "0138020516711657842137252872618414196005736571020784012017239778435034127391935239860358552999031189756351"
        "523101329803466796875"},
    {0x1.fffffffffffffp-816, chars_format::fixed, 868,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000004576711468187349817864536020618518869144106790280578225620705345457313089"
        "9835820488302372034692780070220166935439202440551609044744373191478590613439154251676717257411342925442121"
        "6058362861134980446349971781219105969954784287937745248425305384295296740403826784513743470276467321340186"
        "6073460834152713718305298438061038545461888386946359474365282763537659769733548561092696966712068061594378"
        "7143775310367885717108750959645586125819498988724387856331607899633428958195361685317311416586374778810170"
        "0276041033423315684274505745236828392011473142041568024034479556870068254783870479720717105998062379512703"
        "04620265960693359375"},
    {0x1.fffffffffffffp-815, chars_format::fixed, 867,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000009153422936374699635729072041237037738288213580561156451241410690914626179"
        "9671640976604744069385560140440333870878404881103218089488746382957181226878308503353434514822685850884243"
        "2116725722269960892699943562438211939909568575875490496850610768590593480807653569027486940552934642680373"
        "2146921668305427436610596876122077090923776773892718948730565527075319539467097122185393933424136123188757"
        "4287550620735771434217501919291172251638997977448775712663215799266857916390723370634622833172749557620340"
        "0552082066846631368549011490473656784022946284083136048068959113740136509567740959441434211996124759025406"
        "0924053192138671875"},
    {0x1.fffffffffffffp-814, chars_format::fixed, 866,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000018306845872749399271458144082474075476576427161122312902482821381829252359"
        "9343281953209488138771120280880667741756809762206436178977492765914362453756617006706869029645371701768486"
        "4233451444539921785399887124876423879819137151750980993701221537181186961615307138054973881105869285360746"
        "4293843336610854873221193752244154181847553547785437897461131054150639078934194244370787866848272246377514"
        "8575101241471542868435003838582344503277995954897551425326431598533715832781446741269245666345499115240680"
        "1104164133693262737098022980947313568045892568166272096137918227480273019135481918882868423992249518050812"
        "184810638427734375"},
    {0x1.fffffffffffffp-813, chars_format::fixed, 865,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000036613691745498798542916288164948150953152854322244625804965642763658504719"
        "8686563906418976277542240561761335483513619524412872357954985531828724907513234013413738059290743403536972"
        "8466902889079843570799774249752847759638274303501961987402443074362373923230614276109947762211738570721492"
        "8587686673221709746442387504488308363695107095570875794922262108301278157868388488741575733696544492755029"
        "7150202482943085736870007677164689006555991909795102850652863197067431665562893482538491332690998230481360"
        "2208328267386525474196045961894627136091785136332544192275836454960546038270963837765736847984499036101624"
        "36962127685546875"},
    {0x1.fffffffffffffp-812, chars_format::fixed, 864,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000073227383490997597085832576329896301906305708644489251609931285527317009439"
        "7373127812837952555084481123522670967027239048825744715909971063657449815026468026827476118581486807073945"
        "6933805778159687141599548499505695519276548607003923974804886148724747846461228552219895524423477141442985"
        "7175373346443419492884775008976616727390214191141751589844524216602556315736776977483151467393088985510059"
        "4300404965886171473740015354329378013111983819590205701305726394134863331125786965076982665381996460962720"
        "4416656534773050948392091923789254272183570272665088384551672909921092076541927675531473695968998072203248"
        "7392425537109375"},
    {0x1.fffffffffffffp-811, chars_format::fixed, 863,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000146454766981995194171665152659792603812611417288978503219862571054634018879"
        "4746255625675905110168962247045341934054478097651489431819942127314899630052936053654952237162973614147891"
        "3867611556319374283199096999011391038553097214007847949609772297449495692922457104439791048846954282885971"
        "4350746692886838985769550017953233454780428382283503179689048433205112631473553954966302934786177971020118"
        "8600809931772342947480030708658756026223967639180411402611452788269726662251573930153965330763992921925440"
        "8833313069546101896784183847578508544367140545330176769103345819842184153083855351062947391937996144406497"
        "478485107421875"},
    {0x1.fffffffffffffp-810, chars_format::fixed, 862,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000292909533963990388343330305319585207625222834577957006439725142109268037758"
        "9492511251351810220337924494090683868108956195302978863639884254629799260105872107309904474325947228295782"
        "7735223112638748566398193998022782077106194428015695899219544594898991385844914208879582097693908565771942"
        "8701493385773677971539100035906466909560856764567006359378096866410225262947107909932605869572355942040237"
        "7201619863544685894960061417317512052447935278360822805222905576539453324503147860307930661527985843850881"
        "7666626139092203793568367695157017088734281090660353538206691639684368306167710702125894783875992288812994"
        "95697021484375"},
    {0x1.fffffffffffffp-809, chars_format::fixed, 861,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000585819067927980776686660610639170415250445669155914012879450284218536075517"
        "8985022502703620440675848988181367736217912390605957727279768509259598520211744214619808948651894456591565"
        "5470446225277497132796387996045564154212388856031391798439089189797982771689828417759164195387817131543885"
        "7402986771547355943078200071812933819121713529134012718756193732820450525894215819865211739144711884080475"
        "4403239727089371789920122834635024104895870556721645610445811153078906649006295720615861323055971687701763"
        "5333252278184407587136735390314034177468562181320707076413383279368736612335421404251789567751984577625989"
        "9139404296875"},
    {0x1.fffffffffffffp-808, chars_format::fixed, 860,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000001171638135855961553373321221278340830500891338311828025758900568437072151035"
        "7970045005407240881351697976362735472435824781211915454559537018519197040423488429239617897303788913183131"
        "0940892450554994265592775992091128308424777712062783596878178379595965543379656835518328390775634263087771"
        "4805973543094711886156400143625867638243427058268025437512387465640901051788431639730423478289423768160950"
        "8806479454178743579840245669270048209791741113443291220891622306157813298012591441231722646111943375403527"
        "0666504556368815174273470780628068354937124362641414152826766558737473224670842808503579135503969155251979"
        "827880859375"},
    {0x1.fffffffffffffp-807, chars_format::fixed, 859,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000002343276271711923106746642442556681661001782676623656051517801136874144302071"
        "5940090010814481762703395952725470944871649562423830909119074037038394080846976858479235794607577826366262"
        "1881784901109988531185551984182256616849555424125567193756356759191931086759313671036656781551268526175542"
        "9611947086189423772312800287251735276486854116536050875024774931281802103576863279460846956578847536321901"
        "7612958908357487159680491338540096419583482226886582441783244612315626596025182882463445292223886750807054"
        "1333009112737630348546941561256136709874248725282828305653533117474946449341685617007158271007938310503959"
        "65576171875"},
    {0x1.fffffffffffffp-806, chars_format::fixed, 858,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000004686552543423846213493284885113363322003565353247312103035602273748288604143"
        "1880180021628963525406791905450941889743299124847661818238148074076788161693953716958471589215155652732524"
        "3763569802219977062371103968364513233699110848251134387512713518383862173518627342073313563102537052351085"
        "9223894172378847544625600574503470552973708233072101750049549862563604207153726558921693913157695072643803"
        "5225917816714974319360982677080192839166964453773164883566489224631253192050365764926890584447773501614108"
        "2666018225475260697093883122512273419748497450565656611307066234949892898683371234014316542015876621007919"
        "3115234375"},
    {0x1.fffffffffffffp-805, chars_format::fixed, 857,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000009373105086847692426986569770226726644007130706494624206071204547496577208286"
        "3760360043257927050813583810901883779486598249695323636476296148153576323387907433916943178430311305465048"
        "7527139604439954124742207936729026467398221696502268775025427036767724347037254684146627126205074104702171"
        "8447788344757695089251201149006941105947416466144203500099099725127208414307453117843387826315390145287607"
        "0451835633429948638721965354160385678333928907546329767132978449262506384100731529853781168895547003228216"
        "5332036450950521394187766245024546839496994901131313222614132469899785797366742468028633084031753242015838"
        "623046875"},
    {0x1.fffffffffffffp-804, chars_format::fixed, 856,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000018746210173695384853973139540453453288014261412989248412142409094993154416572"
        "7520720086515854101627167621803767558973196499390647272952592296307152646775814867833886356860622610930097"
        "5054279208879908249484415873458052934796443393004537550050854073535448694074509368293254252410148209404343"
        "6895576689515390178502402298013882211894832932288407000198199450254416828614906235686775652630780290575214"
        "0903671266859897277443930708320771356667857815092659534265956898525012768201463059707562337791094006456433"
        "0664072901901042788375532490049093678993989802262626445228264939799571594733484936057266168063506484031677"
        "24609375"},
    {0x1.fffffffffffffp-803, chars_format::fixed, 855,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000037492420347390769707946279080906906576028522825978496824284818189986308833145"
        "5041440173031708203254335243607535117946392998781294545905184592614305293551629735667772713721245221860195"
        "0108558417759816498968831746916105869592886786009075100101708147070897388149018736586508504820296418808687"
        "3791153379030780357004804596027764423789665864576814000396398900508833657229812471373551305261560581150428"
        "1807342533719794554887861416641542713335715630185319068531913797050025536402926119415124675582188012912866"
        "1328145803802085576751064980098187357987979604525252890456529879599143189466969872114532336127012968063354"
        "4921875"},
    {0x1.fffffffffffffp-802, chars_format::fixed, 854,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000074984840694781539415892558161813813152057045651956993648569636379972617666291"
        "0082880346063416406508670487215070235892785997562589091810369185228610587103259471335545427442490443720390"
        "0217116835519632997937663493832211739185773572018150200203416294141794776298037473173017009640592837617374"
        "7582306758061560714009609192055528847579331729153628000792797801017667314459624942747102610523121162300856"
        "3614685067439589109775722833283085426671431260370638137063827594100051072805852238830249351164376025825732"
        "2656291607604171153502129960196374715975959209050505780913059759198286378933939744229064672254025936126708"
        "984375"},
    {0x1.fffffffffffffp-801, chars_format::fixed, 853,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000149969681389563078831785116323627626304114091303913987297139272759945235332582"
        "0165760692126832813017340974430140471785571995125178183620738370457221174206518942671090854884980887440780"
        "0434233671039265995875326987664423478371547144036300400406832588283589552596074946346034019281185675234749"
        "5164613516123121428019218384111057695158663458307256001585595602035334628919249885494205221046242324601712"
        "7229370134879178219551445666566170853342862520741276274127655188200102145611704477660498702328752051651464"
        "5312583215208342307004259920392749431951918418101011561826119518396572757867879488458129344508051872253417"
        "96875"},
    {0x1.fffffffffffffp-800, chars_format::fixed, 852,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000299939362779126157663570232647255252608228182607827974594278545519890470665164"
        "0331521384253665626034681948860280943571143990250356367241476740914442348413037885342181709769961774881560"
        "0868467342078531991750653975328846956743094288072600800813665176567179105192149892692068038562371350469499"
        "0329227032246242856038436768222115390317326916614512003171191204070669257838499770988410442092484649203425"
        "4458740269758356439102891333132341706685725041482552548255310376400204291223408955320997404657504103302929"
        "0625166430416684614008519840785498863903836836202023123652239036793145515735758976916258689016103744506835"
        "9375"},
    {0x1.fffffffffffffp-799, chars_format::fixed, 851,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000599878725558252315327140465294510505216456365215655949188557091039780941330328"
        "0663042768507331252069363897720561887142287980500712734482953481828884696826075770684363419539923549763120"
        "1736934684157063983501307950657693913486188576145201601627330353134358210384299785384136077124742700938998"
        "0658454064492485712076873536444230780634653833229024006342382408141338515676999541976820884184969298406850"
        "8917480539516712878205782666264683413371450082965105096510620752800408582446817910641994809315008206605858"
        "1250332860833369228017039681570997727807673672404046247304478073586291031471517953832517378032207489013671"
        "875"},
    {0x1.fffffffffffffp-798, chars_format::fixed, 850,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000001199757451116504630654280930589021010432912730431311898377114182079561882660656"
        "1326085537014662504138727795441123774284575961001425468965906963657769393652151541368726839079847099526240"
        "3473869368314127967002615901315387826972377152290403203254660706268716420768599570768272154249485401877996"
        "1316908128984971424153747072888461561269307666458048012684764816282677031353999083953641768369938596813701"
        "7834961079033425756411565332529366826742900165930210193021241505600817164893635821283989618630016413211716"
        "2500665721666738456034079363141995455615347344808092494608956147172582062943035907665034756064414978027343"
        "75"},
    {0x1.fffffffffffffp-797, chars_format::fixed, 849,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000002399514902233009261308561861178042020865825460862623796754228364159123765321312"
        "2652171074029325008277455590882247548569151922002850937931813927315538787304303082737453678159694199052480"
        "6947738736628255934005231802630775653944754304580806406509321412537432841537199141536544308498970803755992"
        "2633816257969942848307494145776923122538615332916096025369529632565354062707998167907283536739877193627403"
        "5669922158066851512823130665058733653485800331860420386042483011201634329787271642567979237260032826423432"
        "5001331443333476912068158726283990911230694689616184989217912294345164125886071815330069512128829956054687"
        "5"},
    {0x1.fffffffffffffp-796, chars_format::fixed, 848,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000004799029804466018522617123722356084041731650921725247593508456728318247530642624"
        "5304342148058650016554911181764495097138303844005701875863627854631077574608606165474907356319388398104961"
        "3895477473256511868010463605261551307889508609161612813018642825074865683074398283073088616997941607511984"
        "5267632515939885696614988291553846245077230665832192050739059265130708125415996335814567073479754387254807"
        "1339844316133703025646261330117467306971600663720840772084966022403268659574543285135958474520065652846865"
        "000266288666695382413631745256798182246138937923236997843582458869032825177214363066013902425765991210937"
        "5"},
    {0x1.fffffffffffffp-795, chars_format::fixed, 847,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000009598059608932037045234247444712168083463301843450495187016913456636495061285249"
        "0608684296117300033109822363528990194276607688011403751727255709262155149217212330949814712638776796209922"
        "7790954946513023736020927210523102615779017218323225626037285650149731366148796566146177233995883215023969"
        "0535265031879771393229976583107692490154461331664384101478118530261416250831992671629134146959508774509614"
        "2679688632267406051292522660234934613943201327441681544169932044806537319149086570271916949040131305693730"
        "00053257733339076482726349051359636449227787584647399568716491773806565035442872613202780485153198242187"
        "5"},
    {0x1.fffffffffffffp-794, chars_format::fixed, 846,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000019196119217864074090468494889424336166926603686900990374033826913272990122570498"
        "1217368592234600066219644727057980388553215376022807503454511418524310298434424661899629425277553592419845"
        "5581909893026047472041854421046205231558034436646451252074571300299462732297593132292354467991766430047938"
        "1070530063759542786459953166215384980308922663328768202956237060522832501663985343258268293919017549019228"
        "5359377264534812102585045320469869227886402654883363088339864089613074638298173140543833898080262611387460"
        "00106515466678152965452698102719272898455575169294799137432983547613130070885745226405560970306396484375"},
    {0x1.fffffffffffffp-793, chars_format::fixed, 845,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000038392238435728148180936989778848672333853207373801980748067653826545980245140996"
        "2434737184469200132439289454115960777106430752045615006909022837048620596868849323799258850555107184839691"
        "1163819786052094944083708842092410463116068873292902504149142600598925464595186264584708935983532860095876"
        "2141060127519085572919906332430769960617845326657536405912474121045665003327970686516536587838035098038457"
        "0718754529069624205170090640939738455772805309766726176679728179226149276596346281087667796160525222774920"
        "0021303093335630593090539620543854579691115033858959827486596709522626014177149045281112194061279296875"},
    {0x1.fffffffffffffp-792, chars_format::fixed, 844,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000076784476871456296361873979557697344667706414747603961496135307653091960490281992"
        "4869474368938400264878578908231921554212861504091230013818045674097241193737698647598517701110214369679382"
        "2327639572104189888167417684184820926232137746585805008298285201197850929190372529169417871967065720191752"
        "4282120255038171145839812664861539921235690653315072811824948242091330006655941373033073175676070196076914"
        "1437509058139248410340181281879476911545610619533452353359456358452298553192692562175335592321050445549840"
        "004260618667126118618107924108770915938223006771791965497319341904525202835429809056222438812255859375"},
    {0x1.fffffffffffffp-791, chars_format::fixed, 843,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000153568953742912592723747959115394689335412829495207922992270615306183920980563984"
        "9738948737876800529757157816463843108425723008182460027636091348194482387475397295197035402220428739358764"
        "4655279144208379776334835368369641852464275493171610016596570402395701858380745058338835743934131440383504"
        "8564240510076342291679625329723079842471381306630145623649896484182660013311882746066146351352140392153828"
        "2875018116278496820680362563758953823091221239066904706718912716904597106385385124350671184642100891099680"
        "00852123733425223723621584821754183187644601354358393099463868380905040567085961811244487762451171875"},
    {0x1.fffffffffffffp-790, chars_format::fixed, 842,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000307137907485825185447495918230789378670825658990415845984541230612367841961127969"
        "9477897475753601059514315632927686216851446016364920055272182696388964774950794590394070804440857478717528"
        "9310558288416759552669670736739283704928550986343220033193140804791403716761490116677671487868262880767009"
        "7128481020152684583359250659446159684942762613260291247299792968365320026623765492132292702704280784307656"
        "5750036232556993641360725127517907646182442478133809413437825433809194212770770248701342369284201782199360"
        "0170424746685044744724316964350836637528920270871678619892773676181008113417192362248897552490234375"},
    {0x1.fffffffffffffp-789, chars_format::fixed, 841,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000614275814971650370894991836461578757341651317980831691969082461224735683922255939"
        "8955794951507202119028631265855372433702892032729840110544365392777929549901589180788141608881714957435057"
        "8621116576833519105339341473478567409857101972686440066386281609582807433522980233355342975736525761534019"
        "4256962040305369166718501318892319369885525226520582494599585936730640053247530984264585405408561568615313"
        "1500072465113987282721450255035815292364884956267618826875650867618388425541540497402684738568403564398720"
        "034084949337008948944863392870167327505784054174335723978554735236201622683438472449779510498046875"},
    {0x1.fffffffffffffp-788, chars_format::fixed, 840,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000001228551629943300741789983672923157514683302635961663383938164922449471367844511879"
        "7911589903014404238057262531710744867405784065459680221088730785555859099803178361576283217763429914870115"
        "7242233153667038210678682946957134819714203945372880132772563219165614867045960466710685951473051523068038"
        "8513924080610738333437002637784638739771050453041164989199171873461280106495061968529170810817123137230626"
        "3000144930227974565442900510071630584729769912535237653751301735236776851083080994805369477136807128797440"
        "06816989867401789788972678574033465501156810834867144795710947047240324536687694489955902099609375"},
    {0x1.fffffffffffffp-787, chars_format::fixed, 839,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000002457103259886601483579967345846315029366605271923326767876329844898942735689023759"
        "5823179806028808476114525063421489734811568130919360442177461571111718199606356723152566435526859829740231"
        "4484466307334076421357365893914269639428407890745760265545126438331229734091920933421371902946103046136077"
        "7027848161221476666874005275569277479542100906082329978398343746922560212990123937058341621634246274461252"
        "6000289860455949130885801020143261169459539825070475307502603470473553702166161989610738954273614257594880"
        "1363397973480357957794535714806693100231362166973428959142189409448064907337538897991180419921875"},
    {0x1.fffffffffffffp-786, chars_format::fixed, 838,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000004914206519773202967159934691692630058733210543846653535752659689797885471378047519"
        "1646359612057616952229050126842979469623136261838720884354923142223436399212713446305132871053719659480462"
        "8968932614668152842714731787828539278856815781491520531090252876662459468183841866842743805892206092272155"
        "4055696322442953333748010551138554959084201812164659956796687493845120425980247874116683243268492548922505"
        "2000579720911898261771602040286522338919079650140950615005206940947107404332323979221477908547228515189760"
        "272679594696071591558907142961338620046272433394685791828437881889612981467507779598236083984375"},
    {0x1.fffffffffffffp-785, chars_format::fixed, 837,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000009828413039546405934319869383385260117466421087693307071505319379595770942756095038"
        "3292719224115233904458100253685958939246272523677441768709846284446872798425426892610265742107439318960925"
        "7937865229336305685429463575657078557713631562983041062180505753324918936367683733685487611784412184544310"
        "8111392644885906667496021102277109918168403624329319913593374987690240851960495748233366486536985097845010"
        "4001159441823796523543204080573044677838159300281901230010413881894214808664647958442955817094457030379520"
        "54535918939214318311781428592267724009254486678937158365687576377922596293501555919647216796875"},
    {0x1.fffffffffffffp-784, chars_format::fixed, 836,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000019656826079092811868639738766770520234932842175386614143010638759191541885512190076"
        "6585438448230467808916200507371917878492545047354883537419692568893745596850853785220531484214878637921851"
        "5875730458672611370858927151314157115427263125966082124361011506649837872735367467370975223568824369088621"
        "6222785289771813334992042204554219836336807248658639827186749975380481703920991496466732973073970195690020"
        "8002318883647593047086408161146089355676318600563802460020827763788429617329295916885911634188914060759041"
        "0907183787842863662356285718453544801850897335787431673137515275584519258700311183929443359375"},
    {0x1.fffffffffffffp-783, chars_format::fixed, 835,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000039313652158185623737279477533541040469865684350773228286021277518383083771024380153"
        "3170876896460935617832401014743835756985090094709767074839385137787491193701707570441062968429757275843703"
        "1751460917345222741717854302628314230854526251932164248722023013299675745470734934741950447137648738177243"
        "2445570579543626669984084409108439672673614497317279654373499950760963407841982992933465946147940391380041"
        "6004637767295186094172816322292178711352637201127604920041655527576859234658591833771823268377828121518082"
        "181436757568572732471257143690708960370179467157486334627503055116903851740062236785888671875"},
    {0x1.fffffffffffffp-782, chars_format::fixed, 834,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000078627304316371247474558955067082080939731368701546456572042555036766167542048760306"
        "6341753792921871235664802029487671513970180189419534149678770275574982387403415140882125936859514551687406"
        "3502921834690445483435708605256628461709052503864328497444046026599351490941469869483900894275297476354486"
        "4891141159087253339968168818216879345347228994634559308746999901521926815683965985866931892295880782760083"
        "2009275534590372188345632644584357422705274402255209840083311055153718469317183667543646536755656243036164"
        "36287351513714546494251428738141792074035893431497266925500611023380770348012447357177734375"},
    {0x1.fffffffffffffp-781, chars_format::fixed, 833,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000157254608632742494949117910134164161879462737403092913144085110073532335084097520613"
        "2683507585843742471329604058975343027940360378839068299357540551149964774806830281764251873719029103374812"
        "7005843669380890966871417210513256923418105007728656994888092053198702981882939738967801788550594952708972"
        "9782282318174506679936337636433758690694457989269118617493999803043853631367931971733863784591761565520166"
        "4018551069180744376691265289168714845410548804510419680166622110307436938634367335087293073511312486072328"
        "7257470302742909298850285747628358414807178686299453385100122204676154069602489471435546875"},
    {0x1.fffffffffffffp-780, chars_format::fixed, 832,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000314509217265484989898235820268328323758925474806185826288170220147064670168195041226"
        "5367015171687484942659208117950686055880720757678136598715081102299929549613660563528503747438058206749625"
        "4011687338761781933742834421026513846836210015457313989776184106397405963765879477935603577101189905417945"
        "9564564636349013359872675272867517381388915978538237234987999606087707262735863943467727569183523131040332"
        "8037102138361488753382530578337429690821097609020839360333244220614873877268734670174586147022624972144657"
        "451494060548581859770057149525671682961435737259890677020024440935230813920497894287109375"},
    {0x1.fffffffffffffp-779, chars_format::fixed, 831,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000629018434530969979796471640536656647517850949612371652576340440294129340336390082453"
        "0734030343374969885318416235901372111761441515356273197430162204599859099227321127057007494876116413499250"
        "8023374677523563867485668842053027693672420030914627979552368212794811927531758955871207154202379810835891"
        "9129129272698026719745350545735034762777831957076474469975999212175414525471727886935455138367046262080665"
        "6074204276722977506765061156674859381642195218041678720666488441229747754537469340349172294045249944289314"
        "90298812109716371954011429905134336592287147451978135404004888187046162784099578857421875"},
    {0x1.fffffffffffffp-778, chars_format::fixed, 830,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000001258036869061939959592943281073313295035701899224743305152680880588258680672780164906"
        "1468060686749939770636832471802744223522883030712546394860324409199718198454642254114014989752232826998501"
        "6046749355047127734971337684106055387344840061829255959104736425589623855063517911742414308404759621671783"
        "8258258545396053439490701091470069525555663914152948939951998424350829050943455773870910276734092524161331"
        "2148408553445955013530122313349718763284390436083357441332976882459495509074938680698344588090499888578629"
        "8059762421943274390802285981026867318457429490395627080800977637409232556819915771484375"},
    {0x1.fffffffffffffp-777, chars_format::fixed, 829,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000002516073738123879919185886562146626590071403798449486610305361761176517361345560329812"
        "2936121373499879541273664943605488447045766061425092789720648818399436396909284508228029979504465653997003"
        "2093498710094255469942675368212110774689680123658511918209472851179247710127035823484828616809519243343567"
        "6516517090792106878981402182940139051111327828305897879903996848701658101886911547741820553468185048322662"
        "4296817106891910027060244626699437526568780872166714882665953764918991018149877361396689176180999777157259"
        "611952484388654878160457196205373463691485898079125416160195527481846511363983154296875"},
    {0x1.fffffffffffffp-776, chars_format::fixed, 828,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000005032147476247759838371773124293253180142807596898973220610723522353034722691120659624"
        "5872242746999759082547329887210976894091532122850185579441297636798872793818569016456059959008931307994006"
        "4186997420188510939885350736424221549379360247317023836418945702358495420254071646969657233619038486687135"
        "3033034181584213757962804365880278102222655656611795759807993697403316203773823095483641106936370096645324"
        "8593634213783820054120489253398875053137561744333429765331907529837982036299754722793378352361999554314519"
        "22390496877730975632091439241074692738297179615825083232039105496369302272796630859375"},
    {0x1.fffffffffffffp-775, chars_format::fixed, 827,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000010064294952495519676743546248586506360285615193797946441221447044706069445382241319249"
        "1744485493999518165094659774421953788183064245700371158882595273597745587637138032912119918017862615988012"
        "8373994840377021879770701472848443098758720494634047672837891404716990840508143293939314467238076973374270"
        "6066068363168427515925608731760556204445311313223591519615987394806632407547646190967282213872740193290649"
        "7187268427567640108240978506797750106275123488666859530663815059675964072599509445586756704723999108629038"
        "4478099375546195126418287848214938547659435923165016646407821099273860454559326171875"},
    {0x1.fffffffffffffp-774, chars_format::fixed, 826,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000020128589904991039353487092497173012720571230387595892882442894089412138890764482638498"
        "3488970987999036330189319548843907576366128491400742317765190547195491175274276065824239836035725231976025"
        "6747989680754043759541402945696886197517440989268095345675782809433981681016286587878628934476153946748541"
        "2132136726336855031851217463521112408890622626447183039231974789613264815095292381934564427745480386581299"
        "4374536855135280216481957013595500212550246977333719061327630119351928145199018891173513409447998217258076"
        "895619875109239025283657569642987709531887184633003329281564219854772090911865234375"},
    {0x1.fffffffffffffp-773, chars_format::fixed, 825,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000040257179809982078706974184994346025441142460775191785764885788178824277781528965276996"
        "6977941975998072660378639097687815152732256982801484635530381094390982350548552131648479672071450463952051"
        "3495979361508087519082805891393772395034881978536190691351565618867963362032573175757257868952307893497082"
        "4264273452673710063702434927042224817781245252894366078463949579226529630190584763869128855490960773162598"
        "8749073710270560432963914027191000425100493954667438122655260238703856290398037782347026818895996434516153"
        "79123975021847805056731513928597541906377436926600665856312843970954418182373046875"},
    {0x1.fffffffffffffp-772, chars_format::fixed, 824,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000080514359619964157413948369988692050882284921550383571529771576357648555563057930553993"
        "3955883951996145320757278195375630305464513965602969271060762188781964701097104263296959344142900927904102"
        "6991958723016175038165611782787544790069763957072381382703131237735926724065146351514515737904615786994164"
        "8528546905347420127404869854084449635562490505788732156927899158453059260381169527738257710981921546325197"
        "7498147420541120865927828054382000850200987909334876245310520477407712580796075564694053637791992869032307"
        "5824795004369561011346302785719508381275487385320133171262568794190883636474609375"},
    {0x1.fffffffffffffp-771, chars_format::fixed, 823,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000161028719239928314827896739977384101764569843100767143059543152715297111126115861107986"
        "7911767903992290641514556390751260610929027931205938542121524377563929402194208526593918688285801855808205"
        "3983917446032350076331223565575089580139527914144762765406262475471853448130292703029031475809231573988329"
        "7057093810694840254809739708168899271124981011577464313855798316906118520762339055476515421963843092650395"
        "4996294841082241731855656108764001700401975818669752490621040954815425161592151129388107275583985738064615"
        "164959000873912202269260557143901676255097477064026634252513758838176727294921875"},
    {0x1.fffffffffffffp-770, chars_format::fixed, 822,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000322057438479856629655793479954768203529139686201534286119086305430594222252231722215973"
        "5823535807984581283029112781502521221858055862411877084243048755127858804388417053187837376571603711616410"
        "7967834892064700152662447131150179160279055828289525530812524950943706896260585406058062951618463147976659"
        "4114187621389680509619479416337798542249962023154928627711596633812237041524678110953030843927686185300790"
        "9992589682164483463711312217528003400803951637339504981242081909630850323184302258776214551167971476129230"
        "32991800174782440453852111428780335251019495412805326850502751767635345458984375"},
    {0x1.fffffffffffffp-769, chars_format::fixed, 821,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000644114876959713259311586959909536407058279372403068572238172610861188444504463444431947"
        "1647071615969162566058225563005042443716111724823754168486097510255717608776834106375674753143207423232821"
        "5935669784129400305324894262300358320558111656579051061625049901887413792521170812116125903236926295953318"
        "8228375242779361019238958832675597084499924046309857255423193267624474083049356221906061687855372370601581"
        "9985179364328966927422624435056006801607903274679009962484163819261700646368604517552429102335942952258460"
        "6598360034956488090770422285756067050203899082561065370100550353527069091796875"},
    {0x1.fffffffffffffp-768, chars_format::fixed, 820,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000001288229753919426518623173919819072814116558744806137144476345221722376889008926888863894"
        "3294143231938325132116451126010084887432223449647508336972195020511435217553668212751349506286414846465643"
        "1871339568258800610649788524600716641116223313158102123250099803774827585042341624232251806473852591906637"
        "6456750485558722038477917665351194168999848092619714510846386535248948166098712443812123375710744741203163"
        "9970358728657933854845248870112013603215806549358019924968327638523401292737209035104858204671885904516921"
        "319672006991297618154084457151213410040779816512213074020110070705413818359375"},
    {0x1.fffffffffffffp-767, chars_format::fixed, 819,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000002576459507838853037246347839638145628233117489612274288952690443444753778017853777727788"
        "6588286463876650264232902252020169774864446899295016673944390041022870435107336425502699012572829692931286"
        "3742679136517601221299577049201433282232446626316204246500199607549655170084683248464503612947705183813275"
        "2913500971117444076955835330702388337999696185239429021692773070497896332197424887624246751421489482406327"
        "9940717457315867709690497740224027206431613098716039849936655277046802585474418070209716409343771809033842"
        "63934401398259523630816891430242682008155963302442614804022014141082763671875"},
    {0x1.fffffffffffffp-766, chars_format::fixed, 818,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000005152919015677706074492695679276291256466234979224548577905380886889507556035707555455577"
        "3176572927753300528465804504040339549728893798590033347888780082045740870214672851005398025145659385862572"
        "7485358273035202442599154098402866564464893252632408493000399215099310340169366496929007225895410367626550"
        "5827001942234888153911670661404776675999392370478858043385546140995792664394849775248493502842978964812655"
        "9881434914631735419380995480448054412863226197432079699873310554093605170948836140419432818687543618067685"
        "2786880279651904726163378286048536401631192660488522960804402828216552734375"},
    {0x1.fffffffffffffp-765, chars_format::fixed, 817,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000010305838031355412148985391358552582512932469958449097155810761773779015112071415110911154"
        "6353145855506601056931609008080679099457787597180066695777560164091481740429345702010796050291318771725145"
        "4970716546070404885198308196805733128929786505264816986000798430198620680338732993858014451790820735253101"
        "1654003884469776307823341322809553351998784740957716086771092281991585328789699550496987005685957929625311"
        "9762869829263470838761990960896108825726452394864159399746621108187210341897672280838865637375087236135370"
        "557376055930380945232675657209707280326238532097704592160880565643310546875"},
    {0x1.fffffffffffffp-764, chars_format::fixed, 816,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000020611676062710824297970782717105165025864939916898194311621523547558030224142830221822309"
        "2706291711013202113863218016161358198915575194360133391555120328182963480858691404021592100582637543450290"
        "9941433092140809770396616393611466257859573010529633972001596860397241360677465987716028903581641470506202"
        "3308007768939552615646682645619106703997569481915432173542184563983170657579399100993974011371915859250623"
        "9525739658526941677523981921792217651452904789728318799493242216374420683795344561677731274750174472270741"
        "11475211186076189046535131441941456065247706419540918432176113128662109375"},
    {0x1.fffffffffffffp-763, chars_format::fixed, 815,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000041223352125421648595941565434210330051729879833796388623243047095116060448285660443644618"
        "5412583422026404227726436032322716397831150388720266783110240656365926961717382808043184201165275086900581"
        "9882866184281619540793232787222932515719146021059267944003193720794482721354931975432057807163282941012404"
        "6616015537879105231293365291238213407995138963830864347084369127966341315158798201987948022743831718501247"
        "9051479317053883355047963843584435302905809579456637598986484432748841367590689123355462549500348944541482"
        "2295042237215237809307026288388291213049541283908183686435222625732421875"},
    {0x1.fffffffffffffp-762, chars_format::fixed, 814,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000082446704250843297191883130868420660103459759667592777246486094190232120896571320887289237"
        "0825166844052808455452872064645432795662300777440533566220481312731853923434765616086368402330550173801163"
        "9765732368563239081586465574445865031438292042118535888006387441588965442709863950864115614326565882024809"
        "3232031075758210462586730582476426815990277927661728694168738255932682630317596403975896045487663437002495"
        "8102958634107766710095927687168870605811619158913275197972968865497682735181378246710925099000697889082964"
        "459008447443047561861405257677658242609908256781636737287044525146484375"},
    {0x1.fffffffffffffp-761, chars_format::fixed, 813,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000164893408501686594383766261736841320206919519335185554492972188380464241793142641774578474"
        "1650333688105616910905744129290865591324601554881067132440962625463707846869531232172736804661100347602327"
        "9531464737126478163172931148891730062876584084237071776012774883177930885419727901728231228653131764049618"
        "6464062151516420925173461164952853631980555855323457388337476511865365260635192807951792090975326874004991"
        "6205917268215533420191855374337741211623238317826550395945937730995365470362756493421850198001395778165928"
        "91801689488609512372281051535531648521981651356327347457408905029296875"},
    {0x1.fffffffffffffp-760, chars_format::fixed, 812,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000329786817003373188767532523473682640413839038670371108985944376760928483586285283549156948"
        "3300667376211233821811488258581731182649203109762134264881925250927415693739062464345473609322200695204655"
        "9062929474252956326345862297783460125753168168474143552025549766355861770839455803456462457306263528099237"
        "2928124303032841850346922329905707263961111710646914776674953023730730521270385615903584181950653748009983"
        "2411834536431066840383710748675482423246476635653100791891875461990730940725512986843700396002791556331857"
        "8360337897721902474456210307106329704396330271265469491481781005859375"},
    {0x1.fffffffffffffp-759, chars_format::fixed, 811,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000659573634006746377535065046947365280827678077340742217971888753521856967172570567098313896"
        "6601334752422467643622976517163462365298406219524268529763850501854831387478124928690947218644401390409311"
        "8125858948505912652691724595566920251506336336948287104051099532711723541678911606912924914612527056198474"
        "5856248606065683700693844659811414527922223421293829553349906047461461042540771231807168363901307496019966"
        "4823669072862133680767421497350964846492953271306201583783750923981461881451025973687400792005583112663715"
        "672067579544380494891242061421265940879266054253093898296356201171875"},
    {0x1.fffffffffffffp-758, chars_format::fixed, 810,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000001319147268013492755070130093894730561655356154681484435943777507043713934345141134196627793"
        "3202669504844935287245953034326924730596812439048537059527701003709662774956249857381894437288802780818623"
        "6251717897011825305383449191133840503012672673896574208102199065423447083357823213825849829225054112396949"
        "1712497212131367401387689319622829055844446842587659106699812094922922085081542463614336727802614992039932"
        "9647338145724267361534842994701929692985906542612403167567501847962923762902051947374801584011166225327431"
        "34413515908876098978248412284253188175853210850618779659271240234375"},
    {0x1.fffffffffffffp-757, chars_format::fixed, 809,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000002638294536026985510140260187789461123310712309362968871887555014087427868690282268393255586"
        "6405339009689870574491906068653849461193624878097074119055402007419325549912499714763788874577605561637247"
        "2503435794023650610766898382267681006025345347793148416204398130846894166715646427651699658450108224793898"
        "3424994424262734802775378639245658111688893685175318213399624189845844170163084927228673455605229984079865"
        "9294676291448534723069685989403859385971813085224806335135003695925847525804103894749603168022332450654862"
        "6882703181775219795649682456850637635170642170123755931854248046875"},
    {0x1.fffffffffffffp-756, chars_format::fixed, 808,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000005276589072053971020280520375578922246621424618725937743775110028174855737380564536786511173"
        "2810678019379741148983812137307698922387249756194148238110804014838651099824999429527577749155211123274494"
        "5006871588047301221533796764535362012050690695586296832408796261693788333431292855303399316900216449587796"
        "6849988848525469605550757278491316223377787370350636426799248379691688340326169854457346911210459968159731"
        "8589352582897069446139371978807718771943626170449612670270007391851695051608207789499206336044664901309725"
        "376540636355043959129936491370127527034128434024751186370849609375"},
    {0x1.fffffffffffffp-755, chars_format::fixed, 807,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000010553178144107942040561040751157844493242849237451875487550220056349711474761129073573022346"
        "5621356038759482297967624274615397844774499512388296476221608029677302199649998859055155498310422246548989"
        "0013743176094602443067593529070724024101381391172593664817592523387576666862585710606798633800432899175593"
        "3699977697050939211101514556982632446755574740701272853598496759383376680652339708914693822420919936319463"
        "7178705165794138892278743957615437543887252340899225340540014783703390103216415578998412672089329802619450"
        "75308127271008791825987298274025505406825686804950237274169921875"},
    {0x1.fffffffffffffp-754, chars_format::fixed, 806,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000021106356288215884081122081502315688986485698474903750975100440112699422949522258147146044693"
        "1242712077518964595935248549230795689548999024776592952443216059354604399299997718110310996620844493097978"
        "0027486352189204886135187058141448048202762782345187329635185046775153333725171421213597267600865798351186"
        "7399955394101878422203029113965264893511149481402545707196993518766753361304679417829387644841839872638927"
        "4357410331588277784557487915230875087774504681798450681080029567406780206432831157996825344178659605238901"
        "5061625454201758365197459654805101081365137360990047454833984375"},
    {0x1.fffffffffffffp-753, chars_format::fixed, 805,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000042212712576431768162244163004631377972971396949807501950200880225398845899044516294292089386"
        "2485424155037929191870497098461591379097998049553185904886432118709208798599995436220621993241688986195956"
        "0054972704378409772270374116282896096405525564690374659270370093550306667450342842427194535201731596702373"
        "4799910788203756844406058227930529787022298962805091414393987037533506722609358835658775289683679745277854"
        "8714820663176555569114975830461750175549009363596901362160059134813560412865662315993650688357319210477803"
        "012325090840351673039491930961020216273027472198009490966796875"},
    {0x1.fffffffffffffp-752, chars_format::fixed, 804,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000084425425152863536324488326009262755945942793899615003900401760450797691798089032588584178772"
        "4970848310075858383740994196923182758195996099106371809772864237418417597199990872441243986483377972391912"
        "0109945408756819544540748232565792192811051129380749318540740187100613334900685684854389070403463193404746"
        "9599821576407513688812116455861059574044597925610182828787974075067013445218717671317550579367359490555709"
        "7429641326353111138229951660923500351098018727193802724320118269627120825731324631987301376714638420955606"
        "02465018168070334607898386192204043254605494439601898193359375"},
    {0x1.fffffffffffffp-751, chars_format::fixed, 803,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000168850850305727072648976652018525511891885587799230007800803520901595383596178065177168357544"
        "9941696620151716767481988393846365516391992198212743619545728474836835194399981744882487972966755944783824"
        "0219890817513639089081496465131584385622102258761498637081480374201226669801371369708778140806926386809493"
        "9199643152815027377624232911722119148089195851220365657575948150134026890437435342635101158734718981111419"
        "4859282652706222276459903321847000702196037454387605448640236539254241651462649263974602753429276841911212"
        "0493003633614066921579677238440808650921098887920379638671875"},
    {0x1.fffffffffffffp-750, chars_format::fixed, 802,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000337701700611454145297953304037051023783771175598460015601607041803190767192356130354336715089"
        "9883393240303433534963976787692731032783984396425487239091456949673670388799963489764975945933511889567648"
        "0439781635027278178162992930263168771244204517522997274162960748402453339602742739417556281613852773618987"
        "8399286305630054755248465823444238296178391702440731315151896300268053780874870685270202317469437962222838"
        "9718565305412444552919806643694001404392074908775210897280473078508483302925298527949205506858553683822424"
        "098600726722813384315935447688161730184219777584075927734375"},
    {0x1.fffffffffffffp-749, chars_format::fixed, 801,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000675403401222908290595906608074102047567542351196920031203214083606381534384712260708673430179"
        "9766786480606867069927953575385462065567968792850974478182913899347340777599926979529951891867023779135296"
        "0879563270054556356325985860526337542488409035045994548325921496804906679205485478835112563227705547237975"
        "6798572611260109510496931646888476592356783404881462630303792600536107561749741370540404634938875924445677"
        "9437130610824889105839613287388002808784149817550421794560946157016966605850597055898411013717107367644848"
        "19720145344562676863187089537632346036843955516815185546875"},
    {0x1.fffffffffffffp-748, chars_format::fixed, 800,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000001350806802445816581191813216148204095135084702393840062406428167212763068769424521417346860359"
        "9533572961213734139855907150770924131135937585701948956365827798694681555199853959059903783734047558270592"
        "1759126540109112712651971721052675084976818070091989096651842993609813358410970957670225126455411094475951"
        "3597145222520219020993863293776953184713566809762925260607585201072215123499482741080809269877751848891355"
        "8874261221649778211679226574776005617568299635100843589121892314033933211701194111796822027434214735289696"
        "3944029068912535372637417907526469207368791103363037109375"},
    {0x1.fffffffffffffp-747, chars_format::fixed, 799,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000002701613604891633162383626432296408190270169404787680124812856334425526137538849042834693720719"
        "9067145922427468279711814301541848262271875171403897912731655597389363110399707918119807567468095116541184"
        "3518253080218225425303943442105350169953636140183978193303685987219626716821941915340450252910822188951902"
        "7194290445040438041987726587553906369427133619525850521215170402144430246998965482161618539755503697782711"
        "7748522443299556423358453149552011235136599270201687178243784628067866423402388223593644054868429470579392"
        "788805813782507074527483581505293841473758220672607421875"},
    {0x1.fffffffffffffp-746, chars_format::fixed, 798,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000005403227209783266324767252864592816380540338809575360249625712668851052275077698085669387441439"
        "8134291844854936559423628603083696524543750342807795825463311194778726220799415836239615134936190233082368"
        "7036506160436450850607886884210700339907272280367956386607371974439253433643883830680900505821644377903805"
        "4388580890080876083975453175107812738854267239051701042430340804288860493997930964323237079511007395565423"
        "5497044886599112846716906299104022470273198540403374356487569256135732846804776447187288109736858941158785"
        "57761162756501414905496716301058768294751644134521484375"},
    {0x1.fffffffffffffp-745, chars_format::fixed, 797,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000010806454419566532649534505729185632761080677619150720499251425337702104550155396171338774882879"
        "6268583689709873118847257206167393049087500685615591650926622389557452441598831672479230269872380466164737"
        "4073012320872901701215773768421400679814544560735912773214743948878506867287767661361801011643288755807610"
        "8777161780161752167950906350215625477708534478103402084860681608577720987995861928646474159022014791130847"
        "0994089773198225693433812598208044940546397080806748712975138512271465693609552894374576219473717882317571"
        "1552232551300282981099343260211753658950328826904296875"},
    {0x1.fffffffffffffp-744, chars_format::fixed, 796,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000021612908839133065299069011458371265522161355238301440998502850675404209100310792342677549765759"
        "2537167379419746237694514412334786098175001371231183301853244779114904883197663344958460539744760932329474"
        "8146024641745803402431547536842801359629089121471825546429487897757013734575535322723602023286577511615221"
        "7554323560323504335901812700431250955417068956206804169721363217155441975991723857292948318044029582261694"
        "1988179546396451386867625196416089881092794161613497425950277024542931387219105788749152438947435764635142"
        "310446510260056596219868652042350731790065765380859375"},
    {0x1.fffffffffffffp-743, chars_format::fixed, 795,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000043225817678266130598138022916742531044322710476602881997005701350808418200621584685355099531518"
        "5074334758839492475389028824669572196350002742462366603706489558229809766395326689916921079489521864658949"
        "6292049283491606804863095073685602719258178242943651092858975795514027469151070645447204046573155023230443"
        "5108647120647008671803625400862501910834137912413608339442726434310883951983447714585896636088059164523388"
        "3976359092792902773735250392832179762185588323226994851900554049085862774438211577498304877894871529270284"
        "62089302052011319243973730408470146358013153076171875"},
    {0x1.fffffffffffffp-742, chars_format::fixed, 794,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000086451635356532261196276045833485062088645420953205763994011402701616836401243169370710199063037"
        "0148669517678984950778057649339144392700005484924733207412979116459619532790653379833842158979043729317899"
        "2584098566983213609726190147371205438516356485887302185717951591028054938302141290894408093146310046460887"
        "0217294241294017343607250801725003821668275824827216678885452868621767903966895429171793272176118329046776"
        "7952718185585805547470500785664359524371176646453989703801108098171725548876423154996609755789743058540569"
        "2417860410402263848794746081694029271602630615234375"},
    {0x1.fffffffffffffp-741, chars_format::fixed, 793,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000172903270713064522392552091666970124177290841906411527988022805403233672802486338741420398126074"
        "0297339035357969901556115298678288785400010969849466414825958232919239065581306759667684317958087458635798"
        "5168197133966427219452380294742410877032712971774604371435903182056109876604282581788816186292620092921774"
        "0434588482588034687214501603450007643336551649654433357770905737243535807933790858343586544352236658093553"
        "5905436371171611094941001571328719048742353292907979407602216196343451097752846309993219511579486117081138"
        "483572082080452769758949216338805854320526123046875"},
    {0x1.fffffffffffffp-740, chars_format::fixed, 792,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000345806541426129044785104183333940248354581683812823055976045610806467345604972677482840796252148"
        "0594678070715939803112230597356577570800021939698932829651916465838478131162613519335368635916174917271597"
        "0336394267932854438904760589484821754065425943549208742871806364112219753208565163577632372585240185843548"
        "0869176965176069374429003206900015286673103299308866715541811474487071615867581716687173088704473316187107"
        "1810872742343222189882003142657438097484706585815958815204432392686902195505692619986439023158972234162276"
        "96714416416090553951789843267761170864105224609375"},
    {0x1.fffffffffffffp-739, chars_format::fixed, 791,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000691613082852258089570208366667880496709163367625646111952091221612934691209945354965681592504296"
        "1189356141431879606224461194713155141600043879397865659303832931676956262325227038670737271832349834543194"
        "0672788535865708877809521178969643508130851887098417485743612728224439506417130327155264745170480371687096"
        "1738353930352138748858006413800030573346206598617733431083622948974143231735163433374346177408946632374214"
        "3621745484686444379764006285314876194969413171631917630408864785373804391011385239972878046317944468324553"
        "9342883283218110790357968653552234172821044921875"},
    {0x1.fffffffffffffp-738, chars_format::fixed, 790,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000001383226165704516179140416733335760993418326735251292223904182443225869382419890709931363185008592"
        "2378712282863759212448922389426310283200087758795731318607665863353912524650454077341474543664699669086388"
        "1345577071731417755619042357939287016261703774196834971487225456448879012834260654310529490340960743374192"
        "3476707860704277497716012827600061146692413197235466862167245897948286463470326866748692354817893264748428"
        "7243490969372888759528012570629752389938826343263835260817729570747608782022770479945756092635888936649107"
        "868576656643622158071593730710446834564208984375"},
    {0x1.fffffffffffffp-737, chars_format::fixed, 789,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000002766452331409032358280833466671521986836653470502584447808364886451738764839781419862726370017184"
        "4757424565727518424897844778852620566400175517591462637215331726707825049300908154682949087329399338172776"
        "2691154143462835511238084715878574032523407548393669942974450912897758025668521308621058980681921486748384"
        "6953415721408554995432025655200122293384826394470933724334491795896572926940653733497384709635786529496857"
        "4486981938745777519056025141259504779877652686527670521635459141495217564045540959891512185271777873298215"
        "73715331328724431614318746142089366912841796875"},
    {0x1.fffffffffffffp-736, chars_format::fixed, 788,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000005532904662818064716561666933343043973673306941005168895616729772903477529679562839725452740034368"
        "9514849131455036849795689557705241132800351035182925274430663453415650098601816309365898174658798676345552"
        "5382308286925671022476169431757148065046815096787339885948901825795516051337042617242117961363842973496769"
        "3906831442817109990864051310400244586769652788941867448668983591793145853881307466994769419271573058993714"
        "8973963877491555038112050282519009559755305373055341043270918282990435128091081919783024370543555746596431"
        "4743066265744886322863749228417873382568359375"},
    {0x1.fffffffffffffp-735, chars_format::fixed, 787,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000011065809325636129433123333866686087947346613882010337791233459545806955059359125679450905480068737"
        "9029698262910073699591379115410482265600702070365850548861326906831300197203632618731796349317597352691105"
        "0764616573851342044952338863514296130093630193574679771897803651591032102674085234484235922727685946993538"
        "7813662885634219981728102620800489173539305577883734897337967183586291707762614933989538838543146117987429"
        "7947927754983110076224100565038019119510610746110682086541836565980870256182163839566048741087111493192862"
        "948613253148977264572749845683574676513671875"},
    {0x1.fffffffffffffp-734, chars_format::fixed, 786,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000022131618651272258866246667733372175894693227764020675582466919091613910118718251358901810960137475"
        "8059396525820147399182758230820964531201404140731701097722653813662600394407265237463592698635194705382210"
        "1529233147702684089904677727028592260187260387149359543795607303182064205348170468968471845455371893987077"
        "5627325771268439963456205241600978347078611155767469794675934367172583415525229867979077677086292235974859"
        "5895855509966220152448201130076038239021221492221364173083673131961740512364327679132097482174222986385725"
        "89722650629795452914549969136714935302734375"},
    {0x1.fffffffffffffp-733, chars_format::fixed, 785,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000044263237302544517732493335466744351789386455528041351164933838183227820237436502717803621920274951"
        "6118793051640294798365516461641929062402808281463402195445307627325200788814530474927185397270389410764420"
        "3058466295405368179809355454057184520374520774298719087591214606364128410696340937936943690910743787974155"
        "1254651542536879926912410483201956694157222311534939589351868734345166831050459735958155354172584471949719"
        "1791711019932440304896402260152076478042442984442728346167346263923481024728655358264194964348445972771451"
        "7944530125959090582909993827342987060546875"},
    {0x1.fffffffffffffp-732, chars_format::fixed, 784,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000088526474605089035464986670933488703578772911056082702329867676366455640474873005435607243840549903"
        "2237586103280589596731032923283858124805616562926804390890615254650401577629060949854370794540778821528840"
        "6116932590810736359618710908114369040749041548597438175182429212728256821392681875873887381821487575948310"
        "2509303085073759853824820966403913388314444623069879178703737468690333662100919471916310708345168943899438"
        "3583422039864880609792804520304152956084885968885456692334692527846962049457310716528389928696891945542903"
        "588906025191818116581998765468597412109375"},
    {0x1.fffffffffffffp-731, chars_format::fixed, 783,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000177052949210178070929973341866977407157545822112165404659735352732911280949746010871214487681099806"
        "4475172206561179193462065846567716249611233125853608781781230509300803155258121899708741589081557643057681"
        "2233865181621472719237421816228738081498083097194876350364858425456513642785363751747774763642975151896620"
        "5018606170147519707649641932807826776628889246139758357407474937380667324201838943832621416690337887798876"
        "7166844079729761219585609040608305912169771937770913384669385055693924098914621433056779857393783891085807"
        "17781205038363623316399753093719482421875"},
    {0x1.fffffffffffffp-730, chars_format::fixed, 782,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000354105898420356141859946683733954814315091644224330809319470705465822561899492021742428975362199612"
        "8950344413122358386924131693135432499222466251707217563562461018601606310516243799417483178163115286115362"
        "4467730363242945438474843632457476162996166194389752700729716850913027285570727503495549527285950303793241"
        "0037212340295039415299283865615653553257778492279516714814949874761334648403677887665242833380675775597753"
        "4333688159459522439171218081216611824339543875541826769338770111387848197829242866113559714787567782171614"
        "3556241007672724663279950618743896484375"},
    {0x1.fffffffffffffp-729, chars_format::fixed, 781,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000708211796840712283719893367467909628630183288448661618638941410931645123798984043484857950724399225"
        "7900688826244716773848263386270864998444932503414435127124922037203212621032487598834966356326230572230724"
        "8935460726485890876949687264914952325992332388779505401459433701826054571141455006991099054571900607586482"
        "0074424680590078830598567731231307106515556984559033429629899749522669296807355775330485666761351551195506"
        "8667376318919044878342436162433223648679087751083653538677540222775696395658485732227119429575135564343228"
        "711248201534544932655990123748779296875"},
    {0x1.fffffffffffffp-728, chars_format::fixed, 780,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000001416423593681424567439786734935819257260366576897323237277882821863290247597968086969715901448798451"
        "5801377652489433547696526772541729996889865006828870254249844074406425242064975197669932712652461144461449"
        "7870921452971781753899374529829904651984664777559010802918867403652109142282910013982198109143801215172964"
        "0148849361180157661197135462462614213031113969118066859259799499045338593614711550660971333522703102391013"
        "7334752637838089756684872324866447297358175502167307077355080445551392791316971464454238859150271128686457"
        "42249640306908986531198024749755859375"},
    {0x1.fffffffffffffp-727, chars_format::fixed, 779,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000002832847187362849134879573469871638514520733153794646474555765643726580495195936173939431802897596903"
        "1602755304978867095393053545083459993779730013657740508499688148812850484129950395339865425304922288922899"
        "5741842905943563507798749059659809303969329555118021605837734807304218284565820027964396218287602430345928"
        "0297698722360315322394270924925228426062227938236133718519598998090677187229423101321942667045406204782027"
        "4669505275676179513369744649732894594716351004334614154710160891102785582633942928908477718300542257372914"
        "8449928061381797306239604949951171875"},
    {0x1.fffffffffffffp-726, chars_format::fixed, 778,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000005665694374725698269759146939743277029041466307589292949111531287453160990391872347878863605795193806"
        "3205510609957734190786107090166919987559460027315481016999376297625700968259900790679730850609844577845799"
        "1483685811887127015597498119319618607938659110236043211675469614608436569131640055928792436575204860691856"
        "0595397444720630644788541849850456852124455876472267437039197996181354374458846202643885334090812409564054"
        "9339010551352359026739489299465789189432702008669228309420321782205571165267885857816955436601084514745829"
        "689985612276359461247920989990234375"},
    {0x1.fffffffffffffp-725, chars_format::fixed, 777,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000011331388749451396539518293879486554058082932615178585898223062574906321980783744695757727211590387612"
        "6411021219915468381572214180333839975118920054630962033998752595251401936519801581359461701219689155691598"
        "2967371623774254031194996238639237215877318220472086423350939229216873138263280111857584873150409721383712"
        "1190794889441261289577083699700913704248911752944534874078395992362708748917692405287770668181624819128109"
        "8678021102704718053478978598931578378865404017338456618840643564411142330535771715633910873202169029491659"
        "37997122455271892249584197998046875"},
    {0x1.fffffffffffffp-724, chars_format::fixed, 776,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000022662777498902793079036587758973108116165865230357171796446125149812643961567489391515454423180775225"
        "2822042439830936763144428360667679950237840109261924067997505190502803873039603162718923402439378311383196"
        "5934743247548508062389992477278474431754636440944172846701878458433746276526560223715169746300819442767424"
        "2381589778882522579154167399401827408497823505889069748156791984725417497835384810575541336363249638256219"
        "7356042205409436106957957197863156757730808034676913237681287128822284661071543431267821746404338058983318"
        "7599424491054378449916839599609375"},
    {0x1.fffffffffffffp-723, chars_format::fixed, 775,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000045325554997805586158073175517946216232331730460714343592892250299625287923134978783030908846361550450"
        "5644084879661873526288856721335359900475680218523848135995010381005607746079206325437846804878756622766393"
        "1869486495097016124779984954556948863509272881888345693403756916867492553053120447430339492601638885534848"
        "4763179557765045158308334798803654816995647011778139496313583969450834995670769621151082672726499276512439"
        "4712084410818872213915914395726313515461616069353826475362574257644569322143086862535643492808676117966637"
        "519884898210875689983367919921875"},
    {0x1.fffffffffffffp-722, chars_format::fixed, 774,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000090651109995611172316146351035892432464663460921428687185784500599250575846269957566061817692723100901"
        "1288169759323747052577713442670719800951360437047696271990020762011215492158412650875693609757513245532786"
        "3738972990194032249559969909113897727018545763776691386807513833734985106106240894860678985203277771069696"
        "9526359115530090316616669597607309633991294023556278992627167938901669991341539242302165345452998553024878"
        "9424168821637744427831828791452627030923232138707652950725148515289138644286173725071286985617352235933275"
        "03976979642175137996673583984375"},
    {0x1.fffffffffffffp-721, chars_format::fixed, 773,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000181302219991222344632292702071784864929326921842857374371569001198501151692539915132123635385446201802"
        "2576339518647494105155426885341439601902720874095392543980041524022430984316825301751387219515026491065572"
        "7477945980388064499119939818227795454037091527553382773615027667469970212212481789721357970406555542139393"
        "9052718231060180633233339195214619267982588047112557985254335877803339982683078484604330690905997106049757"
        "8848337643275488855663657582905254061846464277415305901450297030578277288572347450142573971234704471866550"
        "0795395928435027599334716796875"},
    {0x1.fffffffffffffp-720, chars_format::fixed, 772,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000362604439982444689264585404143569729858653843685714748743138002397002303385079830264247270770892403604"
        "5152679037294988210310853770682879203805441748190785087960083048044861968633650603502774439030052982131145"
        "4955891960776128998239879636455590908074183055106765547230055334939940424424963579442715940813111084278787"
        "8105436462120361266466678390429238535965176094225115970508671755606679965366156969208661381811994212099515"
        "7696675286550977711327315165810508123692928554830611802900594061156554577144694900285147942469408943733100"
        "159079185687005519866943359375"},
    {0x1.fffffffffffffp-719, chars_format::fixed, 771,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000725208879964889378529170808287139459717307687371429497486276004794004606770159660528494541541784807209"
        "0305358074589976420621707541365758407610883496381570175920166096089723937267301207005548878060105964262290"
        "9911783921552257996479759272911181816148366110213531094460110669879880848849927158885431881626222168557575"
        "6210872924240722532933356780858477071930352188450231941017343511213359930732313938417322763623988424199031"
        "5393350573101955422654630331621016247385857109661223605801188122313109154289389800570295884938817887466200"
        "31815837137401103973388671875"},
    {0x1.fffffffffffffp-718, chars_format::fixed, 770,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0001450417759929778757058341616574278919434615374742858994972552009588009213540319321056989083083569614418"
        "0610716149179952841243415082731516815221766992763140351840332192179447874534602414011097756120211928524581"
        "9823567843104515992959518545822363632296732220427062188920221339759761697699854317770863763252444337115151"
        "2421745848481445065866713561716954143860704376900463882034687022426719861464627876834645527247976848398063"
        "0786701146203910845309260663242032494771714219322447211602376244626218308578779601140591769877635774932400"
        "6363167427480220794677734375"},
    {0x1.fffffffffffffp-717, chars_format::fixed, 769,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0002900835519859557514116683233148557838869230749485717989945104019176018427080638642113978166167139228836"
        "1221432298359905682486830165463033630443533985526280703680664384358895749069204828022195512240423857049163"
        "9647135686209031985919037091644727264593464440854124377840442679519523395399708635541727526504888674230302"
        "4843491696962890131733427123433908287721408753800927764069374044853439722929255753669291054495953696796126"
        "1573402292407821690618521326484064989543428438644894423204752489252436617157559202281183539755271549864801"
        "272633485496044158935546875"},
    {0x1.fffffffffffffp-716, chars_format::fixed, 768,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0005801671039719115028233366466297115677738461498971435979890208038352036854161277284227956332334278457672"
        "2442864596719811364973660330926067260887067971052561407361328768717791498138409656044391024480847714098327"
        "9294271372418063971838074183289454529186928881708248755680885359039046790799417271083455053009777348460604"
        "9686983393925780263466854246867816575442817507601855528138748089706879445858511507338582108991907393592252"
        "3146804584815643381237042652968129979086856877289788846409504978504873234315118404562367079510543099729602"
        "54526697099208831787109375"},
    {0x1.fffffffffffffp-715, chars_format::fixed, 767,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0011603342079438230056466732932594231355476922997942871959780416076704073708322554568455912664668556915344"
        "4885729193439622729947320661852134521774135942105122814722657537435582996276819312088782048961695428196655"
        "8588542744836127943676148366578909058373857763416497511361770718078093581598834542166910106019554696921209"
        "9373966787851560526933708493735633150885635015203711056277496179413758891717023014677164217983814787184504"
        "6293609169631286762474085305936259958173713754579577692819009957009746468630236809124734159021086199459205"
        "0905339419841766357421875"},
    {0x1.fffffffffffffp-714, chars_format::fixed, 766,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0023206684158876460112933465865188462710953845995885743919560832153408147416645109136911825329337113830688"
        "9771458386879245459894641323704269043548271884210245629445315074871165992553638624177564097923390856393311"
        "7177085489672255887352296733157818116747715526832995022723541436156187163197669084333820212039109393842419"
        "8747933575703121053867416987471266301771270030407422112554992358827517783434046029354328435967629574369009"
        "2587218339262573524948170611872519916347427509159155385638019914019492937260473618249468318042172398918410"
        "181067883968353271484375"},
    {0x1.fffffffffffffp-713, chars_format::fixed, 765,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0046413368317752920225866931730376925421907691991771487839121664306816294833290218273823650658674227661377"
        "9542916773758490919789282647408538087096543768420491258890630149742331985107277248355128195846781712786623"
        "4354170979344511774704593466315636233495431053665990045447082872312374326395338168667640424078218787684839"
        "7495867151406242107734833974942532603542540060814844225109984717655035566868092058708656871935259148738018"
        "5174436678525147049896341223745039832694855018318310771276039828038985874520947236498936636084344797836820"
        "36213576793670654296875"},
    {0x1.fffffffffffffp-712, chars_format::fixed, 764,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0092826736635505840451733863460753850843815383983542975678243328613632589666580436547647301317348455322755"
        "9085833547516981839578565294817076174193087536840982517781260299484663970214554496710256391693563425573246"
        "8708341958689023549409186932631272466990862107331980090894165744624748652790676337335280848156437575369679"
        "4991734302812484215469667949885065207085080121629688450219969435310071133736184117417313743870518297476037"
        "0348873357050294099792682447490079665389710036636621542552079656077971749041894472997873272168689595673640"
        "7242715358734130859375"},
    {0x1.fffffffffffffp-711, chars_format::fixed, 763,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0185653473271011680903467726921507701687630767967085951356486657227265179333160873095294602634696910645511"
        "8171667095033963679157130589634152348386175073681965035562520598969327940429108993420512783387126851146493"
        "7416683917378047098818373865262544933981724214663960181788331489249497305581352674670561696312875150739358"
        "9983468605624968430939335899770130414170160243259376900439938870620142267472368234834627487741036594952074"
        "0697746714100588199585364894980159330779420073273243085104159312155943498083788945995746544337379191347281"
        "448543071746826171875"},
    {0x1.fffffffffffffp-710, chars_format::fixed, 762,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0371306946542023361806935453843015403375261535934171902712973314454530358666321746190589205269393821291023"
        "6343334190067927358314261179268304696772350147363930071125041197938655880858217986841025566774253702292987"
        "4833367834756094197636747730525089867963448429327920363576662978498994611162705349341123392625750301478717"
        "9966937211249936861878671799540260828340320486518753800879877741240284534944736469669254975482073189904148"
        "1395493428201176399170729789960318661558840146546486170208318624311886996167577891991493088674758382694562"
        "89708614349365234375"},
    {0x1.fffffffffffffp-709, chars_format::fixed, 761,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0742613893084046723613870907686030806750523071868343805425946628909060717332643492381178410538787642582047"
        "2686668380135854716628522358536609393544700294727860142250082395877311761716435973682051133548507404585974"
        "9666735669512188395273495461050179735926896858655840727153325956997989222325410698682246785251500602957435"
        "9933874422499873723757343599080521656680640973037507601759755482480569069889472939338509950964146379808296"
        "2790986856402352798341459579920637323117680293092972340416637248623773992335155783982986177349516765389125"
        "7941722869873046875"},
    {0x1.fffffffffffffp-708, chars_format::fixed, 760,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "1485227786168093447227741815372061613501046143736687610851893257818121434665286984762356821077575285164094"
        "5373336760271709433257044717073218787089400589455720284500164791754623523432871947364102267097014809171949"
        "9333471339024376790546990922100359471853793717311681454306651913995978444650821397364493570503001205914871"
        "9867748844999747447514687198161043313361281946075015203519510964961138139778945878677019901928292759616592"
        "5581973712804705596682919159841274646235360586185944680833274497247547984670311567965972354699033530778251"
        "588344573974609375"},
    {0x1.fffffffffffffp-707, chars_format::fixed, 759,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "2970455572336186894455483630744123227002092287473375221703786515636242869330573969524713642155150570328189"
        "0746673520543418866514089434146437574178801178911440569000329583509247046865743894728204534194029618343899"
        "8666942678048753581093981844200718943707587434623362908613303827991956889301642794728987141006002411829743"
        "9735497689999494895029374396322086626722563892150030407039021929922276279557891757354039803856585519233185"
        "1163947425609411193365838319682549292470721172371889361666548994495095969340623135931944709398067061556503"
        "17668914794921875"},
    {0x1.fffffffffffffp-706, chars_format::fixed, 758,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "5940911144672373788910967261488246454004184574946750443407573031272485738661147939049427284310301140656378"
        "1493347041086837733028178868292875148357602357822881138000659167018494093731487789456409068388059236687799"
        "7333885356097507162187963688401437887415174869246725817226607655983913778603285589457974282012004823659487"
        "9470995379998989790058748792644173253445127784300060814078043859844552559115783514708079607713171038466370"
        "2327894851218822386731676639365098584941442344743778723333097988990191938681246271863889418796134123113006"
        "3533782958984375"},
    {0x1.fffffffffffffp-705, chars_format::fixed, 757,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
        "1881822289344747577821934522976492908008369149893500886815146062544971477322295878098854568620602281312756"
        "2986694082173675466056357736585750296715204715645762276001318334036988187462975578912818136776118473375599"
        "4667770712195014324375927376802875774830349738493451634453215311967827557206571178915948564024009647318975"
        "8941990759997979580117497585288346506890255568600121628156087719689105118231567029416159215426342076932740"
        "4655789702437644773463353278730197169882884689487557446666195977980383877362492543727778837592268246226012"
        "706756591796875"},
    {0x1.fffffffffffffp-704, chars_format::fixed, 756,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002"
        "3763644578689495155643869045952985816016738299787001773630292125089942954644591756197709137241204562625512"
        "5973388164347350932112715473171500593430409431291524552002636668073976374925951157825636273552236946751198"
        "9335541424390028648751854753605751549660699476986903268906430623935655114413142357831897128048019294637951"
        "7883981519995959160234995170576693013780511137200243256312175439378210236463134058832318430852684153865480"
        "9311579404875289546926706557460394339765769378975114893332391955960767754724985087455557675184536492452025"
        "41351318359375"},
    {0x1.fffffffffffffp-703, chars_format::fixed, 755,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004"
        "7527289157378990311287738091905971632033476599574003547260584250179885909289183512395418274482409125251025"
        "1946776328694701864225430946343001186860818862583049104005273336147952749851902315651272547104473893502397"
        "8671082848780057297503709507211503099321398953973806537812861247871310228826284715663794256096038589275903"
        "5767963039991918320469990341153386027561022274400486512624350878756420472926268117664636861705368307730961"
        "8623158809750579093853413114920788679531538757950229786664783911921535509449970174911115350369072984904050"
        "8270263671875"},
    {0x1.fffffffffffffp-702, chars_format::fixed, 754,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009"
        "5054578314757980622575476183811943264066953199148007094521168500359771818578367024790836548964818250502050"
        "3893552657389403728450861892686002373721637725166098208010546672295905499703804631302545094208947787004795"
        "7342165697560114595007419014423006198642797907947613075625722495742620457652569431327588512192077178551807"
        "1535926079983836640939980682306772055122044548800973025248701757512840945852536235329273723410736615461923"
        "7246317619501158187706826229841577359063077515900459573329567823843071018899940349822230700738145969808101"
        "654052734375"},
    {0x1.fffffffffffffp-701, chars_format::fixed, 753,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000019"
        "0109156629515961245150952367623886528133906398296014189042337000719543637156734049581673097929636501004100"
        "7787105314778807456901723785372004747443275450332196416021093344591810999407609262605090188417895574009591"
        "4684331395120229190014838028846012397285595815895226151251444991485240915305138862655177024384154357103614"
        "3071852159967673281879961364613544110244089097601946050497403515025681891705072470658547446821473230923847"
        "4492635239002316375413652459683154718126155031800919146659135647686142037799880699644461401476291939616203"
        "30810546875"},
    {0x1.fffffffffffffp-700, chars_format::fixed, 752,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000038"
        "0218313259031922490301904735247773056267812796592028378084674001439087274313468099163346195859273002008201"
        "5574210629557614913803447570744009494886550900664392832042186689183621998815218525210180376835791148019182"
        "9368662790240458380029676057692024794571191631790452302502889982970481830610277725310354048768308714207228"
        "6143704319935346563759922729227088220488178195203892100994807030051363783410144941317094893642946461847694"
        "8985270478004632750827304919366309436252310063601838293318271295372284075599761399288922802952583879232406"
        "6162109375"},
    {0x1.fffffffffffffp-699, chars_format::fixed, 751,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000076"
        "0436626518063844980603809470495546112535625593184056756169348002878174548626936198326692391718546004016403"
        "1148421259115229827606895141488018989773101801328785664084373378367243997630437050420360753671582296038365"
        "8737325580480916760059352115384049589142383263580904605005779965940963661220555450620708097536617428414457"
        "2287408639870693127519845458454176440976356390407784201989614060102727566820289882634189787285892923695389"
        "7970540956009265501654609838732618872504620127203676586636542590744568151199522798577845605905167758464813"
        "232421875"},
    {0x1.fffffffffffffp-698, chars_format::fixed, 750,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000152"
        "0873253036127689961207618940991092225071251186368113512338696005756349097253872396653384783437092008032806"
        "2296842518230459655213790282976037979546203602657571328168746756734487995260874100840721507343164592076731"
        "7474651160961833520118704230768099178284766527161809210011559931881927322441110901241416195073234856828914"
        "4574817279741386255039690916908352881952712780815568403979228120205455133640579765268379574571785847390779"
        "5941081912018531003309219677465237745009240254407353173273085181489136302399045597155691211810335516929626"
        "46484375"},
    {0x1.fffffffffffffp-697, chars_format::fixed, 749,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000304"
        "1746506072255379922415237881982184450142502372736227024677392011512698194507744793306769566874184016065612"
        "4593685036460919310427580565952075959092407205315142656337493513468975990521748201681443014686329184153463"
        "4949302321923667040237408461536198356569533054323618420023119863763854644882221802482832390146469713657828"
        "9149634559482772510079381833816705763905425561631136807958456240410910267281159530536759149143571694781559"
        "1882163824037062006618439354930475490018480508814706346546170362978272604798091194311382423620671033859252"
        "9296875"},
    {0x1.fffffffffffffp-696, chars_format::fixed, 748,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000608"
        "3493012144510759844830475763964368900285004745472454049354784023025396389015489586613539133748368032131224"
        "9187370072921838620855161131904151918184814410630285312674987026937951981043496403362886029372658368306926"
        "9898604643847334080474816923072396713139066108647236840046239727527709289764443604965664780292939427315657"
        "8299269118965545020158763667633411527810851123262273615916912480821820534562319061073518298287143389563118"
        "3764327648074124013236878709860950980036961017629412693092340725956545209596182388622764847241342067718505"
        "859375"},
    {0x1.fffffffffffffp-695, chars_format::fixed, 747,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001216"
        "6986024289021519689660951527928737800570009490944908098709568046050792778030979173227078267496736064262449"
        "8374740145843677241710322263808303836369628821260570625349974053875903962086992806725772058745316736613853"
        "9797209287694668160949633846144793426278132217294473680092479455055418579528887209931329560585878854631315"
        "6598538237931090040317527335266823055621702246524547231833824961643641069124638122147036596574286779126236"
        "7528655296148248026473757419721901960073922035258825386184681451913090419192364777245529694482684135437011"
        "71875"},
    {0x1.fffffffffffffp-694, chars_format::fixed, 746,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002433"
        "3972048578043039379321903055857475601140018981889816197419136092101585556061958346454156534993472128524899"
        "6749480291687354483420644527616607672739257642521141250699948107751807924173985613451544117490633473227707"
        "9594418575389336321899267692289586852556264434588947360184958910110837159057774419862659121171757709262631"
        "3197076475862180080635054670533646111243404493049094463667649923287282138249276244294073193148573558252473"
        "5057310592296496052947514839443803920147844070517650772369362903826180838384729554491059388965368270874023"
        "4375"},
    {0x1.fffffffffffffp-693, chars_format::fixed, 745,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004866"
        "7944097156086078758643806111714951202280037963779632394838272184203171112123916692908313069986944257049799"
        "3498960583374708966841289055233215345478515285042282501399896215503615848347971226903088234981266946455415"
        "9188837150778672643798535384579173705112528869177894720369917820221674318115548839725318242343515418525262"
        "6394152951724360161270109341067292222486808986098188927335299846574564276498552488588146386297147116504947"
        "0114621184592992105895029678887607840295688141035301544738725807652361676769459108982118777930736541748046"
        "875"},
    {0x1.fffffffffffffp-692, chars_format::fixed, 744,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009733"
        "5888194312172157517287612223429902404560075927559264789676544368406342224247833385816626139973888514099598"
        "6997921166749417933682578110466430690957030570084565002799792431007231696695942453806176469962533892910831"
        "8377674301557345287597070769158347410225057738355789440739835640443348636231097679450636484687030837050525"
        "2788305903448720322540218682134584444973617972196377854670599693149128552997104977176292772594294233009894"
        "0229242369185984211790059357775215680591376282070603089477451615304723353538918217964237555861473083496093"
        "75"},
    {0x1.fffffffffffffp-691, chars_format::fixed, 743,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000019467"
        "1776388624344315034575224446859804809120151855118529579353088736812684448495666771633252279947777028199197"
        "3995842333498835867365156220932861381914061140169130005599584862014463393391884907612352939925067785821663"
        "6755348603114690575194141538316694820450115476711578881479671280886697272462195358901272969374061674101050"
        "5576611806897440645080437364269168889947235944392755709341199386298257105994209954352585545188588466019788"
        "0458484738371968423580118715550431361182752564141206178954903230609446707077836435928475111722946166992187"
        "5"},
    {0x1.fffffffffffffp-690, chars_format::fixed, 742,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000038934"
        "3552777248688630069150448893719609618240303710237059158706177473625368896991333543266504559895554056398394"
        "7991684666997671734730312441865722763828122280338260011199169724028926786783769815224705879850135571643327"
        "3510697206229381150388283076633389640900230953423157762959342561773394544924390717802545938748123348202101"
        "1153223613794881290160874728538337779894471888785511418682398772596514211988419908705171090377176932039576"
        "091696947674393684716023743110086272236550512828241235790980646121889341415567287185695022344589233398437"
        "5"},
    {0x1.fffffffffffffp-689, chars_format::fixed, 741,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000077868"
        "7105554497377260138300897787439219236480607420474118317412354947250737793982667086533009119791108112796789"
        "5983369333995343469460624883731445527656244560676520022398339448057853573567539630449411759700271143286654"
        "7021394412458762300776566153266779281800461906846315525918685123546789089848781435605091877496246696404202"
        "2306447227589762580321749457076675559788943777571022837364797545193028423976839817410342180754353864079152"
        "18339389534878736943204748622017254447310102565648247158196129224377868283113457437139004468917846679687"
        "5"},
    {0x1.fffffffffffffp-688, chars_format::fixed, 740,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000155737"
        "4211108994754520276601795574878438472961214840948236634824709894501475587965334173066018239582216225593579"
        "1966738667990686938921249767462891055312489121353040044796678896115707147135079260898823519400542286573309"
        "4042788824917524601553132306533558563600923813692631051837370247093578179697562871210183754992493392808404"
        "4612894455179525160643498914153351119577887555142045674729595090386056847953679634820684361508707728158304"
        "36678779069757473886409497244034508894620205131296494316392258448755736566226914874278008937835693359375"},
    {0x1.fffffffffffffp-687, chars_format::fixed, 739,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000311474"
        "8422217989509040553203591149756876945922429681896473269649419789002951175930668346132036479164432451187158"
        "3933477335981373877842499534925782110624978242706080089593357792231414294270158521797647038801084573146618"
        "8085577649835049203106264613067117127201847627385262103674740494187156359395125742420367509984986785616808"
        "9225788910359050321286997828306702239155775110284091349459190180772113695907359269641368723017415456316608"
        "7335755813951494777281899448806901778924041026259298863278451689751147313245382974855601787567138671875"},
    {0x1.fffffffffffffp-686, chars_format::fixed, 738,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000622949"
        "6844435979018081106407182299513753891844859363792946539298839578005902351861336692264072958328864902374316"
        "7866954671962747755684999069851564221249956485412160179186715584462828588540317043595294077602169146293237"
        "6171155299670098406212529226134234254403695254770524207349480988374312718790251484840735019969973571233617"
        "8451577820718100642573995656613404478311550220568182698918380361544227391814718539282737446034830912633217"
        "467151162790298955456379889761380355784808205251859772655690337950229462649076594971120357513427734375"},
    {0x1.fffffffffffffp-685, chars_format::fixed, 737,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001245899"
        "3688871958036162212814364599027507783689718727585893078597679156011804703722673384528145916657729804748633"
        "5733909343925495511369998139703128442499912970824320358373431168925657177080634087190588155204338292586475"
        "2342310599340196812425058452268468508807390509541048414698961976748625437580502969681470039939947142467235"
        "6903155641436201285147991313226808956623100441136365397836760723088454783629437078565474892069661825266434"
        "93430232558059791091275977952276071156961641050371954531138067590045892529815318994224071502685546875"},
    {0x1.fffffffffffffp-684, chars_format::fixed, 736,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002491798"
        "7377743916072324425628729198055015567379437455171786157195358312023609407445346769056291833315459609497267"
        "1467818687850991022739996279406256884999825941648640716746862337851314354161268174381176310408676585172950"
        "4684621198680393624850116904536937017614781019082096829397923953497250875161005939362940079879894284934471"
        "3806311282872402570295982626453617913246200882272730795673521446176909567258874157130949784139323650532869"
        "8686046511611958218255195590455214231392328210074390906227613518009178505963063798844814300537109375"},
    {0x1.fffffffffffffp-683, chars_format::fixed, 735,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004983597"
        "4755487832144648851257458396110031134758874910343572314390716624047218814890693538112583666630919218994534"
        "2935637375701982045479992558812513769999651883297281433493724675702628708322536348762352620817353170345900"
        "9369242397360787249700233809073874035229562038164193658795847906994501750322011878725880159759788569868942"
        "7612622565744805140591965252907235826492401764545461591347042892353819134517748314261899568278647301065739"
        "737209302322391643651039118091042846278465642014878181245522703601835701192612759768962860107421875"},
    {0x1.fffffffffffffp-682, chars_format::fixed, 734,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009967194"
        "9510975664289297702514916792220062269517749820687144628781433248094437629781387076225167333261838437989068"
        "5871274751403964090959985117625027539999303766594562866987449351405257416645072697524705241634706340691801"
        "8738484794721574499400467618147748070459124076328387317591695813989003500644023757451760319519577139737885"
        "5225245131489610281183930505814471652984803529090923182694085784707638269035496628523799136557294602131479"
        "47441860464478328730207823618208569255693128402975636249104540720367140238522551953792572021484375"},
    {0x1.fffffffffffffp-681, chars_format::fixed, 733,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000019934389"
        "9021951328578595405029833584440124539035499641374289257562866496188875259562774152450334666523676875978137"
        "1742549502807928181919970235250055079998607533189125733974898702810514833290145395049410483269412681383603"
        "7476969589443148998800935236295496140918248152656774635183391627978007001288047514903520639039154279475771"
        "0450490262979220562367861011628943305969607058181846365388171569415276538070993257047598273114589204262958"
        "9488372092895665746041564723641713851138625680595127249820908144073428047704510390758514404296875"},
    {0x1.fffffffffffffp-680, chars_format::fixed, 732,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000039868779"
        "8043902657157190810059667168880249078070999282748578515125732992377750519125548304900669333047353751956274"
        "3485099005615856363839940470500110159997215066378251467949797405621029666580290790098820966538825362767207"
        "4953939178886297997601870472590992281836496305313549270366783255956014002576095029807041278078308558951542"
        "0900980525958441124735722023257886611939214116363692730776343138830553076141986514095196546229178408525917"
        "897674418579133149208312944728342770227725136119025449964181628814685609540902078151702880859375"},
    {0x1.fffffffffffffp-679, chars_format::fixed, 731,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000079737559"
        "6087805314314381620119334337760498156141998565497157030251465984755501038251096609801338666094707503912548"
        "6970198011231712727679880941000220319994430132756502935899594811242059333160581580197641933077650725534414"
        "9907878357772595995203740945181984563672992610627098540733566511912028005152190059614082556156617117903084"
        "1801961051916882249471444046515773223878428232727385461552686277661106152283973028190393092458356817051835"
        "79534883715826629841662588945668554045545027223805089992836325762937121908180415630340576171875"},
    {0x1.fffffffffffffp-678, chars_format::fixed, 730,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000159475119"
        "2175610628628763240238668675520996312283997130994314060502931969511002076502193219602677332189415007825097"
        "3940396022463425455359761882000440639988860265513005871799189622484118666321163160395283866155301451068829"
        "9815756715545191990407481890363969127345985221254197081467133023824056010304380119228165112313234235806168"
        "3603922103833764498942888093031546447756856465454770923105372555322212304567946056380786184916713634103671"
        "5906976743165325968332517789133710809109005444761017998567265152587424381636083126068115234375"},
    {0x1.fffffffffffffp-677, chars_format::fixed, 729,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000318950238"
        "4351221257257526480477337351041992624567994261988628121005863939022004153004386439205354664378830015650194"
        "7880792044926850910719523764000881279977720531026011743598379244968237332642326320790567732310602902137659"
        "9631513431090383980814963780727938254691970442508394162934266047648112020608760238456330224626468471612336"
        "7207844207667528997885776186063092895513712930909541846210745110644424609135892112761572369833427268207343"
        "181395348633065193666503557826742161821801088952203599713453030517484876327216625213623046875"},
    {0x1.fffffffffffffp-676, chars_format::fixed, 728,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000637900476"
        "8702442514515052960954674702083985249135988523977256242011727878044008306008772878410709328757660031300389"
        "5761584089853701821439047528001762559955441062052023487196758489936474665284652641581135464621205804275319"
        "9263026862180767961629927561455876509383940885016788325868532095296224041217520476912660449252936943224673"
        "4415688415335057995771552372126185791027425861819083692421490221288849218271784225523144739666854536414686"
        "36279069726613038733300711565348432364360217790440719942690606103496975265443325042724609375"},
    {0x1.fffffffffffffp-675, chars_format::fixed, 727,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001275800953"
        "7404885029030105921909349404167970498271977047954512484023455756088016612017545756821418657515320062600779"
        "1523168179707403642878095056003525119910882124104046974393516979872949330569305283162270929242411608550639"
        "8526053724361535923259855122911753018767881770033576651737064190592448082435040953825320898505873886449346"
        "8831376830670115991543104744252371582054851723638167384842980442577698436543568451046289479333709072829372"
        "7255813945322607746660142313069686472872043558088143988538121220699395053088665008544921875"},
    {0x1.fffffffffffffp-674, chars_format::fixed, 726,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002551601907"
        "4809770058060211843818698808335940996543954095909024968046911512176033224035091513642837315030640125201558"
        "3046336359414807285756190112007050239821764248208093948787033959745898661138610566324541858484823217101279"
        "7052107448723071846519710245823506037535763540067153303474128381184896164870081907650641797011747772898693"
        "7662753661340231983086209488504743164109703447276334769685960885155396873087136902092578958667418145658745"
        "451162789064521549332028462613937294574408711617628797707624244139879010617733001708984375"},
    {0x1.fffffffffffffp-673, chars_format::fixed, 725,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005103203814"
        "9619540116120423687637397616671881993087908191818049936093823024352066448070183027285674630061280250403116"
        "6092672718829614571512380224014100479643528496416187897574067919491797322277221132649083716969646434202559"
        "4104214897446143693039420491647012075071527080134306606948256762369792329740163815301283594023495545797387"
        "5325507322680463966172418977009486328219406894552669539371921770310793746174273804185157917334836291317490"
        "90232557812904309866405692522787458914881742323525759541524848827975802123546600341796875"},
    {0x1.fffffffffffffp-672, chars_format::fixed, 724,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010206407629"
        "9239080232240847375274795233343763986175816383636099872187646048704132896140366054571349260122560500806233"
        "2185345437659229143024760448028200959287056992832375795148135838983594644554442265298167433939292868405118"
        "8208429794892287386078840983294024150143054160268613213896513524739584659480327630602567188046991091594775"
        "0651014645360927932344837954018972656438813789105339078743843540621587492348547608370315834669672582634981"
        "8046511562580861973281138504557491782976348464705151908304969765595160424709320068359375"},
    {0x1.fffffffffffffp-671, chars_format::fixed, 723,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000020412815259"
        "8478160464481694750549590466687527972351632767272199744375292097408265792280732109142698520245121001612466"
        "4370690875318458286049520896056401918574113985664751590296271677967189289108884530596334867878585736810237"
        "6416859589784574772157681966588048300286108320537226427793027049479169318960655261205134376093982183189550"
        "1302029290721855864689675908037945312877627578210678157487687081243174984697095216740631669339345165269963"
        "609302312516172394656227700911498356595269692941030381660993953119032084941864013671875"},
    {0x1.fffffffffffffp-670, chars_format::fixed, 722,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000040825630519"
        "6956320928963389501099180933375055944703265534544399488750584194816531584561464218285397040490242003224932"
        "8741381750636916572099041792112803837148227971329503180592543355934378578217769061192669735757171473620475"
        "2833719179569149544315363933176096600572216641074452855586054098958338637921310522410268752187964366379100"
        "2604058581443711729379351816075890625755255156421356314975374162486349969394190433481263338678690330539927"
        "21860462503234478931245540182299671319053938588206076332198790623806416988372802734375"},
    {0x1.fffffffffffffp-669, chars_format::fixed, 721,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000081651261039"
        "3912641857926779002198361866750111889406531069088798977501168389633063169122928436570794080980484006449865"
        "7482763501273833144198083584225607674296455942659006361185086711868757156435538122385339471514342947240950"
        "5667438359138299088630727866352193201144433282148905711172108197916677275842621044820537504375928732758200"
        "5208117162887423458758703632151781251510510312842712629950748324972699938788380866962526677357380661079854"
        "4372092500646895786249108036459934263810787717641215266439758124761283397674560546875"},
    {0x1.fffffffffffffp-668, chars_format::fixed, 720,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000163302522078"
        "7825283715853558004396723733500223778813062138177597955002336779266126338245856873141588161960968012899731"
        "4965527002547666288396167168451215348592911885318012722370173423737514312871076244770678943028685894481901"
        "1334876718276598177261455732704386402288866564297811422344216395833354551685242089641075008751857465516401"
        "0416234325774846917517407264303562503021020625685425259901496649945399877576761733925053354714761322159708"
        "874418500129379157249821607291986852762157543528243053287951624952256679534912109375"},
    {0x1.fffffffffffffp-667, chars_format::fixed, 719,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000326605044157"
        "5650567431707116008793447467000447557626124276355195910004673558532252676491713746283176323921936025799462"
        "9931054005095332576792334336902430697185823770636025444740346847475028625742152489541357886057371788963802"
        "2669753436553196354522911465408772804577733128595622844688432791666709103370484179282150017503714931032802"
        "0832468651549693835034814528607125006042041251370850519802993299890799755153523467850106709429522644319417"
        "74883700025875831449964321458397370552431508705648610657590324990451335906982421875"},
    {0x1.fffffffffffffp-666, chars_format::fixed, 718,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000653210088315"
        "1301134863414232017586894934000895115252248552710391820009347117064505352983427492566352647843872051598925"
        "9862108010190665153584668673804861394371647541272050889480693694950057251484304979082715772114743577927604"
        "5339506873106392709045822930817545609155466257191245689376865583333418206740968358564300035007429862065604"
        "1664937303099387670069629057214250012084082502741701039605986599781599510307046935700213418859045288638835"
        "4976740005175166289992864291679474110486301741129722131518064998090267181396484375"},
    {0x1.fffffffffffffp-665, chars_format::fixed, 717,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001306420176630"
        "2602269726828464035173789868001790230504497105420783640018694234129010705966854985132705295687744103197851"
        "9724216020381330307169337347609722788743295082544101778961387389900114502968609958165431544229487155855209"
        "0679013746212785418091645861635091218310932514382491378753731166666836413481936717128600070014859724131208"
        "3329874606198775340139258114428500024168165005483402079211973199563199020614093871400426837718090577277670"
        "995348001035033257998572858335894822097260348225944426303612999618053436279296875"},
    {0x1.fffffffffffffp-664, chars_format::fixed, 716,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002612840353260"
        "5204539453656928070347579736003580461008994210841567280037388468258021411933709970265410591375488206395703"
        "9448432040762660614338674695219445577486590165088203557922774779800229005937219916330863088458974311710418"
        "1358027492425570836183291723270182436621865028764982757507462333333672826963873434257200140029719448262416"
        "6659749212397550680278516228857000048336330010966804158423946399126398041228187742800853675436181154555341"
        "99069600207006651599714571667178964419452069645188885260722599923610687255859375"},
    {0x1.fffffffffffffp-663, chars_format::fixed, 715,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005225680706521"
        "0409078907313856140695159472007160922017988421683134560074776936516042823867419940530821182750976412791407"
        "8896864081525321228677349390438891154973180330176407115845549559600458011874439832661726176917948623420836"
        "2716054984851141672366583446540364873243730057529965515014924666667345653927746868514400280059438896524833"
        "3319498424795101360557032457714000096672660021933608316847892798252796082456375485601707350872362309110683"
        "9813920041401330319942914333435792883890413929037777052144519984722137451171875"},
    {0x1.fffffffffffffp-662, chars_format::fixed, 714,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010451361413042"
        "0818157814627712281390318944014321844035976843366269120149553873032085647734839881061642365501952825582815"
        "7793728163050642457354698780877782309946360660352814231691099119200916023748879665323452353835897246841672"
        "5432109969702283344733166893080729746487460115059931030029849333334691307855493737028800560118877793049666"
        "6638996849590202721114064915428000193345320043867216633695785596505592164912750971203414701744724618221367"
        "962784008280266063988582866687158576778082785807555410428903996944427490234375"},
    {0x1.fffffffffffffp-661, chars_format::fixed, 713,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000020902722826084"
        "1636315629255424562780637888028643688071953686732538240299107746064171295469679762123284731003905651165631"
        "5587456326101284914709397561755564619892721320705628463382198238401832047497759330646904707671794493683345"
        "0864219939404566689466333786161459492974920230119862060059698666669382615710987474057601120237755586099333"
        "3277993699180405442228129830856000386690640087734433267391571193011184329825501942406829403489449236442735"
        "92556801656053212797716573337431715355616557161511082085780799388885498046875"},
    {0x1.fffffffffffffp-660, chars_format::fixed, 712,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000041805445652168"
        "3272631258510849125561275776057287376143907373465076480598215492128342590939359524246569462007811302331263"
        "1174912652202569829418795123511129239785442641411256926764396476803664094995518661293809415343588987366690"
        "1728439878809133378932667572322918985949840460239724120119397333338765231421974948115202240475511172198666"
        "6555987398360810884456259661712000773381280175468866534783142386022368659651003884813658806978898472885471"
        "8511360331210642559543314667486343071123311432302216417156159877777099609375"},
    {0x1.fffffffffffffp-659, chars_format::fixed, 711,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000083610891304336"
        "6545262517021698251122551552114574752287814746930152961196430984256685181878719048493138924015622604662526"
        "2349825304405139658837590247022258479570885282822513853528792953607328189991037322587618830687177974733380"
        "3456879757618266757865335144645837971899680920479448240238794666677530462843949896230404480951022344397333"
        "3111974796721621768912519323424001546762560350937733069566284772044737319302007769627317613957796945770943"
        "702272066242128511908662933497268614224662286460443283431231975555419921875"},
    {0x1.fffffffffffffp-658, chars_format::fixed, 710,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000167221782608673"
        "3090525034043396502245103104229149504575629493860305922392861968513370363757438096986277848031245209325052"
        "4699650608810279317675180494044516959141770565645027707057585907214656379982074645175237661374355949466760"
        "6913759515236533515730670289291675943799361840958896480477589333355060925687899792460808961902044688794666"
        "6223949593443243537825038646848003093525120701875466139132569544089474638604015539254635227915593891541887"
        "40454413248425702381732586699453722844932457292088656686246395111083984375"},
    {0x1.fffffffffffffp-657, chars_format::fixed, 709,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000334443565217346"
        "6181050068086793004490206208458299009151258987720611844785723937026740727514876193972555696062490418650104"
        "9399301217620558635350360988089033918283541131290055414115171814429312759964149290350475322748711898933521"
        "3827519030473067031461340578583351887598723681917792960955178666710121851375799584921617923804089377589333"
        "2447899186886487075650077293696006187050241403750932278265139088178949277208031078509270455831187783083774"
        "8090882649685140476346517339890744568986491458417731337249279022216796875"},
    {0x1.fffffffffffffp-656, chars_format::fixed, 708,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000668887130434693"
        "2362100136173586008980412416916598018302517975441223689571447874053481455029752387945111392124980837300209"
        "8798602435241117270700721976178067836567082262580110828230343628858625519928298580700950645497423797867042"
        "7655038060946134062922681157166703775197447363835585921910357333420243702751599169843235847608178755178666"
        "4895798373772974151300154587392012374100482807501864556530278176357898554416062157018540911662375566167549"
        "618176529937028095269303467978148913797298291683546267449855804443359375"},
    {0x1.fffffffffffffp-655, chars_format::fixed, 707,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001337774260869386"
        "4724200272347172017960824833833196036605035950882447379142895748106962910059504775890222784249961674600419"
        "7597204870482234541401443952356135673134164525160221656460687257717251039856597161401901290994847595734085"
        "5310076121892268125845362314333407550394894727671171843820714666840487405503198339686471695216357510357332"
        "9791596747545948302600309174784024748200965615003729113060556352715797108832124314037081823324751132335099"
        "23635305987405619053860693595629782759459658336709253489971160888671875"},
    {0x1.fffffffffffffp-654, chars_format::fixed, 706,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002675548521738772"
        "9448400544694344035921649667666392073210071901764894758285791496213925820119009551780445568499923349200839"
        "5194409740964469082802887904712271346268329050320443312921374515434502079713194322803802581989695191468171"
        "0620152243784536251690724628666815100789789455342343687641429333680974811006396679372943390432715020714665"
        "9583193495091896605200618349568049496401931230007458226121112705431594217664248628074163646649502264670198"
        "4727061197481123810772138719125956551891931667341850697994232177734375"},
    {0x1.fffffffffffffp-653, chars_format::fixed, 705,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000005351097043477545"
        "8896801089388688071843299335332784146420143803529789516571582992427851640238019103560891136999846698401679"
        "0388819481928938165605775809424542692536658100640886625842749030869004159426388645607605163979390382936342"
        "1240304487569072503381449257333630201579578910684687375282858667361949622012793358745886780865430041429331"
        "9166386990183793210401236699136098992803862460014916452242225410863188435328497256148327293299004529340396"
        "945412239496224762154427743825191310378386333468370139598846435546875"},
    {0x1.fffffffffffffp-652, chars_format::fixed, 704,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000010702194086955091"
        "7793602178777376143686598670665568292840287607059579033143165984855703280476038207121782273999693396803358"
        "0777638963857876331211551618849085385073316201281773251685498061738008318852777291215210327958780765872684"
        "2480608975138145006762898514667260403159157821369374750565717334723899244025586717491773561730860082858663"
        "8332773980367586420802473398272197985607724920029832904484450821726376870656994512296654586598009058680793"
        "89082447899244952430885548765038262075677266693674027919769287109375"},
    {0x1.fffffffffffffp-651, chars_format::fixed, 703,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000021404388173910183"
        "5587204357554752287373197341331136585680575214119158066286331969711406560952076414243564547999386793606716"
        "1555277927715752662423103237698170770146632402563546503370996123476016637705554582430420655917561531745368"
        "4961217950276290013525797029334520806318315642738749501131434669447798488051173434983547123461720165717327"
        "6665547960735172841604946796544395971215449840059665808968901643452753741313989024593309173196018117361587"
        "7816489579848990486177109753007652415135453338734805583953857421875"},
    {0x1.fffffffffffffp-650, chars_format::fixed, 702,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000042808776347820367"
        "1174408715109504574746394682662273171361150428238316132572663939422813121904152828487129095998773587213432"
        "3110555855431505324846206475396341540293264805127093006741992246952033275411109164860841311835123063490736"
        "9922435900552580027051594058669041612636631285477499002262869338895596976102346869967094246923440331434655"
        "3331095921470345683209893593088791942430899680119331617937803286905507482627978049186618346392036234723175"
        "563297915969798097235421950601530483027090667746961116790771484375"},
    {0x1.fffffffffffffp-649, chars_format::fixed, 701,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000085617552695640734"
        "2348817430219009149492789365324546342722300856476632265145327878845626243808305656974258191997547174426864"
        "6221111710863010649692412950792683080586529610254186013483984493904066550822218329721682623670246126981473"
        "9844871801105160054103188117338083225273262570954998004525738677791193952204693739934188493846880662869310"
        "6662191842940691366419787186177583884861799360238663235875606573811014965255956098373236692784072469446351"
        "12659583193959619447084390120306096605418133549392223358154296875"},
    {0x1.fffffffffffffp-648, chars_format::fixed, 700,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000171235105391281468"
        "4697634860438018298985578730649092685444601712953264530290655757691252487616611313948516383995094348853729"
        "2442223421726021299384825901585366161173059220508372026967968987808133101644436659443365247340492253962947"
        "9689743602210320108206376234676166450546525141909996009051477355582387904409387479868376987693761325738621"
        "3324383685881382732839574372355167769723598720477326471751213147622029930511912196746473385568144938892702"
        "2531916638791923889416878024061219321083626709878444671630859375"},
    {0x1.fffffffffffffp-647, chars_format::fixed, 699,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000342470210782562936"
        "9395269720876036597971157461298185370889203425906529060581311515382504975233222627897032767990188697707458"
        "4884446843452042598769651803170732322346118441016744053935937975616266203288873318886730494680984507925895"
        "9379487204420640216412752469352332901093050283819992018102954711164775808818774959736753975387522651477242"
        "6648767371762765465679148744710335539447197440954652943502426295244059861023824393492946771136289877785404"
        "506383327758384777883375604812243864216725341975688934326171875"},
    {0x1.fffffffffffffp-646, chars_format::fixed, 698,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000684940421565125873"
        "8790539441752073195942314922596370741778406851813058121162623030765009950466445255794065535980377395414916"
        "9768893686904085197539303606341464644692236882033488107871875951232532406577746637773460989361969015851791"
        "8758974408841280432825504938704665802186100567639984036205909422329551617637549919473507950775045302954485"
        "3297534743525530931358297489420671078894394881909305887004852590488119722047648786985893542272579755570809"
        "01276665551676955576675120962448772843345068395137786865234375"},
    {0x1.fffffffffffffp-645, chars_format::fixed, 697,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000001369880843130251747"
        "7581078883504146391884629845192741483556813703626116242325246061530019900932890511588131071960754790829833"
        "9537787373808170395078607212682929289384473764066976215743751902465064813155493275546921978723938031703583"
        "7517948817682560865651009877409331604372201135279968072411818844659103235275099838947015901550090605908970"
        "6595069487051061862716594978841342157788789763818611774009705180976239444095297573971787084545159511141618"
        "0255333110335391115335024192489754568669013679027557373046875"},
    {0x1.fffffffffffffp-644, chars_format::fixed, 696,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000002739761686260503495"
        "5162157767008292783769259690385482967113627407252232484650492123060039801865781023176262143921509581659667"
        "9075574747616340790157214425365858578768947528133952431487503804930129626310986551093843957447876063407167"
        "5035897635365121731302019754818663208744402270559936144823637689318206470550199677894031803100181211817941"
        "3190138974102123725433189957682684315577579527637223548019410361952478888190595147943574169090319022283236"
        "051066622067078223067004838497950913733802735805511474609375"},
    {0x1.fffffffffffffp-643, chars_format::fixed, 695,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000005479523372521006991"
        "0324315534016585567538519380770965934227254814504464969300984246120079603731562046352524287843019163319335"
        "8151149495232681580314428850731717157537895056267904862975007609860259252621973102187687914895752126814335"
        "0071795270730243462604039509637326417488804541119872289647275378636412941100399355788063606200362423635882"
        "6380277948204247450866379915365368631155159055274447096038820723904957776381190295887148338180638044566472"
        "10213324413415644613400967699590182746760547161102294921875"},
    {0x1.fffffffffffffp-642, chars_format::fixed, 694,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000010959046745042013982"
        "0648631068033171135077038761541931868454509629008929938601968492240159207463124092705048575686038326638671"
        "6302298990465363160628857701463434315075790112535809725950015219720518505243946204375375829791504253628670"
        "0143590541460486925208079019274652834977609082239744579294550757272825882200798711576127212400724847271765"
        "2760555896408494901732759830730737262310318110548894192077641447809915552762380591774296676361276089132944"
        "2042664882683128922680193539918036549352109432220458984375"},
    {0x1.fffffffffffffp-641, chars_format::fixed, 693,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000021918093490084027964"
        "1297262136066342270154077523083863736909019258017859877203936984480318414926248185410097151372076653277343"
        "2604597980930726321257715402926868630151580225071619451900030439441037010487892408750751659583008507257340"
        "0287181082920973850416158038549305669955218164479489158589101514545651764401597423152254424801449694543530"
        "5521111792816989803465519661461474524620636221097788384155282895619831105524761183548593352722552178265888"
        "408532976536625784536038707983607309870421886444091796875"},
    {0x1.fffffffffffffp-640, chars_format::fixed, 692,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000043836186980168055928"
        "2594524272132684540308155046167727473818038516035719754407873968960636829852496370820194302744153306554686"
        "5209195961861452642515430805853737260303160450143238903800060878882074020975784817501503319166017014514680"
        "0574362165841947700832316077098611339910436328958978317178203029091303528803194846304508849602899389087061"
        "1042223585633979606931039322922949049241272442195576768310565791239662211049522367097186705445104356531776"
        "81706595307325156907207741596721461974084377288818359375"},
    {0x1.fffffffffffffp-639, chars_format::fixed, 691,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000087672373960336111856"
        "5189048544265369080616310092335454947636077032071439508815747937921273659704992741640388605488306613109373"
        "0418391923722905285030861611707474520606320900286477807600121757764148041951569635003006638332034029029360"
        "1148724331683895401664632154197222679820872657917956634356406058182607057606389692609017699205798778174122"
        "2084447171267959213862078645845898098482544884391153536621131582479324422099044734194373410890208713063553"
        "6341319061465031381441548319344292394816875457763671875"},
    {0x1.fffffffffffffp-638, chars_format::fixed, 690,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000175344747920672223713"
        "0378097088530738161232620184670909895272154064142879017631495875842547319409985483280777210976613226218746"
        "0836783847445810570061723223414949041212641800572955615200243515528296083903139270006013276664068058058720"
        "2297448663367790803329264308394445359641745315835913268712812116365214115212779385218035398411597556348244"
        "4168894342535918427724157291691796196965089768782307073242263164958648844198089468388746821780417426127107"
        "268263812293006276288309663868858478963375091552734375"},
    {0x1.fffffffffffffp-637, chars_format::fixed, 689,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000350689495841344447426"
        "0756194177061476322465240369341819790544308128285758035262991751685094638819970966561554421953226452437492"
        "1673567694891621140123446446829898082425283601145911230400487031056592167806278540012026553328136116117440"
        "4594897326735581606658528616788890719283490631671826537425624232730428230425558770436070796823195112696488"
        "8337788685071836855448314583383592393930179537564614146484526329917297688396178936777493643560834852254214"
        "53652762458601255257661932773771695792675018310546875"},
    {0x1.fffffffffffffp-636, chars_format::fixed, 688,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000701378991682688894852"
        "1512388354122952644930480738683639581088616256571516070525983503370189277639941933123108843906452904874984"
        "3347135389783242280246892893659796164850567202291822460800974062113184335612557080024053106656272232234880"
        "9189794653471163213317057233577781438566981263343653074851248465460856460851117540872141593646390225392977"
        "6675577370143673710896629166767184787860359075129228292969052659834595376792357873554987287121669704508429"
        "0730552491720251051532386554754339158535003662109375"},
    {0x1.fffffffffffffp-635, chars_format::fixed, 687,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000001402757983365377789704"
        "3024776708245905289860961477367279162177232513143032141051967006740378555279883866246217687812905809749968"
        "6694270779566484560493785787319592329701134404583644921601948124226368671225114160048106213312544464469761"
        "8379589306942326426634114467155562877133962526687306149702496930921712921702235081744283187292780450785955"
        "3351154740287347421793258333534369575720718150258456585938105319669190753584715747109974574243339409016858"
        "146110498344050210306477310950867831707000732421875"},
    {0x1.fffffffffffffp-634, chars_format::fixed, 686,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000002805515966730755579408"
        "6049553416491810579721922954734558324354465026286064282103934013480757110559767732492435375625811619499937"
        "3388541559132969120987571574639184659402268809167289843203896248452737342450228320096212426625088928939523"
        "6759178613884652853268228934311125754267925053374612299404993861843425843404470163488566374585560901571910"
        "6702309480574694843586516667068739151441436300516913171876210639338381507169431494219949148486678818033716"
        "29222099668810042061295462190173566341400146484375"},
    {0x1.fffffffffffffp-633, chars_format::fixed, 685,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000005611031933461511158817"
        "2099106832983621159443845909469116648708930052572128564207868026961514221119535464984870751251623238999874"
        "6777083118265938241975143149278369318804537618334579686407792496905474684900456640192424853250177857879047"
        "3518357227769305706536457868622251508535850106749224598809987723686851686808940326977132749171121803143821"
        "3404618961149389687173033334137478302882872601033826343752421278676763014338862988439898296973357636067432"
        "5844419933762008412259092438034713268280029296875"},
    {0x1.fffffffffffffp-632, chars_format::fixed, 684,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000011222063866923022317634"
        "4198213665967242318887691818938233297417860105144257128415736053923028442239070929969741502503246477999749"
        "3554166236531876483950286298556738637609075236669159372815584993810949369800913280384849706500355715758094"
        "7036714455538611413072915737244503017071700213498449197619975447373703373617880653954265498342243606287642"
        "6809237922298779374346066668274956605765745202067652687504842557353526028677725976879796593946715272134865"
        "168883986752401682451818487606942653656005859375"},
    {0x1.fffffffffffffp-631, chars_format::fixed, 683,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000022444127733846044635268"
        "8396427331934484637775383637876466594835720210288514256831472107846056884478141859939483005006492955999498"
        "7108332473063752967900572597113477275218150473338318745631169987621898739601826560769699413000711431516189"
        "4073428911077222826145831474489006034143400426996898395239950894747406747235761307908530996684487212575285"
        "3618475844597558748692133336549913211531490404135305375009685114707052057355451953759593187893430544269730"
        "33776797350480336490363697521388530731201171875"},
    {0x1.fffffffffffffp-630, chars_format::fixed, 682,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000044888255467692089270537"
        "6792854663868969275550767275752933189671440420577028513662944215692113768956283719878966010012985911998997"
        "4216664946127505935801145194226954550436300946676637491262339975243797479203653121539398826001422863032378"
        "8146857822154445652291662948978012068286800853993796790479901789494813494471522615817061993368974425150570"
        "7236951689195117497384266673099826423062980808270610750019370229414104114710903907519186375786861088539460"
        "6755359470096067298072739504277706146240234375"},
    {0x1.fffffffffffffp-629, chars_format::fixed, 681,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000089776510935384178541075"
        "3585709327737938551101534551505866379342880841154057027325888431384227537912567439757932020025971823997994"
        "8433329892255011871602290388453909100872601893353274982524679950487594958407306243078797652002845726064757"
        "6293715644308891304583325897956024136573601707987593580959803578989626988943045231634123986737948850301141"
        "4473903378390234994768533346199652846125961616541221500038740458828208229421807815038372751573722177078921"
        "351071894019213459614547900855541229248046875"},
    {0x1.fffffffffffffp-628, chars_format::fixed, 680,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000179553021870768357082150"
        "7171418655475877102203069103011732758685761682308114054651776862768455075825134879515864040051943647995989"
        "6866659784510023743204580776907818201745203786706549965049359900975189916814612486157595304005691452129515"
        "2587431288617782609166651795912048273147203415975187161919607157979253977886090463268247973475897700602282"
        "8947806756780469989537066692399305692251923233082443000077480917656416458843615630076745503147444354157842"
        "70214378803842691922909580171108245849609375"},
    {0x1.fffffffffffffp-627, chars_format::fixed, 679,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000359106043741536714164301"
        "4342837310951754204406138206023465517371523364616228109303553725536910151650269759031728080103887295991979"
        "3733319569020047486409161553815636403490407573413099930098719801950379833629224972315190608011382904259030"
        "5174862577235565218333303591824096546294406831950374323839214315958507955772180926536495946951795401204565"
        "7895613513560939979074133384798611384503846466164886000154961835312832917687231260153491006294888708315685"
        "4042875760768538384581916034221649169921875"},
    {0x1.fffffffffffffp-626, chars_format::fixed, 678,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000718212087483073428328602"
        "8685674621903508408812276412046931034743046729232456218607107451073820303300539518063456160207774591983958"
        "7466639138040094972818323107631272806980815146826199860197439603900759667258449944630381216022765808518061"
        "0349725154471130436666607183648193092588813663900748647678428631917015911544361853072991893903590802409131"
        "5791227027121879958148266769597222769007692932329772000309923670625665835374462520306982012589777416631370"
        "808575152153707676916383206844329833984375"},
    {0x1.fffffffffffffp-625, chars_format::fixed, 677,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000001436424174966146856657205"
        "7371349243807016817624552824093862069486093458464912437214214902147640606601079036126912320415549183967917"
        "4933278276080189945636646215262545613961630293652399720394879207801519334516899889260762432045531617036122"
        "0699450308942260873333214367296386185177627327801497295356857263834031823088723706145983787807181604818263"
        "1582454054243759916296533539194445538015385864659544000619847341251331670748925040613964025179554833262741"
        "61715030430741535383276641368865966796875"},
    {0x1.fffffffffffffp-624, chars_format::fixed, 676,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000002872848349932293713314411"
        "4742698487614033635249105648187724138972186916929824874428429804295281213202158072253824640831098367935834"
        "9866556552160379891273292430525091227923260587304799440789758415603038669033799778521524864091063234072244"
        "1398900617884521746666428734592772370355254655602994590713714527668063646177447412291967575614363209636526"
        "3164908108487519832593067078388891076030771729319088001239694682502663341497850081227928050359109666525483"
        "2343006086148307076655328273773193359375"},
    {0x1.fffffffffffffp-623, chars_format::fixed, 675,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000005745696699864587426628822"
        "9485396975228067270498211296375448277944373833859649748856859608590562426404316144507649281662196735871669"
        "9733113104320759782546584861050182455846521174609598881579516831206077338067599557043049728182126468144488"
        "2797801235769043493332857469185544740710509311205989181427429055336127292354894824583935151228726419273052"
        "6329816216975039665186134156777782152061543458638176002479389365005326682995700162455856100718219333050966"
        "468601217229661415331065654754638671875"},
    {0x1.fffffffffffffp-622, chars_format::fixed, 674,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000011491393399729174853257645"
        "8970793950456134540996422592750896555888747667719299497713719217181124852808632289015298563324393471743339"
        "9466226208641519565093169722100364911693042349219197763159033662412154676135199114086099456364252936288976"
        "5595602471538086986665714938371089481421018622411978362854858110672254584709789649167870302457452838546105"
        "2659632433950079330372268313555564304123086917276352004958778730010653365991400324911712201436438666101932"
        "93720243445932283066213130950927734375"},
    {0x1.fffffffffffffp-621, chars_format::fixed, 673,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000022982786799458349706515291"
        "7941587900912269081992845185501793111777495335438598995427438434362249705617264578030597126648786943486679"
        "8932452417283039130186339444200729823386084698438395526318067324824309352270398228172198912728505872577953"
        "1191204943076173973331429876742178962842037244823956725709716221344509169419579298335740604914905677092210"
        "5319264867900158660744536627111128608246173834552704009917557460021306731982800649823424402872877332203865"
        "8744048689186456613242626190185546875"},
    {0x1.fffffffffffffp-620, chars_format::fixed, 672,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000045965573598916699413030583"
        "5883175801824538163985690371003586223554990670877197990854876868724499411234529156061194253297573886973359"
        "7864904834566078260372678888401459646772169396876791052636134649648618704540796456344397825457011745155906"
        "2382409886152347946662859753484357925684074489647913451419432442689018338839158596671481209829811354184421"
        "0638529735800317321489073254222257216492347669105408019835114920042613463965601299646848805745754664407731"
        "748809737837291322648525238037109375"},
    {0x1.fffffffffffffp-619, chars_format::fixed, 671,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000091931147197833398826061167"
        "1766351603649076327971380742007172447109981341754395981709753737448998822469058312122388506595147773946719"
        "5729809669132156520745357776802919293544338793753582105272269299297237409081592912688795650914023490311812"
        "4764819772304695893325719506968715851368148979295826902838864885378036677678317193342962419659622708368842"
        "1277059471600634642978146508444514432984695338210816039670229840085226927931202599293697611491509328815463"
        "49761947567458264529705047607421875"},
    {0x1.fffffffffffffp-618, chars_format::fixed, 670,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000183862294395666797652122334"
        "3532703207298152655942761484014344894219962683508791963419507474897997644938116624244777013190295547893439"
        "1459619338264313041490715553605838587088677587507164210544538598594474818163185825377591301828046980623624"
        "9529639544609391786651439013937431702736297958591653805677729770756073355356634386685924839319245416737684"
        "2554118943201269285956293016889028865969390676421632079340459680170453855862405198587395222983018657630926"
        "9952389513491652905941009521484375"},
    {0x1.fffffffffffffp-617, chars_format::fixed, 669,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000367724588791333595304244668"
        "7065406414596305311885522968028689788439925367017583926839014949795995289876233248489554026380591095786878"
        "2919238676528626082981431107211677174177355175014328421089077197188949636326371650755182603656093961247249"
        "9059279089218783573302878027874863405472595917183307611355459541512146710713268773371849678638490833475368"
        "5108237886402538571912586033778057731938781352843264158680919360340907711724810397174790445966037315261853"
        "990477902698330581188201904296875"},
    {0x1.fffffffffffffp-616, chars_format::fixed, 668,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000735449177582667190608489337"
        "4130812829192610623771045936057379576879850734035167853678029899591990579752466496979108052761182191573756"
        "5838477353057252165962862214423354348354710350028656842178154394377899272652743301510365207312187922494499"
        "8118558178437567146605756055749726810945191834366615222710919083024293421426537546743699357276981666950737"
        "0216475772805077143825172067556115463877562705686528317361838720681815423449620794349580891932074630523707"
        "98095580539666116237640380859375"},
    {0x1.fffffffffffffp-615, chars_format::fixed, 667,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000001470898355165334381216978674"
        "8261625658385221247542091872114759153759701468070335707356059799183981159504932993958216105522364383147513"
        "1676954706114504331925724428846708696709420700057313684356308788755798545305486603020730414624375844988999"
        "6237116356875134293211512111499453621890383668733230445421838166048586842853075093487398714553963333901474"
        "0432951545610154287650344135112230927755125411373056634723677441363630846899241588699161783864149261047415"
        "9619116107933223247528076171875"},
    {0x1.fffffffffffffp-614, chars_format::fixed, 666,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000002941796710330668762433957349"
        "6523251316770442495084183744229518307519402936140671414712119598367962319009865987916432211044728766295026"
        "3353909412229008663851448857693417393418841400114627368712617577511597090610973206041460829248751689977999"
        "2474232713750268586423024222998907243780767337466460890843676332097173685706150186974797429107926667802948"
        "0865903091220308575300688270224461855510250822746113269447354882727261693798483177398323567728298522094831"
        "923823221586644649505615234375"},
    {0x1.fffffffffffffp-613, chars_format::fixed, 665,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000005883593420661337524867914699"
        "3046502633540884990168367488459036615038805872281342829424239196735924638019731975832864422089457532590052"
        "6707818824458017327702897715386834786837682800229254737425235155023194181221946412082921658497503379955998"
        "4948465427500537172846048445997814487561534674932921781687352664194347371412300373949594858215853335605896"
        "1731806182440617150601376540448923711020501645492226538894709765454523387596966354796647135456597044189663"
        "84764644317328929901123046875"},
    {0x1.fffffffffffffp-612, chars_format::fixed, 664,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000011767186841322675049735829398"
        "6093005267081769980336734976918073230077611744562685658848478393471849276039463951665728844178915065180105"
        "3415637648916034655405795430773669573675365600458509474850470310046388362443892824165843316995006759911996"
        "9896930855001074345692096891995628975123069349865843563374705328388694742824600747899189716431706671211792"
        "3463612364881234301202753080897847422041003290984453077789419530909046775193932709593294270913194088379327"
        "6952928863465785980224609375"},
    {0x1.fffffffffffffp-611, chars_format::fixed, 663,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000023534373682645350099471658797"
        "2186010534163539960673469953836146460155223489125371317696956786943698552078927903331457688357830130360210"
        "6831275297832069310811590861547339147350731200917018949700940620092776724887785648331686633990013519823993"
        "9793861710002148691384193783991257950246138699731687126749410656777389485649201495798379432863413342423584"
        "6927224729762468602405506161795694844082006581968906155578839061818093550387865419186588541826388176758655"
        "390585772693157196044921875"},
    {0x1.fffffffffffffp-610, chars_format::fixed, 662,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000047068747365290700198943317594"
        "4372021068327079921346939907672292920310446978250742635393913573887397104157855806662915376715660260720421"
        "3662550595664138621623181723094678294701462401834037899401881240185553449775571296663373267980027039647987"
        "9587723420004297382768387567982515900492277399463374253498821313554778971298402991596758865726826684847169"
        "3854449459524937204811012323591389688164013163937812311157678123636187100775730838373177083652776353517310"
        "78117154538631439208984375"},
    {0x1.fffffffffffffp-609, chars_format::fixed, 661,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000094137494730581400397886635188"
        "8744042136654159842693879815344585840620893956501485270787827147774794208315711613325830753431320521440842"
        "7325101191328277243246363446189356589402924803668075798803762480371106899551142593326746535960054079295975"
        "9175446840008594765536775135965031800984554798926748506997642627109557942596805983193517731453653369694338"
        "7708898919049874409622024647182779376328026327875624622315356247272374201551461676746354167305552707034621"
        "5623430907726287841796875"},
    {0x1.fffffffffffffp-608, chars_format::fixed, 660,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000188274989461162800795773270377"
        "7488084273308319685387759630689171681241787913002970541575654295549588416631423226651661506862641042881685"
        "4650202382656554486492726892378713178805849607336151597607524960742213799102285186653493071920108158591951"
        "8350893680017189531073550271930063601969109597853497013995285254219115885193611966387035462907306739388677"
        "5417797838099748819244049294365558752656052655751249244630712494544748403102923353492708334611105414069243"
        "124686181545257568359375"},
    {0x1.fffffffffffffp-607, chars_format::fixed, 659,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000376549978922325601591546540755"
        "4976168546616639370775519261378343362483575826005941083151308591099176833262846453303323013725282085763370"
        "9300404765313108972985453784757426357611699214672303195215049921484427598204570373306986143840216317183903"
        "6701787360034379062147100543860127203938219195706994027990570508438231770387223932774070925814613478777355"
        "0835595676199497638488098588731117505312105311502498489261424989089496806205846706985416669222210828138486"
        "24937236309051513671875"},
    {0x1.fffffffffffffp-606, chars_format::fixed, 658,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000000753099957844651203183093081510"
        "9952337093233278741551038522756686724967151652011882166302617182198353666525692906606646027450564171526741"
        "8600809530626217945970907569514852715223398429344606390430099842968855196409140746613972287680432634367807"
        "3403574720068758124294201087720254407876438391413988055981141016876463540774447865548141851629226957554710"
        "1671191352398995276976197177462235010624210623004996978522849978178993612411693413970833338444421656276972"
        "4987447261810302734375"},
    {0x1.fffffffffffffp-605, chars_format::fixed, 657,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000001506199915689302406366186163021"
        "9904674186466557483102077045513373449934303304023764332605234364396707333051385813213292054901128343053483"
        "7201619061252435891941815139029705430446796858689212780860199685937710392818281493227944575360865268735614"
        "6807149440137516248588402175440508815752876782827976111962282033752927081548895731096283703258453915109420"
        "3342382704797990553952394354924470021248421246009993957045699956357987224823386827941666676888843312553944"
        "997489452362060546875"},
    {0x1.fffffffffffffp-604, chars_format::fixed, 656,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000003012399831378604812732372326043"
        "9809348372933114966204154091026746899868606608047528665210468728793414666102771626426584109802256686106967"
        "4403238122504871783883630278059410860893593717378425561720399371875420785636562986455889150721730537471229"
        "3614298880275032497176804350881017631505753565655952223924564067505854163097791462192567406516907830218840"
        "6684765409595981107904788709848940042496842492019987914091399912715974449646773655883333353777686625107889"
        "99497890472412109375"},
    {0x1.fffffffffffffp-603, chars_format::fixed, 655,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000006024799662757209625464744652087"
        "9618696745866229932408308182053493799737213216095057330420937457586829332205543252853168219604513372213934"
        "8806476245009743567767260556118821721787187434756851123440798743750841571273125972911778301443461074942458"
        "7228597760550064994353608701762035263011507131311904447849128135011708326195582924385134813033815660437681"
        "3369530819191962215809577419697880084993684984039975828182799825431948899293547311766666707555373250215779"
        "9899578094482421875"},
    {0x1.fffffffffffffp-602, chars_format::fixed, 654,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000012049599325514419250929489304175"
        "9237393491732459864816616364106987599474426432190114660841874915173658664411086505706336439209026744427869"
        "7612952490019487135534521112237643443574374869513702246881597487501683142546251945823556602886922149884917"
        "4457195521100129988707217403524070526023014262623808895698256270023416652391165848770269626067631320875362"
        "6739061638383924431619154839395760169987369968079951656365599650863897798587094623533333415110746500431559"
        "979915618896484375"},
    {0x1.fffffffffffffp-601, chars_format::fixed, 653,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000024099198651028838501858978608351"
        "8474786983464919729633232728213975198948852864380229321683749830347317328822173011412672878418053488855739"
        "5225904980038974271069042224475286887148749739027404493763194975003366285092503891647113205773844299769834"
        "8914391042200259977414434807048141052046028525247617791396512540046833304782331697540539252135262641750725"
        "3478123276767848863238309678791520339974739936159903312731199301727795597174189247066666830221493000863119"
        "95983123779296875"},
    {0x1.fffffffffffffp-600, chars_format::fixed, 652,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000048198397302057677003717957216703"
        "6949573966929839459266465456427950397897705728760458643367499660694634657644346022825345756836106977711479"
        "0451809960077948542138084448950573774297499478054808987526389950006732570185007783294226411547688599539669"
        "7828782084400519954828869614096282104092057050495235582793025080093666609564663395081078504270525283501450"
        "6956246553535697726476619357583040679949479872319806625462398603455591194348378494133333660442986001726239"
        "9196624755859375"},
    {0x1.fffffffffffffp-599, chars_format::fixed, 651,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000096396794604115354007435914433407"
        "3899147933859678918532930912855900795795411457520917286734999321389269315288692045650691513672213955422958"
        "0903619920155897084276168897901147548594998956109617975052779900013465140370015566588452823095377199079339"
        "5657564168801039909657739228192564208184114100990471165586050160187333219129326790162157008541050567002901"
        "3912493107071395452953238715166081359898959744639613250924797206911182388696756988266667320885972003452479"
        "839324951171875"},
    {0x1.fffffffffffffp-598, chars_format::fixed, 650,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000192793589208230708014871828866814"
        "7798295867719357837065861825711801591590822915041834573469998642778538630577384091301383027344427910845916"
        "1807239840311794168552337795802295097189997912219235950105559800026930280740031133176905646190754398158679"
        "1315128337602079819315478456385128416368228201980942331172100320374666438258653580324314017082101134005802"
        "7824986214142790905906477430332162719797919489279226501849594413822364777393513976533334641771944006904959"
        "67864990234375"},
    {0x1.fffffffffffffp-597, chars_format::fixed, 649,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000385587178416461416029743657733629"
        "5596591735438715674131723651423603183181645830083669146939997285557077261154768182602766054688855821691832"
        "3614479680623588337104675591604590194379995824438471900211119600053860561480062266353811292381508796317358"
        "2630256675204159638630956912770256832736456403961884662344200640749332876517307160648628034164202268011605"
        "5649972428285581811812954860664325439595838978558453003699188827644729554787027953066669283543888013809919"
        "3572998046875"},
    {0x1.fffffffffffffp-596, chars_format::fixed, 648,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000000771174356832922832059487315467259"
        "1193183470877431348263447302847206366363291660167338293879994571114154522309536365205532109377711643383664"
        "7228959361247176674209351183209180388759991648876943800422239200107721122960124532707622584763017592634716"
        "5260513350408319277261913825540513665472912807923769324688401281498665753034614321297256068328404536023211"
        "1299944856571163623625909721328650879191677957116906007398377655289459109574055906133338567087776027619838"
        "714599609375"},
    {0x1.fffffffffffffp-595, chars_format::fixed, 647,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000001542348713665845664118974630934518"
        "2386366941754862696526894605694412732726583320334676587759989142228309044619072730411064218755423286767329"
        "4457918722494353348418702366418360777519983297753887600844478400215442245920249065415245169526035185269433"
        "0521026700816638554523827651081027330945825615847538649376802562997331506069228642594512136656809072046422"
        "2599889713142327247251819442657301758383355914233812014796755310578918219148111812266677134175552055239677"
        "42919921875"},
    {0x1.fffffffffffffp-594, chars_format::fixed, 646,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000003084697427331691328237949261869036"
        "4772733883509725393053789211388825465453166640669353175519978284456618089238145460822128437510846573534658"
        "8915837444988706696837404732836721555039966595507775201688956800430884491840498130830490339052070370538866"
        "1042053401633277109047655302162054661891651231695077298753605125994663012138457285189024273313618144092844"
        "5199779426284654494503638885314603516766711828467624029593510621157836438296223624533354268351104110479354"
        "8583984375"},
    {0x1.fffffffffffffp-593, chars_format::fixed, 645,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000006169394854663382656475898523738072"
        "9545467767019450786107578422777650930906333281338706351039956568913236178476290921644256875021693147069317"
        "7831674889977413393674809465673443110079933191015550403377913600861768983680996261660980678104140741077732"
        "2084106803266554218095310604324109323783302463390154597507210251989326024276914570378048546627236288185689"
        "0399558852569308989007277770629207033533423656935248059187021242315672876592447249066708536702208220958709"
        "716796875"},
    {0x1.fffffffffffffp-592, chars_format::fixed, 644,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000012338789709326765312951797047476145"
        "9090935534038901572215156845555301861812666562677412702079913137826472356952581843288513750043386294138635"
        "5663349779954826787349618931346886220159866382031100806755827201723537967361992523321961356208281482155464"
        "4168213606533108436190621208648218647566604926780309195014420503978652048553829140756097093254472576371378"
        "0799117705138617978014555541258414067066847313870496118374042484631345753184894498133417073404416441917419"
        "43359375"},
    {0x1.fffffffffffffp-591, chars_format::fixed, 643,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000024677579418653530625903594094952291"
        "8181871068077803144430313691110603723625333125354825404159826275652944713905163686577027500086772588277271"
        "1326699559909653574699237862693772440319732764062201613511654403447075934723985046643922712416562964310928"
        "8336427213066216872381242417296437295133209853560618390028841007957304097107658281512194186508945152742756"
        "1598235410277235956029111082516828134133694627740992236748084969262691506369788996266834146808832883834838"
        "8671875"},
    {0x1.fffffffffffffp-590, chars_format::fixed, 642,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000049355158837307061251807188189904583"
        "6363742136155606288860627382221207447250666250709650808319652551305889427810327373154055000173545176554542"
        "2653399119819307149398475725387544880639465528124403227023308806894151869447970093287845424833125928621857"
        "6672854426132433744762484834592874590266419707121236780057682015914608194215316563024388373017890305485512"
        "3196470820554471912058222165033656268267389255481984473496169938525383012739577992533668293617665767669677"
        "734375"},
    {0x1.fffffffffffffp-589, chars_format::fixed, 641,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000098710317674614122503614376379809167"
        "2727484272311212577721254764442414894501332501419301616639305102611778855620654746308110000347090353109084"
        "5306798239638614298796951450775089761278931056248806454046617613788303738895940186575690849666251857243715"
        "3345708852264867489524969669185749180532839414242473560115364031829216388430633126048776746035780610971024"
        "6392941641108943824116444330067312536534778510963968946992339877050766025479155985067336587235331535339355"
        "46875"},
    {0x1.fffffffffffffp-588, chars_format::fixed, 640,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000197420635349228245007228752759618334"
        "5454968544622425155442509528884829789002665002838603233278610205223557711241309492616220000694180706218169"
        "0613596479277228597593902901550179522557862112497612908093235227576607477791880373151381699332503714487430"
        "6691417704529734979049939338371498361065678828484947120230728063658432776861266252097553492071561221942049"
        "2785883282217887648232888660134625073069557021927937893984679754101532050958311970134673174470663070678710"
        "9375"},
    {0x1.fffffffffffffp-587, chars_format::fixed, 639,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000394841270698456490014457505519236669"
        "0909937089244850310885019057769659578005330005677206466557220410447115422482618985232440001388361412436338"
        "1227192958554457195187805803100359045115724224995225816186470455153214955583760746302763398665007428974861"
        "3382835409059469958099878676742996722131357656969894240461456127316865553722532504195106984143122443884098"
        "5571766564435775296465777320269250146139114043855875787969359508203064101916623940269346348941326141357421"
        "875"},
    {0x1.fffffffffffffp-586, chars_format::fixed, 638,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000000789682541396912980028915011038473338"
        "1819874178489700621770038115539319156010660011354412933114440820894230844965237970464880002776722824872676"
        "2454385917108914390375611606200718090231448449990451632372940910306429911167521492605526797330014857949722"
        "6765670818118939916199757353485993444262715313939788480922912254633731107445065008390213968286244887768197"
        "1143533128871550592931554640538500292278228087711751575938719016406128203833247880538692697882652282714843"
        "75"},
    {0x1.fffffffffffffp-585, chars_format::fixed, 637,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000001579365082793825960057830022076946676"
        "3639748356979401243540076231078638312021320022708825866228881641788461689930475940929760005553445649745352"
        "4908771834217828780751223212401436180462896899980903264745881820612859822335042985211053594660029715899445"
        "3531341636237879832399514706971986888525430627879576961845824509267462214890130016780427936572489775536394"
        "2287066257743101185863109281077000584556456175423503151877438032812256407666495761077385395765304565429687"
        "5"},
    {0x1.fffffffffffffp-584, chars_format::fixed, 636,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000003158730165587651920115660044153893352"
        "7279496713958802487080152462157276624042640045417651732457763283576923379860951881859520011106891299490704"
        "9817543668435657561502446424802872360925793799961806529491763641225719644670085970422107189320059431798890"
        "7062683272475759664799029413943973777050861255759153923691649018534924429780260033560855873144979551072788"
        "457413251548620237172621856215400116911291235084700630375487606562451281533299152215477079153060913085937"
        "5"},
    {0x1.fffffffffffffp-583, chars_format::fixed, 635,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000006317460331175303840231320088307786705"
        "4558993427917604974160304924314553248085280090835303464915526567153846759721903763719040022213782598981409"
        "9635087336871315123004892849605744721851587599923613058983527282451439289340171940844214378640118863597781"
        "4125366544951519329598058827887947554101722511518307847383298037069848859560520067121711746289959102145576"
        "91482650309724047434524371243080023382258247016940126075097521312490256306659830443095415830612182617187"
        "5"},
    {0x1.fffffffffffffp-582, chars_format::fixed, 634,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000012634920662350607680462640176615573410"
        "9117986855835209948320609848629106496170560181670606929831053134307693519443807527438080044427565197962819"
        "9270174673742630246009785699211489443703175199847226117967054564902878578680343881688428757280237727195562"
        "8250733089903038659196117655775895108203445023036615694766596074139697719121040134243423492579918204291153"
        "82965300619448094869048742486160046764516494033880252150195042624980512613319660886190831661224365234375"},
    {0x1.fffffffffffffp-581, chars_format::fixed, 633,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000025269841324701215360925280353231146821"
        "8235973711670419896641219697258212992341120363341213859662106268615387038887615054876160088855130395925639"
        "8540349347485260492019571398422978887406350399694452235934109129805757157360687763376857514560475454391125"
        "6501466179806077318392235311551790216406890046073231389533192148279395438242080268486846985159836408582307"
        "6593060123889618973809748497232009352903298806776050430039008524996102522663932177238166332244873046875"},
    {0x1.fffffffffffffp-580, chars_format::fixed, 632,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000050539682649402430721850560706462293643"
        "6471947423340839793282439394516425984682240726682427719324212537230774077775230109752320177710260791851279"
        "7080698694970520984039142796845957774812700799388904471868218259611514314721375526753715029120950908782251"
        "3002932359612154636784470623103580432813780092146462779066384296558790876484160536973693970319672817164615"
        "318612024777923794761949699446401870580659761355210086007801704999220504532786435447633266448974609375"},
    {0x1.fffffffffffffp-579, chars_format::fixed, 631,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000101079365298804861443701121412924587287"
        "2943894846681679586564878789032851969364481453364855438648425074461548155550460219504640355420521583702559"
        "4161397389941041968078285593691915549625401598777808943736436519223028629442751053507430058241901817564502"
        "6005864719224309273568941246207160865627560184292925558132768593117581752968321073947387940639345634329230"
        "63722404955584758952389939889280374116131952271042017201560340999844100906557287089526653289794921875"},
    {0x1.fffffffffffffp-578, chars_format::fixed, 630,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000202158730597609722887402242825849174574"
        "5887789693363359173129757578065703938728962906729710877296850148923096311100920439009280710841043167405118"
        "8322794779882083936156571187383831099250803197555617887472873038446057258885502107014860116483803635129005"
        "2011729438448618547137882492414321731255120368585851116265537186235163505936642147894775881278691268658461"
        "2744480991116951790477987977856074823226390454208403440312068199968820181311457417905330657958984375"},
    {0x1.fffffffffffffp-577, chars_format::fixed, 629,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000404317461195219445774804485651698349149"
        "1775579386726718346259515156131407877457925813459421754593700297846192622201840878018561421682086334810237"
        "6645589559764167872313142374767662198501606395111235774945746076892114517771004214029720232967607270258010"
        "4023458876897237094275764984828643462510240737171702232531074372470327011873284295789551762557382537316922"
        "548896198223390358095597595571214964645278090841680688062413639993764036262291483581066131591796875"},
    {0x1.fffffffffffffp-576, chars_format::fixed, 628,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000000808634922390438891549608971303396698298"
        "3551158773453436692519030312262815754915851626918843509187400595692385244403681756037122843364172669620475"
        "3291179119528335744626284749535324397003212790222471549891492153784229035542008428059440465935214540516020"
        "8046917753794474188551529969657286925020481474343404465062148744940654023746568591579103525114765074633845"
        "09779239644678071619119519114242992929055618168336137612482727998752807252458296716213226318359375"},
    {0x1.fffffffffffffp-575, chars_format::fixed, 627,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000001617269844780877783099217942606793396596"
        "7102317546906873385038060624525631509831703253837687018374801191384770488807363512074245686728345339240950"
        "6582358239056671489252569499070648794006425580444943099782984307568458071084016856118880931870429081032041"
        "6093835507588948377103059939314573850040962948686808930124297489881308047493137183158207050229530149267690"
        "1955847928935614323823903822848598585811123633667227522496545599750561450491659343242645263671875"},
    {0x1.fffffffffffffp-574, chars_format::fixed, 626,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000003234539689561755566198435885213586793193"
        "4204635093813746770076121249051263019663406507675374036749602382769540977614727024148491373456690678481901"
        "3164716478113342978505138998141297588012851160889886199565968615136916142168033712237761863740858162064083"
        "2187671015177896754206119878629147700081925897373617860248594979762616094986274366316414100459060298535380"
        "391169585787122864764780764569719717162224726733445504499309119950112290098331868648529052734375"},
    {0x1.fffffffffffffp-573, chars_format::fixed, 625,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000006469079379123511132396871770427173586386"
        "8409270187627493540152242498102526039326813015350748073499204765539081955229454048296982746913381356963802"
        "6329432956226685957010277996282595176025702321779772399131937230273832284336067424475523727481716324128166"
        "4375342030355793508412239757258295400163851794747235720497189959525232189972548732632828200918120597070760"
        "78233917157424572952956152913943943432444945346689100899861823990022458019666373729705810546875"},
    {0x1.fffffffffffffp-572, chars_format::fixed, 624,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000012938158758247022264793743540854347172773"
        "6818540375254987080304484996205052078653626030701496146998409531078163910458908096593965493826762713927605"
        "2658865912453371914020555992565190352051404643559544798263874460547664568672134848951047454963432648256332"
        "8750684060711587016824479514516590800327703589494471440994379919050464379945097465265656401836241194141521"
        "5646783431484914590591230582788788686488989069337820179972364798004491603933274745941162109375"},
    {0x1.fffffffffffffp-571, chars_format::fixed, 623,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000025876317516494044529587487081708694345547"
        "3637080750509974160608969992410104157307252061402992293996819062156327820917816193187930987653525427855210"
        "5317731824906743828041111985130380704102809287119089596527748921095329137344269697902094909926865296512665"
        "7501368121423174033648959029033181600655407178988942881988759838100928759890194930531312803672482388283043"
        "129356686296982918118246116557757737297797813867564035994472959600898320786654949188232421875"},
    {0x1.fffffffffffffp-570, chars_format::fixed, 622,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000051752635032988089059174974163417388691094"
        "7274161501019948321217939984820208314614504122805984587993638124312655641835632386375861975307050855710421"
        "0635463649813487656082223970260761408205618574238179193055497842190658274688539395804189819853730593025331"
        "5002736242846348067297918058066363201310814357977885763977519676201857519780389861062625607344964776566086"
        "25871337259396583623649223311551547459559562773512807198894591920179664157330989837646484375"},
    {0x1.fffffffffffffp-569, chars_format::fixed, 621,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000103505270065976178118349948326834777382189"
        "4548323002039896642435879969640416629229008245611969175987276248625311283671264772751723950614101711420842"
        "1270927299626975312164447940521522816411237148476358386110995684381316549377078791608379639707461186050663"
        "0005472485692696134595836116132726402621628715955771527955039352403715039560779722125251214689929553132172"
        "5174267451879316724729844662310309491911912554702561439778918384035932831466197967529296875"},
    {0x1.fffffffffffffp-568, chars_format::fixed, 620,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000207010540131952356236699896653669554764378"
        "9096646004079793284871759939280833258458016491223938351974552497250622567342529545503447901228203422841684"
        "2541854599253950624328895881043045632822474296952716772221991368762633098754157583216759279414922372101326"
        "0010944971385392269191672232265452805243257431911543055910078704807430079121559444250502429379859106264345"
        "034853490375863344945968932462061898382382510940512287955783676807186566293239593505859375"},
    {0x1.fffffffffffffp-567, chars_format::fixed, 619,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000414021080263904712473399793307339109528757"
        "8193292008159586569743519878561666516916032982447876703949104994501245134685059091006895802456406845683368"
        "5083709198507901248657791762086091265644948593905433544443982737525266197508315166433518558829844744202652"
        "0021889942770784538383344464530905610486514863823086111820157409614860158243118888501004858759718212528690"
        "06970698075172668989193786492412379676476502188102457591156735361437313258647918701171875"},
    {0x1.fffffffffffffp-566, chars_format::fixed, 618,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000828042160527809424946799586614678219057515"
        "6386584016319173139487039757123333033832065964895753407898209989002490269370118182013791604912813691366737"
        "0167418397015802497315583524172182531289897187810867088887965475050532395016630332867037117659689488405304"
        "0043779885541569076766688929061811220973029727646172223640314819229720316486237777002009717519436425057380"
        "1394139615034533797838757298482475935295300437620491518231347072287462651729583740234375"},
    {0x1.fffffffffffffp-565, chars_format::fixed, 617,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000001656084321055618849893599173229356438115031"
        "2773168032638346278974079514246666067664131929791506815796419978004980538740236364027583209825627382733474"
        "0334836794031604994631167048344365062579794375621734177775930950101064790033260665734074235319378976810608"
        "0087559771083138153533377858123622441946059455292344447280629638459440632972475554004019435038872850114760"
        "278827923006906759567751459696495187059060087524098303646269414457492530345916748046875"},
    {0x1.fffffffffffffp-564, chars_format::fixed, 616,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000003312168642111237699787198346458712876230062"
        "5546336065276692557948159028493332135328263859583013631592839956009961077480472728055166419651254765466948"
        "0669673588063209989262334096688730125159588751243468355551861900202129580066521331468148470638757953621216"
        "0175119542166276307066755716247244883892118910584688894561259276918881265944951108008038870077745700229520"
        "55765584601381351913550291939299037411812017504819660729253882891498506069183349609375"},
    {0x1.fffffffffffffp-563, chars_format::fixed, 615,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000006624337284222475399574396692917425752460125"
        "1092672130553385115896318056986664270656527719166027263185679912019922154960945456110332839302509530933896"
        "1339347176126419978524668193377460250319177502486936711103723800404259160133042662936296941277515907242432"
        "0350239084332552614133511432494489767784237821169377789122518553837762531889902216016077740155491400459041"
        "1153116920276270382710058387859807482362403500963932145850776578299701213836669921875"},
    {0x1.fffffffffffffp-562, chars_format::fixed, 614,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000013248674568444950799148793385834851504920250"
        "2185344261106770231792636113973328541313055438332054526371359824039844309921890912220665678605019061867792"
        "2678694352252839957049336386754920500638355004973873422207447600808518320266085325872593882555031814484864"
        "0700478168665105228267022864988979535568475642338755578245037107675525063779804432032155480310982800918082"
        "230623384055254076542011677571961496472480700192786429170155315659940242767333984375"},
    {0x1.fffffffffffffp-561, chars_format::fixed, 613,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000026497349136889901598297586771669703009840500"
        "4370688522213540463585272227946657082626110876664109052742719648079688619843781824441331357210038123735584"
        "5357388704505679914098672773509841001276710009947746844414895201617036640532170651745187765110063628969728"
        "1400956337330210456534045729977959071136951284677511156490074215351050127559608864064310960621965601836164"
        "46124676811050815308402335514392299294496140038557285834031063131988048553466796875"},
    {0x1.fffffffffffffp-560, chars_format::fixed, 612,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000052994698273779803196595173543339406019681000"
        "8741377044427080927170544455893314165252221753328218105485439296159377239687563648882662714420076247471169"
        "0714777409011359828197345547019682002553420019895493688829790403234073281064341303490375530220127257939456"
        "2801912674660420913068091459955918142273902569355022312980148430702100255119217728128621921243931203672328"
        "9224935362210163061680467102878459858899228007711457166806212626397609710693359375"},
    {0x1.fffffffffffffp-559, chars_format::fixed, 611,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000105989396547559606393190347086678812039362001"
        "7482754088854161854341088911786628330504443506656436210970878592318754479375127297765325428840152494942338"
        "1429554818022719656394691094039364005106840039790987377659580806468146562128682606980751060440254515878912"
        "5603825349320841826136182919911836284547805138710044625960296861404200510238435456257243842487862407344657"
        "844987072442032612336093420575691971779845601542291433361242525279521942138671875"},
    {0x1.fffffffffffffp-558, chars_format::fixed, 610,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000211978793095119212786380694173357624078724003"
        "4965508177708323708682177823573256661008887013312872421941757184637508958750254595530650857680304989884676"
        "2859109636045439312789382188078728010213680079581974755319161612936293124257365213961502120880509031757825"
        "1207650698641683652272365839823672569095610277420089251920593722808401020476870912514487684975724814689315"
        "68997414488406522467218684115138394355969120308458286672248505055904388427734375"},
    {0x1.fffffffffffffp-557, chars_format::fixed, 609,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000423957586190238425572761388346715248157448006"
        "9931016355416647417364355647146513322017774026625744843883514369275017917500509191061301715360609979769352"
        "5718219272090878625578764376157456020427360159163949510638323225872586248514730427923004241761018063515650"
        "2415301397283367304544731679647345138191220554840178503841187445616802040953741825028975369951449629378631"
        "3799482897681304493443736823027678871193824061691657334449701011180877685546875"},
    {0x1.fffffffffffffp-556, chars_format::fixed, 608,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000847915172380476851145522776693430496314896013"
        "9862032710833294834728711294293026644035548053251489687767028738550035835001018382122603430721219959538705"
        "1436438544181757251157528752314912040854720318327899021276646451745172497029460855846008483522036127031300"
        "4830602794566734609089463359294690276382441109680357007682374891233604081907483650057950739902899258757262"
        "759896579536260898688747364605535774238764812338331466889940202236175537109375"},
    {0x1.fffffffffffffp-555, chars_format::fixed, 607,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000001695830344760953702291045553386860992629792027"
        "9724065421666589669457422588586053288071096106502979375534057477100071670002036764245206861442439919077410"
        "2872877088363514502315057504629824081709440636655798042553292903490344994058921711692016967044072254062600"
        "9661205589133469218178926718589380552764882219360714015364749782467208163814967300115901479805798517514525"
        "51979315907252179737749472921107154847752962467666293377988040447235107421875"},
    {0x1.fffffffffffffp-554, chars_format::fixed, 606,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000003391660689521907404582091106773721985259584055"
        "9448130843333179338914845177172106576142192213005958751068114954200143340004073528490413722884879838154820"
        "5745754176727029004630115009259648163418881273311596085106585806980689988117843423384033934088144508125201"
        "9322411178266938436357853437178761105529764438721428030729499564934416327629934600231802959611597035029051"
        "0395863181450435947549894584221430969550592493533258675597608089447021484375"},
    {0x1.fffffffffffffp-553, chars_format::fixed, 605,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000006783321379043814809164182213547443970519168111"
        "8896261686666358677829690354344213152284384426011917502136229908400286680008147056980827445769759676309641"
        "1491508353454058009260230018519296326837762546623192170213171613961379976235686846768067868176289016250403"
        "8644822356533876872715706874357522211059528877442856061458999129868832655259869200463605919223194070058102"
        "079172636290087189509978916844286193910118498706651735119521617889404296875"},
    {0x1.fffffffffffffp-552, chars_format::fixed, 604,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000013566642758087629618328364427094887941038336223"
        "7792523373332717355659380708688426304568768852023835004272459816800573360016294113961654891539519352619282"
        "2983016706908116018520460037038592653675525093246384340426343227922759952471373693536135736352578032500807"
        "7289644713067753745431413748715044422119057754885712122917998259737665310519738400927211838446388140116204"
        "15834527258017437901995783368857238782023699741330347023904323577880859375"},
    {0x1.fffffffffffffp-551, chars_format::fixed, 603,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000027133285516175259236656728854189775882076672447"
        "5585046746665434711318761417376852609137537704047670008544919633601146720032588227923309783079038705238564"
        "5966033413816232037040920074077185307351050186492768680852686455845519904942747387072271472705156065001615"
        "4579289426135507490862827497430088844238115509771424245835996519475330621039476801854423676892776280232408"
        "3166905451603487580399156673771447756404739948266069404780864715576171875"},
    {0x1.fffffffffffffp-550, chars_format::fixed, 602,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000054266571032350518473313457708379551764153344895"
        "1170093493330869422637522834753705218275075408095340017089839267202293440065176455846619566158077410477129"
        "1932066827632464074081840148154370614702100372985537361705372911691039809885494774144542945410312130003230"
        "9158578852271014981725654994860177688476231019542848491671993038950661242078953603708847353785552560464816"
        "633381090320697516079831334754289551280947989653213880956172943115234375"},
    {0x1.fffffffffffffp-549, chars_format::fixed, 601,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000108533142064701036946626915416759103528306689790"
        "2340186986661738845275045669507410436550150816190680034179678534404586880130352911693239132316154820954258"
        "3864133655264928148163680296308741229404200745971074723410745823382079619770989548289085890820624260006461"
        "8317157704542029963451309989720355376952462039085696983343986077901322484157907207417694707571105120929633"
        "26676218064139503215966266950857910256189597930642776191234588623046875"},
    {0x1.fffffffffffffp-548, chars_format::fixed, 600,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000217066284129402073893253830833518207056613379580"
        "4680373973323477690550091339014820873100301632381360068359357068809173760260705823386478264632309641908516"
        "7728267310529856296327360592617482458808401491942149446821491646764159239541979096578171781641248520012923"
        "6634315409084059926902619979440710753904924078171393966687972155802644968315814414835389415142210241859266"
        "5335243612827900643193253390171582051237919586128555238246917724609375"},
    {0x1.fffffffffffffp-547, chars_format::fixed, 599,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000434132568258804147786507661667036414113226759160"
        "9360747946646955381100182678029641746200603264762720136718714137618347520521411646772956529264619283817033"
        "5456534621059712592654721185234964917616802983884298893642983293528318479083958193156343563282497040025847"
        "3268630818168119853805239958881421507809848156342787933375944311605289936631628829670778830284420483718533"
        "067048722565580128638650678034316410247583917225711047649383544921875"},
    {0x1.fffffffffffffp-546, chars_format::fixed, 598,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000868265136517608295573015323334072828226453518321"
        "8721495893293910762200365356059283492401206529525440273437428275236695041042823293545913058529238567634067"
        "0913069242119425185309442370469929835233605967768597787285966587056636958167916386312687126564994080051694"
        "6537261636336239707610479917762843015619696312685575866751888623210579873263257659341557660568840967437066"
        "13409744513116025727730135606863282049516783445142209529876708984375"},
    {0x1.fffffffffffffp-545, chars_format::fixed, 597,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000001736530273035216591146030646668145656452907036643"
        "7442991786587821524400730712118566984802413059050880546874856550473390082085646587091826117058477135268134"
        "1826138484238850370618884740939859670467211935537195574571933174113273916335832772625374253129988160103389"
        "3074523272672479415220959835525686031239392625371151733503777246421159746526515318683115321137681934874132"
        "2681948902623205145546027121372656409903356689028441905975341796875"},
    {0x1.fffffffffffffp-544, chars_format::fixed, 596,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000003473060546070433182292061293336291312905814073287"
        "4885983573175643048801461424237133969604826118101761093749713100946780164171293174183652234116954270536268"
        "3652276968477700741237769481879719340934423871074391149143866348226547832671665545250748506259976320206778"
        "6149046545344958830441919671051372062478785250742303467007554492842319493053030637366230642275363869748264"
        "536389780524641029109205424274531281980671337805688381195068359375"},
    {0x1.fffffffffffffp-543, chars_format::fixed, 595,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000006946121092140866364584122586672582625811628146574"
        "9771967146351286097602922848474267939209652236203522187499426201893560328342586348367304468233908541072536"
        "7304553936955401482475538963759438681868847742148782298287732696453095665343331090501497012519952640413557"
        "2298093090689917660883839342102744124957570501484606934015108985684638986106061274732461284550727739496529"
        "07277956104928205821841084854906256396134267561137676239013671875"},
    {0x1.fffffffffffffp-542, chars_format::fixed, 594,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000013892242184281732729168245173345165251623256293149"
        "9543934292702572195205845696948535878419304472407044374998852403787120656685172696734608936467817082145073"
        "4609107873910802964951077927518877363737695484297564596575465392906191330686662181002994025039905280827114"
        "4596186181379835321767678684205488249915141002969213868030217971369277972212122549464922569101455478993058"
        "1455591220985641164368216970981251279226853512227535247802734375"},
    {0x1.fffffffffffffp-541, chars_format::fixed, 593,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000027784484368563465458336490346690330503246512586299"
        "9087868585405144390411691393897071756838608944814088749997704807574241313370345393469217872935634164290146"
        "9218215747821605929902155855037754727475390968595129193150930785812382661373324362005988050079810561654228"
        "9192372362759670643535357368410976499830282005938427736060435942738555944424245098929845138202910957986116"
        "291118244197128232873643394196250255845370702445507049560546875"},
    {0x1.fffffffffffffp-540, chars_format::fixed, 592,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000055568968737126930916672980693380661006493025172599"
        "8175737170810288780823382787794143513677217889628177499995409615148482626740690786938435745871268328580293"
        "8436431495643211859804311710075509454950781937190258386301861571624765322746648724011976100159621123308457"
        "8384744725519341287070714736821952999660564011876855472120871885477111888848490197859690276405821915972232"
        "58223648839425646574728678839250051169074140489101409912109375"},
    {0x1.fffffffffffffp-539, chars_format::fixed, 591,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000111137937474253861833345961386761322012986050345199"
        "6351474341620577561646765575588287027354435779256354999990819230296965253481381573876871491742536657160587"
        "6872862991286423719608623420151018909901563874380516772603723143249530645493297448023952200319242246616915"
        "6769489451038682574141429473643905999321128023753710944241743770954223777696980395719380552811643831944465"
        "1644729767885129314945735767850010233814828097820281982421875"},
    {0x1.fffffffffffffp-538, chars_format::fixed, 590,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000222275874948507723666691922773522644025972100690399"
        "2702948683241155123293531151176574054708871558512709999981638460593930506962763147753742983485073314321175"
        "3745725982572847439217246840302037819803127748761033545207446286499061290986594896047904400638484493233831"
        "3538978902077365148282858947287811998642256047507421888483487541908447555393960791438761105623287663888930"
        "328945953577025862989147153570002046762965619564056396484375"},
    {0x1.fffffffffffffp-537, chars_format::fixed, 589,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000444551749897015447333383845547045288051944201380798"
        "5405897366482310246587062302353148109417743117025419999963276921187861013925526295507485966970146628642350"
        "7491451965145694878434493680604075639606255497522067090414892572998122581973189792095808801276968986467662"
        "7077957804154730296565717894575623997284512095014843776966975083816895110787921582877522211246575327777860"
        "65789190715405172597829430714000409352593123912811279296875"},
    {0x1.fffffffffffffp-536, chars_format::fixed, 588,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000889103499794030894666767691094090576103888402761597"
        "0811794732964620493174124604706296218835486234050839999926553842375722027851052591014971933940293257284701"
        "4982903930291389756868987361208151279212510995044134180829785145996245163946379584191617602553937972935325"
        "4155915608309460593131435789151247994569024190029687553933950167633790221575843165755044422493150655555721"
        "3157838143081034519565886142800081870518624782562255859375"},
    {0x1.fffffffffffffp-535, chars_format::fixed, 587,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000001778206999588061789333535382188181152207776805523194"
        "1623589465929240986348249209412592437670972468101679999853107684751444055702105182029943867880586514569402"
        "9965807860582779513737974722416302558425021990088268361659570291992490327892759168383235205107875945870650"
        "8311831216618921186262871578302495989138048380059375107867900335267580443151686331510088844986301311111442"
        "631567628616206903913177228560016374103724956512451171875"},
    {0x1.fffffffffffffp-534, chars_format::fixed, 586,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000003556413999176123578667070764376362304415553611046388"
        "3247178931858481972696498418825184875341944936203359999706215369502888111404210364059887735761173029138805"
        "9931615721165559027475949444832605116850043980176536723319140583984980655785518336766470410215751891741301"
        "6623662433237842372525743156604991978276096760118750215735800670535160886303372663020177689972602622222885"
        "26313525723241380782635445712003274820744991302490234375"},
    {0x1.fffffffffffffp-533, chars_format::fixed, 585,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000007112827998352247157334141528752724608831107222092776"
        "6494357863716963945392996837650369750683889872406719999412430739005776222808420728119775471522346058277611"
        "9863231442331118054951898889665210233700087960353073446638281167969961311571036673532940820431503783482603"
        "3247324866475684745051486313209983956552193520237500431471601341070321772606745326040355379945205244445770"
        "5262705144648276156527089142400654964148998260498046875"},
    {0x1.fffffffffffffp-532, chars_format::fixed, 584,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000014225655996704494314668283057505449217662214444185553"
        "2988715727433927890785993675300739501367779744813439998824861478011552445616841456239550943044692116555223"
        "9726462884662236109903797779330420467400175920706146893276562335939922623142073347065881640863007566965206"
        "6494649732951369490102972626419967913104387040475000862943202682140643545213490652080710759890410488891541"
        "052541028929655231305417828480130992829799652099609375"},
    {0x1.fffffffffffffp-531, chars_format::fixed, 583,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000028451311993408988629336566115010898435324428888371106"
        "5977431454867855781571987350601479002735559489626879997649722956023104891233682912479101886089384233110447"
        "9452925769324472219807595558660840934800351841412293786553124671879845246284146694131763281726015133930413"
        "2989299465902738980205945252839935826208774080950001725886405364281287090426981304161421519780820977783082"
        "10508205785931046261083565696026198565959930419921875"},
    {0x1.fffffffffffffp-530, chars_format::fixed, 582,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000056902623986817977258673132230021796870648857776742213"
        "1954862909735711563143974701202958005471118979253759995299445912046209782467365824958203772178768466220895"
        "8905851538648944439615191117321681869600703682824587573106249343759690492568293388263526563452030267860826"
        "5978598931805477960411890505679871652417548161900003451772810728562574180853962608322843039561641955566164"
        "2101641157186209252216713139205239713191986083984375"},
    {0x1.fffffffffffffp-529, chars_format::fixed, 581,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000113805247973635954517346264460043593741297715553484426"
        "3909725819471423126287949402405916010942237958507519990598891824092419564934731649916407544357536932441791"
        "7811703077297888879230382234643363739201407365649175146212498687519380985136586776527053126904060535721653"
        "1957197863610955920823781011359743304835096323800006903545621457125148361707925216645686079123283911132328"
        "420328231437241850443342627841047942638397216796875"},
    {0x1.fffffffffffffp-528, chars_format::fixed, 580,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000227610495947271909034692528920087187482595431106968852"
        "7819451638942846252575898804811832021884475917015039981197783648184839129869463299832815088715073864883583"
        "5623406154595777758460764469286727478402814731298350292424997375038761970273173553054106253808121071443306"
        "3914395727221911841647562022719486609670192647600013807091242914250296723415850433291372158246567822264656"
        "84065646287448370088668525568209588527679443359375"},
    {0x1.fffffffffffffp-527, chars_format::fixed, 579,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000455220991894543818069385057840174374965190862213937705"
        "5638903277885692505151797609623664043768951834030079962395567296369678259738926599665630177430147729767167"
        "1246812309191555516921528938573454956805629462596700584849994750077523940546347106108212507616242142886612"
        "7828791454443823683295124045438973219340385295200027614182485828500593446831700866582744316493135644529313"
        "6813129257489674017733705113641917705535888671875"},
    {0x1.fffffffffffffp-526, chars_format::fixed, 578,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000910441983789087636138770115680348749930381724427875411"
        "1277806555771385010303595219247328087537903668060159924791134592739356519477853199331260354860295459534334"
        "2493624618383111033843057877146909913611258925193401169699989500155047881092694212216425015232484285773225"
        "5657582908887647366590248090877946438680770590400055228364971657001186893663401733165488632986271289058627"
        "362625851497934803546741022728383541107177734375"},
    {0x1.fffffffffffffp-525, chars_format::fixed, 577,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000001820883967578175272277540231360697499860763448855750822"
        "2555613111542770020607190438494656175075807336120319849582269185478713038955706398662520709720590919068668"
        "4987249236766222067686115754293819827222517850386802339399979000310095762185388424432850030464968571546451"
        "1315165817775294733180496181755892877361541180800110456729943314002373787326803466330977265972542578117254"
        "72525170299586960709348204545676708221435546875"},
    {0x1.fffffffffffffp-524, chars_format::fixed, 576,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000003641767935156350544555080462721394999721526897711501644"
        "5111226223085540041214380876989312350151614672240639699164538370957426077911412797325041419441181838137336"
        "9974498473532444135372231508587639654445035700773604678799958000620191524370776848865700060929937143092902"
        "2630331635550589466360992363511785754723082361600220913459886628004747574653606932661954531945085156234509"
        "4505034059917392141869640909135341644287109375"},
    {0x1.fffffffffffffp-523, chars_format::fixed, 575,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000007283535870312701089110160925442789999443053795423003289"
        "0222452446171080082428761753978624700303229344481279398329076741914852155822825594650082838882363676274673"
        "9948996947064888270744463017175279308890071401547209357599916001240383048741553697731400121859874286185804"
        "5260663271101178932721984727023571509446164723200441826919773256009495149307213865323909063890170312469018"
        "901006811983478428373928181827068328857421875"},
    {0x1.fffffffffffffp-522, chars_format::fixed, 574,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000014567071740625402178220321850885579998886107590846006578"
        "0444904892342160164857523507957249400606458688962558796658153483829704311645651189300165677764727352549347"
        "9897993894129776541488926034350558617780142803094418715199832002480766097483107395462800243719748572371609"
        "0521326542202357865443969454047143018892329446400883653839546512018990298614427730647818127780340624938037"
        "80201362396695685674785636365413665771484375"},
    {0x1.fffffffffffffp-521, chars_format::fixed, 573,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000029134143481250804356440643701771159997772215181692013156"
        "0889809784684320329715047015914498801212917377925117593316306967659408623291302378600331355529454705098695"
        "9795987788259553082977852068701117235560285606188837430399664004961532194966214790925600487439497144743218"
        "1042653084404715730887938908094286037784658892801767307679093024037980597228855461295636255560681249876075"
        "6040272479339137134957127273082733154296875"},
    {0x1.fffffffffffffp-520, chars_format::fixed, 572,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000058268286962501608712881287403542319995544430363384026312"
        "1779619569368640659430094031828997602425834755850235186632613935318817246582604757200662711058909410197391"
        "9591975576519106165955704137402234471120571212377674860799328009923064389932429581851200974878994289486436"
        "2085306168809431461775877816188572075569317785603534615358186048075961194457710922591272511121362499752151"
        "208054495867827426991425454616546630859375"},
    {0x1.fffffffffffffp-519, chars_format::fixed, 571,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000116536573925003217425762574807084639991088860726768052624"
        "3559239138737281318860188063657995204851669511700470373265227870637634493165209514401325422117818820394783"
        "9183951153038212331911408274804468942241142424755349721598656019846128779864859163702401949757988578972872"
        "4170612337618862923551755632377144151138635571207069230716372096151922388915421845182545022242724999504302"
        "41610899173565485398285090923309326171875"},
    {0x1.fffffffffffffp-518, chars_format::fixed, 570,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000233073147850006434851525149614169279982177721453536105248"
        "7118478277474562637720376127315990409703339023400940746530455741275268986330419028802650844235637640789567"
        "8367902306076424663822816549608937884482284849510699443197312039692257559729718327404803899515977157945744"
        "8341224675237725847103511264754288302277271142414138461432744192303844777830843690365090044485449999008604"
        "8322179834713097079657018184661865234375"},
    {0x1.fffffffffffffp-517, chars_format::fixed, 569,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000466146295700012869703050299228338559964355442907072210497"
        "4236956554949125275440752254631980819406678046801881493060911482550537972660838057605301688471275281579135"
        "6735804612152849327645633099217875768964569699021398886394624079384515119459436654809607799031954315891489"
        "6682449350475451694207022529508576604554542284828276922865488384607689555661687380730180088970899998017209"
        "664435966942619415931403636932373046875"},
    {0x1.fffffffffffffp-516, chars_format::fixed, 568,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000932292591400025739406100598456677119928710885814144420994"
        "8473913109898250550881504509263961638813356093603762986121822965101075945321676115210603376942550563158271"
        "3471609224305698655291266198435751537929139398042797772789248158769030238918873309619215598063908631782979"
        "3364898700950903388414045059017153209109084569656553845730976769215379111323374761460360177941799996034419"
        "32887193388523883186280727386474609375"},
    {0x1.fffffffffffffp-515, chars_format::fixed, 567,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000001864585182800051478812201196913354239857421771628288841989"
        "6947826219796501101763009018527923277626712187207525972243645930202151890643352230421206753885101126316542"
        "6943218448611397310582532396871503075858278796085595545578496317538060477837746619238431196127817263565958"
        "6729797401901806776828090118034306418218169139313107691461953538430758222646749522920720355883599992068838"
        "6577438677704776637256145477294921875"},
    {0x1.fffffffffffffp-514, chars_format::fixed, 566,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000003729170365600102957624402393826708479714843543256577683979"
        "3895652439593002203526018037055846555253424374415051944487291860404303781286704460842413507770202252633085"
        "3886436897222794621165064793743006151716557592171191091156992635076120955675493238476862392255634527131917"
        "3459594803803613553656180236068612836436338278626215382923907076861516445293499045841440711767199984137677"
        "315487735540955327451229095458984375"},
    {0x1.fffffffffffffp-513, chars_format::fixed, 565,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000007458340731200205915248804787653416959429687086513155367958"
        "7791304879186004407052036074111693110506848748830103888974583720808607562573408921684827015540404505266170"
        "7772873794445589242330129587486012303433115184342382182313985270152241911350986476953724784511269054263834"
        "6919189607607227107312360472137225672872676557252430765847814153723032890586998091682881423534399968275354"
        "63097547108191065490245819091796875"},
    {0x1.fffffffffffffp-512, chars_format::fixed, 564,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000014916681462400411830497609575306833918859374173026310735917"
        "5582609758372008814104072148223386221013697497660207777949167441617215125146817843369654031080809010532341"
        "5545747588891178484660259174972024606866230368684764364627970540304483822701972953907449569022538108527669"
        "3838379215214454214624720944274451345745353114504861531695628307446065781173996183365762847068799936550709"
        "2619509421638213098049163818359375"},
    {0x1.fffffffffffffp-511, chars_format::fixed, 563,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000029833362924800823660995219150613667837718748346052621471835"
        "1165219516744017628208144296446772442027394995320415555898334883234430250293635686739308062161618021064683"
        "1091495177782356969320518349944049213732460737369528729255941080608967645403945907814899138045076217055338"
        "7676758430428908429249441888548902691490706229009723063391256614892131562347992366731525694137599873101418"
        "523901884327642619609832763671875"},
    {0x1.fffffffffffffp-510, chars_format::fixed, 562,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000059666725849601647321990438301227335675437496692105242943670"
        "2330439033488035256416288592893544884054789990640831111796669766468860500587271373478616124323236042129366"
        "2182990355564713938641036699888098427464921474739057458511882161217935290807891815629798276090152434110677"
        "5353516860857816858498883777097805382981412458019446126782513229784263124695984733463051388275199746202837"
        "04780376865528523921966552734375"},
    {0x1.fffffffffffffp-509, chars_format::fixed, 561,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000119333451699203294643980876602454671350874993384210485887340"
        "4660878066976070512832577185787089768109579981281662223593339532937721001174542746957232248646472084258732"
        "4365980711129427877282073399776196854929842949478114917023764322435870581615783631259596552180304868221355"
        "0707033721715633716997767554195610765962824916038892253565026459568526249391969466926102776550399492405674"
        "0956075373105704784393310546875"},
    {0x1.fffffffffffffp-508, chars_format::fixed, 560,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000238666903398406589287961753204909342701749986768420971774680"
        "9321756133952141025665154371574179536219159962563324447186679065875442002349085493914464497292944168517464"
        "8731961422258855754564146799552393709859685898956229834047528644871741163231567262519193104360609736442710"
        "1414067443431267433995535108391221531925649832077784507130052919137052498783938933852205553100798984811348"
        "191215074621140956878662109375"},
    {0x1.fffffffffffffp-507, chars_format::fixed, 559,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000477333806796813178575923506409818685403499973536841943549361"
        "8643512267904282051330308743148359072438319925126648894373358131750884004698170987828928994585888337034929"
        "7463922844517711509128293599104787419719371797912459668095057289743482326463134525038386208721219472885420"
        "2828134886862534867991070216782443063851299664155569014260105838274104997567877867704411106201597969622696"
        "38243014924228191375732421875"},
    {0x1.fffffffffffffp-506, chars_format::fixed, 558,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000954667613593626357151847012819637370806999947073683887098723"
        "7287024535808564102660617486296718144876639850253297788746716263501768009396341975657857989171776674069859"
        "4927845689035423018256587198209574839438743595824919336190114579486964652926269050076772417442438945770840"
        "5656269773725069735982140433564886127702599328311138028520211676548209995135755735408822212403195939245392"
        "7648602984845638275146484375"},
    {0x1.fffffffffffffp-505, chars_format::fixed, 557,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000001909335227187252714303694025639274741613999894147367774197447"
        "4574049071617128205321234972593436289753279700506595577493432527003536018792683951315715978343553348139718"
        "9855691378070846036513174396419149678877487191649838672380229158973929305852538100153544834884877891541681"
        "1312539547450139471964280867129772255405198656622276057040423353096419990271511470817644424806391878490785"
        "529720596969127655029296875"},
    {0x1.fffffffffffffp-504, chars_format::fixed, 556,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000003818670454374505428607388051278549483227999788294735548394894"
        "9148098143234256410642469945186872579506559401013191154986865054007072037585367902631431956687106696279437"
        "9711382756141692073026348792838299357754974383299677344760458317947858611705076200307089669769755783083362"
        "2625079094900278943928561734259544510810397313244552114080846706192839980543022941635288849612783756981571"
        "05944119393825531005859375"},
    {0x1.fffffffffffffp-503, chars_format::fixed, 555,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000007637340908749010857214776102557098966455999576589471096789789"
        "8296196286468512821284939890373745159013118802026382309973730108014144075170735805262863913374213392558875"
        "9422765512283384146052697585676598715509948766599354689520916635895717223410152400614179339539511566166724"
        "5250158189800557887857123468519089021620794626489104228161693412385679961086045883270577699225567513963142"
        "1188823878765106201171875"},
    {0x1.fffffffffffffp-502, chars_format::fixed, 554,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000015274681817498021714429552205114197932911999153178942193579579"
        "6592392572937025642569879780747490318026237604052764619947460216028288150341471610525727826748426785117751"
        "8845531024566768292105395171353197431019897533198709379041833271791434446820304801228358679079023132333449"
        "0500316379601115775714246937038178043241589252978208456323386824771359922172091766541155398451135027926284"
        "237764775753021240234375"},
    {0x1.fffffffffffffp-501, chars_format::fixed, 553,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000030549363634996043428859104410228395865823998306357884387159159"
        "3184785145874051285139759561494980636052475208105529239894920432056576300682943221051455653496853570235503"
        "7691062049133536584210790342706394862039795066397418758083666543582868893640609602456717358158046264666898"
        "1000632759202231551428493874076356086483178505956416912646773649542719844344183533082310796902270055852568"
        "47552955150604248046875"},
    {0x1.fffffffffffffp-500, chars_format::fixed, 552,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000061098727269992086857718208820456791731647996612715768774318318"
        "6369570291748102570279519122989961272104950416211058479789840864113152601365886442102911306993707140471007"
        "5382124098267073168421580685412789724079590132794837516167333087165737787281219204913434716316092529333796"
        "2001265518404463102856987748152712172966357011912833825293547299085439688688367066164621593804540111705136"
        "9510591030120849609375"},
    {0x1.fffffffffffffp-499, chars_format::fixed, 551,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000122197454539984173715436417640913583463295993225431537548636637"
        "2739140583496205140559038245979922544209900832422116959579681728226305202731772884205822613987414280942015"
        "0764248196534146336843161370825579448159180265589675032334666174331475574562438409826869432632185058667592"
        "4002531036808926205713975496305424345932714023825667650587094598170879377376734132329243187609080223410273"
        "902118206024169921875"},
    {0x1.fffffffffffffp-498, chars_format::fixed, 550,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000244394909079968347430872835281827166926591986450863075097273274"
        "5478281166992410281118076491959845088419801664844233919159363456452610405463545768411645227974828561884030"
        "1528496393068292673686322741651158896318360531179350064669332348662951149124876819653738865264370117335184"
        "8005062073617852411427950992610848691865428047651335301174189196341758754753468264658486375218160446820547"
        "80423641204833984375"},
    {0x1.fffffffffffffp-497, chars_format::fixed, 549,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000488789818159936694861745670563654333853183972901726150194546549"
        "0956562333984820562236152983919690176839603329688467838318726912905220810927091536823290455949657123768060"
        "3056992786136585347372645483302317792636721062358700129338664697325902298249753639307477730528740234670369"
        "6010124147235704822855901985221697383730856095302670602348378392683517509506936529316972750436320893641095"
        "6084728240966796875"},
    {0x1.fffffffffffffp-496, chars_format::fixed, 548,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000977579636319873389723491341127308667706367945803452300389093098"
        "1913124667969641124472305967839380353679206659376935676637453825810441621854183073646580911899314247536120"
        "6113985572273170694745290966604635585273442124717400258677329394651804596499507278614955461057480469340739"
        "2020248294471409645711803970443394767461712190605341204696756785367035019013873058633945500872641787282191"
        "216945648193359375"},
    {0x1.fffffffffffffp-495, chars_format::fixed, 547,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000001955159272639746779446982682254617335412735891606904600778186196"
        "3826249335939282248944611935678760707358413318753871353274907651620883243708366147293161823798628495072241"
        "2227971144546341389490581933209271170546884249434800517354658789303609192999014557229910922114960938681478"
        "4040496588942819291423607940886789534923424381210682409393513570734070038027746117267891001745283574564382"
        "43389129638671875"},
    {0x1.fffffffffffffp-494, chars_format::fixed, 546,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000003910318545279493558893965364509234670825471783213809201556372392"
        "7652498671878564497889223871357521414716826637507742706549815303241766487416732294586323647597256990144482"
        "4455942289092682778981163866418542341093768498869601034709317578607218385998029114459821844229921877362956"
        "8080993177885638582847215881773579069846848762421364818787027141468140076055492234535782003490567149128764"
        "8677825927734375"},
    {0x1.fffffffffffffp-493, chars_format::fixed, 545,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000007820637090558987117787930729018469341650943566427618403112744785"
        "5304997343757128995778447742715042829433653275015485413099630606483532974833464589172647295194513980288964"
        "8911884578185365557962327732837084682187536997739202069418635157214436771996058228919643688459843754725913"
        "6161986355771277165694431763547158139693697524842729637574054282936280152110984469071564006981134298257529"
        "735565185546875"},
    {0x1.fffffffffffffp-492, chars_format::fixed, 544,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000015641274181117974235575861458036938683301887132855236806225489571"
        "0609994687514257991556895485430085658867306550030970826199261212967065949666929178345294590389027960577929"
        "7823769156370731115924655465674169364375073995478404138837270314428873543992116457839287376919687509451827"
        "2323972711542554331388863527094316279387395049685459275148108565872560304221968938143128013962268596515059"
        "47113037109375"},
    {0x1.fffffffffffffp-491, chars_format::fixed, 543,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000031282548362235948471151722916073877366603774265710473612450979142"
        "1219989375028515983113790970860171317734613100061941652398522425934131899333858356690589180778055921155859"
        "5647538312741462231849310931348338728750147990956808277674540628857747087984232915678574753839375018903654"
        "4647945423085108662777727054188632558774790099370918550296217131745120608443937876286256027924537193030118"
        "9422607421875"},
    {0x1.fffffffffffffp-490, chars_format::fixed, 542,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000062565096724471896942303445832147754733207548531420947224901958284"
        "2439978750057031966227581941720342635469226200123883304797044851868263798667716713381178361556111842311719"
        "1295076625482924463698621862696677457500295981913616555349081257715494175968465831357149507678750037807308"
        "9295890846170217325555454108377265117549580198741837100592434263490241216887875752572512055849074386060237"
        "884521484375"},
    {0x1.fffffffffffffp-489, chars_format::fixed, 541,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000125130193448943793884606891664295509466415097062841894449803916568"
        "4879957500114063932455163883440685270938452400247766609594089703736527597335433426762356723112223684623438"
        "2590153250965848927397243725393354915000591963827233110698162515430988351936931662714299015357500075614617"
        "8591781692340434651110908216754530235099160397483674201184868526980482433775751505145024111698148772120475"
        "76904296875"},
    {0x1.fffffffffffffp-488, chars_format::fixed, 540,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000250260386897887587769213783328591018932830194125683788899607833136"
        "9759915000228127864910327766881370541876904800495533219188179407473055194670866853524713446224447369246876"
        "5180306501931697854794487450786709830001183927654466221396325030861976703873863325428598030715000151229235"
        "7183563384680869302221816433509060470198320794967348402369737053960964867551503010290048223396297544240951"
        "5380859375"},
    {0x1.fffffffffffffp-487, chars_format::fixed, 539,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000500520773795775175538427566657182037865660388251367577799215666273"
        "9519830000456255729820655533762741083753809600991066438376358814946110389341733707049426892448894738493753"
        "0360613003863395709588974901573419660002367855308932442792650061723953407747726650857196061430000302458471"
        "4367126769361738604443632867018120940396641589934696804739474107921929735103006020580096446792595088481903"
        "076171875"},
    {0x1.fffffffffffffp-486, chars_format::fixed, 538,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000001001041547591550351076855133314364075731320776502735155598431332547"
        "9039660000912511459641311067525482167507619201982132876752717629892220778683467414098853784897789476987506"
        "0721226007726791419177949803146839320004735710617864885585300123447906815495453301714392122860000604916942"
        "8734253538723477208887265734036241880793283179869393609478948215843859470206012041160192893585190176963806"
        "15234375"},
    {0x1.fffffffffffffp-485, chars_format::fixed, 537,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000002002083095183100702153710266628728151462641553005470311196862665095"
        "8079320001825022919282622135050964335015238403964265753505435259784441557366934828197707569795578953975012"
        "1442452015453582838355899606293678640009471421235729771170600246895813630990906603428784245720001209833885"
        "7468507077446954417774531468072483761586566359738787218957896431687718940412024082320385787170380353927612"
        "3046875"},
    {0x1.fffffffffffffp-484, chars_format::fixed, 536,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000004004166190366201404307420533257456302925283106010940622393725330191"
        "6158640003650045838565244270101928670030476807928531507010870519568883114733869656395415139591157907950024"
        "2884904030907165676711799212587357280018942842471459542341200493791627261981813206857568491440002419667771"
        "4937014154893908835549062936144967523173132719477574437915792863375437880824048164640771574340760707855224"
        "609375"},
    {0x1.fffffffffffffp-483, chars_format::fixed, 535,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000008008332380732402808614841066514912605850566212021881244787450660383"
        "2317280007300091677130488540203857340060953615857063014021741039137766229467739312790830279182315815900048"
        "5769808061814331353423598425174714560037885684942919084682400987583254523963626413715136982880004839335542"
        "9874028309787817671098125872289935046346265438955148875831585726750875761648096329281543148681521415710449"
        "21875"},
    {0x1.fffffffffffffp-482, chars_format::fixed, 534,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000016016664761464805617229682133029825211701132424043762489574901320766"
        "4634560014600183354260977080407714680121907231714126028043482078275532458935478625581660558364631631800097"
        "1539616123628662706847196850349429120075771369885838169364801975166509047927252827430273965760009678671085"
        "9748056619575635342196251744579870092692530877910297751663171453501751523296192658563086297363042831420898"
        "4375"},
    {0x1.fffffffffffffp-481, chars_format::fixed, 533,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000032033329522929611234459364266059650423402264848087524979149802641532"
        "9269120029200366708521954160815429360243814463428252056086964156551064917870957251163321116729263263600194"
        "3079232247257325413694393700698858240151542739771676338729603950333018095854505654860547931520019357342171"
        "9496113239151270684392503489159740185385061755820595503326342907003503046592385317126172594726085662841796"
        "875"},
    {0x1.fffffffffffffp-480, chars_format::fixed, 532,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000064066659045859222468918728532119300846804529696175049958299605283065"
        "8538240058400733417043908321630858720487628926856504112173928313102129835741914502326642233458526527200388"
        "6158464494514650827388787401397716480303085479543352677459207900666036191709011309721095863040038714684343"
        "8992226478302541368785006978319480370770123511641191006652685814007006093184770634252345189452171325683593"
        "75"},
    {0x1.fffffffffffffp-479, chars_format::fixed, 531,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000128133318091718444937837457064238601693609059392350099916599210566131"
        "7076480116801466834087816643261717440975257853713008224347856626204259671483829004653284466917053054400777"
        "2316928989029301654777574802795432960606170959086705354918415801332072383418022619442191726080077429368687"
        "7984452956605082737570013956638960741540247023282382013305371628014012186369541268504690378904342651367187"
        "5"},
    {0x1.fffffffffffffp-478, chars_format::fixed, 530,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000256266636183436889875674914128477203387218118784700199833198421132263"
        "4152960233602933668175633286523434881950515707426016448695713252408519342967658009306568933834106108801554"
        "4633857978058603309555149605590865921212341918173410709836831602664144766836045238884383452160154858737375"
        "596890591321016547514002791327792148308049404656476402661074325602802437273908253700938075780868530273437"
        "5"},
    {0x1.fffffffffffffp-477, chars_format::fixed, 529,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000512533272366873779751349828256954406774436237569400399666396842264526"
        "8305920467205867336351266573046869763901031414852032897391426504817038685935316018613137867668212217603108"
        "9267715956117206619110299211181731842424683836346821419673663205328289533672090477768766904320309717474751"
        "19378118264203309502800558265558429661609880931295280532214865120560487454781650740187615156173706054687"
        "5"},
    {0x1.fffffffffffffp-476, chars_format::fixed, 528,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000001025066544733747559502699656513908813548872475138800799332793684529053"
        "6611840934411734672702533146093739527802062829704065794782853009634077371870632037226275735336424435206217"
        "8535431912234413238220598422363463684849367672693642839347326410656579067344180955537533808640619434949502"
        "38756236528406619005601116531116859323219761862590561064429730241120974909563301480375230312347412109375"},
    {0x1.fffffffffffffp-475, chars_format::fixed, 527,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000002050133089467495119005399313027817627097744950277601598665587369058107"
        "3223681868823469345405066292187479055604125659408131589565706019268154743741264074452551470672848870412435"
        "7070863824468826476441196844726927369698735345387285678694652821313158134688361911075067617281238869899004"
        "7751247305681323801120223306223371864643952372518112212885946048224194981912660296075046062469482421875"},
    {0x1.fffffffffffffp-474, chars_format::fixed, 526,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000004100266178934990238010798626055635254195489900555203197331174738116214"
        "6447363737646938690810132584374958111208251318816263179131412038536309487482528148905102941345697740824871"
        "4141727648937652952882393689453854739397470690774571357389305642626316269376723822150135234562477739798009"
        "550249461136264760224044661244674372928790474503622442577189209644838996382532059215009212493896484375"},
    {0x1.fffffffffffffp-473, chars_format::fixed, 525,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000008200532357869980476021597252111270508390979801110406394662349476232429"
        "2894727475293877381620265168749916222416502637632526358262824077072618974965056297810205882691395481649742"
        "8283455297875305905764787378907709478794941381549142714778611285252632538753447644300270469124955479596019"
        "10049892227252952044808932248934874585758094900724488515437841928967799276506411843001842498779296875"},
    {0x1.fffffffffffffp-472, chars_format::fixed, 524,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000016401064715739960952043194504222541016781959602220812789324698952464858"
        "5789454950587754763240530337499832444833005275265052716525648154145237949930112595620411765382790963299485"
        "6566910595750611811529574757815418957589882763098285429557222570505265077506895288600540938249910959192038"
        "2009978445450590408961786449786974917151618980144897703087568385793559855301282368600368499755859375"},
    {0x1.fffffffffffffp-471, chars_format::fixed, 523,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000032802129431479921904086389008445082033563919204441625578649397904929717"
        "1578909901175509526481060674999664889666010550530105433051296308290475899860225191240823530765581926598971"
        "3133821191501223623059149515630837915179765526196570859114445141010530155013790577201081876499821918384076"
        "401995689090118081792357289957394983430323796028979540617513677158711971060256473720073699951171875"},
    {0x1.fffffffffffffp-470, chars_format::fixed, 522,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000065604258862959843808172778016890164067127838408883251157298795809859434"
        "3157819802351019052962121349999329779332021101060210866102592616580951799720450382481647061531163853197942"
        "6267642383002447246118299031261675830359531052393141718228890282021060310027581154402163752999643836768152"
        "80399137818023616358471457991478996686064759205795908123502735431742394212051294744014739990234375"},
    {0x1.fffffffffffffp-469, chars_format::fixed, 521,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000131208517725919687616345556033780328134255676817766502314597591619718868"
        "6315639604702038105924242699998659558664042202120421732205185233161903599440900764963294123062327706395885"
        "2535284766004894492236598062523351660719062104786283436457780564042120620055162308804327505999287673536305"
        "6079827563604723271694291598295799337212951841159181624700547086348478842410258948802947998046875"},
    {0x1.fffffffffffffp-468, chars_format::fixed, 520,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000262417035451839375232691112067560656268511353635533004629195183239437737"
        "2631279209404076211848485399997319117328084404240843464410370466323807198881801529926588246124655412791770"
        "5070569532009788984473196125046703321438124209572566872915561128084241240110324617608655011998575347072611"
        "215965512720944654338858319659159867442590368231836324940109417269695768482051789760589599609375"},
    {0x1.fffffffffffffp-467, chars_format::fixed, 519,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000524834070903678750465382224135121312537022707271066009258390366478875474"
        "5262558418808152423696970799994638234656168808481686928820740932647614397763603059853176492249310825583541"
        "0141139064019577968946392250093406642876248419145133745831122256168482480220649235217310023997150694145222"
        "43193102544188930867771663931831973488518073646367264988021883453939153696410357952117919921875"},
    {0x1.fffffffffffffp-466, chars_format::fixed, 518,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000001049668141807357500930764448270242625074045414542132018516780732957750949"
        "0525116837616304847393941599989276469312337616963373857641481865295228795527206119706352984498621651167082"
        "0282278128039155937892784500186813285752496838290267491662244512336964960441298470434620047994301388290444"
        "8638620508837786173554332786366394697703614729273452997604376690787830739282071590423583984375"},
    {0x1.fffffffffffffp-465, chars_format::fixed, 517,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000002099336283614715001861528896540485250148090829084264037033561465915501898"
        "1050233675232609694787883199978552938624675233926747715282963730590457591054412239412705968997243302334164"
        "0564556256078311875785569000373626571504993676580534983324489024673929920882596940869240095988602776580889"
        "727724101767557234710866557273278939540722945854690599520875338157566147856414318084716796875"},
    {0x1.fffffffffffffp-464, chars_format::fixed, 516,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000004198672567229430003723057793080970500296181658168528074067122931831003796"
        "2100467350465219389575766399957105877249350467853495430565927461180915182108824478825411937994486604668328"
        "1129112512156623751571138000747253143009987353161069966648978049347859841765193881738480191977205553161779"
        "45544820353511446942173311454655787908144589170938119904175067631513229571282863616943359375"},
    {0x1.fffffffffffffp-463, chars_format::fixed, 515,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000008397345134458860007446115586161941000592363316337056148134245863662007592"
        "4200934700930438779151532799914211754498700935706990861131854922361830364217648957650823875988973209336656"
        "2258225024313247503142276001494506286019974706322139933297956098695719683530387763476960383954411106323558"
        "9108964070702289388434662290931157581628917834187623980835013526302645914256572723388671875"},
    {0x1.fffffffffffffp-462, chars_format::fixed, 514,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000016794690268917720014892231172323882001184726632674112296268491727324015184"
        "8401869401860877558303065599828423508997401871413981722263709844723660728435297915301647751977946418673312"
        "4516450048626495006284552002989012572039949412644279866595912197391439367060775526953920767908822212647117"
        "821792814140457877686932458186231516325783566837524796167002705260529182851314544677734375"},
    {0x1.fffffffffffffp-461, chars_format::fixed, 513,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000033589380537835440029784462344647764002369453265348224592536983454648030369"
        "6803738803721755116606131199656847017994803742827963444527419689447321456870595830603295503955892837346624"
        "9032900097252990012569104005978025144079898825288559733191824394782878734121551053907841535817644425294235"
        "64358562828091575537386491637246303265156713367504959233400541052105836570262908935546875"},
    {0x1.fffffffffffffp-460, chars_format::fixed, 512,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000067178761075670880059568924689295528004738906530696449185073966909296060739"
        "3607477607443510233212262399313694035989607485655926889054839378894642913741191661206591007911785674693249"
        "8065800194505980025138208011956050288159797650577119466383648789565757468243102107815683071635288850588471"
        "2871712565618315107477298327449260653031342673500991846680108210421167314052581787109375"},
    {0x1.fffffffffffffp-459, chars_format::fixed, 511,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000134357522151341760119137849378591056009477813061392898370147933818592121478"
        "7214955214887020466424524798627388071979214971311853778109678757789285827482383322413182015823571349386499"
        "6131600389011960050276416023912100576319595301154238932767297579131514936486204215631366143270577701176942"
        "574342513123663021495459665489852130606268534700198369336021642084233462810516357421875"},
    {0x1.fffffffffffffp-458, chars_format::fixed, 510,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000268715044302683520238275698757182112018955626122785796740295867637184242957"
        "4429910429774040932849049597254776143958429942623707556219357515578571654964766644826364031647142698772999"
        "2263200778023920100552832047824201152639190602308477865534595158263029872972408431262732286541155402353885"
        "14868502624732604299091933097970426121253706940039673867204328416846692562103271484375"},
    {0x1.fffffffffffffp-457, chars_format::fixed, 509,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000537430088605367040476551397514364224037911252245571593480591735274368485914"
        "8859820859548081865698099194509552287916859885247415112438715031157143309929533289652728063294285397545998"
        "4526401556047840201105664095648402305278381204616955731069190316526059745944816862525464573082310804707770"
        "2973700524946520859818386619594085224250741388007934773440865683369338512420654296875"},
    {0x1.fffffffffffffp-456, chars_format::fixed, 508,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000001074860177210734080953102795028728448075822504491143186961183470548736971829"
        "7719641719096163731396198389019104575833719770494830224877430062314286619859066579305456126588570795091996"
        "9052803112095680402211328191296804610556762409233911462138380633052119491889633725050929146164621609415540"
        "594740104989304171963677323918817044850148277601586954688173136673867702484130859375"},
    {0x1.fffffffffffffp-455, chars_format::fixed, 507,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000002149720354421468161906205590057456896151645008982286373922366941097473943659"
        "5439283438192327462792396778038209151667439540989660449754860124628573239718133158610912253177141590183993"
        "8105606224191360804422656382593609221113524818467822924276761266104238983779267450101858292329243218831081"
        "18948020997860834392735464783763408970029655520317390937634627334773540496826171875"},
    {0x1.fffffffffffffp-454, chars_format::fixed, 506,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000004299440708842936323812411180114913792303290017964572747844733882194947887319"
        "0878566876384654925584793556076418303334879081979320899509720249257146479436266317221824506354283180367987"
        "6211212448382721608845312765187218442227049636935645848553522532208477967558534900203716584658486437662162"
        "3789604199572166878547092956752681794005931104063478187526925466954708099365234375"},
    {0x1.fffffffffffffp-453, chars_format::fixed, 505,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000008598881417685872647624822360229827584606580035929145495689467764389895774638"
        "1757133752769309851169587112152836606669758163958641799019440498514292958872532634443649012708566360735975"
        "2422424896765443217690625530374436884454099273871291697107045064416955935117069800407433169316972875324324"
        "757920839914433375709418591350536358801186220812695637505385093390941619873046875"},
    {0x1.fffffffffffffp-452, chars_format::fixed, 504,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000017197762835371745295249644720459655169213160071858290991378935528779791549276"
        "3514267505538619702339174224305673213339516327917283598038880997028585917745065268887298025417132721471950"
        "4844849793530886435381251060748873768908198547742583394214090128833911870234139600814866338633945750648649"
        "51584167982886675141883718270107271760237244162539127501077018678188323974609375"},
    {0x1.fffffffffffffp-451, chars_format::fixed, 503,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000034395525670743490590499289440919310338426320143716581982757871057559583098552"
        "7028535011077239404678348448611346426679032655834567196077761994057171835490130537774596050834265442943900"
        "9689699587061772870762502121497747537816397095485166788428180257667823740468279201629732677267891501297299"
        "0316833596577335028376743654021454352047448832507825500215403735637664794921875"},
    {0x1.fffffffffffffp-450, chars_format::fixed, 502,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000068791051341486981180998578881838620676852640287433163965515742115119166197105"
        "4057070022154478809356696897222692853358065311669134392155523988114343670980261075549192101668530885887801"
        "9379399174123545741525004242995495075632794190970333576856360515335647480936558403259465354535783002594598"
        "063366719315467005675348730804290870409489766501565100043080747127532958984375"},
    {0x1.fffffffffffffp-449, chars_format::fixed, 501,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000137582102682973962361997157763677241353705280574866327931031484230238332394210"
        "8114140044308957618713393794445385706716130623338268784311047976228687341960522151098384203337061771775603"
        "8758798348247091483050008485990990151265588381940667153712721030671294961873116806518930709071566005189196"
        "12673343863093401135069746160858174081897953300313020008616149425506591796875"},
    {0x1.fffffffffffffp-448, chars_format::fixed, 500,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000275164205365947924723994315527354482707410561149732655862062968460476664788421"
        "6228280088617915237426787588890771413432261246676537568622095952457374683921044302196768406674123543551207"
        "7517596696494182966100016971981980302531176763881334307425442061342589923746233613037861418143132010378392"
        "2534668772618680227013949232171634816379590660062604001723229885101318359375"},
    {0x1.fffffffffffffp-447, chars_format::fixed, 499,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000550328410731895849447988631054708965414821122299465311724125936920953329576843"
        "2456560177235830474853575177781542826864522493353075137244191904914749367842088604393536813348247087102415"
        "5035193392988365932200033943963960605062353527762668614850884122685179847492467226075722836286264020756784"
        "506933754523736045402789846434326963275918132012520800344645977020263671875"},
    {0x1.fffffffffffffp-446, chars_format::fixed, 498,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000001100656821463791698895977262109417930829642244598930623448251873841906659153686"
        "4913120354471660949707150355563085653729044986706150274488383809829498735684177208787073626696494174204831"
        "0070386785976731864400067887927921210124707055525337229701768245370359694984934452151445672572528041513569"
        "01386750904747209080557969286865392655183626402504160068929195404052734375"},
    {0x1.fffffffffffffp-445, chars_format::fixed, 497,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000002201313642927583397791954524218835861659284489197861246896503747683813318307372"
        "9826240708943321899414300711126171307458089973412300548976767619658997471368354417574147253392988348409662"
        "0140773571953463728800135775855842420249414111050674459403536490740719389969868904302891345145056083027138"
        "0277350180949441816111593857373078531036725280500832013785839080810546875"},
    {0x1.fffffffffffffp-444, chars_format::fixed, 496,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000004402627285855166795583909048437671723318568978395722493793007495367626636614745"
        "9652481417886643798828601422252342614916179946824601097953535239317994942736708835148294506785976696819324"
        "0281547143906927457600271551711684840498828222101348918807072981481438779939737808605782690290112166054276"
        "055470036189888363222318771474615706207345056100166402757167816162109375"},
    {0x1.fffffffffffffp-443, chars_format::fixed, 495,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000008805254571710333591167818096875343446637137956791444987586014990735253273229491"
        "9304962835773287597657202844504685229832359893649202195907070478635989885473417670296589013571953393638648"
        "0563094287813854915200543103423369680997656444202697837614145962962877559879475617211565380580224332108552"
        "11094007237977672644463754294923141241469011220033280551433563232421875"},
    {0x1.fffffffffffffp-442, chars_format::fixed, 494,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000017610509143420667182335636193750686893274275913582889975172029981470506546458983"
        "8609925671546575195314405689009370459664719787298404391814140957271979770946835340593178027143906787277296"
        "1126188575627709830401086206846739361995312888405395675228291925925755119758951234423130761160448664217104"
        "2218801447595534528892750858984628248293802244006656110286712646484375"},
    {0x1.fffffffffffffp-441, chars_format::fixed, 493,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000035221018286841334364671272387501373786548551827165779950344059962941013092917967"
        "7219851343093150390628811378018740919329439574596808783628281914543959541893670681186356054287813574554592"
        "2252377151255419660802172413693478723990625776810791350456583851851510239517902468846261522320897328434208"
        "443760289519106905778550171796925649658760448801331222057342529296875"},
    {0x1.fffffffffffffp-440, chars_format::fixed, 492,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000070442036573682668729342544775002747573097103654331559900688119925882026185835935"
        "4439702686186300781257622756037481838658879149193617567256563829087919083787341362372712108575627149109184"
        "4504754302510839321604344827386957447981251553621582700913167703703020479035804937692523044641794656868416"
        "88752057903821381155710034359385129931752089760266244411468505859375"},
    {0x1.fffffffffffffp-439, chars_format::fixed, 491,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000140884073147365337458685089550005495146194207308663119801376239851764052371671870"
        "8879405372372601562515245512074963677317758298387235134513127658175838167574682724745424217151254298218368"
        "9009508605021678643208689654773914895962503107243165401826335407406040958071609875385046089283589313736833"
        "7750411580764276231142006871877025986350417952053248882293701171875"},
    {0x1.fffffffffffffp-438, chars_format::fixed, 490,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000281768146294730674917370179100010990292388414617326239602752479703528104743343741"
        "7758810744745203125030491024149927354635516596774470269026255316351676335149365449490848434302508596436737"
        "8019017210043357286417379309547829791925006214486330803652670814812081916143219750770092178567178627473667"
        "550082316152855246228401374375405197270083590410649776458740234375"},
    {0x1.fffffffffffffp-437, chars_format::fixed, 489,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000563536292589461349834740358200021980584776829234652479205504959407056209486687483"
        "5517621489490406250060982048299854709271033193548940538052510632703352670298730898981696868605017192873475"
        "6038034420086714572834758619095659583850012428972661607305341629624163832286439501540184357134357254947335"
        "10016463230571049245680274875081039454016718082129955291748046875"},
    {0x1.fffffffffffffp-436, chars_format::fixed, 488,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000001127072585178922699669480716400043961169553658469304958411009918814112418973374967"
        "1035242978980812500121964096599709418542066387097881076105021265406705340597461797963393737210034385746951"
        "2076068840173429145669517238191319167700024857945323214610683259248327664572879003080368714268714509894670"
        "2003292646114209849136054975016207890803343616425991058349609375"},
    {0x1.fffffffffffffp-435, chars_format::fixed, 487,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000002254145170357845399338961432800087922339107316938609916822019837628224837946749934"
        "2070485957961625000243928193199418837084132774195762152210042530813410681194923595926787474420068771493902"
        "4152137680346858291339034476382638335400049715890646429221366518496655329145758006160737428537429019789340"
        "400658529222841969827210995003241578160668723285198211669921875"},
    {0x1.fffffffffffffp-434, chars_format::fixed, 486,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000004508290340715690798677922865600175844678214633877219833644039675256449675893499868"
        "4140971915923250000487856386398837674168265548391524304420085061626821362389847191853574948840137542987804"
        "8304275360693716582678068952765276670800099431781292858442733036993310658291516012321474857074858039578680"
        "80131705844568393965442199000648315632133744657039642333984375"},
    {0x1.fffffffffffffp-433, chars_format::fixed, 485,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000009016580681431381597355845731200351689356429267754439667288079350512899351786999736"
        "8281943831846500000975712772797675348336531096783048608840170123253642724779694383707149897680275085975609"
        "6608550721387433165356137905530553341600198863562585716885466073986621316583032024642949714149716079157361"
        "6026341168913678793088439800129663126426748931407928466796875"},
    {0x1.fffffffffffffp-432, chars_format::fixed, 484,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000018033161362862763194711691462400703378712858535508879334576158701025798703573999473"
        "6563887663693000001951425545595350696673062193566097217680340246507285449559388767414299795360550171951219"
        "3217101442774866330712275811061106683200397727125171433770932147973242633166064049285899428299432158314723"
        "205268233782735758617687960025932625285349786281585693359375"},
    {0x1.fffffffffffffp-431, chars_format::fixed, 483,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000036066322725725526389423382924801406757425717071017758669152317402051597407147998947"
        "3127775327386000003902851091190701393346124387132194435360680493014570899118777534828599590721100343902438"
        "6434202885549732661424551622122213366400795454250342867541864295946485266332128098571798856598864316629446"
        "41053646756547151723537592005186525057069957256317138671875"},
    {0x1.fffffffffffffp-430, chars_format::fixed, 482,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000072132645451451052778846765849602813514851434142035517338304634804103194814295997894"
        "6255550654772000007805702182381402786692248774264388870721360986029141798237555069657199181442200687804877"
        "2868405771099465322849103244244426732801590908500685735083728591892970532664256197143597713197728633258892"
        "8210729351309430344707518401037305011413991451263427734375"},
    {0x1.fffffffffffffp-429, chars_format::fixed, 481,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000144265290902902105557693531699205627029702868284071034676609269608206389628591995789"
        "2511101309544000015611404364762805573384497548528777741442721972058283596475110139314398362884401375609754"
        "5736811542198930645698206488488853465603181817001371470167457183785941065328512394287195426395457266517785"
        "642145870261886068941503680207461002282798290252685546875"},
    {0x1.fffffffffffffp-428, chars_format::fixed, 480,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000288530581805804211115387063398411254059405736568142069353218539216412779257183991578"
        "5022202619088000031222808729525611146768995097057555482885443944116567192950220278628796725768802751219509"
        "1473623084397861291396412976977706931206363634002742940334914367571882130657024788574390852790914533035571"
        "28429174052377213788300736041492200456559658050537109375"},
    {0x1.fffffffffffffp-427, chars_format::fixed, 479,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000577061163611608422230774126796822508118811473136284138706437078432825558514367983157"
        "0044405238176000062445617459051222293537990194115110965770887888233134385900440557257593451537605502439018"
        "2947246168795722582792825953955413862412727268005485880669828735143764261314049577148781705581829066071142"
        "5685834810475442757660147208298440091311931610107421875"},
    {0x1.fffffffffffffp-426, chars_format::fixed, 478,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000001154122327223216844461548253593645016237622946272568277412874156865651117028735966314"
        "0088810476352000124891234918102444587075980388230221931541775776466268771800881114515186903075211004878036"
        "5894492337591445165585651907910827724825454536010971761339657470287528522628099154297563411163658132142285"
        "137166962095088551532029441659688018262386322021484375"},
    {0x1.fffffffffffffp-425, chars_format::fixed, 477,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000002308244654446433688923096507187290032475245892545136554825748313731302234057471932628"
        "0177620952704000249782469836204889174151960776460443863083551552932537543601762229030373806150422009756073"
        "1788984675182890331171303815821655449650909072021943522679314940575057045256198308595126822327316264284570"
        "27433392419017710306405888331937603652477264404296875"},
    {0x1.fffffffffffffp-424, chars_format::fixed, 476,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000004616489308892867377846193014374580064950491785090273109651496627462604468114943865256"
        "0355241905408000499564939672409778348303921552920887726167103105865075087203524458060747612300844019512146"
        "3577969350365780662342607631643310899301818144043887045358629881150114090512396617190253644654632528569140"
        "5486678483803542061281177666387520730495452880859375"},
    {0x1.fffffffffffffp-423, chars_format::fixed, 475,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000009232978617785734755692386028749160129900983570180546219302993254925208936229887730512"
        "0710483810816000999129879344819556696607843105841775452334206211730150174407048916121495224601688039024292"
        "7155938700731561324685215263286621798603636288087774090717259762300228181024793234380507289309265057138281"
        "097335696760708412256235533277504146099090576171875"},
    {0x1.fffffffffffffp-422, chars_format::fixed, 474,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000018465957235571469511384772057498320259801967140361092438605986509850417872459775461024"
        "1420967621632001998259758689639113393215686211683550904668412423460300348814097832242990449203376078048585"
        "4311877401463122649370430526573243597207272576175548181434519524600456362049586468761014578618530114276562"
        "19467139352141682451247106655500829219818115234375"},
    {0x1.fffffffffffffp-421, chars_format::fixed, 473,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000036931914471142939022769544114996640519603934280722184877211973019700835744919550922048"
        "2841935243264003996519517379278226786431372423367101809336824846920600697628195664485980898406752156097170"
        "8623754802926245298740861053146487194414545152351096362869039049200912724099172937522029157237060228553124"
        "3893427870428336490249421331100165843963623046875"},
    {0x1.fffffffffffffp-420, chars_format::fixed, 472,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000073863828942285878045539088229993281039207868561444369754423946039401671489839101844096"
        "5683870486528007993039034758556453572862744846734203618673649693841201395256391328971961796813504312194341"
        "7247509605852490597481722106292974388829090304702192725738078098401825448198345875044058314474120457106248"
        "778685574085667298049884266220033168792724609375"},
    {0x1.fffffffffffffp-419, chars_format::fixed, 471,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000147727657884571756091078176459986562078415737122888739508847892078803342979678203688193"
        "1367740973056015986078069517112907145725489693468407237347299387682402790512782657943923593627008624388683"
        "4495019211704981194963444212585948777658180609404385451476156196803650896396691750088116628948240914212497"
        "55737114817133459609976853244006633758544921875"},
    {0x1.fffffffffffffp-418, chars_format::fixed, 470,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000295455315769143512182156352919973124156831474245777479017695784157606685959356407376386"
        "2735481946112031972156139034225814291450979386936814474694598775364805581025565315887847187254017248777366"
        "8990038423409962389926888425171897555316361218808770902952312393607301792793383500176233257896481828424995"
        "1147422963426691921995370648801326751708984375"},
    {0x1.fffffffffffffp-417, chars_format::fixed, 469,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000590910631538287024364312705839946248313662948491554958035391568315213371918712814752772"
        "5470963892224063944312278068451628582901958773873628949389197550729611162051130631775694374508034497554733"
        "7980076846819924779853776850343795110632722437617541805904624787214603585586767000352466515792963656849990"
        "229484592685338384399074129760265350341796875"},
    {0x1.fffffffffffffp-416, chars_format::fixed, 468,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000001181821263076574048728625411679892496627325896983109916070783136630426743837425629505545"
        "0941927784448127888624556136903257165803917547747257898778395101459222324102261263551388749016068995109467"
        "5960153693639849559707553700687590221265444875235083611809249574429207171173534000704933031585927313699980"
        "45896918537067676879814825952053070068359375"},
    {0x1.fffffffffffffp-415, chars_format::fixed, 467,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000002363642526153148097457250823359784993254651793966219832141566273260853487674851259011090"
        "1883855568896255777249112273806514331607835095494515797556790202918444648204522527102777498032137990218935"
        "1920307387279699119415107401375180442530889750470167223618499148858414342347068001409866063171854627399960"
        "9179383707413535375962965190410614013671875"},
    {0x1.fffffffffffffp-414, chars_format::fixed, 466,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000004727285052306296194914501646719569986509303587932439664283132546521706975349702518022180"
        "3767711137792511554498224547613028663215670190989031595113580405836889296409045054205554996064275980437870"
        "3840614774559398238830214802750360885061779500940334447236998297716828684694136002819732126343709254799921"
        "835876741482707075192593038082122802734375"},
    {0x1.fffffffffffffp-413, chars_format::fixed, 465,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000009454570104612592389829003293439139973018607175864879328566265093043413950699405036044360"
        "7535422275585023108996449095226057326431340381978063190227160811673778592818090108411109992128551960875740"
        "7681229549118796477660429605500721770123559001880668894473996595433657369388272005639464252687418509599843"
        "67175348296541415038518607616424560546875"},
    {0x1.fffffffffffffp-412, chars_format::fixed, 464,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000018909140209225184779658006586878279946037214351729758657132530186086827901398810072088721"
        "5070844551170046217992898190452114652862680763956126380454321623347557185636180216822219984257103921751481"
        "5362459098237592955320859211001443540247118003761337788947993190867314738776544011278928505374837019199687"
        "3435069659308283007703721523284912109375"},
    {0x1.fffffffffffffp-411, chars_format::fixed, 463,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000037818280418450369559316013173756559892074428703459517314265060372173655802797620144177443"
        "0141689102340092435985796380904229305725361527912252760908643246695114371272360433644439968514207843502963"
        "0724918196475185910641718422002887080494236007522675577895986381734629477553088022557857010749674038399374"
        "687013931861656601540744304656982421875"},
    {0x1.fffffffffffffp-410, chars_format::fixed, 462,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000075636560836900739118632026347513119784148857406919034628530120744347311605595240288354886"
        "0283378204680184871971592761808458611450723055824505521817286493390228742544720867288879937028415687005926"
        "1449836392950371821283436844005774160988472015045351155791972763469258955106176045115714021499348076798749"
        "37402786372331320308148860931396484375"},
    {0x1.fffffffffffffp-409, chars_format::fixed, 461,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000151273121673801478237264052695026239568297714813838069257060241488694623211190480576709772"
        "0566756409360369743943185523616917222901446111649011043634572986780457485089441734577759874056831374011852"
        "2899672785900743642566873688011548321976944030090702311583945526938517910212352090231428042998696153597498"
        "7480557274466264061629772186279296875"},
    {0x1.fffffffffffffp-408, chars_format::fixed, 460,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000302546243347602956474528105390052479136595429627676138514120482977389246422380961153419544"
        "1133512818720739487886371047233834445802892223298022087269145973560914970178883469155519748113662748023704"
        "5799345571801487285133747376023096643953888060181404623167891053877035820424704180462856085997392307194997"
        "496111454893252812325954437255859375"},
    {0x1.fffffffffffffp-407, chars_format::fixed, 459,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000605092486695205912949056210780104958273190859255352277028240965954778492844761922306839088"
        "2267025637441478975772742094467668891605784446596044174538291947121829940357766938311039496227325496047409"
        "1598691143602974570267494752046193287907776120362809246335782107754071640849408360925712171994784614389994"
        "99222290978650562465190887451171875"},
    {0x1.fffffffffffffp-406, chars_format::fixed, 458,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000001210184973390411825898112421560209916546381718510704554056481931909556985689523844613678176"
        "4534051274882957951545484188935337783211568893192088349076583894243659880715533876622078992454650992094818"
        "3197382287205949140534989504092386575815552240725618492671564215508143281698816721851424343989569228779989"
        "9844458195730112493038177490234375"},
    {0x1.fffffffffffffp-405, chars_format::fixed, 457,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000002420369946780823651796224843120419833092763437021409108112963863819113971379047689227356352"
        "9068102549765915903090968377870675566423137786384176698153167788487319761431067753244157984909301984189636"
        "6394764574411898281069979008184773151631104481451236985343128431016286563397633443702848687979138457559979"
        "968891639146022498607635498046875"},
    {0x1.fffffffffffffp-404, chars_format::fixed, 456,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000004840739893561647303592449686240839666185526874042818216225927727638227942758095378454712705"
        "8136205099531831806181936755741351132846275572768353396306335576974639522862135506488315969818603968379273"
        "2789529148823796562139958016369546303262208962902473970686256862032573126795266887405697375958276915119959"
        "93778327829204499721527099609375"},
    {0x1.fffffffffffffp-403, chars_format::fixed, 455,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000009681479787123294607184899372481679332371053748085636432451855455276455885516190756909425411"
        "6272410199063663612363873511482702265692551145536706792612671153949279045724271012976631939637207936758546"
        "5579058297647593124279916032739092606524417925804947941372513724065146253590533774811394751916553830239919"
        "8755665565840899944305419921875"},
    {0x1.fffffffffffffp-402, chars_format::fixed, 454,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000019362959574246589214369798744963358664742107496171272864903710910552911771032381513818850823"
        "2544820398127327224727747022965404531385102291073413585225342307898558091448542025953263879274415873517093"
        "1158116595295186248559832065478185213048835851609895882745027448130292507181067549622789503833107660479839"
        "751133113168179988861083984375"},
    {0x1.fffffffffffffp-401, chars_format::fixed, 453,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000038725919148493178428739597489926717329484214992342545729807421821105823542064763027637701646"
        "5089640796254654449455494045930809062770204582146827170450684615797116182897084051906527758548831747034186"
        "2316233190590372497119664130956370426097671703219791765490054896260585014362135099245579007666215320959679"
        "50226622633635997772216796875"},
    {0x1.fffffffffffffp-400, chars_format::fixed, 452,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000077451838296986356857479194979853434658968429984685091459614843642211647084129526055275403293"
        "0179281592509308898910988091861618125540409164293654340901369231594232365794168103813055517097663494068372"
        "4632466381180744994239328261912740852195343406439583530980109792521170028724270198491158015332430641919359"
        "0045324526727199554443359375"},
    {0x1.fffffffffffffp-399, chars_format::fixed, 451,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000154903676593972713714958389959706869317936859969370182919229687284423294168259052110550806586"
        "0358563185018617797821976183723236251080818328587308681802738463188464731588336207626111034195326988136744"
        "9264932762361489988478656523825481704390686812879167061960219585042340057448540396982316030664861283838718"
        "009064905345439910888671875"},
    {0x1.fffffffffffffp-398, chars_format::fixed, 450,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000309807353187945427429916779919413738635873719938740365838459374568846588336518104221101613172"
        "0717126370037235595643952367446472502161636657174617363605476926376929463176672415252222068390653976273489"
        "8529865524722979976957313047650963408781373625758334123920439170084680114897080793964632061329722567677436"
        "01812981069087982177734375"},
    {0x1.fffffffffffffp-397, chars_format::fixed, 449,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000619614706375890854859833559838827477271747439877480731676918749137693176673036208442203226344"
        "1434252740074471191287904734892945004323273314349234727210953852753858926353344830504444136781307952546979"
        "7059731049445959953914626095301926817562747251516668247840878340169360229794161587929264122659445135354872"
        "0362596213817596435546875"},
    {0x1.fffffffffffffp-396, chars_format::fixed, 448,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000001239229412751781709719667119677654954543494879754961463353837498275386353346072416884406452688"
        "2868505480148942382575809469785890008646546628698469454421907705507717852706689661008888273562615905093959"
        "4119462098891919907829252190603853635125494503033336495681756680338720459588323175858528245318890270709744"
        "072519242763519287109375"},
    {0x1.fffffffffffffp-395, chars_format::fixed, 447,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000002478458825503563419439334239355309909086989759509922926707674996550772706692144833768812905376"
        "5737010960297884765151618939571780017293093257396938908843815411015435705413379322017776547125231810187918"
        "8238924197783839815658504381207707270250989006066672991363513360677440919176646351717056490637780541419488"
        "14503848552703857421875"},
    {0x1.fffffffffffffp-394, chars_format::fixed, 446,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000004956917651007126838878668478710619818173979519019845853415349993101545413384289667537625810753"
        "1474021920595769530303237879143560034586186514793877817687630822030871410826758644035553094250463620375837"
        "6477848395567679631317008762415414540501978012133345982727026721354881838353292703434112981275561082838976"
        "2900769710540771484375"},
    {0x1.fffffffffffffp-393, chars_format::fixed, 445,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000009913835302014253677757336957421239636347959038039691706830699986203090826768579335075251621506"
        "2948043841191539060606475758287120069172373029587755635375261644061742821653517288071106188500927240751675"
        "2955696791135359262634017524830829081003956024266691965454053442709763676706585406868225962551122165677952"
        "580153942108154296875"},
    {0x1.fffffffffffffp-392, chars_format::fixed, 444,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000019827670604028507355514673914842479272695918076079383413661399972406181653537158670150503243012"
        "5896087682383078121212951516574240138344746059175511270750523288123485643307034576142212377001854481503350"
        "5911393582270718525268035049661658162007912048533383930908106885419527353413170813736451925102244331355905"
        "16030788421630859375"},
    {0x1.fffffffffffffp-391, chars_format::fixed, 443,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000039655341208057014711029347829684958545391836152158766827322799944812363307074317340301006486025"
        "1792175364766156242425903033148480276689492118351022541501046576246971286614069152284424754003708963006701"
        "1822787164541437050536070099323316324015824097066767861816213770839054706826341627472903850204488662711810"
        "3206157684326171875"},
    {0x1.fffffffffffffp-390, chars_format::fixed, 442,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000079310682416114029422058695659369917090783672304317533654645599889624726614148634680602012972050"
        "3584350729532312484851806066296960553378984236702045083002093152493942573228138304568849508007417926013402"
        "3645574329082874101072140198646632648031648194133535723632427541678109413652683254945807700408977325423620"
        "641231536865234375"},
    {0x1.fffffffffffffp-389, chars_format::fixed, 441,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000158621364832228058844117391318739834181567344608635067309291199779249453228297269361204025944100"
        "7168701459064624969703612132593921106757968473404090166004186304987885146456276609137699016014835852026804"
        "7291148658165748202144280397293265296063296388267071447264855083356218827305366509891615400817954650847241"
        "28246307373046875"},
    {0x1.fffffffffffffp-388, chars_format::fixed, 440,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000317242729664456117688234782637479668363134689217270134618582399558498906456594538722408051888201"
        "4337402918129249939407224265187842213515936946808180332008372609975770292912553218275398032029671704053609"
        "4582297316331496404288560794586530592126592776534142894529710166712437654610733019783230801635909301694482"
        "5649261474609375"},
    {0x1.fffffffffffffp-387, chars_format::fixed, 439,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000634485459328912235376469565274959336726269378434540269237164799116997812913189077444816103776402"
        "8674805836258499878814448530375684427031873893616360664016745219951540585825106436550796064059343408107218"
        "9164594632662992808577121589173061184253185553068285789059420333424875309221466039566461603271818603388965"
        "129852294921875"},
    {0x1.fffffffffffffp-386, chars_format::fixed, 438,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000001268970918657824470752939130549918673452538756869080538474329598233995625826378154889632207552805"
        "7349611672516999757628897060751368854063747787232721328033490439903081171650212873101592128118686816214437"
        "8329189265325985617154243178346122368506371106136571578118840666849750618442932079132923206543637206777930"
        "25970458984375"},
    {0x1.fffffffffffffp-385, chars_format::fixed, 437,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000002537941837315648941505878261099837346905077513738161076948659196467991251652756309779264415105611"
        "4699223345033999515257794121502737708127495574465442656066980879806162343300425746203184256237373632428875"
        "6658378530651971234308486356692244737012742212273143156237681333699501236885864158265846413087274413555860"
        "5194091796875"},
    {0x1.fffffffffffffp-384, chars_format::fixed, 436,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000005075883674631297883011756522199674693810155027476322153897318392935982503305512619558528830211222"
        "9398446690067999030515588243005475416254991148930885312133961759612324686600851492406368512474747264857751"
        "3316757061303942468616972713384489474025484424546286312475362667399002473771728316531692826174548827111721"
        "038818359375"},
    {0x1.fffffffffffffp-383, chars_format::fixed, 435,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000010151767349262595766023513044399349387620310054952644307794636785871965006611025239117057660422445"
        "8796893380135998061031176486010950832509982297861770624267923519224649373201702984812737024949494529715502"
        "6633514122607884937233945426768978948050968849092572624950725334798004947543456633063385652349097654223442"
        "07763671875"},
    {0x1.fffffffffffffp-382, chars_format::fixed, 434,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000020303534698525191532047026088798698775240620109905288615589273571743930013222050478234115320844891"
        "7593786760271996122062352972021901665019964595723541248535847038449298746403405969625474049898989059431005"
        "3267028245215769874467890853537957896101937698185145249901450669596009895086913266126771304698195308446884"
        "1552734375"},
    {0x1.fffffffffffffp-381, chars_format::fixed, 433,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000040607069397050383064094052177597397550481240219810577231178547143487860026444100956468230641689783"
        "5187573520543992244124705944043803330039929191447082497071694076898597492806811939250948099797978118862010"
        "6534056490431539748935781707075915792203875396370290499802901339192019790173826532253542609396390616893768"
        "310546875"},
    {0x1.fffffffffffffp-380, chars_format::fixed, 432,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000081214138794100766128188104355194795100962480439621154462357094286975720052888201912936461283379567"
        "0375147041087984488249411888087606660079858382894164994143388153797194985613623878501896199595956237724021"
        "3068112980863079497871563414151831584407750792740580999605802678384039580347653064507085218792781233787536"
        "62109375"},
    {0x1.fffffffffffffp-379, chars_format::fixed, 431,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000162428277588201532256376208710389590201924960879242308924714188573951440105776403825872922566759134"
        "0750294082175968976498823776175213320159716765788329988286776307594389971227247757003792399191912475448042"
        "6136225961726158995743126828303663168815501585481161999211605356768079160695306129014170437585562467575073"
        "2421875"},
    {0x1.fffffffffffffp-378, chars_format::fixed, 430,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000324856555176403064512752417420779180403849921758484617849428377147902880211552807651745845133518268"
        "1500588164351937952997647552350426640319433531576659976573552615188779942454495514007584798383824950896085"
        "2272451923452317991486253656607326337631003170962323998423210713536158321390612258028340875171124935150146"
        "484375"},
    {0x1.fffffffffffffp-377, chars_format::fixed, 429,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000649713110352806129025504834841558360807699843516969235698856754295805760423105615303491690267036536"
        "3001176328703875905995295104700853280638867063153319953147105230377559884908991028015169596767649901792170"
        "4544903846904635982972507313214652675262006341924647996846421427072316642781224516056681750342249870300292"
        "96875"},
    {0x1.fffffffffffffp-376, chars_format::fixed, 428,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000001299426220705612258051009669683116721615399687033938471397713508591611520846211230606983380534073072"
        "6002352657407751811990590209401706561277734126306639906294210460755119769817982056030339193535299803584340"
        "9089807693809271965945014626429305350524012683849295993692842854144633285562449032113363500684499740600585"
        "9375"},
    {0x1.fffffffffffffp-375, chars_format::fixed, 427,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000002598852441411224516102019339366233443230799374067876942795427017183223041692422461213966761068146145"
        "2004705314815503623981180418803413122555468252613279812588420921510239539635964112060678387070599607168681"
        "8179615387618543931890029252858610701048025367698591987385685708289266571124898064226727001368999481201171"
        "875"},
    {0x1.fffffffffffffp-374, chars_format::fixed, 426,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000005197704882822449032204038678732466886461598748135753885590854034366446083384844922427933522136292290"
        "4009410629631007247962360837606826245110936505226559625176841843020479079271928224121356774141199214337363"
        "6359230775237087863780058505717221402096050735397183974771371416578533142249796128453454002737998962402343"
        "75"},
    {0x1.fffffffffffffp-373, chars_format::fixed, 425,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000010395409765644898064408077357464933772923197496271507771181708068732892166769689844855867044272584580"
        "8018821259262014495924721675213652490221873010453119250353683686040958158543856448242713548282398428674727"
        "2718461550474175727560117011434442804192101470794367949542742833157066284499592256906908005475997924804687"
        "5"},
    {0x1.fffffffffffffp-372, chars_format::fixed, 424,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000020790819531289796128816154714929867545846394992543015542363416137465784333539379689711734088545169161"
        "6037642518524028991849443350427304980443746020906238500707367372081916317087712896485427096564796857349454"
        "543692310094835145512023402286888560838420294158873589908548566631413256899918451381381601095199584960937"
        "5"},
    {0x1.fffffffffffffp-371, chars_format::fixed, 423,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000041581639062579592257632309429859735091692789985086031084726832274931568667078759379423468177090338323"
        "2075285037048057983698886700854609960887492041812477001414734744163832634175425792970854193129593714698909"
        "08738462018967029102404680457377712167684058831774717981709713326282651379983690276276320219039916992187"
        "5"},
    {0x1.fffffffffffffp-370, chars_format::fixed, 422,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000083163278125159184515264618859719470183385579970172062169453664549863137334157518758846936354180676646"
        "4150570074096115967397773401709219921774984083624954002829469488327665268350851585941708386259187429397818"
        "17476924037934058204809360914755424335368117663549435963419426652565302759967380552552640438079833984375"},
    {0x1.fffffffffffffp-369, chars_format::fixed, 421,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000166326556250318369030529237719438940366771159940344124338907329099726274668315037517693872708361353292"
        "8301140148192231934795546803418439843549968167249908005658938976655330536701703171883416772518374858795636"
        "3495384807586811640961872182951084867073623532709887192683885330513060551993476110510528087615966796875"},
    {0x1.fffffffffffffp-368, chars_format::fixed, 420,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000332653112500636738061058475438877880733542319880688248677814658199452549336630075035387745416722706585"
        "6602280296384463869591093606836879687099936334499816011317877953310661073403406343766833545036749717591272"
        "699076961517362328192374436590216973414724706541977438536777066102612110398695222102105617523193359375"},
    {0x1.fffffffffffffp-367, chars_format::fixed, 419,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000665306225001273476122116950877755761467084639761376497355629316398905098673260150070775490833445413171"
        "3204560592768927739182187213673759374199872668999632022635755906621322146806812687533667090073499435182545"
        "39815392303472465638474887318043394682944941308395487707355413220522422079739044420421123504638671875"},
    {0x1.fffffffffffffp-366, chars_format::fixed, 418,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0001330612450002546952244233901755511522934169279522752994711258632797810197346520300141550981666890826342"
        "6409121185537855478364374427347518748399745337999264045271511813242644293613625375067334180146998870365090"
        "7963078460694493127694977463608678936588988261679097541471082644104484415947808884084224700927734375"},
    {0x1.fffffffffffffp-365, chars_format::fixed, 417,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0002661224900005093904488467803511023045868338559045505989422517265595620394693040600283101963333781652685"
        "2818242371075710956728748854695037496799490675998528090543023626485288587227250750134668360293997740730181"
        "592615692138898625538995492721735787317797652335819508294216528820896883189561776816844940185546875"},
    {0x1.fffffffffffffp-364, chars_format::fixed, 416,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0005322449800010187808976935607022046091736677118091011978845034531191240789386081200566203926667563305370"
        "5636484742151421913457497709390074993598981351997056181086047252970577174454501500269336720587995481460363"
        "18523138427779725107799098544347157463559530467163901658843305764179376637912355363368988037109375"},
    {0x1.fffffffffffffp-363, chars_format::fixed, 415,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0010644899600020375617953871214044092183473354236182023957690069062382481578772162401132407853335126610741"
        "1272969484302843826914995418780149987197962703994112362172094505941154348909003000538673441175990962920726"
        "3704627685555945021559819708869431492711906093432780331768661152835875327582471072673797607421875"},
    {0x1.fffffffffffffp-362, chars_format::fixed, 414,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0021289799200040751235907742428088184366946708472364047915380138124764963157544324802264815706670253221482"
        "2545938968605687653829990837560299974395925407988224724344189011882308697818006001077346882351981925841452"
        "740925537111189004311963941773886298542381218686556066353732230567175065516494214534759521484375"},
    {0x1.fffffffffffffp-361, chars_format::fixed, 413,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0042579598400081502471815484856176368733893416944728095830760276249529926315088649604529631413340506442964"
        "5091877937211375307659981675120599948791850815976449448688378023764617395636012002154693764703963851682905"
        "48185107422237800862392788354777259708476243737311213270746446113435013103298842906951904296875"},
    {0x1.fffffffffffffp-360, chars_format::fixed, 412,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0085159196800163004943630969712352737467786833889456191661520552499059852630177299209059262826681012885929"
        "0183755874422750615319963350241199897583701631952898897376756047529234791272024004309387529407927703365810"
        "9637021484447560172478557670955451941695248747462242654149289222687002620659768581390380859375"},
    {0x1.fffffffffffffp-359, chars_format::fixed, 411,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0170318393600326009887261939424705474935573667778912383323041104998119705260354598418118525653362025771858"
        "0367511748845501230639926700482399795167403263905797794753512095058469582544048008618775058815855406731621"
        "927404296889512034495711534191090388339049749492448530829857844537400524131953716278076171875"},
    {0x1.fffffffffffffp-358, chars_format::fixed, 410,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0340636787200652019774523878849410949871147335557824766646082209996239410520709196836237051306724051543716"
        "0735023497691002461279853400964799590334806527811595589507024190116939165088096017237550117631710813463243"
        "85480859377902406899142306838218077667809949898489706165971568907480104826390743255615234375"},
    {0x1.fffffffffffffp-357, chars_format::fixed, 409,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0681273574401304039549047757698821899742294671115649533292164419992478821041418393672474102613448103087432"
        "1470046995382004922559706801929599180669613055623191179014048380233878330176192034475100235263421626926487"
        "7096171875580481379828461367643615533561989979697941233194313781496020965278148651123046875"},
    {0x1.fffffffffffffp-356, chars_format::fixed, 408,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "1362547148802608079098095515397643799484589342231299066584328839984957642082836787344948205226896206174864"
        "2940093990764009845119413603859198361339226111246382358028096760467756660352384068950200470526843253852975"
        "419234375116096275965692273528723106712397995939588246638862756299204193055629730224609375"},
    {0x1.fffffffffffffp-355, chars_format::fixed, 407,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "2725094297605216158196191030795287598969178684462598133168657679969915284165673574689896410453792412349728"
        "5880187981528019690238827207718396722678452222492764716056193520935513320704768137900400941053686507705950"
        "83846875023219255193138454705744621342479599187917649327772551259840838611125946044921875"},
    {0x1.fffffffffffffp-354, chars_format::fixed, 406,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "5450188595210432316392382061590575197938357368925196266337315359939830568331347149379792820907584824699457"
        "1760375963056039380477654415436793445356904444985529432112387041871026641409536275800801882107373015411901"
        "6769375004643851038627690941148924268495919837583529865554510251968167722225189208984375"},
    {0x1.fffffffffffffp-353, chars_format::fixed, 405,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001"
        "0900377190420864632784764123181150395876714737850392532674630719879661136662694298759585641815169649398914"
        "3520751926112078760955308830873586890713808889971058864224774083742053282819072551601603764214746030823803"
        "353875000928770207725538188229784853699183967516705973110902050393633544445037841796875"},
    {0x1.fffffffffffffp-352, chars_format::fixed, 404,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002"
        "1800754380841729265569528246362300791753429475700785065349261439759322273325388597519171283630339298797828"
        "7041503852224157521910617661747173781427617779942117728449548167484106565638145103203207528429492061647606"
        "70775000185754041545107637645956970739836793503341194622180410078726708889007568359375"},
    {0x1.fffffffffffffp-351, chars_format::fixed, 403,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004"
        "3601508761683458531139056492724601583506858951401570130698522879518644546650777195038342567260678597595657"
        "4083007704448315043821235323494347562855235559884235456899096334968213131276290206406415056858984123295213"
        "4155000037150808309021527529191394147967358700668238924436082015745341777801513671875"},
    {0x1.fffffffffffffp-350, chars_format::fixed, 402,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008"
        "7203017523366917062278112985449203167013717902803140261397045759037289093301554390076685134521357195191314"
        "8166015408896630087642470646988695125710471119768470913798192669936426262552580412812830113717968246590426"
        "831000007430161661804305505838278829593471740133647784887216403149068355560302734375"},
    {0x1.fffffffffffffp-349, chars_format::fixed, 401,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000017"
        "4406035046733834124556225970898406334027435805606280522794091518074578186603108780153370269042714390382629"
        "6332030817793260175284941293977390251420942239536941827596385339872852525105160825625660227435936493180853"
        "66200001486032332360861101167655765918694348026729556977443280629813671112060546875"},
    {0x1.fffffffffffffp-348, chars_format::fixed, 400,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000034"
        "8812070093467668249112451941796812668054871611212561045588183036149156373206217560306740538085428780765259"
        "2664061635586520350569882587954780502841884479073883655192770679745705050210321651251320454871872986361707"
        "3240000297206466472172220233531153183738869605345911395488656125962734222412109375"},
    {0x1.fffffffffffffp-347, chars_format::fixed, 399,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000069"
        "7624140186935336498224903883593625336109743222425122091176366072298312746412435120613481076170857561530518"
        "5328123271173040701139765175909561005683768958147767310385541359491410100420643302502640909743745972723414"
        "648000059441293294434444046706230636747773921069182279097731225192546844482421875"},
    {0x1.fffffffffffffp-346, chars_format::fixed, 398,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000139"
        "5248280373870672996449807767187250672219486444850244182352732144596625492824870241226962152341715123061037"
        "0656246542346081402279530351819122011367537916295534620771082718982820200841286605005281819487491945446829"
        "29600011888258658886888809341246127349554784213836455819546245038509368896484375"},
    {0x1.fffffffffffffp-345, chars_format::fixed, 397,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000279"
        "0496560747741345992899615534374501344438972889700488364705464289193250985649740482453924304683430246122074"
        "1312493084692162804559060703638244022735075832591069241542165437965640401682573210010563638974983890893658"
        "5920002377651731777377761868249225469910956842767291163909249007701873779296875"},
    {0x1.fffffffffffffp-344, chars_format::fixed, 396,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000558"
        "0993121495482691985799231068749002688877945779400976729410928578386501971299480964907848609366860492244148"
        "2624986169384325609118121407276488045470151665182138483084330875931280803365146420021127277949967781787317"
        "184000475530346355475552373649845093982191368553458232781849801540374755859375"},
    {0x1.fffffffffffffp-343, chars_format::fixed, 395,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001116"
        "1986242990965383971598462137498005377755891558801953458821857156773003942598961929815697218733720984488296"
        "5249972338768651218236242814552976090940303330364276966168661751862561606730292840042254555899935563574634"
        "36800095106069271095110474729969018796438273710691646556369960308074951171875"},
    {0x1.fffffffffffffp-342, chars_format::fixed, 394,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002232"
        "3972485981930767943196924274996010755511783117603906917643714313546007885197923859631394437467441968976593"
        "0499944677537302436472485629105952181880606660728553932337323503725123213460585680084509111799871127149268"
        "7360019021213854219022094945993803759287654742138329311273992061614990234375"},
    {0x1.fffffffffffffp-341, chars_format::fixed, 393,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004464"
        "7944971963861535886393848549992021511023566235207813835287428627092015770395847719262788874934883937953186"
        "0999889355074604872944971258211904363761213321457107864674647007450246426921171360169018223599742254298537"
        "472003804242770843804418989198760751857530948427665862254798412322998046875"},
    {0x1.fffffffffffffp-340, chars_format::fixed, 392,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000008929"
        "5889943927723071772787697099984043022047132470415627670574857254184031540791695438525577749869767875906372"
        "1999778710149209745889942516423808727522426642914215729349294014900492853842342720338036447199484508597074"
        "94400760848554168760883797839752150371506189685533172450959682464599609375"},
    {0x1.fffffffffffffp-339, chars_format::fixed, 391,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000017859"
        "1779887855446143545575394199968086044094264940831255341149714508368063081583390877051155499739535751812744"
        "3999557420298419491779885032847617455044853285828431458698588029800985707684685440676072894398969017194149"
        "8880152169710833752176759567950430074301237937106634490191936492919921875"},
    {0x1.fffffffffffffp-338, chars_format::fixed, 390,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000035718"
        "3559775710892287091150788399936172088188529881662510682299429016736126163166781754102310999479071503625488"
        "7999114840596838983559770065695234910089706571656862917397176059601971415369370881352145788797938034388299"
        "776030433942166750435351913590086014860247587421326898038387298583984375"},
    {0x1.fffffffffffffp-337, chars_format::fixed, 389,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000071436"
        "7119551421784574182301576799872344176377059763325021364598858033472252326333563508204621998958143007250977"
        "5998229681193677967119540131390469820179413143313725834794352119203942830738741762704291577595876068776599"
        "55206086788433350087070382718017202972049517484265379607677459716796875"},
    {0x1.fffffffffffffp-336, chars_format::fixed, 388,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000142873"
        "4239102843569148364603153599744688352754119526650042729197716066944504652667127016409243997916286014501955"
        "1996459362387355934239080262780939640358826286627451669588704238407885661477483525408583155191752137553199"
        "1041217357686670017414076543603440594409903496853075921535491943359375"},
    {0x1.fffffffffffffp-335, chars_format::fixed, 387,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000285746"
        "8478205687138296729206307199489376705508239053300085458395432133889009305334254032818487995832572029003910"
        "3992918724774711868478160525561879280717652573254903339177408476815771322954967050817166310383504275106398"
        "208243471537334003482815308720688118881980699370615184307098388671875"},
    {0x1.fffffffffffffp-334, chars_format::fixed, 386,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000571493"
        "6956411374276593458412614398978753411016478106600170916790864267778018610668508065636975991665144058007820"
        "7985837449549423736956321051123758561435305146509806678354816953631542645909934101634332620767008550212796"
        "41648694307466800696563061744137623776396139874123036861419677734375"},
    {0x1.fffffffffffffp-333, chars_format::fixed, 385,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001142987"
        "3912822748553186916825228797957506822032956213200341833581728535556037221337016131273951983330288116015641"
        "5971674899098847473912642102247517122870610293019613356709633907263085291819868203268665241534017100425592"
        "8329738861493360139312612348827524755279227974824607372283935546875"},
    {0x1.fffffffffffffp-332, chars_format::fixed, 384,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002285974"
        "7825645497106373833650457595915013644065912426400683667163457071112074442674032262547903966660576232031283"
        "1943349798197694947825284204495034245741220586039226713419267814526170583639736406537330483068034200851185"
        "665947772298672027862522469765504951055845594964921474456787109375"},
    {0x1.fffffffffffffp-331, chars_format::fixed, 383,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004571949"
        "5651290994212747667300915191830027288131824852801367334326914142224148885348064525095807933321152464062566"
        "3886699596395389895650568408990068491482441172078453426838535629052341167279472813074660966136068401702371"
        "33189554459734405572504493953100990211169118992984294891357421875"},
    {0x1.fffffffffffffp-330, chars_format::fixed, 382,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009143899"
        "1302581988425495334601830383660054576263649705602734668653828284448297770696129050191615866642304928125132"
        "7773399192790779791301136817980136982964882344156906853677071258104682334558945626149321932272136803404742"
        "6637910891946881114500898790620198042233823798596858978271484375"},
    {0x1.fffffffffffffp-329, chars_format::fixed, 381,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000018287798"
        "2605163976850990669203660767320109152527299411205469337307656568896595541392258100383231733284609856250265"
        "5546798385581559582602273635960273965929764688313813707354142516209364669117891252298643864544273606809485"
        "327582178389376222900179758124039608446764759719371795654296875"},
    {0x1.fffffffffffffp-328, chars_format::fixed, 380,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000036575596"
        "5210327953701981338407321534640218305054598822410938674615313137793191082784516200766463466569219712500531"
        "1093596771163119165204547271920547931859529376627627414708285032418729338235782504597287729088547213618970"
        "65516435677875244580035951624807921689352951943874359130859375"},
    {0x1.fffffffffffffp-327, chars_format::fixed, 379,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000073151193"
        "0420655907403962676814643069280436610109197644821877349230626275586382165569032401532926933138439425001062"
        "2187193542326238330409094543841095863719058753255254829416570064837458676471565009194575458177094427237941"
        "3103287135575048916007190324961584337870590388774871826171875"},
    {0x1.fffffffffffffp-326, chars_format::fixed, 378,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000146302386"
        "0841311814807925353629286138560873220218395289643754698461252551172764331138064803065853866276878850002124"
        "4374387084652476660818189087682191727438117506510509658833140129674917352943130018389150916354188854475882"
        "620657427115009783201438064992316867574118077754974365234375"},
    {0x1.fffffffffffffp-325, chars_format::fixed, 377,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000292604772"
        "1682623629615850707258572277121746440436790579287509396922505102345528662276129606131707732553757700004248"
        "8748774169304953321636378175364383454876235013021019317666280259349834705886260036778301832708377708951765"
        "24131485423001956640287612998463373514823615550994873046875"},
    {0x1.fffffffffffffp-324, chars_format::fixed, 376,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000585209544"
        "3365247259231701414517144554243492880873581158575018793845010204691057324552259212263415465107515400008497"
        "7497548338609906643272756350728766909752470026042038635332560518699669411772520073556603665416755417903530"
        "4826297084600391328057522599692674702964723110198974609375"},
    {0x1.fffffffffffffp-323, chars_format::fixed, 375,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001170419088"
        "6730494518463402829034289108486985761747162317150037587690020409382114649104518424526830930215030800016995"
        "4995096677219813286545512701457533819504940052084077270665121037399338823545040147113207330833510835807060"
        "965259416920078265611504519938534940592944622039794921875"},
    {0x1.fffffffffffffp-322, chars_format::fixed, 374,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002340838177"
        "3460989036926805658068578216973971523494324634300075175380040818764229298209036849053661860430061600033990"
        "9990193354439626573091025402915067639009880104168154541330242074798677647090080294226414661667021671614121"
        "93051883384015653122300903987706988118588924407958984375"},
    {0x1.fffffffffffffp-321, chars_format::fixed, 373,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004681676354"
        "6921978073853611316137156433947943046988649268600150350760081637528458596418073698107323720860123200067981"
        "9980386708879253146182050805830135278019760208336309082660484149597355294180160588452829323334043343228243"
        "8610376676803130624460180797541397623717784881591796875"},
    {0x1.fffffffffffffp-320, chars_format::fixed, 372,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009363352709"
        "3843956147707222632274312867895886093977298537200300701520163275056917192836147396214647441720246400135963"
        "9960773417758506292364101611660270556039520416672618165320968299194710588360321176905658646668086686456487"
        "722075335360626124892036159508279524743556976318359375"},
    {0x1.fffffffffffffp-319, chars_format::fixed, 371,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000018726705418"
        "7687912295414445264548625735791772187954597074400601403040326550113834385672294792429294883440492800271927"
        "9921546835517012584728203223320541112079040833345236330641936598389421176720642353811317293336173372912975"
        "44415067072125224978407231901655904948711395263671875"},
    {0x1.fffffffffffffp-318, chars_format::fixed, 370,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000037453410837"
        "5375824590828890529097251471583544375909194148801202806080653100227668771344589584858589766880985600543855"
        "9843093671034025169456406446641082224158081666690472661283873196778842353441284707622634586672346745825950"
        "8883013414425044995681446380331180989742279052734375"},
    {0x1.fffffffffffffp-317, chars_format::fixed, 369,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000074906821675"
        "0751649181657781058194502943167088751818388297602405612161306200455337542689179169717179533761971201087711"
        "9686187342068050338912812893282164448316163333380945322567746393557684706882569415245269173344693491651901"
        "776602682885008999136289276066236197948455810546875"},
    {0x1.fffffffffffffp-316, chars_format::fixed, 368,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000149813643350"
        "1503298363315562116389005886334177503636776595204811224322612400910675085378358339434359067523942402175423"
        "9372374684136100677825625786564328896632326666761890645135492787115369413765138830490538346689386983303803"
        "55320536577001799827257855213247239589691162109375"},
    {0x1.fffffffffffffp-315, chars_format::fixed, 367,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000299627286700"
        "3006596726631124232778011772668355007273553190409622448645224801821350170756716678868718135047884804350847"
        "8744749368272201355651251573128657793264653333523781290270985574230738827530277660981076693378773966607607"
        "1064107315400359965451571042649447917938232421875"},
    {0x1.fffffffffffffp-314, chars_format::fixed, 366,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000599254573400"
        "6013193453262248465556023545336710014547106380819244897290449603642700341513433357737436270095769608701695"
        "7489498736544402711302503146257315586529306667047562580541971148461477655060555321962153386757547933215214"
        "212821463080071993090314208529889583587646484375"},
    {0x1.fffffffffffffp-313, chars_format::fixed, 365,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001198509146801"
        "2026386906524496931112047090673420029094212761638489794580899207285400683026866715474872540191539217403391"
        "4978997473088805422605006292514631173058613334095125161083942296922955310121110643924306773515095866430428"
        "42564292616014398618062841705977916717529296875"},
    {0x1.fffffffffffffp-312, chars_format::fixed, 364,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002397018293602"
        "4052773813048993862224094181346840058188425523276979589161798414570801366053733430949745080383078434806782"
        "9957994946177610845210012585029262346117226668190250322167884593845910620242221287848613547030191732860856"
        "8512858523202879723612568341195583343505859375"},
    {0x1.fffffffffffffp-311, chars_format::fixed, 363,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004794036587204"
        "8105547626097987724448188362693680116376851046553959178323596829141602732107466861899490160766156869613565"
        "9915989892355221690420025170058524692234453336380500644335769187691821240484442575697227094060383465721713"
        "702571704640575944722513668239116668701171875"},
    {0x1.fffffffffffffp-310, chars_format::fixed, 362,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009588073174409"
        "6211095252195975448896376725387360232753702093107918356647193658283205464214933723798980321532313739227131"
        "9831979784710443380840050340117049384468906672761001288671538375383642480968885151394454188120766931443427"
        "40514340928115188944502733647823333740234375"},
    {0x1.fffffffffffffp-309, chars_format::fixed, 361,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000019176146348819"
        "2422190504391950897792753450774720465507404186215836713294387316566410928429867447597960643064627478454263"
        "9663959569420886761680100680234098768937813345522002577343076750767284961937770302788908376241533862886854"
        "8102868185623037788900546729564666748046875"},
    {0x1.fffffffffffffp-308, chars_format::fixed, 360,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000038352292697638"
        "4844381008783901795585506901549440931014808372431673426588774633132821856859734895195921286129254956908527"
        "9327919138841773523360201360468197537875626691044005154686153501534569923875540605577816752483067725773709"
        "620573637124607557780109345912933349609375"},
    {0x1.fffffffffffffp-307, chars_format::fixed, 359,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000076704585395276"
        "9688762017567803591171013803098881862029616744863346853177549266265643713719469790391842572258509913817055"
        "8655838277683547046720402720936395075751253382088010309372307003069139847751081211155633504966135451547419"
        "24114727424921511556021869182586669921875"},
    {0x1.fffffffffffffp-306, chars_format::fixed, 358,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000153409170790553"
        "9377524035135607182342027606197763724059233489726693706355098532531287427438939580783685144517019827634111"
        "7311676555367094093440805441872790151502506764176020618744614006138279695502162422311267009932270903094838"
        "4822945484984302311204373836517333984375"},
    {0x1.fffffffffffffp-305, chars_format::fixed, 357,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000306818341581107"
        "8755048070271214364684055212395527448118466979453387412710197065062574854877879161567370289034039655268223"
        "4623353110734188186881610883745580303005013528352041237489228012276559391004324844622534019864541806189676"
        "964589096996860462240874767303466796875"},
    {0x1.fffffffffffffp-304, chars_format::fixed, 356,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000613636683162215"
        "7510096140542428729368110424791054896236933958906774825420394130125149709755758323134740578068079310536446"
        "9246706221468376373763221767491160606010027056704082474978456024553118782008649689245068039729083612379353"
        "92917819399372092448174953460693359375"},
    {0x1.fffffffffffffp-303, chars_format::fixed, 355,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001227273366324431"
        "5020192281084857458736220849582109792473867917813549650840788260250299419511516646269481156136158621072893"
        "8493412442936752747526443534982321212020054113408164949956912049106237564017299378490136079458167224758707"
        "8583563879874418489634990692138671875"},
    {0x1.fffffffffffffp-302, chars_format::fixed, 354,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002454546732648863"
        "0040384562169714917472441699164219584947735835627099301681576520500598839023033292538962312272317242145787"
        "6986824885873505495052887069964642424040108226816329899913824098212475128034598756980272158916334449517415"
        "716712775974883697926998138427734375"},
    {0x1.fffffffffffffp-301, chars_format::fixed, 353,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004909093465297726"
        "0080769124339429834944883398328439169895471671254198603363153041001197678046066585077924624544634484291575"
        "3973649771747010990105774139929284848080216453632659799827648196424950256069197513960544317832668899034831"
        "43342555194976739585399627685546875"},
    {0x1.fffffffffffffp-300, chars_format::fixed, 352,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000009818186930595452"
        "0161538248678859669889766796656878339790943342508397206726306082002395356092133170155849249089268968583150"
        "7947299543494021980211548279858569696160432907265319599655296392849900512138395027921088635665337798069662"
        "8668511038995347917079925537109375"},
    {0x1.fffffffffffffp-299, chars_format::fixed, 351,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000019636373861190904"
        "0323076497357719339779533593313756679581886685016794413452612164004790712184266340311698498178537937166301"
        "5894599086988043960423096559717139392320865814530639199310592785699801024276790055842177271330675596139325"
        "733702207799069583415985107421875"},
    {0x1.fffffffffffffp-298, chars_format::fixed, 350,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000039272747722381808"
        "0646152994715438679559067186627513359163773370033588826905224328009581424368532680623396996357075874332603"
        "1789198173976087920846193119434278784641731629061278398621185571399602048553580111684354542661351192278651"
        "46740441559813916683197021484375"},
    {0x1.fffffffffffffp-297, chars_format::fixed, 349,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000078545495444763616"
        "1292305989430877359118134373255026718327546740067177653810448656019162848737065361246793992714151748665206"
        "3578396347952175841692386238868557569283463258122556797242371142799204097107160223368709085322702384557302"
        "9348088311962783336639404296875"},
    {0x1.fffffffffffffp-296, chars_format::fixed, 348,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000157090990889527232"
        "2584611978861754718236268746510053436655093480134355307620897312038325697474130722493587985428303497330412"
        "7156792695904351683384772477737115138566926516245113594484742285598408194214320446737418170645404769114605"
        "869617662392556667327880859375"},
    {0x1.fffffffffffffp-295, chars_format::fixed, 347,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000314181981779054464"
        "5169223957723509436472537493020106873310186960268710615241794624076651394948261444987175970856606994660825"
        "4313585391808703366769544955474230277133853032490227188969484571196816388428640893474836341290809538229211"
        "73923532478511333465576171875"},
    {0x1.fffffffffffffp-294, chars_format::fixed, 346,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000628363963558108929"
        "0338447915447018872945074986040213746620373920537421230483589248153302789896522889974351941713213989321650"
        "8627170783617406733539089910948460554267706064980454377938969142393632776857281786949672682581619076458423"
        "4784706495702266693115234375"},
    {0x1.fffffffffffffp-293, chars_format::fixed, 345,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000001256727927116217858"
        "0676895830894037745890149972080427493240747841074842460967178496306605579793045779948703883426427978643301"
        "7254341567234813467078179821896921108535412129960908755877938284787265553714563573899345365163238152916846"
        "956941299140453338623046875"},
    {0x1.fffffffffffffp-292, chars_format::fixed, 344,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000002513455854232435716"
        "1353791661788075491780299944160854986481495682149684921934356992613211159586091559897407766852855957286603"
        "4508683134469626934156359643793842217070824259921817511755876569574531107429127147798690730326476305833693"
        "91388259828090667724609375"},
    {0x1.fffffffffffffp-291, chars_format::fixed, 343,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000005026911708464871432"
        "2707583323576150983560599888321709972962991364299369843868713985226422319172183119794815533705711914573206"
        "9017366268939253868312719287587684434141648519843635023511753139149062214858254295597381460652952611667387"
        "8277651965618133544921875"},
    {0x1.fffffffffffffp-290, chars_format::fixed, 342,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000010053823416929742864"
        "5415166647152301967121199776643419945925982728598739687737427970452844638344366239589631067411423829146413"
        "8034732537878507736625438575175368868283297039687270047023506278298124429716508591194762921305905223334775"
        "655530393123626708984375"},
    {0x1.fffffffffffffp-289, chars_format::fixed, 341,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000020107646833859485729"
        "0830333294304603934242399553286839891851965457197479375474855940905689276688732479179262134822847658292827"
        "6069465075757015473250877150350737736566594079374540094047012556596248859433017182389525842611810446669551"
        "31106078624725341796875"},
    {0x1.fffffffffffffp-288, chars_format::fixed, 340,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000040215293667718971458"
        "1660666588609207868484799106573679783703930914394958750949711881811378553377464958358524269645695316585655"
        "2138930151514030946501754300701475473133188158749080188094025113192497718866034364779051685223620893339102"
        "6221215724945068359375"},
    {0x1.fffffffffffffp-287, chars_format::fixed, 339,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000080430587335437942916"
        "3321333177218415736969598213147359567407861828789917501899423763622757106754929916717048539291390633171310"
        "4277860303028061893003508601402950946266376317498160376188050226384995437732068729558103370447241786678205"
        "244243144989013671875"},
    {0x1.fffffffffffffp-286, chars_format::fixed, 338,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000160861174670875885832"
        "6642666354436831473939196426294719134815723657579835003798847527245514213509859833434097078582781266342620"
        "8555720606056123786007017202805901892532752634996320752376100452769990875464137459116206740894483573356410"
        "48848628997802734375"},
    {0x1.fffffffffffffp-285, chars_format::fixed, 337,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000321722349341751771665"
        "3285332708873662947878392852589438269631447315159670007597695054491028427019719666868194157165562532685241"
        "7111441212112247572014034405611803785065505269992641504752200905539981750928274918232413481788967146712820"
        "9769725799560546875"},
    {0x1.fffffffffffffp-284, chars_format::fixed, 336,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000643444698683503543330"
        "6570665417747325895756785705178876539262894630319340015195390108982056854039439333736388314331125065370483"
        "4222882424224495144028068811223607570131010539985283009504401811079963501856549836464826963577934293425641"
        "953945159912109375"},
    {0x1.fffffffffffffp-283, chars_format::fixed, 335,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000001286889397367007086661"
        "3141330835494651791513571410357753078525789260638680030390780217964113708078878667472776628662250130740966"
        "8445764848448990288056137622447215140262021079970566019008803622159927003713099672929653927155868586851283"
        "90789031982421875"},
    {0x1.fffffffffffffp-282, chars_format::fixed, 334,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000002573778794734014173322"
        "6282661670989303583027142820715506157051578521277360060781560435928227416157757334945553257324500261481933"
        "6891529696897980576112275244894430280524042159941132038017607244319854007426199345859307854311737173702567"
        "8157806396484375"},
    {0x1.fffffffffffffp-281, chars_format::fixed, 333,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000005147557589468028346645"
        "2565323341978607166054285641431012314103157042554720121563120871856454832315514669891106514649000522963867"
        "3783059393795961152224550489788860561048084319882264076035214488639708014852398691718615708623474347405135"
        "631561279296875"},
    {0x1.fffffffffffffp-280, chars_format::fixed, 332,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000010295115178936056693290"
        "5130646683957214332108571282862024628206314085109440243126241743712909664631029339782213029298001045927734"
        "7566118787591922304449100979577721122096168639764528152070428977279416029704797383437231417246948694810271"
        "26312255859375"},
    {0x1.fffffffffffffp-279, chars_format::fixed, 331,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000020590230357872113386581"
        "0261293367914428664217142565724049256412628170218880486252483487425819329262058679564426058596002091855469"
        "5132237575183844608898201959155442244192337279529056304140857954558832059409594766874462834493897389620542"
        "5262451171875"},
    {0x1.fffffffffffffp-278, chars_format::fixed, 330,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000041180460715744226773162"
        "0522586735828857328434285131448098512825256340437760972504966974851638658524117359128852117192004183710939"
        "0264475150367689217796403918310884488384674559058112608281715909117664118819189533748925668987794779241085"
        "052490234375"},
    {0x1.fffffffffffffp-277, chars_format::fixed, 329,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000082360921431488453546324"
        "1045173471657714656868570262896197025650512680875521945009933949703277317048234718257704234384008367421878"
        "0528950300735378435592807836621768976769349118116225216563431818235328237638379067497851337975589558482170"
        "10498046875"},
    {0x1.fffffffffffffp-276, chars_format::fixed, 328,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000164721842862976907092648"
        "2090346943315429313737140525792394051301025361751043890019867899406554634096469436515408468768016734843756"
        "1057900601470756871185615673243537953538698236232450433126863636470656475276758134995702675951179116964340"
        "2099609375"},
    {0x1.fffffffffffffp-275, chars_format::fixed, 327,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000329443685725953814185296"
        "4180693886630858627474281051584788102602050723502087780039735798813109268192938873030816937536033469687512"
        "2115801202941513742371231346487075907077396472464900866253727272941312950553516269991405351902358233928680"
        "419921875"},
    {0x1.fffffffffffffp-274, chars_format::fixed, 326,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000000658887371451907628370592"
        "8361387773261717254948562103169576205204101447004175560079471597626218536385877746061633875072066939375024"
        "4231602405883027484742462692974151814154792944929801732507454545882625901107032539982810703804716467857360"
        "83984375"},
    {0x1.fffffffffffffp-273, chars_format::fixed, 325,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000001317774742903815256741185"
        "6722775546523434509897124206339152410408202894008351120158943195252437072771755492123267750144133878750048"
        "8463204811766054969484925385948303628309585889859603465014909091765251802214065079965621407609432935714721"
        "6796875"},
    {0x1.fffffffffffffp-272, chars_format::fixed, 324,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000002635549485807630513482371"
        "3445551093046869019794248412678304820816405788016702240317886390504874145543510984246535500288267757500097"
        "6926409623532109938969850771896607256619171779719206930029818183530503604428130159931242815218865871429443"
        "359375"},
    {0x1.fffffffffffffp-271, chars_format::fixed, 323,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000005271098971615261026964742"
        "6891102186093738039588496825356609641632811576033404480635772781009748291087021968493071000576535515000195"
        "3852819247064219877939701543793214513238343559438413860059636367061007208856260319862485630437731742858886"
        "71875"},
    {0x1.fffffffffffffp-270, chars_format::fixed, 322,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000010542197943230522053929485"
        "3782204372187476079176993650713219283265623152066808961271545562019496582174043936986142001153071030000390"
        "7705638494128439755879403087586429026476687118876827720119272734122014417712520639724971260875463485717773"
        "4375"},
    {0x1.fffffffffffffp-269, chars_format::fixed, 321,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000021084395886461044107858970"
        "7564408744374952158353987301426438566531246304133617922543091124038993164348087873972284002306142060000781"
        "5411276988256879511758806175172858052953374237753655440238545468244028835425041279449942521750926971435546"
        "875"},
    {0x1.fffffffffffffp-268, chars_format::fixed, 320,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000042168791772922088215717941"
        "5128817488749904316707974602852877133062492608267235845086182248077986328696175747944568004612284120001563"
        "0822553976513759023517612350345716105906748475507310880477090936488057670850082558899885043501853942871093"
        "75"},
    {0x1.fffffffffffffp-267, chars_format::fixed, 319,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000084337583545844176431435883"
        "0257634977499808633415949205705754266124985216534471690172364496155972657392351495889136009224568240003126"
        "1645107953027518047035224700691432211813496951014621760954181872976115341700165117799770087003707885742187"
        "5"},
    {0x1.fffffffffffffp-266, chars_format::fixed, 318,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000168675167091688352862871766"
        "0515269954999617266831898411411508532249970433068943380344728992311945314784702991778272018449136480006252"
        "329021590605503609407044940138286442362699390202924352190836374595223068340033023559954017400741577148437"
        "5"},
    {0x1.fffffffffffffp-265, chars_format::fixed, 317,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000337350334183376705725743532"
        "1030539909999234533663796822823017064499940866137886760689457984623890629569405983556544036898272960012504"
        "65804318121100721881408988027657288472539878040584870438167274919044613668006604711990803480148315429687"
        "5"},
    {0x1.fffffffffffffp-264, chars_format::fixed, 316,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000000674700668366753411451487064"
        "2061079819998469067327593645646034128999881732275773521378915969247781259138811967113088073796545920025009"
        "31608636242201443762817976055314576945079756081169740876334549838089227336013209423981606960296630859375"},
    {0x1.fffffffffffffp-263, chars_format::fixed, 315,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000001349401336733506822902974128"
        "4122159639996938134655187291292068257999763464551547042757831938495562518277623934226176147593091840050018"
        "6321727248440288752563595211062915389015951216233948175266909967617845467202641884796321392059326171875"},
    {0x1.fffffffffffffp-262, chars_format::fixed, 314,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000002698802673467013645805948256"
        "8244319279993876269310374582584136515999526929103094085515663876991125036555247868452352295186183680100037"
        "264345449688057750512719042212583077803190243246789635053381993523569093440528376959264278411865234375"},
    {0x1.fffffffffffffp-261, chars_format::fixed, 313,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000005397605346934027291611896513"
        "6488638559987752538620749165168273031999053858206188171031327753982250073110495736904704590372367360200074"
        "52869089937611550102543808442516615560638048649357927010676398704713818688105675391852855682373046875"},
    {0x1.fffffffffffffp-260, chars_format::fixed, 312,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000010795210693868054583223793027"
        "2977277119975505077241498330336546063998107716412376342062655507964500146220991473809409180744734720400149"
        "0573817987522310020508761688503323112127609729871585402135279740942763737621135078370571136474609375"},
    {0x1.fffffffffffffp-259, chars_format::fixed, 311,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000021590421387736109166447586054"
        "5954554239951010154482996660673092127996215432824752684125311015929000292441982947618818361489469440800298"
        "114763597504462004101752337700664622425521945974317080427055948188552747524227015674114227294921875"},
    {0x1.fffffffffffffp-258, chars_format::fixed, 310,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000043180842775472218332895172109"
        "1909108479902020308965993321346184255992430865649505368250622031858000584883965895237636722978938881600596"
        "22952719500892400820350467540132924485104389194863416085411189637710549504845403134822845458984375"},
    {0x1.fffffffffffffp-257, chars_format::fixed, 309,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000086361685550944436665790344218"
        "3818216959804040617931986642692368511984861731299010736501244063716001169767931790475273445957877763201192"
        "4590543900178480164070093508026584897020877838972683217082237927542109900969080626964569091796875"},
    {0x1.fffffffffffffp-256, chars_format::fixed, 308,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000172723371101888873331580688436"
        "7636433919608081235863973285384737023969723462598021473002488127432002339535863580950546891915755526402384"
        "918108780035696032814018701605316979404175567794536643416447585508421980193816125392913818359375"},
    {0x1.fffffffffffffp-255, chars_format::fixed, 307,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000345446742203777746663161376873"
        "5272867839216162471727946570769474047939446925196042946004976254864004679071727161901093783831511052804769"
        "83621756007139206562803740321063395880835113558907328683289517101684396038763225078582763671875"},
    {0x1.fffffffffffffp-254, chars_format::fixed, 306,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000000690893484407555493326322753747"
        "0545735678432324943455893141538948095878893850392085892009952509728009358143454323802187567663022105609539"
        "6724351201427841312560748064212679176167022711781465736657903420336879207752645015716552734375"},
    {0x1.fffffffffffffp-253, chars_format::fixed, 305,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000001381786968815110986652645507494"
        "1091471356864649886911786283077896191757787700784171784019905019456018716286908647604375135326044211219079"
        "344870240285568262512149612842535835233404542356293147331580684067375841550529003143310546875"},
    {0x1.fffffffffffffp-252, chars_format::fixed, 304,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000002763573937630221973305291014988"
        "2182942713729299773823572566155792383515575401568343568039810038912037432573817295208750270652088422438158"
        "68974048057113652502429922568507167046680908471258629466316136813475168310105800628662109375"},
    {0x1.fffffffffffffp-251, chars_format::fixed, 303,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000005527147875260443946610582029976"
        "4365885427458599547647145132311584767031150803136687136079620077824074865147634590417500541304176844876317"
        "3794809611422730500485984513701433409336181694251725893263227362695033662021160125732421875"},
    {0x1.fffffffffffffp-250, chars_format::fixed, 302,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000011054295750520887893221164059952"
        "8731770854917199095294290264623169534062301606273374272159240155648149730295269180835001082608353689752634"
        "758961922284546100097196902740286681867236338850345178652645472539006732404232025146484375"},
    {0x1.fffffffffffffp-249, chars_format::fixed, 301,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000022108591501041775786442328119905"
        "7463541709834398190588580529246339068124603212546748544318480311296299460590538361670002165216707379505269"
        "51792384456909220019439380548057336373447267770069035730529094507801346480846405029296875"},
    {0x1.fffffffffffffp-248, chars_format::fixed, 300,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000044217183002083551572884656239811"
        "4927083419668796381177161058492678136249206425093497088636960622592598921181076723340004330433414759010539"
        "0358476891381844003887876109611467274689453554013807146105818901560269296169281005859375"},
    {0x1.fffffffffffffp-247, chars_format::fixed, 299,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000088434366004167103145769312479622"
        "9854166839337592762354322116985356272498412850186994177273921245185197842362153446680008660866829518021078"
        "071695378276368800777575221922293454937890710802761429221163780312053859233856201171875"},
    {0x1.fffffffffffffp-246, chars_format::fixed, 298,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000176868732008334206291538624959245"
        "9708333678675185524708644233970712544996825700373988354547842490370395684724306893360017321733659036042156"
        "14339075655273760155515044384458690987578142160552285844232756062410771846771240234375"},
    {0x1.fffffffffffffp-245, chars_format::fixed, 297,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000353737464016668412583077249918491"
        "9416667357350371049417288467941425089993651400747976709095684980740791369448613786720034643467318072084312"
        "2867815131054752031103008876891738197515628432110457168846551212482154369354248046875"},
    {0x1.fffffffffffffp-244, chars_format::fixed, 296,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000000707474928033336825166154499836983"
        "8833334714700742098834576935882850179987302801495953418191369961481582738897227573440069286934636144168624"
        "573563026210950406220601775378347639503125686422091433769310242496430873870849609375"},
    {0x1.fffffffffffffp-243, chars_format::fixed, 295,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000001414949856066673650332308999673967"
        "7666669429401484197669153871765700359974605602991906836382739922963165477794455146880138573869272288337249"
        "14712605242190081244120355075669527900625137284418286753862048499286174774169921875"},
    {0x1.fffffffffffffp-242, chars_format::fixed, 294,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000002829899712133347300664617999347935"
        "5333338858802968395338307743531400719949211205983813672765479845926330955588910293760277147738544576674498"
        "2942521048438016248824071015133905580125027456883657350772409699857234954833984375"},
    {0x1.fffffffffffffp-241, chars_format::fixed, 293,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000005659799424266694601329235998695871"
        "0666677717605936790676615487062801439898422411967627345530959691852661911177820587520554295477089153348996"
        "588504209687603249764814203026781116025005491376731470154481939971446990966796875"},
    {0x1.fffffffffffffp-240, chars_format::fixed, 292,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000011319598848533389202658471997391742"
        "1333355435211873581353230974125602879796844823935254691061919383705323822355641175041108590954178306697993"
        "17700841937520649952962840605356223205001098275346294030896387994289398193359375"},
    {0x1.fffffffffffffp-239, chars_format::fixed, 291,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000022639197697066778405316943994783484"
        "2666710870423747162706461948251205759593689647870509382123838767410647644711282350082217181908356613395986"
        "3540168387504129990592568121071244641000219655069258806179277598857879638671875"},
    {0x1.fffffffffffffp-238, chars_format::fixed, 290,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000045278395394133556810633887989566968"
        "5333421740847494325412923896502411519187379295741018764247677534821295289422564700164434363816713226791972"
        "708033677500825998118513624214248928200043931013851761235855519771575927734375"},
    {0x1.fffffffffffffp-237, chars_format::fixed, 289,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000090556790788267113621267775979133937"
        "0666843481694988650825847793004823038374758591482037528495355069642590578845129400328868727633426453583945"
        "41606735500165199623702724842849785640008786202770352247171103954315185546875"},
    {0x1.fffffffffffffp-236, chars_format::fixed, 288,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000181113581576534227242535551958267874"
        "1333686963389977301651695586009646076749517182964075056990710139285181157690258800657737455266852907167890"
        "8321347100033039924740544968569957128001757240554070449434220790863037109375"},
    {0x1.fffffffffffffp-235, chars_format::fixed, 287,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000362227163153068454485071103916535748"
        "2667373926779954603303391172019292153499034365928150113981420278570362315380517601315474910533705814335781"
        "664269420006607984948108993713991425600351448110814089886844158172607421875"},
    {0x1.fffffffffffffp-234, chars_format::fixed, 286,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000000724454326306136908970142207833071496"
        "5334747853559909206606782344038584306998068731856300227962840557140724630761035202630949821067411628671563"
        "32853884001321596989621798742798285120070289622162817977368831634521484375"},
    {0x1.fffffffffffffp-233, chars_format::fixed, 285,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000001448908652612273817940284415666142993"
        "0669495707119818413213564688077168613996137463712600455925681114281449261522070405261899642134823257343126"
        "6570776800264319397924359748559657024014057924432563595473766326904296875"},
    {0x1.fffffffffffffp-232, chars_format::fixed, 284,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000002897817305224547635880568831332285986"
        "1338991414239636826427129376154337227992274927425200911851362228562898523044140810523799284269646514686253"
        "314155360052863879584871949711931404802811584886512719094753265380859375"},
    {0x1.fffffffffffffp-231, chars_format::fixed, 283,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000005795634610449095271761137662664571972"
        "2677982828479273652854258752308674455984549854850401823702724457125797046088281621047598568539293029372506"
        "62831072010572775916974389942386280960562316977302543818950653076171875"},
    {0x1.fffffffffffffp-230, chars_format::fixed, 282,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000011591269220898190543522275325329143944"
        "5355965656958547305708517504617348911969099709700803647405448914251594092176563242095197137078586058745013"
        "2566214402114555183394877988477256192112463395460508763790130615234375"},
    {0x1.fffffffffffffp-229, chars_format::fixed, 281,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000023182538441796381087044550650658287889"
        "0711931313917094611417035009234697823938199419401607294810897828503188184353126484190394274157172117490026"
        "513242880422911036678975597695451238422492679092101752758026123046875"},
    {0x1.fffffffffffffp-228, chars_format::fixed, 280,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000046365076883592762174089101301316575778"
        "1423862627834189222834070018469395647876398838803214589621795657006376368706252968380788548314344234980053"
        "02648576084582207335795119539090247684498535818420350551605224609375"},
    {0x1.fffffffffffffp-227, chars_format::fixed, 279,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000092730153767185524348178202602633151556"
        "2847725255668378445668140036938791295752797677606429179243591314012752737412505936761577096628688469960106"
        "0529715216916441467159023907818049536899707163684070110321044921875"},
    {0x1.fffffffffffffp-226, chars_format::fixed, 278,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000185460307534371048696356405205266303112"
        "5695450511336756891336280073877582591505595355212858358487182628025505474825011873523154193257376939920212"
        "105943043383288293431804781563609907379941432736814022064208984375"},
    {0x1.fffffffffffffp-225, chars_format::fixed, 277,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000370920615068742097392712810410532606225"
        "1390901022673513782672560147755165183011190710425716716974365256051010949650023747046308386514753879840424"
        "21188608676657658686360956312721981475988286547362804412841796875"},
    {0x1.fffffffffffffp-224, chars_format::fixed, 276,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000000741841230137484194785425620821065212450"
        "2781802045347027565345120295510330366022381420851433433948730512102021899300047494092616773029507759680848"
        "4237721735331531737272191262544396295197657309472560882568359375"},
    {0x1.fffffffffffffp-223, chars_format::fixed, 275,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000001483682460274968389570851241642130424900"
        "5563604090694055130690240591020660732044762841702866867897461024204043798600094988185233546059015519361696"
        "847544347066306347454438252508879259039531461894512176513671875"},
    {0x1.fffffffffffffp-222, chars_format::fixed, 274,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000002967364920549936779141702483284260849801"
        "1127208181388110261380481182041321464089525683405733735794922048408087597200189976370467092118031038723393"
        "69508869413261269490887650501775851807906292378902435302734375"},
    {0x1.fffffffffffffp-221, chars_format::fixed, 273,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000005934729841099873558283404966568521699602"
        "2254416362776220522760962364082642928179051366811467471589844096816175194400379952740934184236062077446787"
        "3901773882652253898177530100355170361581258475780487060546875"},
    {0x1.fffffffffffffp-220, chars_format::fixed, 272,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000011869459682199747116566809933137043399204"
        "4508832725552441045521924728165285856358102733622934943179688193632350388800759905481868368472124154893574"
        "780354776530450779635506020071034072316251695156097412109375"},
    {0x1.fffffffffffffp-219, chars_format::fixed, 271,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000023738919364399494233133619866274086798408"
        "9017665451104882091043849456330571712716205467245869886359376387264700777601519810963736736944248309787149"
        "56070955306090155927101204014206814463250339031219482421875"},
    {0x1.fffffffffffffp-218, chars_format::fixed, 270,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000047477838728798988466267239732548173596817"
        "8035330902209764182087698912661143425432410934491739772718752774529401555203039621927473473888496619574299"
        "1214191061218031185420240802841362892650067806243896484375"},
    {0x1.fffffffffffffp-217, chars_format::fixed, 269,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000094955677457597976932534479465096347193635"
        "6070661804419528364175397825322286850864821868983479545437505549058803110406079243854946947776993239148598"
        "242838212243606237084048160568272578530013561248779296875"},
    {0x1.fffffffffffffp-216, chars_format::fixed, 268,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000189911354915195953865068958930192694387271"
        "2141323608839056728350795650644573701729643737966959090875011098117606220812158487709893895553986478297196"
        "48567642448721247416809632113654515706002712249755859375"},
    {0x1.fffffffffffffp-215, chars_format::fixed, 267,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000379822709830391907730137917860385388774542"
        "4282647217678113456701591301289147403459287475933918181750022196235212441624316975419787791107972956594392"
        "9713528489744249483361926422730903141200542449951171875"},
    {0x1.fffffffffffffp-214, chars_format::fixed, 266,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000000759645419660783815460275835720770777549084"
        "8565294435356226913403182602578294806918574951867836363500044392470424883248633950839575582215945913188785"
        "942705697948849896672385284546180628240108489990234375"},
    {0x1.fffffffffffffp-213, chars_format::fixed, 265,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000001519290839321567630920551671441541555098169"
        "7130588870712453826806365205156589613837149903735672727000088784940849766497267901679151164431891826377571"
        "88541139589769979334477056909236125648021697998046875"},
    {0x1.fffffffffffffp-212, chars_format::fixed, 264,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000003038581678643135261841103342883083110196339"
        "4261177741424907653612730410313179227674299807471345454000177569881699532994535803358302328863783652755143"
        "7708227917953995866895411381847225129604339599609375"},
    {0x1.fffffffffffffp-211, chars_format::fixed, 263,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000006077163357286270523682206685766166220392678"
        "8522355482849815307225460820626358455348599614942690908000355139763399065989071606716604657727567305510287"
        "541645583590799173379082276369445025920867919921875"},
    {0x1.fffffffffffffp-210, chars_format::fixed, 262,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000012154326714572541047364413371532332440785357"
        "7044710965699630614450921641252716910697199229885381816000710279526798131978143213433209315455134611020575"
        "08329116718159834675816455273889005184173583984375"},
    {0x1.fffffffffffffp-209, chars_format::fixed, 261,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000024308653429145082094728826743064664881570715"
        "4089421931399261228901843282505433821394398459770763632001420559053596263956286426866418630910269222041150"
        "1665823343631966935163291054777801036834716796875"},
    {0x1.fffffffffffffp-208, chars_format::fixed, 260,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000048617306858290164189457653486129329763141430"
        "8178843862798522457803686565010867642788796919541527264002841118107192527912572853732837261820538444082300"
        "333164668726393387032658210955560207366943359375"},
    {0x1.fffffffffffffp-207, chars_format::fixed, 259,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000097234613716580328378915306972258659526282861"
        "6357687725597044915607373130021735285577593839083054528005682236214385055825145707465674523641076888164600"
        "66632933745278677406531642191112041473388671875"},
    {0x1.fffffffffffffp-206, chars_format::fixed, 258,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000194469227433160656757830613944517319052565723"
        "2715375451194089831214746260043470571155187678166109056011364472428770111650291414931349047282153776329201"
        "3326586749055735481306328438222408294677734375"},
    {0x1.fffffffffffffp-205, chars_format::fixed, 257,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000388938454866321313515661227889034638105131446"
        "5430750902388179662429492520086941142310375356332218112022728944857540223300582829862698094564307552658402"
        "665317349811147096261265687644481658935546875"},
    {0x1.fffffffffffffp-204, chars_format::fixed, 256,
        "0."
        "0000000000000000000000000000000000000000000000000000000000000777876909732642627031322455778069276210262893"
        "0861501804776359324858985040173882284620750712664436224045457889715080446601165659725396189128615105316805"
        "33063469962229419252253137528896331787109375"},
    {0x1.fffffffffffffp-203, chars_format::fixed, 255,
        "0."
        "0000000000000000000000000000000000000000000000000000000000001555753819465285254062644911556138552420525786"
        "1723003609552718649717970080347764569241501425328872448090915779430160893202331319450792378257230210633610"
        "6612693992445883850450627505779266357421875"},
    {0x1.fffffffffffffp-202, chars_format::fixed, 254,
        "0."
        "0000000000000000000000000000000000000000000000000000000000003111507638930570508125289823112277104841051572"
        "3446007219105437299435940160695529138483002850657744896181831558860321786404662638901584756514460421267221"
        "322538798489176770090125501155853271484375"},
    {0x1.fffffffffffffp-201, chars_format::fixed, 253,
        "0."
        "0000000000000000000000000000000000000000000000000000000000006223015277861141016250579646224554209682103144"
        "6892014438210874598871880321391058276966005701315489792363663117720643572809325277803169513028920842534442"
        "64507759697835354018025100231170654296875"},
    {0x1.fffffffffffffp-200, chars_format::fixed, 252,
        "0."
        "0000000000000000000000000000000000000000000000000000000000012446030555722282032501159292449108419364206289"
        "3784028876421749197743760642782116553932011402630979584727326235441287145618650555606339026057841685068885"
        "2901551939567070803605020046234130859375"},
    {0x1.fffffffffffffp-199, chars_format::fixed, 251,
        "0."
        "0000000000000000000000000000000000000000000000000000000000024892061111444564065002318584898216838728412578"
        "7568057752843498395487521285564233107864022805261959169454652470882574291237301111212678052115683370137770"
        "580310387913414160721004009246826171875"},
    {0x1.fffffffffffffp-198, chars_format::fixed, 250,
        "0."
        "0000000000000000000000000000000000000000000000000000000000049784122222889128130004637169796433677456825157"
        "5136115505686996790975042571128466215728045610523918338909304941765148582474602222425356104231366740275541"
        "16062077582682832144200801849365234375"},
    {0x1.fffffffffffffp-197, chars_format::fixed, 249,
        "0."
        "0000000000000000000000000000000000000000000000000000000000099568244445778256260009274339592867354913650315"
        "0272231011373993581950085142256932431456091221047836677818609883530297164949204444850712208462733480551082"
        "3212415516536566428840160369873046875"},
    {0x1.fffffffffffffp-196, chars_format::fixed, 248,
        "0."
        "0000000000000000000000000000000000000000000000000000000000199136488891556512520018548679185734709827300630"
        "0544462022747987163900170284513864862912182442095673355637219767060594329898408889701424416925466961102164"
        "642483103307313285768032073974609375"},
    {0x1.fffffffffffffp-195, chars_format::fixed, 247,
        "0."
        "0000000000000000000000000000000000000000000000000000000000398272977783113025040037097358371469419654601260"
        "1088924045495974327800340569027729725824364884191346711274439534121188659796817779402848833850933922204329"
        "28496620661462657153606414794921875"},
    {0x1.fffffffffffffp-194, chars_format::fixed, 246,
        "0."
        "0000000000000000000000000000000000000000000000000000000000796545955566226050080074194716742938839309202520"
        "2177848090991948655600681138055459451648729768382693422548879068242377319593635558805697667701867844408658"
        "5699324132292531430721282958984375"},
    {0x1.fffffffffffffp-193, chars_format::fixed, 245,
        "0."
        "0000000000000000000000000000000000000000000000000000000001593091911132452100160148389433485877678618405040"
        "4355696181983897311201362276110918903297459536765386845097758136484754639187271117611395335403735688817317"
        "139864826458506286144256591796875"},
    {0x1.fffffffffffffp-192, chars_format::fixed, 244,
        "0."
        "0000000000000000000000000000000000000000000000000000000003186183822264904200320296778866971755357236810080"
        "8711392363967794622402724552221837806594919073530773690195516272969509278374542235222790670807471377634634"
        "27972965291701257228851318359375"},
    {0x1.fffffffffffffp-191, chars_format::fixed, 243,
        "0."
        "0000000000000000000000000000000000000000000000000000000006372367644529808400640593557733943510714473620161"
        "7422784727935589244805449104443675613189838147061547380391032545939018556749084470445581341614942755269268"
        "5594593058340251445770263671875"},
    {0x1.fffffffffffffp-190, chars_format::fixed, 242,
        "0."
        "0000000000000000000000000000000000000000000000000000000012744735289059616801281187115467887021428947240323"
        "4845569455871178489610898208887351226379676294123094760782065091878037113498168940891162683229885510538537"
        "118918611668050289154052734375"},
    {0x1.fffffffffffffp-189, chars_format::fixed, 241,
        "0."
        "0000000000000000000000000000000000000000000000000000000025489470578119233602562374230935774042857894480646"
        "9691138911742356979221796417774702452759352588246189521564130183756074226996337881782325366459771021077074"
        "23783722333610057830810546875"},
    {0x1.fffffffffffffp-188, chars_format::fixed, 240,
        "0."
        "0000000000000000000000000000000000000000000000000000000050978941156238467205124748461871548085715788961293"
        "9382277823484713958443592835549404905518705176492379043128260367512148453992675763564650732919542042154148"
        "4756744466722011566162109375"},
    {0x1.fffffffffffffp-187, chars_format::fixed, 239,
        "0."
        "0000000000000000000000000000000000000000000000000000000101957882312476934410249496923743096171431577922587"
        "8764555646969427916887185671098809811037410352984758086256520735024296907985351527129301465839084084308296"
        "951348893344402313232421875"},
    {0x1.fffffffffffffp-186, chars_format::fixed, 238,
        "0."
        "0000000000000000000000000000000000000000000000000000000203915764624953868820498993847486192342863155845175"
        "7529111293938855833774371342197619622074820705969516172513041470048593815970703054258602931678168168616593"
        "90269778668880462646484375"},
    {0x1.fffffffffffffp-185, chars_format::fixed, 237,
        "0."
        "0000000000000000000000000000000000000000000000000000000407831529249907737640997987694972384685726311690351"
        "5058222587877711667548742684395239244149641411939032345026082940097187631941406108517205863356336337233187"
        "8053955733776092529296875"},
    {0x1.fffffffffffffp-184, chars_format::fixed, 236,
        "0."
        "0000000000000000000000000000000000000000000000000000000815663058499815475281995975389944769371452623380703"
        "0116445175755423335097485368790478488299282823878064690052165880194375263882812217034411726712672674466375"
        "610791146755218505859375"},
    {0x1.fffffffffffffp-183, chars_format::fixed, 235,
        "0."
        "0000000000000000000000000000000000000000000000000000001631326116999630950563991950779889538742905246761406"
        "0232890351510846670194970737580956976598565647756129380104331760388750527765624434068823453425345348932751"
        "22158229351043701171875"},
    {0x1.fffffffffffffp-182, chars_format::fixed, 234,
        "0."
        "0000000000000000000000000000000000000000000000000000003262652233999261901127983901559779077485810493522812"
        "0465780703021693340389941475161913953197131295512258760208663520777501055531248868137646906850690697865502"
        "4431645870208740234375"},
    {0x1.fffffffffffffp-181, chars_format::fixed, 233,
        "0."
        "0000000000000000000000000000000000000000000000000000006525304467998523802255967803119558154971620987045624"
        "0931561406043386680779882950323827906394262591024517520417327041555002111062497736275293813701381395731004"
        "886329174041748046875"},
    {0x1.fffffffffffffp-180, chars_format::fixed, 232,
        "0."
        "0000000000000000000000000000000000000000000000000000013050608935997047604511935606239116309943241974091248"
        "1863122812086773361559765900647655812788525182049035040834654083110004222124995472550587627402762791462009"
        "77265834808349609375"},
    {0x1.fffffffffffffp-179, chars_format::fixed, 231,
        "0."
        "0000000000000000000000000000000000000000000000000000026101217871994095209023871212478232619886483948182496"
        "3726245624173546723119531801295311625577050364098070081669308166220008444249990945101175254805525582924019"
        "5453166961669921875"},
    {0x1.fffffffffffffp-178, chars_format::fixed, 230,
        "0."
        "0000000000000000000000000000000000000000000000000000052202435743988190418047742424956465239772967896364992"
        "7452491248347093446239063602590623251154100728196140163338616332440016888499981890202350509611051165848039"
        "090633392333984375"},
    {0x1.fffffffffffffp-177, chars_format::fixed, 229,
        "0."
        "0000000000000000000000000000000000000000000000000000104404871487976380836095484849912930479545935792729985"
        "4904982496694186892478127205181246502308201456392280326677232664880033776999963780404701019222102331696078"
        "18126678466796875"},
    {0x1.fffffffffffffp-176, chars_format::fixed, 228,
        "0."
        "0000000000000000000000000000000000000000000000000000208809742975952761672190969699825860959091871585459970"
        "9809964993388373784956254410362493004616402912784560653354465329760067553999927560809402038444204663392156"
        "3625335693359375"},
    {0x1.fffffffffffffp-175, chars_format::fixed, 227,
        "0."
        "0000000000000000000000000000000000000000000000000000417619485951905523344381939399651721918183743170919941"
        "9619929986776747569912508820724986009232805825569121306708930659520135107999855121618804076888409326784312"
        "725067138671875"},
    {0x1.fffffffffffffp-174, chars_format::fixed, 226,
        "0."
        "0000000000000000000000000000000000000000000000000000835238971903811046688763878799303443836367486341839883"
        "9239859973553495139825017641449972018465611651138242613417861319040270215999710243237608153776818653568625"
        "45013427734375"},
    {0x1.fffffffffffffp-173, chars_format::fixed, 225,
        "0."
        "0000000000000000000000000000000000000000000000000001670477943807622093377527757598606887672734972683679767"
        "8479719947106990279650035282899944036931223302276485226835722638080540431999420486475216307553637307137250"
        "9002685546875"},
    {0x1.fffffffffffffp-172, chars_format::fixed, 224,
        "0."
        "0000000000000000000000000000000000000000000000000003340955887615244186755055515197213775345469945367359535"
        "6959439894213980559300070565799888073862446604552970453671445276161080863998840972950432615107274614274501"
        "800537109375"},
    {0x1.fffffffffffffp-171, chars_format::fixed, 223,
        "0."
        "0000000000000000000000000000000000000000000000000006681911775230488373510111030394427550690939890734719071"
        "3918879788427961118600141131599776147724893209105940907342890552322161727997681945900865230214549228549003"
        "60107421875"},
    {0x1.fffffffffffffp-170, chars_format::fixed, 222,
        "0."
        "0000000000000000000000000000000000000000000000000013363823550460976747020222060788855101381879781469438142"
        "7837759576855922237200282263199552295449786418211881814685781104644323455995363891801730460429098457098007"
        "2021484375"},
    {0x1.fffffffffffffp-169, chars_format::fixed, 221,
        "0."
        "0000000000000000000000000000000000000000000000000026727647100921953494040444121577710202763759562938876285"
        "5675519153711844474400564526399104590899572836423763629371562209288646911990727783603460920858196914196014"
        "404296875"},
    {0x1.fffffffffffffp-168, chars_format::fixed, 220,
        "0."
        "0000000000000000000000000000000000000000000000000053455294201843906988080888243155420405527519125877752571"
        "1351038307423688948801129052798209181799145672847527258743124418577293823981455567206921841716393828392028"
        "80859375"},
    {0x1.fffffffffffffp-167, chars_format::fixed, 219,
        "0."
        "0000000000000000000000000000000000000000000000000106910588403687813976161776486310840811055038251755505142"
        "2702076614847377897602258105596418363598291345695054517486248837154587647962911134413843683432787656784057"
        "6171875"},
    {0x1.fffffffffffffp-166, chars_format::fixed, 218,
        "0."
        "0000000000000000000000000000000000000000000000000213821176807375627952323552972621681622110076503511010284"
        "5404153229694755795204516211192836727196582691390109034972497674309175295925822268827687366865575313568115"
        "234375"},
    {0x1.fffffffffffffp-165, chars_format::fixed, 217,
        "0."
        "0000000000000000000000000000000000000000000000000427642353614751255904647105945243363244220153007022020569"
        "0808306459389511590409032422385673454393165382780218069944995348618350591851644537655374733731150627136230"
        "46875"},
    {0x1.fffffffffffffp-164, chars_format::fixed, 216,
        "0."
        "0000000000000000000000000000000000000000000000000855284707229502511809294211890486726488440306014044041138"
        "1616612918779023180818064844771346908786330765560436139889990697236701183703289075310749467462301254272460"
        "9375"},
    {0x1.fffffffffffffp-163, chars_format::fixed, 215,
        "0."
        "0000000000000000000000000000000000000000000000001710569414459005023618588423780973452976880612028088082276"
        "3233225837558046361636129689542693817572661531120872279779981394473402367406578150621498934924602508544921"
        "875"},
    {0x1.fffffffffffffp-162, chars_format::fixed, 214,
        "0."
        "0000000000000000000000000000000000000000000000003421138828918010047237176847561946905953761224056176164552"
        "6466451675116092723272259379085387635145323062241744559559962788946804734813156301242997869849205017089843"
        "75"},
    {0x1.fffffffffffffp-161, chars_format::fixed, 213,
        "0."
        "0000000000000000000000000000000000000000000000006842277657836020094474353695123893811907522448112352329105"
        "2932903350232185446544518758170775270290646124483489119119925577893609469626312602485995739698410034179687"
        "5"},
    {0x1.fffffffffffffp-160, chars_format::fixed, 212,
        "0."
        "0000000000000000000000000000000000000000000000013684555315672040188948707390247787623815044896224704658210"
        "586580670046437089308903751634155054058129224896697823823985115578721893925262520497199147939682006835937"
        "5"},
    {0x1.fffffffffffffp-159, chars_format::fixed, 211,
        "0."
        "0000000000000000000000000000000000000000000000027369110631344080377897414780495575247630089792449409316421"
        "17316134009287417861780750326831010811625844979339564764797023115744378785052504099439829587936401367187"
        "5"},
    {0x1.fffffffffffffp-158, chars_format::fixed, 210,
        "0."
        "0000000000000000000000000000000000000000000000054738221262688160755794829560991150495260179584898818632842"
        "34632268018574835723561500653662021623251689958679129529594046231488757570105008198879659175872802734375"},
    {0x1.fffffffffffffp-157, chars_format::fixed, 209,
        "0."
        "0000000000000000000000000000000000000000000000109476442525376321511589659121982300990520359169797637265684"
        "6926453603714967144712300130732404324650337991735825905918809246297751514021001639775931835174560546875"},
    {0x1.fffffffffffffp-156, chars_format::fixed, 208,
        "0."
        "0000000000000000000000000000000000000000000000218952885050752643023179318243964601981040718339595274531369"
        "385290720742993428942460026146480864930067598347165181183761849259550302804200327955186367034912109375"},
    {0x1.fffffffffffffp-155, chars_format::fixed, 207,
        "0."
        "0000000000000000000000000000000000000000000000437905770101505286046358636487929203962081436679190549062738"
        "77058144148598685788492005229296172986013519669433036236752369851910060560840065591037273406982421875"},
    {0x1.fffffffffffffp-154, chars_format::fixed, 206,
        "0."
        "0000000000000000000000000000000000000000000000875811540203010572092717272975858407924162873358381098125477"
        "5411628829719737157698401045859234597202703933886607247350473970382012112168013118207454681396484375"},
    {0x1.fffffffffffffp-153, chars_format::fixed, 205,
        "0."
        "0000000000000000000000000000000000000000000001751623080406021144185434545951716815848325746716762196250955"
        "082325765943947431539680209171846919440540786777321449470094794076402422433602623641490936279296875"},
    {0x1.fffffffffffffp-152, chars_format::fixed, 204,
        "0."
        "0000000000000000000000000000000000000000000003503246160812042288370869091903433631696651493433524392501910"
        "16465153188789486307936041834369383888108157355464289894018958815280484486720524728298187255859375"},
    {0x1.fffffffffffffp-151, chars_format::fixed, 203,
        "0."
        "0000000000000000000000000000000000000000000007006492321624084576741738183806867263393302986867048785003820"
        "3293030637757897261587208366873876777621631471092857978803791763056096897344104945659637451171875"},
    {0x1.fffffffffffffp-150, chars_format::fixed, 202,
        "0."
        "0000000000000000000000000000000000000000000014012984643248169153483476367613734526786605973734097570007640"
        "658606127551579452317441673374775355524326294218571595760758352611219379468820989131927490234375"},
    {0x1.fffffffffffffp-149, chars_format::fixed, 201,
        "0."
        "0000000000000000000000000000000000000000000028025969286496338306966952735227469053573211947468195140015281"
        "31721225510315890463488334674955071104865258843714319152151670522243875893764197826385498046875"},
    {0x1.fffffffffffffp-148, chars_format::fixed, 200,
        "0."
        "0000000000000000000000000000000000000000000056051938572992676613933905470454938107146423894936390280030562"
        "6344245102063178092697666934991014220973051768742863830430334104448775178752839565277099609375"},
    {0x1.fffffffffffffp-147, chars_format::fixed, 199,
        "0."
        "0000000000000000000000000000000000000000000112103877145985353227867810940909876214292847789872780560061125"
        "268849020412635618539533386998202844194610353748572766086066820889755035750567913055419921875"},
    {0x1.fffffffffffffp-146, chars_format::fixed, 198,
        "0."
        "0000000000000000000000000000000000000000000224207754291970706455735621881819752428585695579745561120122250"
        "53769804082527123707906677399640568838922070749714553217213364177951007150113582611083984375"},
    {0x1.fffffffffffffp-145, chars_format::fixed, 197,
        "0."
        "0000000000000000000000000000000000000000000448415508583941412911471243763639504857171391159491122240244501"
        "0753960816505424741581335479928113767784414149942910643442672835590201430022716522216796875"},
    {0x1.fffffffffffffp-144, chars_format::fixed, 196,
        "0."
        "0000000000000000000000000000000000000000000896831017167882825822942487527279009714342782318982244480489002"
        "150792163301084948316267095985622753556882829988582128688534567118040286004543304443359375"},
    {0x1.fffffffffffffp-143, chars_format::fixed, 195,
        "0."
        "0000000000000000000000000000000000000000001793662034335765651645884975054558019428685564637964488960978004"
        "30158432660216989663253419197124550711376565997716425737706913423608057200908660888671875"},
    {0x1.fffffffffffffp-142, chars_format::fixed, 194,
        "0."
        "0000000000000000000000000000000000000000003587324068671531303291769950109116038857371129275928977921956008"
        "6031686532043397932650683839424910142275313199543285147541382684721611440181732177734375"},
    {0x1.fffffffffffffp-141, chars_format::fixed, 193,
        "0."
        "0000000000000000000000000000000000000000007174648137343062606583539900218232077714742258551857955843912017"
        "206337306408679586530136767884982028455062639908657029508276536944322288036346435546875"},
    {0x1.fffffffffffffp-140, chars_format::fixed, 192,
        "0."
        "0000000000000000000000000000000000000000014349296274686125213167079800436464155429484517103715911687824034"
        "41267461281735917306027353576996405691012527981731405901655307388864457607269287109375"},
    {0x1.fffffffffffffp-139, chars_format::fixed, 191,
        "0."
        "0000000000000000000000000000000000000000028698592549372250426334159600872928310858969034207431823375648068"
        "8253492256347183461205470715399281138202505596346281180331061477772891521453857421875"},
    {0x1.fffffffffffffp-138, chars_format::fixed, 190,
        "0."
        "0000000000000000000000000000000000000000057397185098744500852668319201745856621717938068414863646751296137"
        "650698451269436692241094143079856227640501119269256236066212295554578304290771484375"},
    {0x1.fffffffffffffp-137, chars_format::fixed, 189,
        "0."
        "0000000000000000000000000000000000000000114794370197489001705336638403491713243435876136829727293502592275"
        "30139690253887338448218828615971245528100223853851247213242459110915660858154296875"},
    {0x1.fffffffffffffp-136, chars_format::fixed, 188,
        "0."
        "0000000000000000000000000000000000000000229588740394978003410673276806983426486871752273659454587005184550"
        "6027938050777467689643765723194249105620044770770249442648491822183132171630859375"},
    {0x1.fffffffffffffp-135, chars_format::fixed, 187,
        "0."
        "0000000000000000000000000000000000000000459177480789956006821346553613966852973743504547318909174010369101"
        "205587610155493537928753144638849821124008954154049888529698364436626434326171875"},
    {0x1.fffffffffffffp-134, chars_format::fixed, 186,
        "0."
        "0000000000000000000000000000000000000000918354961579912013642693107227933705947487009094637818348020738202"
        "41117522031098707585750628927769964224801790830809977705939672887325286865234375"},
    {0x1.fffffffffffffp-133, chars_format::fixed, 185,
        "0."
        "0000000000000000000000000000000000000001836709923159824027285386214455867411894974018189275636696041476404"
        "8223504406219741517150125785553992844960358166161995541187934577465057373046875"},
    {0x1.fffffffffffffp-132, chars_format::fixed, 184,
        "0."
        "0000000000000000000000000000000000000003673419846319648054570772428911734823789948036378551273392082952809"
        "644700881243948303430025157110798568992071633232399108237586915493011474609375"},
    {0x1.fffffffffffffp-131, chars_format::fixed, 183,
        "0."
        "0000000000000000000000000000000000000007346839692639296109141544857823469647579896072757102546784165905619"
        "28940176248789660686005031422159713798414326646479821647517383098602294921875"},
    {0x1.fffffffffffffp-130, chars_format::fixed, 182,
        "0."
        "0000000000000000000000000000000000000014693679385278592218283089715646939295159792145514205093568331811238"
        "5788035249757932137201006284431942759682865329295964329503476619720458984375"},
    {0x1.fffffffffffffp-129, chars_format::fixed, 181,
        "0."
        "0000000000000000000000000000000000000029387358770557184436566179431293878590319584291028410187136663622477"
        "157607049951586427440201256886388551936573065859192865900695323944091796875"},
    {0x1.fffffffffffffp-128, chars_format::fixed, 180,
        "0."
        "0000000000000000000000000000000000000058774717541114368873132358862587757180639168582056820374273327244954"
        "31521409990317285488040251377277710387314613171838573180139064788818359375"},
    {0x1.fffffffffffffp-127, chars_format::fixed, 179,
        "0."
        "0000000000000000000000000000000000000117549435082228737746264717725175514361278337164113640748546654489908"
        "6304281998063457097608050275455542077462922634367714636027812957763671875"},
    {0x1.fffffffffffffp-126, chars_format::fixed, 178,
        "0."
        "0000000000000000000000000000000000000235098870164457475492529435450351028722556674328227281497093308979817"
        "260856399612691419521610055091108415492584526873542927205562591552734375"},
    {0x1.fffffffffffffp-125, chars_format::fixed, 177,
        "0."
        "0000000000000000000000000000000000000470197740328914950985058870900702057445113348656454562994186617959634"
        "52171279922538283904322011018221683098516905374708585441112518310546875"},
    {0x1.fffffffffffffp-124, chars_format::fixed, 176,
        "0."
        "0000000000000000000000000000000000000940395480657829901970117741801404114890226697312909125988373235919269"
        "0434255984507656780864402203644336619703381074941717088222503662109375"},
    {0x1.fffffffffffffp-123, chars_format::fixed, 175,
        "0."
        "0000000000000000000000000000000000001880790961315659803940235483602808229780453394625818251976746471838538"
        "086851196901531356172880440728867323940676214988343417644500732421875"},
    {0x1.fffffffffffffp-122, chars_format::fixed, 174,
        "0."
        "0000000000000000000000000000000000003761581922631319607880470967205616459560906789251636503953492943677076"
        "17370239380306271234576088145773464788135242997668683528900146484375"},
    {0x1.fffffffffffffp-121, chars_format::fixed, 173,
        "0."
        "0000000000000000000000000000000000007523163845262639215760941934411232919121813578503273007906985887354152"
        "3474047876061254246915217629154692957627048599533736705780029296875"},
    {0x1.fffffffffffffp-120, chars_format::fixed, 172,
        "0."
        "0000000000000000000000000000000000015046327690525278431521883868822465838243627157006546015813971774708304"
        "694809575212250849383043525830938591525409719906747341156005859375"},
    {0x1.fffffffffffffp-119, chars_format::fixed, 171,
        "0."
        "0000000000000000000000000000000000030092655381050556863043767737644931676487254314013092031627943549416609"
        "38961915042450169876608705166187718305081943981349468231201171875"},
    {0x1.fffffffffffffp-118, chars_format::fixed, 170,
        "0."
        "0000000000000000000000000000000000060185310762101113726087535475289863352974508628026184063255887098833218"
        "7792383008490033975321741033237543661016388796269893646240234375"},
    {0x1.fffffffffffffp-117, chars_format::fixed, 169,
        "0."
        "0000000000000000000000000000000000120370621524202227452175070950579726705949017256052368126511774197666437"
        "558476601698006795064348206647508732203277759253978729248046875"},
    {0x1.fffffffffffffp-116, chars_format::fixed, 168,
        "0."
        "0000000000000000000000000000000000240741243048404454904350141901159453411898034512104736253023548395332875"
        "11695320339601359012869641329501746440655551850795745849609375"},
    {0x1.fffffffffffffp-115, chars_format::fixed, 167,
        "0."
        "0000000000000000000000000000000000481482486096808909808700283802318906823796069024209472506047096790665750"
        "2339064067920271802573928265900349288131110370159149169921875"},
    {0x1.fffffffffffffp-114, chars_format::fixed, 166,
        "0."
        "0000000000000000000000000000000000962964972193617819617400567604637813647592138048418945012094193581331500"
        "467812813584054360514785653180069857626222074031829833984375"},
    {0x1.fffffffffffffp-113, chars_format::fixed, 165,
        "0."
        "0000000000000000000000000000000001925929944387235639234801135209275627295184276096837890024188387162663000"
        "93562562716810872102957130636013971525244414806365966796875"},
    {0x1.fffffffffffffp-112, chars_format::fixed, 164,
        "0."
        "0000000000000000000000000000000003851859888774471278469602270418551254590368552193675780048376774325326001"
        "8712512543362174420591426127202794305048882961273193359375"},
    {0x1.fffffffffffffp-111, chars_format::fixed, 163,
        "0."
        "0000000000000000000000000000000007703719777548942556939204540837102509180737104387351560096753548650652003"
        "742502508672434884118285225440558861009776592254638671875"},
    {0x1.fffffffffffffp-110, chars_format::fixed, 162,
        "0."
        "0000000000000000000000000000000015407439555097885113878409081674205018361474208774703120193507097301304007"
        "48500501734486976823657045088111772201955318450927734375"},
    {0x1.fffffffffffffp-109, chars_format::fixed, 161,
        "0."
        "0000000000000000000000000000000030814879110195770227756818163348410036722948417549406240387014194602608014"
        "9700100346897395364731409017622354440391063690185546875"},
    {0x1.fffffffffffffp-108, chars_format::fixed, 160,
        "0."
        "0000000000000000000000000000000061629758220391540455513636326696820073445896835098812480774028389205216029"
        "940020069379479072946281803524470888078212738037109375"},
    {0x1.fffffffffffffp-107, chars_format::fixed, 159,
        "0."
        "0000000000000000000000000000000123259516440783080911027272653393640146891793670197624961548056778410432059"
        "88004013875895814589256360704894177615642547607421875"},
    {0x1.fffffffffffffp-106, chars_format::fixed, 158,
        "0."
        "0000000000000000000000000000000246519032881566161822054545306787280293783587340395249923096113556820864119"
        "7600802775179162917851272140978835523128509521484375"},
    {0x1.fffffffffffffp-105, chars_format::fixed, 157,
        "0."
        "0000000000000000000000000000000493038065763132323644109090613574560587567174680790499846192227113641728239"
        "520160555035832583570254428195767104625701904296875"},
    {0x1.fffffffffffffp-104, chars_format::fixed, 156,
        "0."
        "0000000000000000000000000000000986076131526264647288218181227149121175134349361580999692384454227283456479"
        "04032111007166516714050885639153420925140380859375"},
    {0x1.fffffffffffffp-103, chars_format::fixed, 155,
        "0."
        "0000000000000000000000000000001972152263052529294576436362454298242350268698723161999384768908454566912958"
        "0806422201433303342810177127830684185028076171875"},
    {0x1.fffffffffffffp-102, chars_format::fixed, 154,
        "0."
        "0000000000000000000000000000003944304526105058589152872724908596484700537397446323998769537816909133825916"
        "161284440286660668562035425566136837005615234375"},
    {0x1.fffffffffffffp-101, chars_format::fixed, 153,
        "0."
        "0000000000000000000000000000007888609052210117178305745449817192969401074794892647997539075633818267651832"
        "32256888057332133712407085113227367401123046875"},
    {0x1.fffffffffffffp-100, chars_format::fixed, 152,
        "0."
        "0000000000000000000000000000015777218104420234356611490899634385938802149589785295995078151267636535303664"
        "6451377611466426742481417022645473480224609375"},
    {0x1.fffffffffffffp-99, chars_format::fixed, 151,
        "0."
        "0000000000000000000000000000031554436208840468713222981799268771877604299179570591990156302535273070607329"
        "290275522293285348496283404529094696044921875"},
    {0x1.fffffffffffffp-98, chars_format::fixed, 150,
        "0."
        "0000000000000000000000000000063108872417680937426445963598537543755208598359141183980312605070546141214658"
        "58055104458657069699256680905818939208984375"},
    {0x1.fffffffffffffp-97, chars_format::fixed, 149,
        "0."
        "0000000000000000000000000000126217744835361874852891927197075087510417196718282367960625210141092282429317"
        "1611020891731413939851336181163787841796875"},
    {0x1.fffffffffffffp-96, chars_format::fixed, 148,
        "0."
        "0000000000000000000000000000252435489670723749705783854394150175020834393436564735921250420282184564858634"
        "322204178346282787970267236232757568359375"},
    {0x1.fffffffffffffp-95, chars_format::fixed, 147,
        "0."
        "0000000000000000000000000000504870979341447499411567708788300350041668786873129471842500840564369129717268"
        "64440835669256557594053447246551513671875"},
    {0x1.fffffffffffffp-94, chars_format::fixed, 146,
        "0."
        "0000000000000000000000000001009741958682894998823135417576600700083337573746258943685001681128738259434537"
        "2888167133851311518810689449310302734375"},
    {0x1.fffffffffffffp-93, chars_format::fixed, 145,
        "0."
        "0000000000000000000000000002019483917365789997646270835153201400166675147492517887370003362257476518869074"
        "577633426770262303762137889862060546875"},
    {0x1.fffffffffffffp-92, chars_format::fixed, 144,
        "0."
        "0000000000000000000000000004038967834731579995292541670306402800333350294985035774740006724514953037738149"
        "15526685354052460752427577972412109375"},
    {0x1.fffffffffffffp-91, chars_format::fixed, 143,
        "0."
        "0000000000000000000000000008077935669463159990585083340612805600666700589970071549480013449029906075476298"
        "3105337070810492150485515594482421875"},
    {0x1.fffffffffffffp-90, chars_format::fixed, 142,
        "0."
        "0000000000000000000000000016155871338926319981170166681225611201333401179940143098960026898059812150952596"
        "621067414162098430097103118896484375"},
    {0x1.fffffffffffffp-89, chars_format::fixed, 141,
        "0."
        "0000000000000000000000000032311742677852639962340333362451222402666802359880286197920053796119624301905193"
        "24213482832419686019420623779296875"},
    {0x1.fffffffffffffp-88, chars_format::fixed, 140,
        "0."
        "0000000000000000000000000064623485355705279924680666724902444805333604719760572395840107592239248603810386"
        "4842696566483937203884124755859375"},
    {0x1.fffffffffffffp-87, chars_format::fixed, 139,
        "0."
        "0000000000000000000000000129246970711410559849361333449804889610667209439521144791680215184478497207620772"
        "968539313296787440776824951171875"},
    {0x1.fffffffffffffp-86, chars_format::fixed, 138,
        "0."
        "0000000000000000000000000258493941422821119698722666899609779221334418879042289583360430368956994415241545"
        "93707862659357488155364990234375"},
    {0x1.fffffffffffffp-85, chars_format::fixed, 137,
        "0."
        "0000000000000000000000000516987882845642239397445333799219558442668837758084579166720860737913988830483091"
        "8741572531871497631072998046875"},
    {0x1.fffffffffffffp-84, chars_format::fixed, 136,
        "0."
        "0000000000000000000000001033975765691284478794890667598439116885337675516169158333441721475827977660966183"
        "748314506374299526214599609375"},
    {0x1.fffffffffffffp-83, chars_format::fixed, 135,
        "0."
        "0000000000000000000000002067951531382568957589781335196878233770675351032338316666883442951655955321932367"
        "49662901274859905242919921875"},
    {0x1.fffffffffffffp-82, chars_format::fixed, 134,
        "0."
        "0000000000000000000000004135903062765137915179562670393756467541350702064676633333766885903311910643864734"
        "9932580254971981048583984375"},
    {0x1.fffffffffffffp-81, chars_format::fixed, 133,
        "0."
        "0000000000000000000000008271806125530275830359125340787512935082701404129353266667533771806623821287729469"
        "986516050994396209716796875"},
    {0x1.fffffffffffffp-80, chars_format::fixed, 132,
        "0."
        "0000000000000000000000016543612251060551660718250681575025870165402808258706533335067543613247642575458939"
        "97303210198879241943359375"},
    {0x1.fffffffffffffp-79, chars_format::fixed, 131,
        "0."
        "0000000000000000000000033087224502121103321436501363150051740330805616517413066670135087226495285150917879"
        "9460642039775848388671875"},
    {0x1.fffffffffffffp-78, chars_format::fixed, 130,
        "0."
        "0000000000000000000000066174449004242206642873002726300103480661611233034826133340270174452990570301835759"
        "892128407955169677734375"},
    {0x1.fffffffffffffp-77, chars_format::fixed, 129,
        "0."
        "0000000000000000000000132348898008484413285746005452600206961323222466069652266680540348905981140603671519"
        "78425681591033935546875"},
    {0x1.fffffffffffffp-76, chars_format::fixed, 128,
        "0."
        "0000000000000000000000264697796016968826571492010905200413922646444932139304533361080697811962281207343039"
        "5685136318206787109375"},
    {0x1.fffffffffffffp-75, chars_format::fixed, 127,
        "0."
        "0000000000000000000000529395592033937653142984021810400827845292889864278609066722161395623924562414686079"
        "137027263641357421875"},
    {0x1.fffffffffffffp-74, chars_format::fixed, 126,
        "0."
        "0000000000000000000001058791184067875306285968043620801655690585779728557218133444322791247849124829372158"
        "27405452728271484375"},
    {0x1.fffffffffffffp-73, chars_format::fixed, 125,
        "0."
        "0000000000000000000002117582368135750612571936087241603311381171559457114436266888645582495698249658744316"
        "5481090545654296875"},
    {0x1.fffffffffffffp-72, chars_format::fixed, 124,
        "0."
        "0000000000000000000004235164736271501225143872174483206622762343118914228872533777291164991396499317488633"
        "096218109130859375"},
    {0x1.fffffffffffffp-71, chars_format::fixed, 123,
        "0."
        "0000000000000000000008470329472543002450287744348966413245524686237828457745067554582329982792998634977266"
        "19243621826171875"},
    {0x1.fffffffffffffp-70, chars_format::fixed, 122,
        "0."
        "0000000000000000000016940658945086004900575488697932826491049372475656915490135109164659965585997269954532"
        "3848724365234375"},
    {0x1.fffffffffffffp-69, chars_format::fixed, 121,
        "0."
        "0000000000000000000033881317890172009801150977395865652982098744951313830980270218329319931171994539909064"
        "769744873046875"},
    {0x1.fffffffffffffp-68, chars_format::fixed, 120,
        "0."
        "0000000000000000000067762635780344019602301954791731305964197489902627661960540436658639862343989079818129"
        "53948974609375"},
    {0x1.fffffffffffffp-67, chars_format::fixed, 119,
        "0."
        "0000000000000000000135525271560688039204603909583462611928394979805255323921080873317279724687978159636259"
        "0789794921875"},
    {0x1.fffffffffffffp-66, chars_format::fixed, 118,
        "0."
        "0000000000000000000271050543121376078409207819166925223856789959610510647842161746634559449375956319272518"
        "157958984375"},
    {0x1.fffffffffffffp-65, chars_format::fixed, 117,
        "0."
        "0000000000000000000542101086242752156818415638333850447713579919221021295684323493269118898751912638545036"
        "31591796875"},
    {0x1.fffffffffffffp-64, chars_format::fixed, 116,
        "0."
        "0000000000000000001084202172485504313636831276667700895427159838442042591368646986538237797503825277090072"
        "6318359375"},
    {0x1.fffffffffffffp-63, chars_format::fixed, 115,
        "0."
        "0000000000000000002168404344971008627273662553335401790854319676884085182737293973076475595007650554180145"
        "263671875"},
    {0x1.fffffffffffffp-62, chars_format::fixed, 114,
        "0."
        "0000000000000000004336808689942017254547325106670803581708639353768170365474587946152951190015301108360290"
        "52734375"},
    {0x1.fffffffffffffp-61, chars_format::fixed, 113,
        "0."
        "0000000000000000008673617379884034509094650213341607163417278707536340730949175892305902380030602216720581"
        "0546875"},
    {0x1.fffffffffffffp-60, chars_format::fixed, 112,
        "0."
        "0000000000000000017347234759768069018189300426683214326834557415072681461898351784611804760061204433441162"
        "109375"},
    {0x1.fffffffffffffp-59, chars_format::fixed, 111,
        "0."
        "0000000000000000034694469519536138036378600853366428653669114830145362923796703569223609520122408866882324"
        "21875"},
    {0x1.fffffffffffffp-58, chars_format::fixed, 110,
        "0."
        "0000000000000000069388939039072276072757201706732857307338229660290725847593407138447219040244817733764648"
        "4375"},
    {0x1.fffffffffffffp-57, chars_format::fixed, 109,
        "0."
        "0000000000000000138777878078144552145514403413465714614676459320581451695186814276894438080489635467529296"
        "875"},
    {0x1.fffffffffffffp-56, chars_format::fixed, 108,
        "0."
        "0000000000000000277555756156289104291028806826931429229352918641162903390373628553788876160979270935058593"
        "75"},
    {0x1.fffffffffffffp-55, chars_format::fixed, 107,
        "0."
        "0000000000000000555111512312578208582057613653862858458705837282325806780747257107577752321958541870117187"
        "5"},
    {0x1.fffffffffffffp-54, chars_format::fixed, 106,
        "0."
        "000000000000000111022302462515641716411522730772571691741167456465161356149451421515550464391708374023437"
        "5"},
    {0x1.fffffffffffffp-53, chars_format::fixed, 105,
        "0."
        "00000000000000022204460492503128343282304546154514338348233491293032271229890284303110092878341674804687"
        "5"},
    {0x1.fffffffffffffp-52, chars_format::fixed, 104,
        "0."
        "00000000000000044408920985006256686564609092309028676696466982586064542459780568606220185756683349609375"},
    {0x1.fffffffffffffp-51, chars_format::fixed, 103,
        "0."
        "0000000000000008881784197001251337312921818461805735339293396517212908491956113721244037151336669921875"},
    {0x1.fffffffffffffp-50, chars_format::fixed, 102,
        "0.000000000000001776356839400250267462584363692361147067858679303442581698391222744248807430267333984375"},
    {0x1.fffffffffffffp-49, chars_format::fixed, 101,
        "0.00000000000000355271367880050053492516872738472229413571735860688516339678244548849761486053466796875"},
    {0x1.fffffffffffffp-48, chars_format::fixed, 100,
        "0.0000000000000071054273576010010698503374547694445882714347172137703267935648909769952297210693359375"},
    {0x1.fffffffffffffp-47, chars_format::fixed, 99,
        "0.000000000000014210854715202002139700674909538889176542869434427540653587129781953990459442138671875"},
    {0x1.fffffffffffffp-46, chars_format::fixed, 98,
        "0.00000000000002842170943040400427940134981907777835308573886885508130717425956390798091888427734375"},
    {0x1.fffffffffffffp-45, chars_format::fixed, 97,
        "0.0000000000000568434188608080085588026996381555567061714777377101626143485191278159618377685546875"},
    {0x1.fffffffffffffp-44, chars_format::fixed, 96,
        "0.000000000000113686837721616017117605399276311113412342955475420325228697038255631923675537109375"},
    {0x1.fffffffffffffp-43, chars_format::fixed, 95,
        "0.00000000000022737367544323203423521079855262222682468591095084065045739407651126384735107421875"},
    {0x1.fffffffffffffp-42, chars_format::fixed, 94,
        "0.0000000000004547473508864640684704215971052444536493718219016813009147881530225276947021484375"},
    {0x1.fffffffffffffp-41, chars_format::fixed, 93,
        "0.000000000000909494701772928136940843194210488907298743643803362601829576306045055389404296875"},
    {0x1.fffffffffffffp-40, chars_format::fixed, 92,
        "0.00000000000181898940354585627388168638842097781459748728760672520365915261209011077880859375"},
    {0x1.fffffffffffffp-39, chars_format::fixed, 91,
        "0.0000000000036379788070917125477633727768419556291949745752134504073183052241802215576171875"},
    {0x1.fffffffffffffp-38, chars_format::fixed, 90,
        "0.000000000007275957614183425095526745553683911258389949150426900814636610448360443115234375"},
    {0x1.fffffffffffffp-37, chars_format::fixed, 89,
        "0.00000000001455191522836685019105349110736782251677989830085380162927322089672088623046875"},
    {0x1.fffffffffffffp-36, chars_format::fixed, 88,
        "0.0000000000291038304567337003821069822147356450335597966017076032585464417934417724609375"},
    {0x1.fffffffffffffp-35, chars_format::fixed, 87,
        "0.000000000058207660913467400764213964429471290067119593203415206517092883586883544921875"},
    {0x1.fffffffffffffp-34, chars_format::fixed, 86,
        "0.00000000011641532182693480152842792885894258013423918640683041303418576717376708984375"},
    {0x1.fffffffffffffp-33, chars_format::fixed, 85,
        "0.0000000002328306436538696030568558577178851602684783728136608260683715343475341796875"},
    {0x1.fffffffffffffp-32, chars_format::fixed, 84,
        "0.000000000465661287307739206113711715435770320536956745627321652136743068695068359375"},
    {0x1.fffffffffffffp-31, chars_format::fixed, 83,
        "0.00000000093132257461547841222742343087154064107391349125464330427348613739013671875"},
    {0x1.fffffffffffffp-30, chars_format::fixed, 82,
        "0.0000000018626451492309568244548468617430812821478269825092866085469722747802734375"},
    {0x1.fffffffffffffp-29, chars_format::fixed, 81,
        "0.000000003725290298461913648909693723486162564295653965018573217093944549560546875"},
    {0x1.fffffffffffffp-28, chars_format::fixed, 80,
        "0.00000000745058059692382729781938744697232512859130793003714643418788909912109375"},
    {0x1.fffffffffffffp-27, chars_format::fixed, 79,
        "0.0000000149011611938476545956387748939446502571826158600742928683757781982421875"},
    {0x1.fffffffffffffp-26, chars_format::fixed, 78,
        "0.000000029802322387695309191277549787889300514365231720148585736751556396484375"},
    {0x1.fffffffffffffp-25, chars_format::fixed, 77,
        "0.00000005960464477539061838255509957577860102873046344029717147350311279296875"},
    {0x1.fffffffffffffp-24, chars_format::fixed, 76,
        "0.0000001192092895507812367651101991515572020574609268805943429470062255859375"},
    {0x1.fffffffffffffp-23, chars_format::fixed, 75,
        "0.000000238418579101562473530220398303114404114921853761188685894012451171875"},
    {0x1.fffffffffffffp-22, chars_format::fixed, 74,
        "0.00000047683715820312494706044079660622880822984370752237737178802490234375"},
    {0x1.fffffffffffffp-21, chars_format::fixed, 73,
        "0.0000009536743164062498941208815932124576164596874150447547435760498046875"},
    {0x1.fffffffffffffp-20, chars_format::fixed, 72,
        "0.000001907348632812499788241763186424915232919374830089509487152099609375"},
    {0x1.fffffffffffffp-19, chars_format::fixed, 71,
        "0.00000381469726562499957648352637284983046583874966017901897430419921875"},
    {0x1.fffffffffffffp-18, chars_format::fixed, 70,
        "0.0000076293945312499991529670527456996609316774993203580379486083984375"},
    {0x1.fffffffffffffp-17, chars_format::fixed, 69,
        "0.000015258789062499998305934105491399321863354998640716075897216796875"},
    {0x1.fffffffffffffp-16, chars_format::fixed, 68,
        "0.00003051757812499999661186821098279864372670999728143215179443359375"},
    {0x1.fffffffffffffp-15, chars_format::fixed, 67,
        "0.0000610351562499999932237364219655972874534199945628643035888671875"},
    {0x1.fffffffffffffp-14, chars_format::fixed, 66,
        "0.000122070312499999986447472843931194574906839989125728607177734375"},
    {0x1.fffffffffffffp-13, chars_format::fixed, 65,
        "0.00024414062499999997289494568786238914981367997825145721435546875"},
    {0x1.fffffffffffffp-12, chars_format::fixed, 64,
        "0.0004882812499999999457898913757247782996273599565029144287109375"},
    {0x1.fffffffffffffp-11, chars_format::fixed, 63,
        "0.000976562499999999891579782751449556599254719913005828857421875"},
    {0x1.fffffffffffffp-10, chars_format::fixed, 62,
        "0.00195312499999999978315956550289911319850943982601165771484375"},
    {0x1.fffffffffffffp-9, chars_format::fixed, 61, "0.0039062499999999995663191310057982263970188796520233154296875"},
    {0x1.fffffffffffffp-8, chars_format::fixed, 60, "0.007812499999999999132638262011596452794037759304046630859375"},
    {0x1.fffffffffffffp-7, chars_format::fixed, 59, "0.01562499999999999826527652402319290558807551860809326171875"},
    {0x1.fffffffffffffp-6, chars_format::fixed, 58, "0.0312499999999999965305530480463858111761510372161865234375"},
    {0x1.fffffffffffffp-5, chars_format::fixed, 57, "0.062499999999999993061106096092771622352302074432373046875"},
    {0x1.fffffffffffffp-4, chars_format::fixed, 56, "0.12499999999999998612221219218554324470460414886474609375"},
    {0x1.fffffffffffffp-3, chars_format::fixed, 55, "0.2499999999999999722444243843710864894092082977294921875"},
    {0x1.fffffffffffffp-2, chars_format::fixed, 54, "0.499999999999999944488848768742172978818416595458984375"},
    {0x1.fffffffffffffp-1, chars_format::fixed, 53, "0.99999999999999988897769753748434595763683319091796875"},
    {0x1.fffffffffffffp+0, chars_format::fixed, 52, "1.9999999999999997779553950749686919152736663818359375"},
    {0x1.fffffffffffffp+1, chars_format::fixed, 51, "3.999999999999999555910790149937383830547332763671875"},
    {0x1.fffffffffffffp+2, chars_format::fixed, 50, "7.99999999999999911182158029987476766109466552734375"},
    {0x1.fffffffffffffp+3, chars_format::fixed, 49, "15.9999999999999982236431605997495353221893310546875"},
    {0x1.fffffffffffffp+4, chars_format::fixed, 48, "31.999999999999996447286321199499070644378662109375"},
    {0x1.fffffffffffffp+5, chars_format::fixed, 47, "63.99999999999999289457264239899814128875732421875"},
    {0x1.fffffffffffffp+6, chars_format::fixed, 46, "127.9999999999999857891452847979962825775146484375"},
    {0x1.fffffffffffffp+7, chars_format::fixed, 45, "255.999999999999971578290569595992565155029296875"},
    {0x1.fffffffffffffp+8, chars_format::fixed, 44, "511.99999999999994315658113919198513031005859375"},
    {0x1.fffffffffffffp+9, chars_format::fixed, 43, "1023.9999999999998863131622783839702606201171875"},
    {0x1.fffffffffffffp+10, chars_format::fixed, 42, "2047.999999999999772626324556767940521240234375"},
    {0x1.fffffffffffffp+11, chars_format::fixed, 41, "4095.99999999999954525264911353588104248046875"},
    {0x1.fffffffffffffp+12, chars_format::fixed, 40, "8191.9999999999990905052982270717620849609375"},
    {0x1.fffffffffffffp+13, chars_format::fixed, 39, "16383.999999999998181010596454143524169921875"},
    {0x1.fffffffffffffp+14, chars_format::fixed, 38, "32767.99999999999636202119290828704833984375"},
    {0x1.fffffffffffffp+15, chars_format::fixed, 37, "65535.9999999999927240423858165740966796875"},
    {0x1.fffffffffffffp+16, chars_format::fixed, 36, "131071.999999999985448084771633148193359375"},
    {0x1.fffffffffffffp+17, chars_format::fixed, 35, "262143.99999999997089616954326629638671875"},
    {0x1.fffffffffffffp+18, chars_format::fixed, 34, "524287.9999999999417923390865325927734375"},
    {0x1.fffffffffffffp+19, chars_format::fixed, 33, "1048575.999999999883584678173065185546875"},
    {0x1.fffffffffffffp+20, chars_format::fixed, 32, "2097151.99999999976716935634613037109375"},
    {0x1.fffffffffffffp+21, chars_format::fixed, 31, "4194303.9999999995343387126922607421875"},
    {0x1.fffffffffffffp+22, chars_format::fixed, 30, "8388607.999999999068677425384521484375"},
    {0x1.fffffffffffffp+23, chars_format::fixed, 29, "16777215.99999999813735485076904296875"},
    {0x1.fffffffffffffp+24, chars_format::fixed, 28, "33554431.9999999962747097015380859375"},
    {0x1.fffffffffffffp+25, chars_format::fixed, 27, "67108863.999999992549419403076171875"},
    {0x1.fffffffffffffp+26, chars_format::fixed, 26, "134217727.99999998509883880615234375"},
    {0x1.fffffffffffffp+27, chars_format::fixed, 25, "268435455.9999999701976776123046875"},
    {0x1.fffffffffffffp+28, chars_format::fixed, 24, "536870911.999999940395355224609375"},
    {0x1.fffffffffffffp+29, chars_format::fixed, 23, "1073741823.99999988079071044921875"},
    {0x1.fffffffffffffp+30, chars_format::fixed, 22, "2147483647.9999997615814208984375"},
    {0x1.fffffffffffffp+31, chars_format::fixed, 21, "4294967295.999999523162841796875"},
    {0x1.fffffffffffffp+32, chars_format::fixed, 20, "8589934591.99999904632568359375"},
    {0x1.fffffffffffffp+33, chars_format::fixed, 19, "17179869183.9999980926513671875"},
    {0x1.fffffffffffffp+34, chars_format::fixed, 18, "34359738367.999996185302734375"},
    {0x1.fffffffffffffp+35, chars_format::fixed, 17, "68719476735.99999237060546875"},
    {0x1.fffffffffffffp+36, chars_format::fixed, 16, "137438953471.9999847412109375"},
    {0x1.fffffffffffffp+37, chars_format::fixed, 15, "274877906943.999969482421875"},
    {0x1.fffffffffffffp+38, chars_format::fixed, 14, "549755813887.99993896484375"},
    {0x1.fffffffffffffp+39, chars_format::fixed, 13, "1099511627775.9998779296875"},
    {0x1.fffffffffffffp+40, chars_format::fixed, 12, "2199023255551.999755859375"},
    {0x1.fffffffffffffp+41, chars_format::fixed, 11, "4398046511103.99951171875"},
    {0x1.fffffffffffffp+42, chars_format::fixed, 10, "8796093022207.9990234375"},
    {0x1.fffffffffffffp+43, chars_format::fixed, 9, "17592186044415.998046875"},
    {0x1.fffffffffffffp+44, chars_format::fixed, 8, "35184372088831.99609375"},
    {0x1.fffffffffffffp+45, chars_format::fixed, 7, "70368744177663.9921875"},
    {0x1.fffffffffffffp+46, chars_format::fixed, 6, "140737488355327.984375"},
    {0x1.fffffffffffffp+47, chars_format::fixed, 5, "281474976710655.96875"},
    {0x1.fffffffffffffp+48, chars_format::fixed, 4, "562949953421311.9375"},
    {0x1.fffffffffffffp+49, chars_format::fixed, 3, "1125899906842623.875"},
    {0x1.fffffffffffffp+50, chars_format::fixed, 2, "2251799813685247.75"},
    {0x1.fffffffffffffp+51, chars_format::fixed, 1, "4503599627370495.5"},
    {0x1.fffffffffffffp+52, chars_format::fixed, 0, "9007199254740991"},
    {0x1.fffffffffffffp+53, chars_format::fixed, 0, "18014398509481982"},
    {0x1.fffffffffffffp+54, chars_format::fixed, 0, "36028797018963964"},
    {0x1.fffffffffffffp+55, chars_format::fixed, 0, "72057594037927928"},
    {0x1.fffffffffffffp+56, chars_format::fixed, 0, "144115188075855856"},
    {0x1.fffffffffffffp+57, chars_format::fixed, 0, "288230376151711712"},
    {0x1.fffffffffffffp+58, chars_format::fixed, 0, "576460752303423424"},
    {0x1.fffffffffffffp+59, chars_format::fixed, 0, "1152921504606846848"},
    {0x1.fffffffffffffp+60, chars_format::fixed, 0, "2305843009213693696"},
    {0x1.fffffffffffffp+61, chars_format::fixed, 0, "4611686018427387392"},
    {0x1.fffffffffffffp+62, chars_format::fixed, 0, "9223372036854774784"},
    {0x1.fffffffffffffp+63, chars_format::fixed, 0, "18446744073709549568"},
    {0x1.fffffffffffffp+64, chars_format::fixed, 0, "36893488147419099136"},
    {0x1.fffffffffffffp+65, chars_format::fixed, 0, "73786976294838198272"},
    {0x1.fffffffffffffp+66, chars_format::fixed, 0, "147573952589676396544"},
    {0x1.fffffffffffffp+67, chars_format::fixed, 0, "295147905179352793088"},
    {0x1.fffffffffffffp+68, chars_format::fixed, 0, "590295810358705586176"},
    {0x1.fffffffffffffp+69, chars_format::fixed, 0, "1180591620717411172352"},
    {0x1.fffffffffffffp+70, chars_format::fixed, 0, "2361183241434822344704"},
    {0x1.fffffffffffffp+71, chars_format::fixed, 0, "4722366482869644689408"},
    {0x1.fffffffffffffp+72, chars_format::fixed, 0, "9444732965739289378816"},
    {0x1.fffffffffffffp+73, chars_format::fixed, 0, "18889465931478578757632"},
    {0x1.fffffffffffffp+74, chars_format::fixed, 0, "37778931862957157515264"},
    {0x1.fffffffffffffp+75, chars_format::fixed, 0, "75557863725914315030528"},
    {0x1.fffffffffffffp+76, chars_format::fixed, 0, "151115727451828630061056"},
    {0x1.fffffffffffffp+77, chars_format::fixed, 0, "302231454903657260122112"},
    {0x1.fffffffffffffp+78, chars_format::fixed, 0, "604462909807314520244224"},
    {0x1.fffffffffffffp+79, chars_format::fixed, 0, "1208925819614629040488448"},
    {0x1.fffffffffffffp+80, chars_format::fixed, 0, "2417851639229258080976896"},
    {0x1.fffffffffffffp+81, chars_format::fixed, 0, "4835703278458516161953792"},
    {0x1.fffffffffffffp+82, chars_format::fixed, 0, "9671406556917032323907584"},
    {0x1.fffffffffffffp+83, chars_format::fixed, 0, "19342813113834064647815168"},
    {0x1.fffffffffffffp+84, chars_format::fixed, 0, "38685626227668129295630336"},
    {0x1.fffffffffffffp+85, chars_format::fixed, 0, "77371252455336258591260672"},
    {0x1.fffffffffffffp+86, chars_format::fixed, 0, "154742504910672517182521344"},
    {0x1.fffffffffffffp+87, chars_format::fixed, 0, "309485009821345034365042688"},
    {0x1.fffffffffffffp+88, chars_format::fixed, 0, "618970019642690068730085376"},
    {0x1.fffffffffffffp+89, chars_format::fixed, 0, "1237940039285380137460170752"},
    {0x1.fffffffffffffp+90, chars_format::fixed, 0, "2475880078570760274920341504"},
    {0x1.fffffffffffffp+91, chars_format::fixed, 0, "4951760157141520549840683008"},
    {0x1.fffffffffffffp+92, chars_format::fixed, 0, "9903520314283041099681366016"},
    {0x1.fffffffffffffp+93, chars_format::fixed, 0, "19807040628566082199362732032"},
    {0x1.fffffffffffffp+94, chars_format::fixed, 0, "39614081257132164398725464064"},
    {0x1.fffffffffffffp+95, chars_format::fixed, 0, "79228162514264328797450928128"},
    {0x1.fffffffffffffp+96, chars_format::fixed, 0, "158456325028528657594901856256"},
    {0x1.fffffffffffffp+97, chars_format::fixed, 0, "316912650057057315189803712512"},
    {0x1.fffffffffffffp+98, chars_format::fixed, 0, "633825300114114630379607425024"},
    {0x1.fffffffffffffp+99, chars_format::fixed, 0, "1267650600228229260759214850048"},
    {0x1.fffffffffffffp+100, chars_format::fixed, 0, "2535301200456458521518429700096"},
    {0x1.fffffffffffffp+101, chars_format::fixed, 0, "5070602400912917043036859400192"},
    {0x1.fffffffffffffp+102, chars_format::fixed, 0, "10141204801825834086073718800384"},
    {0x1.fffffffffffffp+103, chars_format::fixed, 0, "20282409603651668172147437600768"},
    {0x1.fffffffffffffp+104, chars_format::fixed, 0, "40564819207303336344294875201536"},
    {0x1.fffffffffffffp+105, chars_format::fixed, 0, "81129638414606672688589750403072"},
    {0x1.fffffffffffffp+106, chars_format::fixed, 0, "162259276829213345377179500806144"},
    {0x1.fffffffffffffp+107, chars_format::fixed, 0, "324518553658426690754359001612288"},
    {0x1.fffffffffffffp+108, chars_format::fixed, 0, "649037107316853381508718003224576"},
    {0x1.fffffffffffffp+109, chars_format::fixed, 0, "1298074214633706763017436006449152"},
    {0x1.fffffffffffffp+110, chars_format::fixed, 0, "2596148429267413526034872012898304"},
    {0x1.fffffffffffffp+111, chars_format::fixed, 0, "5192296858534827052069744025796608"},
    {0x1.fffffffffffffp+112, chars_format::fixed, 0, "10384593717069654104139488051593216"},
    {0x1.fffffffffffffp+113, chars_format::fixed, 0, "20769187434139308208278976103186432"},
    {0x1.fffffffffffffp+114, chars_format::fixed, 0, "41538374868278616416557952206372864"},
    {0x1.fffffffffffffp+115, chars_format::fixed, 0, "83076749736557232833115904412745728"},
    {0x1.fffffffffffffp+116, chars_format::fixed, 0, "166153499473114465666231808825491456"},
    {0x1.fffffffffffffp+117, chars_format::fixed, 0, "332306998946228931332463617650982912"},
    {0x1.fffffffffffffp+118, chars_format::fixed, 0, "664613997892457862664927235301965824"},
    {0x1.fffffffffffffp+119, chars_format::fixed, 0, "1329227995784915725329854470603931648"},
    {0x1.fffffffffffffp+120, chars_format::fixed, 0, "2658455991569831450659708941207863296"},
    {0x1.fffffffffffffp+121, chars_format::fixed, 0, "5316911983139662901319417882415726592"},
    {0x1.fffffffffffffp+122, chars_format::fixed, 0, "10633823966279325802638835764831453184"},
    {0x1.fffffffffffffp+123, chars_format::fixed, 0, "21267647932558651605277671529662906368"},
    {0x1.fffffffffffffp+124, chars_format::fixed, 0, "42535295865117303210555343059325812736"},
    {0x1.fffffffffffffp+125, chars_format::fixed, 0, "85070591730234606421110686118651625472"},
    {0x1.fffffffffffffp+126, chars_format::fixed, 0, "170141183460469212842221372237303250944"},
    {0x1.fffffffffffffp+127, chars_format::fixed, 0, "340282366920938425684442744474606501888"},
    {0x1.fffffffffffffp+128, chars_format::fixed, 0, "680564733841876851368885488949213003776"},
    {0x1.fffffffffffffp+129, chars_format::fixed, 0, "1361129467683753702737770977898426007552"},
    {0x1.fffffffffffffp+130, chars_format::fixed, 0, "2722258935367507405475541955796852015104"},
    {0x1.fffffffffffffp+131, chars_format::fixed, 0, "5444517870735014810951083911593704030208"},
    {0x1.fffffffffffffp+132, chars_format::fixed, 0, "10889035741470029621902167823187408060416"},
    {0x1.fffffffffffffp+133, chars_format::fixed, 0, "21778071482940059243804335646374816120832"},
    {0x1.fffffffffffffp+134, chars_format::fixed, 0, "43556142965880118487608671292749632241664"},
    {0x1.fffffffffffffp+135, chars_format::fixed, 0, "87112285931760236975217342585499264483328"},
    {0x1.fffffffffffffp+136, chars_format::fixed, 0, "174224571863520473950434685170998528966656"},
    {0x1.fffffffffffffp+137, chars_format::fixed, 0, "348449143727040947900869370341997057933312"},
    {0x1.fffffffffffffp+138, chars_format::fixed, 0, "696898287454081895801738740683994115866624"},
    {0x1.fffffffffffffp+139, chars_format::fixed, 0, "1393796574908163791603477481367988231733248"},
    {0x1.fffffffffffffp+140, chars_format::fixed, 0, "2787593149816327583206954962735976463466496"},
    {0x1.fffffffffffffp+141, chars_format::fixed, 0, "5575186299632655166413909925471952926932992"},
    {0x1.fffffffffffffp+142, chars_format::fixed, 0, "11150372599265310332827819850943905853865984"},
    {0x1.fffffffffffffp+143, chars_format::fixed, 0, "22300745198530620665655639701887811707731968"},
    {0x1.fffffffffffffp+144, chars_format::fixed, 0, "44601490397061241331311279403775623415463936"},
    {0x1.fffffffffffffp+145, chars_format::fixed, 0, "89202980794122482662622558807551246830927872"},
    {0x1.fffffffffffffp+146, chars_format::fixed, 0, "178405961588244965325245117615102493661855744"},
    {0x1.fffffffffffffp+147, chars_format::fixed, 0, "356811923176489930650490235230204987323711488"},
    {0x1.fffffffffffffp+148, chars_format::fixed, 0, "713623846352979861300980470460409974647422976"},
    {0x1.fffffffffffffp+149, chars_format::fixed, 0, "1427247692705959722601960940920819949294845952"},
    {0x1.fffffffffffffp+150, chars_format::fixed, 0, "2854495385411919445203921881841639898589691904"},
    {0x1.fffffffffffffp+151, chars_format::fixed, 0, "5708990770823838890407843763683279797179383808"},
    {0x1.fffffffffffffp+152, chars_format::fixed, 0, "11417981541647677780815687527366559594358767616"},
    {0x1.fffffffffffffp+153, chars_format::fixed, 0, "22835963083295355561631375054733119188717535232"},
    {0x1.fffffffffffffp+154, chars_format::fixed, 0, "45671926166590711123262750109466238377435070464"},
    {0x1.fffffffffffffp+155, chars_format::fixed, 0, "91343852333181422246525500218932476754870140928"},
    {0x1.fffffffffffffp+156, chars_format::fixed, 0, "182687704666362844493051000437864953509740281856"},
    {0x1.fffffffffffffp+157, chars_format::fixed, 0, "365375409332725688986102000875729907019480563712"},
    {0x1.fffffffffffffp+158, chars_format::fixed, 0, "730750818665451377972204001751459814038961127424"},
    {0x1.fffffffffffffp+159, chars_format::fixed, 0, "1461501637330902755944408003502919628077922254848"},
    {0x1.fffffffffffffp+160, chars_format::fixed, 0, "2923003274661805511888816007005839256155844509696"},
    {0x1.fffffffffffffp+161, chars_format::fixed, 0, "5846006549323611023777632014011678512311689019392"},
    {0x1.fffffffffffffp+162, chars_format::fixed, 0, "11692013098647222047555264028023357024623378038784"},
    {0x1.fffffffffffffp+163, chars_format::fixed, 0, "23384026197294444095110528056046714049246756077568"},
    {0x1.fffffffffffffp+164, chars_format::fixed, 0, "46768052394588888190221056112093428098493512155136"},
    {0x1.fffffffffffffp+165, chars_format::fixed, 0, "93536104789177776380442112224186856196987024310272"},
    {0x1.fffffffffffffp+166, chars_format::fixed, 0, "187072209578355552760884224448373712393974048620544"},
    {0x1.fffffffffffffp+167, chars_format::fixed, 0, "374144419156711105521768448896747424787948097241088"},
    {0x1.fffffffffffffp+168, chars_format::fixed, 0, "748288838313422211043536897793494849575896194482176"},
    {0x1.fffffffffffffp+169, chars_format::fixed, 0, "1496577676626844422087073795586989699151792388964352"},
    {0x1.fffffffffffffp+170, chars_format::fixed, 0, "2993155353253688844174147591173979398303584777928704"},
    {0x1.fffffffffffffp+171, chars_format::fixed, 0, "5986310706507377688348295182347958796607169555857408"},
    {0x1.fffffffffffffp+172, chars_format::fixed, 0, "11972621413014755376696590364695917593214339111714816"},
    {0x1.fffffffffffffp+173, chars_format::fixed, 0, "23945242826029510753393180729391835186428678223429632"},
    {0x1.fffffffffffffp+174, chars_format::fixed, 0, "47890485652059021506786361458783670372857356446859264"},
    {0x1.fffffffffffffp+175, chars_format::fixed, 0, "95780971304118043013572722917567340745714712893718528"},
    {0x1.fffffffffffffp+176, chars_format::fixed, 0, "191561942608236086027145445835134681491429425787437056"},
    {0x1.fffffffffffffp+177, chars_format::fixed, 0, "383123885216472172054290891670269362982858851574874112"},
    {0x1.fffffffffffffp+178, chars_format::fixed, 0, "766247770432944344108581783340538725965717703149748224"},
    {0x1.fffffffffffffp+179, chars_format::fixed, 0, "1532495540865888688217163566681077451931435406299496448"},
    {0x1.fffffffffffffp+180, chars_format::fixed, 0, "3064991081731777376434327133362154903862870812598992896"},
    {0x1.fffffffffffffp+181, chars_format::fixed, 0, "6129982163463554752868654266724309807725741625197985792"},
    {0x1.fffffffffffffp+182, chars_format::fixed, 0, "12259964326927109505737308533448619615451483250395971584"},
    {0x1.fffffffffffffp+183, chars_format::fixed, 0, "24519928653854219011474617066897239230902966500791943168"},
    {0x1.fffffffffffffp+184, chars_format::fixed, 0, "49039857307708438022949234133794478461805933001583886336"},
    {0x1.fffffffffffffp+185, chars_format::fixed, 0, "98079714615416876045898468267588956923611866003167772672"},
    {0x1.fffffffffffffp+186, chars_format::fixed, 0, "196159429230833752091796936535177913847223732006335545344"},
    {0x1.fffffffffffffp+187, chars_format::fixed, 0, "392318858461667504183593873070355827694447464012671090688"},
    {0x1.fffffffffffffp+188, chars_format::fixed, 0, "784637716923335008367187746140711655388894928025342181376"},
    {0x1.fffffffffffffp+189, chars_format::fixed, 0, "1569275433846670016734375492281423310777789856050684362752"},
    {0x1.fffffffffffffp+190, chars_format::fixed, 0, "3138550867693340033468750984562846621555579712101368725504"},
    {0x1.fffffffffffffp+191, chars_format::fixed, 0, "6277101735386680066937501969125693243111159424202737451008"},
    {0x1.fffffffffffffp+192, chars_format::fixed, 0, "12554203470773360133875003938251386486222318848405474902016"},
    {0x1.fffffffffffffp+193, chars_format::fixed, 0, "25108406941546720267750007876502772972444637696810949804032"},
    {0x1.fffffffffffffp+194, chars_format::fixed, 0, "50216813883093440535500015753005545944889275393621899608064"},
    {0x1.fffffffffffffp+195, chars_format::fixed, 0, "100433627766186881071000031506011091889778550787243799216128"},
    {0x1.fffffffffffffp+196, chars_format::fixed, 0, "200867255532373762142000063012022183779557101574487598432256"},
    {0x1.fffffffffffffp+197, chars_format::fixed, 0, "401734511064747524284000126024044367559114203148975196864512"},
    {0x1.fffffffffffffp+198, chars_format::fixed, 0, "803469022129495048568000252048088735118228406297950393729024"},
    {0x1.fffffffffffffp+199, chars_format::fixed, 0, "1606938044258990097136000504096177470236456812595900787458048"},
    {0x1.fffffffffffffp+200, chars_format::fixed, 0, "3213876088517980194272001008192354940472913625191801574916096"},
    {0x1.fffffffffffffp+201, chars_format::fixed, 0, "6427752177035960388544002016384709880945827250383603149832192"},
    {0x1.fffffffffffffp+202, chars_format::fixed, 0, "12855504354071920777088004032769419761891654500767206299664384"},
    {0x1.fffffffffffffp+203, chars_format::fixed, 0, "25711008708143841554176008065538839523783309001534412599328768"},
    {0x1.fffffffffffffp+204, chars_format::fixed, 0, "51422017416287683108352016131077679047566618003068825198657536"},
    {0x1.fffffffffffffp+205, chars_format::fixed, 0, "102844034832575366216704032262155358095133236006137650397315072"},
    {0x1.fffffffffffffp+206, chars_format::fixed, 0, "205688069665150732433408064524310716190266472012275300794630144"},
    {0x1.fffffffffffffp+207, chars_format::fixed, 0, "411376139330301464866816129048621432380532944024550601589260288"},
    {0x1.fffffffffffffp+208, chars_format::fixed, 0, "822752278660602929733632258097242864761065888049101203178520576"},
    {0x1.fffffffffffffp+209, chars_format::fixed, 0,
        "1645504557321205859467264516194485729522131776098202406357041152"},
    {0x1.fffffffffffffp+210, chars_format::fixed, 0,
        "3291009114642411718934529032388971459044263552196404812714082304"},
    {0x1.fffffffffffffp+211, chars_format::fixed, 0,
        "6582018229284823437869058064777942918088527104392809625428164608"},
    {0x1.fffffffffffffp+212, chars_format::fixed, 0,
        "13164036458569646875738116129555885836177054208785619250856329216"},
    {0x1.fffffffffffffp+213, chars_format::fixed, 0,
        "26328072917139293751476232259111771672354108417571238501712658432"},
    {0x1.fffffffffffffp+214, chars_format::fixed, 0,
        "52656145834278587502952464518223543344708216835142477003425316864"},
    {0x1.fffffffffffffp+215, chars_format::fixed, 0,
        "105312291668557175005904929036447086689416433670284954006850633728"},
    {0x1.fffffffffffffp+216, chars_format::fixed, 0,
        "210624583337114350011809858072894173378832867340569908013701267456"},
    {0x1.fffffffffffffp+217, chars_format::fixed, 0,
        "421249166674228700023619716145788346757665734681139816027402534912"},
    {0x1.fffffffffffffp+218, chars_format::fixed, 0,
        "842498333348457400047239432291576693515331469362279632054805069824"},
    {0x1.fffffffffffffp+219, chars_format::fixed, 0,
        "1684996666696914800094478864583153387030662938724559264109610139648"},
    {0x1.fffffffffffffp+220, chars_format::fixed, 0,
        "3369993333393829600188957729166306774061325877449118528219220279296"},
    {0x1.fffffffffffffp+221, chars_format::fixed, 0,
        "6739986666787659200377915458332613548122651754898237056438440558592"},
    {0x1.fffffffffffffp+222, chars_format::fixed, 0,
        "13479973333575318400755830916665227096245303509796474112876881117184"},
    {0x1.fffffffffffffp+223, chars_format::fixed, 0,
        "26959946667150636801511661833330454192490607019592948225753762234368"},
    {0x1.fffffffffffffp+224, chars_format::fixed, 0,
        "53919893334301273603023323666660908384981214039185896451507524468736"},
    {0x1.fffffffffffffp+225, chars_format::fixed, 0,
        "107839786668602547206046647333321816769962428078371792903015048937472"},
    {0x1.fffffffffffffp+226, chars_format::fixed, 0,
        "215679573337205094412093294666643633539924856156743585806030097874944"},
    {0x1.fffffffffffffp+227, chars_format::fixed, 0,
        "431359146674410188824186589333287267079849712313487171612060195749888"},
    {0x1.fffffffffffffp+228, chars_format::fixed, 0,
        "862718293348820377648373178666574534159699424626974343224120391499776"},
    {0x1.fffffffffffffp+229, chars_format::fixed, 0,
        "1725436586697640755296746357333149068319398849253948686448240782999552"},
    {0x1.fffffffffffffp+230, chars_format::fixed, 0,
        "3450873173395281510593492714666298136638797698507897372896481565999104"},
    {0x1.fffffffffffffp+231, chars_format::fixed, 0,
        "6901746346790563021186985429332596273277595397015794745792963131998208"},
    {0x1.fffffffffffffp+232, chars_format::fixed, 0,
        "13803492693581126042373970858665192546555190794031589491585926263996416"},
    {0x1.fffffffffffffp+233, chars_format::fixed, 0,
        "27606985387162252084747941717330385093110381588063178983171852527992832"},
    {0x1.fffffffffffffp+234, chars_format::fixed, 0,
        "55213970774324504169495883434660770186220763176126357966343705055985664"},
    {0x1.fffffffffffffp+235, chars_format::fixed, 0,
        "110427941548649008338991766869321540372441526352252715932687410111971328"},
    {0x1.fffffffffffffp+236, chars_format::fixed, 0,
        "220855883097298016677983533738643080744883052704505431865374820223942656"},
    {0x1.fffffffffffffp+237, chars_format::fixed, 0,
        "441711766194596033355967067477286161489766105409010863730749640447885312"},
    {0x1.fffffffffffffp+238, chars_format::fixed, 0,
        "883423532389192066711934134954572322979532210818021727461499280895770624"},
    {0x1.fffffffffffffp+239, chars_format::fixed, 0,
        "1766847064778384133423868269909144645959064421636043454922998561791541248"},
    {0x1.fffffffffffffp+240, chars_format::fixed, 0,
        "3533694129556768266847736539818289291918128843272086909845997123583082496"},
    {0x1.fffffffffffffp+241, chars_format::fixed, 0,
        "7067388259113536533695473079636578583836257686544173819691994247166164992"},
    {0x1.fffffffffffffp+242, chars_format::fixed, 0,
        "14134776518227073067390946159273157167672515373088347639383988494332329984"},
    {0x1.fffffffffffffp+243, chars_format::fixed, 0,
        "28269553036454146134781892318546314335345030746176695278767976988664659968"},
    {0x1.fffffffffffffp+244, chars_format::fixed, 0,
        "56539106072908292269563784637092628670690061492353390557535953977329319936"},
    {0x1.fffffffffffffp+245, chars_format::fixed, 0,
        "113078212145816584539127569274185257341380122984706781115071907954658639872"},
    {0x1.fffffffffffffp+246, chars_format::fixed, 0,
        "226156424291633169078255138548370514682760245969413562230143815909317279744"},
    {0x1.fffffffffffffp+247, chars_format::fixed, 0,
        "452312848583266338156510277096741029365520491938827124460287631818634559488"},
    {0x1.fffffffffffffp+248, chars_format::fixed, 0,
        "904625697166532676313020554193482058731040983877654248920575263637269118976"},
    {0x1.fffffffffffffp+249, chars_format::fixed, 0,
        "1809251394333065352626041108386964117462081967755308497841150527274538237952"},
    {0x1.fffffffffffffp+250, chars_format::fixed, 0,
        "3618502788666130705252082216773928234924163935510616995682301054549076475904"},
    {0x1.fffffffffffffp+251, chars_format::fixed, 0,
        "7237005577332261410504164433547856469848327871021233991364602109098152951808"},
    {0x1.fffffffffffffp+252, chars_format::fixed, 0,
        "14474011154664522821008328867095712939696655742042467982729204218196305903616"},
    {0x1.fffffffffffffp+253, chars_format::fixed, 0,
        "28948022309329045642016657734191425879393311484084935965458408436392611807232"},
    {0x1.fffffffffffffp+254, chars_format::fixed, 0,
        "57896044618658091284033315468382851758786622968169871930916816872785223614464"},
    {0x1.fffffffffffffp+255, chars_format::fixed, 0,
        "115792089237316182568066630936765703517573245936339743861833633745570447228928"},
    {0x1.fffffffffffffp+256, chars_format::fixed, 0,
        "231584178474632365136133261873531407035146491872679487723667267491140894457856"},
    {0x1.fffffffffffffp+257, chars_format::fixed, 0,
        "463168356949264730272266523747062814070292983745358975447334534982281788915712"},
    {0x1.fffffffffffffp+258, chars_format::fixed, 0,
        "926336713898529460544533047494125628140585967490717950894669069964563577831424"},
    {0x1.fffffffffffffp+259, chars_format::fixed, 0,
        "1852673427797058921089066094988251256281171934981435901789338139929127155662848"},
    {0x1.fffffffffffffp+260, chars_format::fixed, 0,
        "3705346855594117842178132189976502512562343869962871803578676279858254311325696"},
    {0x1.fffffffffffffp+261, chars_format::fixed, 0,
        "7410693711188235684356264379953005025124687739925743607157352559716508622651392"},
    {0x1.fffffffffffffp+262, chars_format::fixed, 0,
        "14821387422376471368712528759906010050249375479851487214314705119433017245302784"},
    {0x1.fffffffffffffp+263, chars_format::fixed, 0,
        "29642774844752942737425057519812020100498750959702974428629410238866034490605568"},
    {0x1.fffffffffffffp+264, chars_format::fixed, 0,
        "59285549689505885474850115039624040200997501919405948857258820477732068981211136"},
    {0x1.fffffffffffffp+265, chars_format::fixed, 0,
        "118571099379011770949700230079248080401995003838811897714517640955464137962422272"},
    {0x1.fffffffffffffp+266, chars_format::fixed, 0,
        "237142198758023541899400460158496160803990007677623795429035281910928275924844544"},
    {0x1.fffffffffffffp+267, chars_format::fixed, 0,
        "474284397516047083798800920316992321607980015355247590858070563821856551849689088"},
    {0x1.fffffffffffffp+268, chars_format::fixed, 0,
        "948568795032094167597601840633984643215960030710495181716141127643713103699378176"},
    {0x1.fffffffffffffp+269, chars_format::fixed, 0,
        "1897137590064188335195203681267969286431920061420990363432282255287426207398756352"},
    {0x1.fffffffffffffp+270, chars_format::fixed, 0,
        "3794275180128376670390407362535938572863840122841980726864564510574852414797512704"},
    {0x1.fffffffffffffp+271, chars_format::fixed, 0,
        "7588550360256753340780814725071877145727680245683961453729129021149704829595025408"},
    {0x1.fffffffffffffp+272, chars_format::fixed, 0,
        "15177100720513506681561629450143754291455360491367922907458258042299409659190050816"},
    {0x1.fffffffffffffp+273, chars_format::fixed, 0,
        "30354201441027013363123258900287508582910720982735845814916516084598819318380101632"},
    {0x1.fffffffffffffp+274, chars_format::fixed, 0,
        "60708402882054026726246517800575017165821441965471691629833032169197638636760203264"},
    {0x1.fffffffffffffp+275, chars_format::fixed, 0,
        "121416805764108053452493035601150034331642883930943383259666064338395277273520406528"},
    {0x1.fffffffffffffp+276, chars_format::fixed, 0,
        "242833611528216106904986071202300068663285767861886766519332128676790554547040813056"},
    {0x1.fffffffffffffp+277, chars_format::fixed, 0,
        "485667223056432213809972142404600137326571535723773533038664257353581109094081626112"},
    {0x1.fffffffffffffp+278, chars_format::fixed, 0,
        "971334446112864427619944284809200274653143071447547066077328514707162218188163252224"},
    {0x1.fffffffffffffp+279, chars_format::fixed, 0,
        "1942668892225728855239888569618400549306286142895094132154657029414324436376326504448"},
    {0x1.fffffffffffffp+280, chars_format::fixed, 0,
        "3885337784451457710479777139236801098612572285790188264309314058828648872752653008896"},
    {0x1.fffffffffffffp+281, chars_format::fixed, 0,
        "7770675568902915420959554278473602197225144571580376528618628117657297745505306017792"},
    {0x1.fffffffffffffp+282, chars_format::fixed, 0,
        "15541351137805830841919108556947204394450289143160753057237256235314595491010612035584"},
    {0x1.fffffffffffffp+283, chars_format::fixed, 0,
        "31082702275611661683838217113894408788900578286321506114474512470629190982021224071168"},
    {0x1.fffffffffffffp+284, chars_format::fixed, 0,
        "62165404551223323367676434227788817577801156572643012228949024941258381964042448142336"},
    {0x1.fffffffffffffp+285, chars_format::fixed, 0,
        "124330809102446646735352868455577635155602313145286024457898049882516763928084896284672"},
    {0x1.fffffffffffffp+286, chars_format::fixed, 0,
        "248661618204893293470705736911155270311204626290572048915796099765033527856169792569344"},
    {0x1.fffffffffffffp+287, chars_format::fixed, 0,
        "497323236409786586941411473822310540622409252581144097831592199530067055712339585138688"},
    {0x1.fffffffffffffp+288, chars_format::fixed, 0,
        "994646472819573173882822947644621081244818505162288195663184399060134111424679170277376"},
    {0x1.fffffffffffffp+289, chars_format::fixed, 0,
        "1989292945639146347765645895289242162489637010324576391326368798120268222849358340554752"},
    {0x1.fffffffffffffp+290, chars_format::fixed, 0,
        "3978585891278292695531291790578484324979274020649152782652737596240536445698716681109504"},
    {0x1.fffffffffffffp+291, chars_format::fixed, 0,
        "7957171782556585391062583581156968649958548041298305565305475192481072891397433362219008"},
    {0x1.fffffffffffffp+292, chars_format::fixed, 0,
        "15914343565113170782125167162313937299917096082596611130610950384962145782794866724438016"},
    {0x1.fffffffffffffp+293, chars_format::fixed, 0,
        "31828687130226341564250334324627874599834192165193222261221900769924291565589733448876032"},
    {0x1.fffffffffffffp+294, chars_format::fixed, 0,
        "63657374260452683128500668649255749199668384330386444522443801539848583131179466897752064"},
    {0x1.fffffffffffffp+295, chars_format::fixed, 0,
        "127314748520905366257001337298511498399336768660772889044887603079697166262358933795504128"},
    {0x1.fffffffffffffp+296, chars_format::fixed, 0,
        "254629497041810732514002674597022996798673537321545778089775206159394332524717867591008256"},
    {0x1.fffffffffffffp+297, chars_format::fixed, 0,
        "509258994083621465028005349194045993597347074643091556179550412318788665049435735182016512"},
    {0x1.fffffffffffffp+298, chars_format::fixed, 0,
        "1018517988167242930056010698388091987194694149286183112359100824637577330098871470364033024"},
    {0x1.fffffffffffffp+299, chars_format::fixed, 0,
        "2037035976334485860112021396776183974389388298572366224718201649275154660197742940728066048"},
    {0x1.fffffffffffffp+300, chars_format::fixed, 0,
        "4074071952668971720224042793552367948778776597144732449436403298550309320395485881456132096"},
    {0x1.fffffffffffffp+301, chars_format::fixed, 0,
        "8148143905337943440448085587104735897557553194289464898872806597100618640790971762912264192"},
    {0x1.fffffffffffffp+302, chars_format::fixed, 0,
        "16296287810675886880896171174209471795115106388578929797745613194201237281581943525824528384"},
    {0x1.fffffffffffffp+303, chars_format::fixed, 0,
        "32592575621351773761792342348418943590230212777157859595491226388402474563163887051649056768"},
    {0x1.fffffffffffffp+304, chars_format::fixed, 0,
        "65185151242703547523584684696837887180460425554315719190982452776804949126327774103298113536"},
    {0x1.fffffffffffffp+305, chars_format::fixed, 0,
        "130370302485407095047169369393675774360920851108631438381964905553609898252655548206596227072"},
    {0x1.fffffffffffffp+306, chars_format::fixed, 0,
        "260740604970814190094338738787351548721841702217262876763929811107219796505311096413192454144"},
    {0x1.fffffffffffffp+307, chars_format::fixed, 0,
        "521481209941628380188677477574703097443683404434525753527859622214439593010622192826384908288"},
    {0x1.fffffffffffffp+308, chars_format::fixed, 0,
        "1042962419883256760377354955149406194887366808869051507055719244428879186021244385652769816576"},
    {0x1.fffffffffffffp+309, chars_format::fixed, 0,
        "2085924839766513520754709910298812389774733617738103014111438488857758372042488771305539633152"},
    {0x1.fffffffffffffp+310, chars_format::fixed, 0,
        "4171849679533027041509419820597624779549467235476206028222876977715516744084977542611079266304"},
    {0x1.fffffffffffffp+311, chars_format::fixed, 0,
        "8343699359066054083018839641195249559098934470952412056445753955431033488169955085222158532608"},
    {0x1.fffffffffffffp+312, chars_format::fixed, 0,
        "16687398718132108166037679282390499118197868941904824112891507910862066976339910170444317065216"},
    {0x1.fffffffffffffp+313, chars_format::fixed, 0,
        "33374797436264216332075358564780998236395737883809648225783015821724133952679820340888634130432"},
    {0x1.fffffffffffffp+314, chars_format::fixed, 0,
        "66749594872528432664150717129561996472791475767619296451566031643448267905359640681777268260864"},
    {0x1.fffffffffffffp+315, chars_format::fixed, 0,
        "133499189745056865328301434259123992945582951535238592903132063286896535810719281363554536521728"},
    {0x1.fffffffffffffp+316, chars_format::fixed, 0,
        "266998379490113730656602868518247985891165903070477185806264126573793071621438562727109073043456"},
    {0x1.fffffffffffffp+317, chars_format::fixed, 0,
        "533996758980227461313205737036495971782331806140954371612528253147586143242877125454218146086912"},
    {0x1.fffffffffffffp+318, chars_format::fixed, 0,
        "1067993517960454922626411474072991943564663612281908743225056506295172286485754250908436292173824"},
    {0x1.fffffffffffffp+319, chars_format::fixed, 0,
        "2135987035920909845252822948145983887129327224563817486450113012590344572971508501816872584347648"},
    {0x1.fffffffffffffp+320, chars_format::fixed, 0,
        "4271974071841819690505645896291967774258654449127634972900226025180689145943017003633745168695296"},
    {0x1.fffffffffffffp+321, chars_format::fixed, 0,
        "8543948143683639381011291792583935548517308898255269945800452050361378291886034007267490337390592"},
    {0x1.fffffffffffffp+322, chars_format::fixed, 0,
        "17087896287367278762022583585167871097034617796510539891600904100722756583772068014534980674781184"},
    {0x1.fffffffffffffp+323, chars_format::fixed, 0,
        "34175792574734557524045167170335742194069235593021079783201808201445513167544136029069961349562368"},
    {0x1.fffffffffffffp+324, chars_format::fixed, 0,
        "68351585149469115048090334340671484388138471186042159566403616402891026335088272058139922699124736"},
    {0x1.fffffffffffffp+325, chars_format::fixed, 0,
        "136703170298938230096180668681342968776276942372084319132807232805782052670176544116279845398249472"},
    {0x1.fffffffffffffp+326, chars_format::fixed, 0,
        "273406340597876460192361337362685937552553884744168638265614465611564105340353088232559690796498944"},
    {0x1.fffffffffffffp+327, chars_format::fixed, 0,
        "546812681195752920384722674725371875105107769488337276531228931223128210680706176465119381592997888"},
    {0x1.fffffffffffffp+328, chars_format::fixed, 0,
        "1093625362391505840769445349450743750210215538976674553062457862446256421361412352930238763185995776"},
    {0x1.fffffffffffffp+329, chars_format::fixed, 0,
        "2187250724783011681538890698901487500420431077953349106124915724892512842722824705860477526371991552"},
    {0x1.fffffffffffffp+330, chars_format::fixed, 0,
        "4374501449566023363077781397802975000840862155906698212249831449785025685445649411720955052743983104"},
    {0x1.fffffffffffffp+331, chars_format::fixed, 0,
        "8749002899132046726155562795605950001681724311813396424499662899570051370891298823441910105487966208"},
    {0x1.fffffffffffffp+332, chars_format::fixed, 0,
        "17498005798264093452311125591211900003363448623626792848999325799140102741782597646883820210975932416"},
    {0x1.fffffffffffffp+333, chars_format::fixed, 0,
        "34996011596528186904622251182423800006726897247253585697998651598280205483565195293767640421951864832"},
    {0x1.fffffffffffffp+334, chars_format::fixed, 0,
        "69992023193056373809244502364847600013453794494507171395997303196560410967130390587535280843903729664"},
    {0x1.fffffffffffffp+335, chars_format::fixed, 0,
        "139984046386112747618489004729695200026907588989014342791994606393120821934260781175070561687807459328"},
    {0x1.fffffffffffffp+336, chars_format::fixed, 0,
        "279968092772225495236978009459390400053815177978028685583989212786241643868521562350141123375614918656"},
    {0x1.fffffffffffffp+337, chars_format::fixed, 0,
        "559936185544450990473956018918780800107630355956057371167978425572483287737043124700282246751229837312"},
    {0x1.fffffffffffffp+338, chars_format::fixed, 0,
        "1119872371088901980947912037837561600215260711912114742335956851144966575474086249400564493502459674624"},
    {0x1.fffffffffffffp+339, chars_format::fixed, 0,
        "2239744742177803961895824075675123200430521423824229484671913702289933150948172498801128987004919349248"},
    {0x1.fffffffffffffp+340, chars_format::fixed, 0,
        "4479489484355607923791648151350246400861042847648458969343827404579866301896344997602257974009838698496"},
    {0x1.fffffffffffffp+341, chars_format::fixed, 0,
        "8958978968711215847583296302700492801722085695296917938687654809159732603792689995204515948019677396992"},
    {0x1.fffffffffffffp+342, chars_format::fixed, 0,
        "17917957937422431695166592605400985603444171390593835877375309618319465207585379990409031896039354793984"},
    {0x1.fffffffffffffp+343, chars_format::fixed, 0,
        "35835915874844863390333185210801971206888342781187671754750619236638930415170759980818063792078709587968"},
    {0x1.fffffffffffffp+344, chars_format::fixed, 0,
        "71671831749689726780666370421603942413776685562375343509501238473277860830341519961636127584157419175936"},
    {0x1.fffffffffffffp+345, chars_format::fixed, 0,
        "14334366349937945356133274084320788482755337112475068701900247694655572166068303992327225516831483835187"
        "2"},
    {0x1.fffffffffffffp+346, chars_format::fixed, 0,
        "28668732699875890712266548168641576965510674224950137403800495389311144332136607984654451033662967670374"
        "4"},
    {0x1.fffffffffffffp+347, chars_format::fixed, 0,
        "57337465399751781424533096337283153931021348449900274807600990778622288664273215969308902067325935340748"
        "8"},
    {0x1.fffffffffffffp+348, chars_format::fixed, 0,
        "114674930799503562849066192674566307862042696899800549615201981557244577328546431938617804134651870681497"
        "6"},
    {0x1.fffffffffffffp+349, chars_format::fixed, 0,
        "229349861599007125698132385349132615724085393799601099230403963114489154657092863877235608269303741362995"
        "2"},
    {0x1.fffffffffffffp+350, chars_format::fixed, 0,
        "458699723198014251396264770698265231448170787599202198460807926228978309314185727754471216538607482725990"
        "4"},
    {0x1.fffffffffffffp+351, chars_format::fixed, 0,
        "917399446396028502792529541396530462896341575198404396921615852457956618628371455508942433077214965451980"
        "8"},
    {0x1.fffffffffffffp+352, chars_format::fixed, 0,
        "1834798892792057005585059082793060925792683150396808793843231704915913237256742911017884866154429930903961"
        "6"},
    {0x1.fffffffffffffp+353, chars_format::fixed, 0,
        "3669597785584114011170118165586121851585366300793617587686463409831826474513485822035769732308859861807923"
        "2"},
    {0x1.fffffffffffffp+354, chars_format::fixed, 0,
        "7339195571168228022340236331172243703170732601587235175372926819663652949026971644071539464617719723615846"
        "4"},
    {0x1.fffffffffffffp+355, chars_format::fixed, 0,
        "1467839114233645604468047266234448740634146520317447035074585363932730589805394328814307892923543944723169"
        "28"},
    {0x1.fffffffffffffp+356, chars_format::fixed, 0,
        "2935678228467291208936094532468897481268293040634894070149170727865461179610788657628615785847087889446338"
        "56"},
    {0x1.fffffffffffffp+357, chars_format::fixed, 0,
        "5871356456934582417872189064937794962536586081269788140298341455730922359221577315257231571694175778892677"
        "12"},
    {0x1.fffffffffffffp+358, chars_format::fixed, 0,
        "1174271291386916483574437812987558992507317216253957628059668291146184471844315463051446314338835155778535"
        "424"},
    {0x1.fffffffffffffp+359, chars_format::fixed, 0,
        "2348542582773832967148875625975117985014634432507915256119336582292368943688630926102892628677670311557070"
        "848"},
    {0x1.fffffffffffffp+360, chars_format::fixed, 0,
        "4697085165547665934297751251950235970029268865015830512238673164584737887377261852205785257355340623114141"
        "696"},
    {0x1.fffffffffffffp+361, chars_format::fixed, 0,
        "9394170331095331868595502503900471940058537730031661024477346329169475774754523704411570514710681246228283"
        "392"},
    {0x1.fffffffffffffp+362, chars_format::fixed, 0,
        "1878834066219066373719100500780094388011707546006332204895469265833895154950904740882314102942136249245656"
        "6784"},
    {0x1.fffffffffffffp+363, chars_format::fixed, 0,
        "3757668132438132747438201001560188776023415092012664409790938531667790309901809481764628205884272498491313"
        "3568"},
    {0x1.fffffffffffffp+364, chars_format::fixed, 0,
        "7515336264876265494876402003120377552046830184025328819581877063335580619803618963529256411768544996982626"
        "7136"},
    {0x1.fffffffffffffp+365, chars_format::fixed, 0,
        "1503067252975253098975280400624075510409366036805065763916375412667116123960723792705851282353708999396525"
        "34272"},
    {0x1.fffffffffffffp+366, chars_format::fixed, 0,
        "3006134505950506197950560801248151020818732073610131527832750825334232247921447585411702564707417998793050"
        "68544"},
    {0x1.fffffffffffffp+367, chars_format::fixed, 0,
        "6012269011901012395901121602496302041637464147220263055665501650668464495842895170823405129414835997586101"
        "37088"},
    {0x1.fffffffffffffp+368, chars_format::fixed, 0,
        "1202453802380202479180224320499260408327492829444052611133100330133692899168579034164681025882967199517220"
        "274176"},
    {0x1.fffffffffffffp+369, chars_format::fixed, 0,
        "2404907604760404958360448640998520816654985658888105222266200660267385798337158068329362051765934399034440"
        "548352"},
    {0x1.fffffffffffffp+370, chars_format::fixed, 0,
        "4809815209520809916720897281997041633309971317776210444532401320534771596674316136658724103531868798068881"
        "096704"},
    {0x1.fffffffffffffp+371, chars_format::fixed, 0,
        "9619630419041619833441794563994083266619942635552420889064802641069543193348632273317448207063737596137762"
        "193408"},
    {0x1.fffffffffffffp+372, chars_format::fixed, 0,
        "1923926083808323966688358912798816653323988527110484177812960528213908638669726454663489641412747519227552"
        "4386816"},
    {0x1.fffffffffffffp+373, chars_format::fixed, 0,
        "3847852167616647933376717825597633306647977054220968355625921056427817277339452909326979282825495038455104"
        "8773632"},
    {0x1.fffffffffffffp+374, chars_format::fixed, 0,
        "7695704335233295866753435651195266613295954108441936711251842112855634554678905818653958565650990076910209"
        "7547264"},
    {0x1.fffffffffffffp+375, chars_format::fixed, 0,
        "1539140867046659173350687130239053322659190821688387342250368422571126910935781163730791713130198015382041"
        "95094528"},
    {0x1.fffffffffffffp+376, chars_format::fixed, 0,
        "3078281734093318346701374260478106645318381643376774684500736845142253821871562327461583426260396030764083"
        "90189056"},
    {0x1.fffffffffffffp+377, chars_format::fixed, 0,
        "6156563468186636693402748520956213290636763286753549369001473690284507643743124654923166852520792061528167"
        "80378112"},
    {0x1.fffffffffffffp+378, chars_format::fixed, 0,
        "1231312693637327338680549704191242658127352657350709873800294738056901528748624930984633370504158412305633"
        "560756224"},
    {0x1.fffffffffffffp+379, chars_format::fixed, 0,
        "2462625387274654677361099408382485316254705314701419747600589476113803057497249861969266741008316824611267"
        "121512448"},
    {0x1.fffffffffffffp+380, chars_format::fixed, 0,
        "4925250774549309354722198816764970632509410629402839495201178952227606114994499723938533482016633649222534"
        "243024896"},
    {0x1.fffffffffffffp+381, chars_format::fixed, 0,
        "9850501549098618709444397633529941265018821258805678990402357904455212229988999447877066964033267298445068"
        "486049792"},
    {0x1.fffffffffffffp+382, chars_format::fixed, 0,
        "1970100309819723741888879526705988253003764251761135798080471580891042445997799889575413392806653459689013"
        "6972099584"},
    {0x1.fffffffffffffp+383, chars_format::fixed, 0,
        "3940200619639447483777759053411976506007528503522271596160943161782084891995599779150826785613306919378027"
        "3944199168"},
    {0x1.fffffffffffffp+384, chars_format::fixed, 0,
        "7880401239278894967555518106823953012015057007044543192321886323564169783991199558301653571226613838756054"
        "7888398336"},
    {0x1.fffffffffffffp+385, chars_format::fixed, 0,
        "1576080247855778993511103621364790602403011401408908638464377264712833956798239911660330714245322767751210"
        "95776796672"},
    {0x1.fffffffffffffp+386, chars_format::fixed, 0,
        "3152160495711557987022207242729581204806022802817817276928754529425667913596479823320661428490645535502421"
        "91553593344"},
    {0x1.fffffffffffffp+387, chars_format::fixed, 0,
        "6304320991423115974044414485459162409612045605635634553857509058851335827192959646641322856981291071004843"
        "83107186688"},
    {0x1.fffffffffffffp+388, chars_format::fixed, 0,
        "1260864198284623194808882897091832481922409121127126910771501811770267165438591929328264571396258214200968"
        "766214373376"},
    {0x1.fffffffffffffp+389, chars_format::fixed, 0,
        "2521728396569246389617765794183664963844818242254253821543003623540534330877183858656529142792516428401937"
        "532428746752"},
    {0x1.fffffffffffffp+390, chars_format::fixed, 0,
        "5043456793138492779235531588367329927689636484508507643086007247081068661754367717313058285585032856803875"
        "064857493504"},
    {0x1.fffffffffffffp+391, chars_format::fixed, 0,
        "1008691358627698555847106317673465985537927296901701528617201449416213732350873543462611657117006571360775"
        "0129714987008"},
    {0x1.fffffffffffffp+392, chars_format::fixed, 0,
        "2017382717255397111694212635346931971075854593803403057234402898832427464701747086925223314234013142721550"
        "0259429974016"},
    {0x1.fffffffffffffp+393, chars_format::fixed, 0,
        "4034765434510794223388425270693863942151709187606806114468805797664854929403494173850446628468026285443100"
        "0518859948032"},
    {0x1.fffffffffffffp+394, chars_format::fixed, 0,
        "8069530869021588446776850541387727884303418375213612228937611595329709858806988347700893256936052570886200"
        "1037719896064"},
    {0x1.fffffffffffffp+395, chars_format::fixed, 0,
        "1613906173804317689355370108277545576860683675042722445787522319065941971761397669540178651387210514177240"
        "02075439792128"},
    {0x1.fffffffffffffp+396, chars_format::fixed, 0,
        "3227812347608635378710740216555091153721367350085444891575044638131883943522795339080357302774421028354480"
        "04150879584256"},
    {0x1.fffffffffffffp+397, chars_format::fixed, 0,
        "6455624695217270757421480433110182307442734700170889783150089276263767887045590678160714605548842056708960"
        "08301759168512"},
    {0x1.fffffffffffffp+398, chars_format::fixed, 0,
        "1291124939043454151484296086622036461488546940034177956630017855252753577409118135632142921109768411341792"
        "016603518337024"},
    {0x1.fffffffffffffp+399, chars_format::fixed, 0,
        "2582249878086908302968592173244072922977093880068355913260035710505507154818236271264285842219536822683584"
        "033207036674048"},
    {0x1.fffffffffffffp+400, chars_format::fixed, 0,
        "5164499756173816605937184346488145845954187760136711826520071421011014309636472542528571684439073645367168"
        "066414073348096"},
    {0x1.fffffffffffffp+401, chars_format::fixed, 0,
        "1032899951234763321187436869297629169190837552027342365304014284202202861927294508505714336887814729073433"
        "6132828146696192"},
    {0x1.fffffffffffffp+402, chars_format::fixed, 0,
        "2065799902469526642374873738595258338381675104054684730608028568404405723854589017011428673775629458146867"
        "2265656293392384"},
    {0x1.fffffffffffffp+403, chars_format::fixed, 0,
        "4131599804939053284749747477190516676763350208109369461216057136808811447709178034022857347551258916293734"
        "4531312586784768"},
    {0x1.fffffffffffffp+404, chars_format::fixed, 0,
        "8263199609878106569499494954381033353526700416218738922432114273617622895418356068045714695102517832587468"
        "9062625173569536"},
    {0x1.fffffffffffffp+405, chars_format::fixed, 0,
        "1652639921975621313899898990876206670705340083243747784486422854723524579083671213609142939020503566517493"
        "78125250347139072"},
    {0x1.fffffffffffffp+406, chars_format::fixed, 0,
        "3305279843951242627799797981752413341410680166487495568972845709447049158167342427218285878041007133034987"
        "56250500694278144"},
    {0x1.fffffffffffffp+407, chars_format::fixed, 0,
        "6610559687902485255599595963504826682821360332974991137945691418894098316334684854436571756082014266069975"
        "12501001388556288"},
    {0x1.fffffffffffffp+408, chars_format::fixed, 0,
        "1322111937580497051119919192700965336564272066594998227589138283778819663266936970887314351216402853213995"
        "025002002777112576"},
    {0x1.fffffffffffffp+409, chars_format::fixed, 0,
        "2644223875160994102239838385401930673128544133189996455178276567557639326533873941774628702432805706427990"
        "050004005554225152"},
    {0x1.fffffffffffffp+410, chars_format::fixed, 0,
        "5288447750321988204479676770803861346257088266379992910356553135115278653067747883549257404865611412855980"
        "100008011108450304"},
    {0x1.fffffffffffffp+411, chars_format::fixed, 0,
        "1057689550064397640895935354160772269251417653275998582071310627023055730613549576709851480973122282571196"
        "0200016022216900608"},
    {0x1.fffffffffffffp+412, chars_format::fixed, 0,
        "2115379100128795281791870708321544538502835306551997164142621254046111461227099153419702961946244565142392"
        "0400032044433801216"},
    {0x1.fffffffffffffp+413, chars_format::fixed, 0,
        "4230758200257590563583741416643089077005670613103994328285242508092222922454198306839405923892489130284784"
        "0800064088867602432"},
    {0x1.fffffffffffffp+414, chars_format::fixed, 0,
        "8461516400515181127167482833286178154011341226207988656570485016184445844908396613678811847784978260569568"
        "1600128177735204864"},
    {0x1.fffffffffffffp+415, chars_format::fixed, 0,
        "1692303280103036225433496566657235630802268245241597731314097003236889168981679322735762369556995652113913"
        "63200256355470409728"},
    {0x1.fffffffffffffp+416, chars_format::fixed, 0,
        "3384606560206072450866993133314471261604536490483195462628194006473778337963358645471524739113991304227827"
        "26400512710940819456"},
    {0x1.fffffffffffffp+417, chars_format::fixed, 0,
        "6769213120412144901733986266628942523209072980966390925256388012947556675926717290943049478227982608455654"
        "52801025421881638912"},
    {0x1.fffffffffffffp+418, chars_format::fixed, 0,
        "1353842624082428980346797253325788504641814596193278185051277602589511335185343458188609895645596521691130"
        "905602050843763277824"},
    {0x1.fffffffffffffp+419, chars_format::fixed, 0,
        "2707685248164857960693594506651577009283629192386556370102555205179022670370686916377219791291193043382261"
        "811204101687526555648"},
    {0x1.fffffffffffffp+420, chars_format::fixed, 0,
        "5415370496329715921387189013303154018567258384773112740205110410358045340741373832754439582582386086764523"
        "622408203375053111296"},
    {0x1.fffffffffffffp+421, chars_format::fixed, 0,
        "1083074099265943184277437802660630803713451676954622548041022082071609068148274766550887916516477217352904"
        "7244816406750106222592"},
    {0x1.fffffffffffffp+422, chars_format::fixed, 0,
        "2166148198531886368554875605321261607426903353909245096082044164143218136296549533101775833032954434705809"
        "4489632813500212445184"},
    {0x1.fffffffffffffp+423, chars_format::fixed, 0,
        "4332296397063772737109751210642523214853806707818490192164088328286436272593099066203551666065908869411618"
        "8979265627000424890368"},
    {0x1.fffffffffffffp+424, chars_format::fixed, 0,
        "8664592794127545474219502421285046429707613415636980384328176656572872545186198132407103332131817738823237"
        "7958531254000849780736"},
    {0x1.fffffffffffffp+425, chars_format::fixed, 0,
        "1732918558825509094843900484257009285941522683127396076865635331314574509037239626481420666426363547764647"
        "55917062508001699561472"},
    {0x1.fffffffffffffp+426, chars_format::fixed, 0,
        "3465837117651018189687800968514018571883045366254792153731270662629149018074479252962841332852727095529295"
        "11834125016003399122944"},
    {0x1.fffffffffffffp+427, chars_format::fixed, 0,
        "6931674235302036379375601937028037143766090732509584307462541325258298036148958505925682665705454191058590"
        "23668250032006798245888"},
    {0x1.fffffffffffffp+428, chars_format::fixed, 0,
        "1386334847060407275875120387405607428753218146501916861492508265051659607229791701185136533141090838211718"
        "047336500064013596491776"},
    {0x1.fffffffffffffp+429, chars_format::fixed, 0,
        "2772669694120814551750240774811214857506436293003833722985016530103319214459583402370273066282181676423436"
        "094673000128027192983552"},
    {0x1.fffffffffffffp+430, chars_format::fixed, 0,
        "5545339388241629103500481549622429715012872586007667445970033060206638428919166804740546132564363352846872"
        "189346000256054385967104"},
    {0x1.fffffffffffffp+431, chars_format::fixed, 0,
        "1109067877648325820700096309924485943002574517201533489194006612041327685783833360948109226512872670569374"
        "4378692000512108771934208"},
    {0x1.fffffffffffffp+432, chars_format::fixed, 0,
        "2218135755296651641400192619848971886005149034403066978388013224082655371567666721896218453025745341138748"
        "8757384001024217543868416"},
    {0x1.fffffffffffffp+433, chars_format::fixed, 0,
        "4436271510593303282800385239697943772010298068806133956776026448165310743135333443792436906051490682277497"
        "7514768002048435087736832"},
    {0x1.fffffffffffffp+434, chars_format::fixed, 0,
        "8872543021186606565600770479395887544020596137612267913552052896330621486270666887584873812102981364554995"
        "5029536004096870175473664"},
    {0x1.fffffffffffffp+435, chars_format::fixed, 0,
        "1774508604237321313120154095879177508804119227522453582710410579266124297254133377516974762420596272910999"
        "10059072008193740350947328"},
    {0x1.fffffffffffffp+436, chars_format::fixed, 0,
        "3549017208474642626240308191758355017608238455044907165420821158532248594508266755033949524841192545821998"
        "20118144016387480701894656"},
    {0x1.fffffffffffffp+437, chars_format::fixed, 0,
        "7098034416949285252480616383516710035216476910089814330841642317064497189016533510067899049682385091643996"
        "40236288032774961403789312"},
    {0x1.fffffffffffffp+438, chars_format::fixed, 0,
        "1419606883389857050496123276703342007043295382017962866168328463412899437803306702013579809936477018328799"
        "280472576065549922807578624"},
    {0x1.fffffffffffffp+439, chars_format::fixed, 0,
        "2839213766779714100992246553406684014086590764035925732336656926825798875606613404027159619872954036657598"
        "560945152131099845615157248"},
    {0x1.fffffffffffffp+440, chars_format::fixed, 0,
        "5678427533559428201984493106813368028173181528071851464673313853651597751213226808054319239745908073315197"
        "121890304262199691230314496"},
    {0x1.fffffffffffffp+441, chars_format::fixed, 0,
        "1135685506711885640396898621362673605634636305614370292934662770730319550242645361610863847949181614663039"
        "4243780608524399382460628992"},
    {0x1.fffffffffffffp+442, chars_format::fixed, 0,
        "2271371013423771280793797242725347211269272611228740585869325541460639100485290723221727695898363229326078"
        "8487561217048798764921257984"},
    {0x1.fffffffffffffp+443, chars_format::fixed, 0,
        "4542742026847542561587594485450694422538545222457481171738651082921278200970581446443455391796726458652157"
        "6975122434097597529842515968"},
    {0x1.fffffffffffffp+444, chars_format::fixed, 0,
        "9085484053695085123175188970901388845077090444914962343477302165842556401941162892886910783593452917304315"
        "3950244868195195059685031936"},
    {0x1.fffffffffffffp+445, chars_format::fixed, 0,
        "1817096810739017024635037794180277769015418088982992468695460433168511280388232578577382156718690583460863"
        "07900489736390390119370063872"},
    {0x1.fffffffffffffp+446, chars_format::fixed, 0,
        "3634193621478034049270075588360555538030836177965984937390920866337022560776465157154764313437381166921726"
        "15800979472780780238740127744"},
    {0x1.fffffffffffffp+447, chars_format::fixed, 0,
        "7268387242956068098540151176721111076061672355931969874781841732674045121552930314309528626874762333843452"
        "31601958945561560477480255488"},
    {0x1.fffffffffffffp+448, chars_format::fixed, 0,
        "1453677448591213619708030235344222215212334471186393974956368346534809024310586062861905725374952466768690"
        "463203917891123120954960510976"},
    {0x1.fffffffffffffp+449, chars_format::fixed, 0,
        "2907354897182427239416060470688444430424668942372787949912736693069618048621172125723811450749904933537380"
        "926407835782246241909921021952"},
    {0x1.fffffffffffffp+450, chars_format::fixed, 0,
        "5814709794364854478832120941376888860849337884745575899825473386139236097242344251447622901499809867074761"
        "852815671564492483819842043904"},
    {0x1.fffffffffffffp+451, chars_format::fixed, 0,
        "1162941958872970895766424188275377772169867576949115179965094677227847219448468850289524580299961973414952"
        "3705631343128984967639684087808"},
    {0x1.fffffffffffffp+452, chars_format::fixed, 0,
        "2325883917745941791532848376550755544339735153898230359930189354455694438896937700579049160599923946829904"
        "7411262686257969935279368175616"},
    {0x1.fffffffffffffp+453, chars_format::fixed, 0,
        "4651767835491883583065696753101511088679470307796460719860378708911388877793875401158098321199847893659809"
        "4822525372515939870558736351232"},
    {0x1.fffffffffffffp+454, chars_format::fixed, 0,
        "9303535670983767166131393506203022177358940615592921439720757417822777755587750802316196642399695787319618"
        "9645050745031879741117472702464"},
    {0x1.fffffffffffffp+455, chars_format::fixed, 0,
        "1860707134196753433226278701240604435471788123118584287944151483564555551117550160463239328479939157463923"
        "79290101490063759482234945404928"},
    {0x1.fffffffffffffp+456, chars_format::fixed, 0,
        "3721414268393506866452557402481208870943576246237168575888302967129111102235100320926478656959878314927847"
        "58580202980127518964469890809856"},
    {0x1.fffffffffffffp+457, chars_format::fixed, 0,
        "7442828536787013732905114804962417741887152492474337151776605934258222204470200641852957313919756629855695"
        "17160405960255037928939781619712"},
    {0x1.fffffffffffffp+458, chars_format::fixed, 0,
        "1488565707357402746581022960992483548377430498494867430355321186851644440894040128370591462783951325971139"
        "034320811920510075857879563239424"},
    {0x1.fffffffffffffp+459, chars_format::fixed, 0,
        "2977131414714805493162045921984967096754860996989734860710642373703288881788080256741182925567902651942278"
        "068641623841020151715759126478848"},
    {0x1.fffffffffffffp+460, chars_format::fixed, 0,
        "5954262829429610986324091843969934193509721993979469721421284747406577763576160513482365851135805303884556"
        "137283247682040303431518252957696"},
    {0x1.fffffffffffffp+461, chars_format::fixed, 0,
        "1190852565885922197264818368793986838701944398795893944284256949481315552715232102696473170227161060776911"
        "2274566495364080606863036505915392"},
    {0x1.fffffffffffffp+462, chars_format::fixed, 0,
        "2381705131771844394529636737587973677403888797591787888568513898962631105430464205392946340454322121553822"
        "4549132990728161213726073011830784"},
    {0x1.fffffffffffffp+463, chars_format::fixed, 0,
        "4763410263543688789059273475175947354807777595183575777137027797925262210860928410785892680908644243107644"
        "9098265981456322427452146023661568"},
    {0x1.fffffffffffffp+464, chars_format::fixed, 0,
        "9526820527087377578118546950351894709615555190367151554274055595850524421721856821571785361817288486215289"
        "8196531962912644854904292047323136"},
    {0x1.fffffffffffffp+465, chars_format::fixed, 0,
        "1905364105417475515623709390070378941923111038073430310854811119170104884344371364314357072363457697243057"
        "96393063925825289709808584094646272"},
    {0x1.fffffffffffffp+466, chars_format::fixed, 0,
        "3810728210834951031247418780140757883846222076146860621709622238340209768688742728628714144726915394486115"
        "92786127851650579419617168189292544"},
    {0x1.fffffffffffffp+467, chars_format::fixed, 0,
        "7621456421669902062494837560281515767692444152293721243419244476680419537377485457257428289453830788972231"
        "85572255703301158839234336378585088"},
    {0x1.fffffffffffffp+468, chars_format::fixed, 0,
        "1524291284333980412498967512056303153538488830458744248683848895336083907475497091451485657890766157794446"
        "371144511406602317678468672757170176"},
    {0x1.fffffffffffffp+469, chars_format::fixed, 0,
        "3048582568667960824997935024112606307076977660917488497367697790672167814950994182902971315781532315588892"
        "742289022813204635356937345514340352"},
    {0x1.fffffffffffffp+470, chars_format::fixed, 0,
        "6097165137335921649995870048225212614153955321834976994735395581344335629901988365805942631563064631177785"
        "484578045626409270713874691028680704"},
    {0x1.fffffffffffffp+471, chars_format::fixed, 0,
        "1219433027467184329999174009645042522830791064366995398947079116268867125980397673161188526312612926235557"
        "0969156091252818541427749382057361408"},
    {0x1.fffffffffffffp+472, chars_format::fixed, 0,
        "2438866054934368659998348019290085045661582128733990797894158232537734251960795346322377052625225852471114"
        "1938312182505637082855498764114722816"},
    {0x1.fffffffffffffp+473, chars_format::fixed, 0,
        "4877732109868737319996696038580170091323164257467981595788316465075468503921590692644754105250451704942228"
        "3876624365011274165710997528229445632"},
    {0x1.fffffffffffffp+474, chars_format::fixed, 0,
        "9755464219737474639993392077160340182646328514935963191576632930150937007843181385289508210500903409884456"
        "7753248730022548331421995056458891264"},
    {0x1.fffffffffffffp+475, chars_format::fixed, 0,
        "1951092843947494927998678415432068036529265702987192638315326586030187401568636277057901642100180681976891"
        "35506497460045096662843990112917782528"},
    {0x1.fffffffffffffp+476, chars_format::fixed, 0,
        "3902185687894989855997356830864136073058531405974385276630653172060374803137272554115803284200361363953782"
        "71012994920090193325687980225835565056"},
    {0x1.fffffffffffffp+477, chars_format::fixed, 0,
        "7804371375789979711994713661728272146117062811948770553261306344120749606274545108231606568400722727907565"
        "42025989840180386651375960451671130112"},
    {0x1.fffffffffffffp+478, chars_format::fixed, 0,
        "1560874275157995942398942732345654429223412562389754110652261268824149921254909021646321313680144545581513"
        "084051979680360773302751920903342260224"},
    {0x1.fffffffffffffp+479, chars_format::fixed, 0,
        "3121748550315991884797885464691308858446825124779508221304522537648299842509818043292642627360289091163026"
        "168103959360721546605503841806684520448"},
    {0x1.fffffffffffffp+480, chars_format::fixed, 0,
        "6243497100631983769595770929382617716893650249559016442609045075296599685019636086585285254720578182326052"
        "336207918721443093211007683613369040896"},
    {0x1.fffffffffffffp+481, chars_format::fixed, 0,
        "1248699420126396753919154185876523543378730049911803288521809015059319937003927217317057050944115636465210"
        "4672415837442886186422015367226738081792"},
    {0x1.fffffffffffffp+482, chars_format::fixed, 0,
        "2497398840252793507838308371753047086757460099823606577043618030118639874007854434634114101888231272930420"
        "9344831674885772372844030734453476163584"},
    {0x1.fffffffffffffp+483, chars_format::fixed, 0,
        "4994797680505587015676616743506094173514920199647213154087236060237279748015708869268228203776462545860841"
        "8689663349771544745688061468906952327168"},
    {0x1.fffffffffffffp+484, chars_format::fixed, 0,
        "9989595361011174031353233487012188347029840399294426308174472120474559496031417738536456407552925091721683"
        "7379326699543089491376122937813904654336"},
    {0x1.fffffffffffffp+485, chars_format::fixed, 0,
        "1997919072202234806270646697402437669405968079858885261634894424094911899206283547707291281510585018344336"
        "74758653399086178982752245875627809308672"},
    {0x1.fffffffffffffp+486, chars_format::fixed, 0,
        "3995838144404469612541293394804875338811936159717770523269788848189823798412567095414582563021170036688673"
        "49517306798172357965504491751255618617344"},
    {0x1.fffffffffffffp+487, chars_format::fixed, 0,
        "7991676288808939225082586789609750677623872319435541046539577696379647596825134190829165126042340073377346"
        "99034613596344715931008983502511237234688"},
    {0x1.fffffffffffffp+488, chars_format::fixed, 0,
        "1598335257761787845016517357921950135524774463887108209307915539275929519365026838165833025208468014675469"
        "398069227192689431862017967005022474469376"},
    {0x1.fffffffffffffp+489, chars_format::fixed, 0,
        "3196670515523575690033034715843900271049548927774216418615831078551859038730053676331666050416936029350938"
        "796138454385378863724035934010044948938752"},
    {0x1.fffffffffffffp+490, chars_format::fixed, 0,
        "6393341031047151380066069431687800542099097855548432837231662157103718077460107352663332100833872058701877"
        "592276908770757727448071868020089897877504"},
    {0x1.fffffffffffffp+491, chars_format::fixed, 0,
        "1278668206209430276013213886337560108419819571109686567446332431420743615492021470532666420166774411740375"
        "5184553817541515454896143736040179795755008"},
    {0x1.fffffffffffffp+492, chars_format::fixed, 0,
        "2557336412418860552026427772675120216839639142219373134892664862841487230984042941065332840333548823480751"
        "0369107635083030909792287472080359591510016"},
    {0x1.fffffffffffffp+493, chars_format::fixed, 0,
        "5114672824837721104052855545350240433679278284438746269785329725682974461968085882130665680667097646961502"
        "0738215270166061819584574944160719183020032"},
    {0x1.fffffffffffffp+494, chars_format::fixed, 0,
        "1022934564967544220810571109070048086735855656887749253957065945136594892393617176426133136133419529392300"
        "41476430540332123639169149888321438366040064"},
    {0x1.fffffffffffffp+495, chars_format::fixed, 0,
        "2045869129935088441621142218140096173471711313775498507914131890273189784787234352852266272266839058784600"
        "82952861080664247278338299776642876732080128"},
    {0x1.fffffffffffffp+496, chars_format::fixed, 0,
        "4091738259870176883242284436280192346943422627550997015828263780546379569574468705704532544533678117569201"
        "65905722161328494556676599553285753464160256"},
    {0x1.fffffffffffffp+497, chars_format::fixed, 0,
        "8183476519740353766484568872560384693886845255101994031656527561092759139148937411409065089067356235138403"
        "31811444322656989113353199106571506928320512"},
    {0x1.fffffffffffffp+498, chars_format::fixed, 0,
        "1636695303948070753296913774512076938777369051020398806331305512218551827829787482281813017813471247027680"
        "663622888645313978226706398213143013856641024"},
    {0x1.fffffffffffffp+499, chars_format::fixed, 0,
        "3273390607896141506593827549024153877554738102040797612662611024437103655659574964563626035626942494055361"
        "327245777290627956453412796426286027713282048"},
    {0x1.fffffffffffffp+500, chars_format::fixed, 0,
        "6546781215792283013187655098048307755109476204081595225325222048874207311319149929127252071253884988110722"
        "654491554581255912906825592852572055426564096"},
    {0x1.fffffffffffffp+501, chars_format::fixed, 0,
        "1309356243158456602637531019609661551021895240816319045065044409774841462263829985825450414250776997622144"
        "5308983109162511825813651185705144110853128192"},
    {0x1.fffffffffffffp+502, chars_format::fixed, 0,
        "2618712486316913205275062039219323102043790481632638090130088819549682924527659971650900828501553995244289"
        "0617966218325023651627302371410288221706256384"},
    {0x1.fffffffffffffp+503, chars_format::fixed, 0,
        "5237424972633826410550124078438646204087580963265276180260177639099365849055319943301801657003107990488578"
        "1235932436650047303254604742820576443412512768"},
    {0x1.fffffffffffffp+504, chars_format::fixed, 0,
        "1047484994526765282110024815687729240817516192653055236052035527819873169811063988660360331400621598097715"
        "62471864873300094606509209485641152886825025536"},
    {0x1.fffffffffffffp+505, chars_format::fixed, 0,
        "2094969989053530564220049631375458481635032385306110472104071055639746339622127977320720662801243196195431"
        "24943729746600189213018418971282305773650051072"},
    {0x1.fffffffffffffp+506, chars_format::fixed, 0,
        "4189939978107061128440099262750916963270064770612220944208142111279492679244255954641441325602486392390862"
        "49887459493200378426036837942564611547300102144"},
    {0x1.fffffffffffffp+507, chars_format::fixed, 0,
        "8379879956214122256880198525501833926540129541224441888416284222558985358488511909282882651204972784781724"
        "99774918986400756852073675885129223094600204288"},
    {0x1.fffffffffffffp+508, chars_format::fixed, 0,
        "1675975991242824451376039705100366785308025908244888377683256844511797071697702381856576530240994556956344"
        "999549837972801513704147351770258446189200408576"},
    {0x1.fffffffffffffp+509, chars_format::fixed, 0,
        "3351951982485648902752079410200733570616051816489776755366513689023594143395404763713153060481989113912689"
        "999099675945603027408294703540516892378400817152"},
    {0x1.fffffffffffffp+510, chars_format::fixed, 0,
        "6703903964971297805504158820401467141232103632979553510733027378047188286790809527426306120963978227825379"
        "998199351891206054816589407081033784756801634304"},
    {0x1.fffffffffffffp+511, chars_format::fixed, 0,
        "1340780792994259561100831764080293428246420726595910702146605475609437657358161905485261224192795645565075"
        "9996398703782412109633178814162067569513603268608"},
    {0x1.fffffffffffffp+512, chars_format::fixed, 0,
        "2681561585988519122201663528160586856492841453191821404293210951218875314716323810970522448385591291130151"
        "9992797407564824219266357628324135139027206537216"},
    {0x1.fffffffffffffp+513, chars_format::fixed, 0,
        "5363123171977038244403327056321173712985682906383642808586421902437750629432647621941044896771182582260303"
        "9985594815129648438532715256648270278054413074432"},
    {0x1.fffffffffffffp+514, chars_format::fixed, 0,
        "1072624634395407648880665411264234742597136581276728561717284380487550125886529524388208979354236516452060"
        "79971189630259296877065430513296540556108826148864"},
    {0x1.fffffffffffffp+515, chars_format::fixed, 0,
        "2145249268790815297761330822528469485194273162553457123434568760975100251773059048776417958708473032904121"
        "59942379260518593754130861026593081112217652297728"},
    {0x1.fffffffffffffp+516, chars_format::fixed, 0,
        "4290498537581630595522661645056938970388546325106914246869137521950200503546118097552835917416946065808243"
        "19884758521037187508261722053186162224435304595456"},
    {0x1.fffffffffffffp+517, chars_format::fixed, 0,
        "8580997075163261191045323290113877940777092650213828493738275043900401007092236195105671834833892131616486"
        "39769517042074375016523444106372324448870609190912"},
    {0x1.fffffffffffffp+518, chars_format::fixed, 0,
        "1716199415032652238209064658022775588155418530042765698747655008780080201418447239021134366966778426323297"
        "279539034084148750033046888212744648897741218381824"},
    {0x1.fffffffffffffp+519, chars_format::fixed, 0,
        "3432398830065304476418129316045551176310837060085531397495310017560160402836894478042268733933556852646594"
        "559078068168297500066093776425489297795482436763648"},
    {0x1.fffffffffffffp+520, chars_format::fixed, 0,
        "6864797660130608952836258632091102352621674120171062794990620035120320805673788956084537467867113705293189"
        "118156136336595000132187552850978595590964873527296"},
    {0x1.fffffffffffffp+521, chars_format::fixed, 0,
        "1372959532026121790567251726418220470524334824034212558998124007024064161134757791216907493573422741058637"
        "8236312272673190000264375105701957191181929747054592"},
    {0x1.fffffffffffffp+522, chars_format::fixed, 0,
        "2745919064052243581134503452836440941048669648068425117996248014048128322269515582433814987146845482117275"
        "6472624545346380000528750211403914382363859494109184"},
    {0x1.fffffffffffffp+523, chars_format::fixed, 0,
        "5491838128104487162269006905672881882097339296136850235992496028096256644539031164867629974293690964234551"
        "2945249090692760001057500422807828764727718988218368"},
    {0x1.fffffffffffffp+524, chars_format::fixed, 0,
        "1098367625620897432453801381134576376419467859227370047198499205619251328907806232973525994858738192846910"
        "25890498181385520002115000845615657529455437976436736"},
    {0x1.fffffffffffffp+525, chars_format::fixed, 0,
        "2196735251241794864907602762269152752838935718454740094396998411238502657815612465947051989717476385693820"
        "51780996362771040004230001691231315058910875952873472"},
    {0x1.fffffffffffffp+526, chars_format::fixed, 0,
        "4393470502483589729815205524538305505677871436909480188793996822477005315631224931894103979434952771387641"
        "03561992725542080008460003382462630117821751905746944"},
    {0x1.fffffffffffffp+527, chars_format::fixed, 0,
        "8786941004967179459630411049076611011355742873818960377587993644954010631262449863788207958869905542775282"
        "07123985451084160016920006764925260235643503811493888"},
    {0x1.fffffffffffffp+528, chars_format::fixed, 0,
        "1757388200993435891926082209815322202271148574763792075517598728990802126252489972757641591773981108555056"
        "414247970902168320033840013529850520471287007622987776"},
    {0x1.fffffffffffffp+529, chars_format::fixed, 0,
        "3514776401986871783852164419630644404542297149527584151035197457981604252504979945515283183547962217110112"
        "828495941804336640067680027059701040942574015245975552"},
    {0x1.fffffffffffffp+530, chars_format::fixed, 0,
        "7029552803973743567704328839261288809084594299055168302070394915963208505009959891030566367095924434220225"
        "656991883608673280135360054119402081885148030491951104"},
    {0x1.fffffffffffffp+531, chars_format::fixed, 0,
        "1405910560794748713540865767852257761816918859811033660414078983192641701001991978206113273419184886844045"
        "1313983767217346560270720108238804163770296060983902208"},
    {0x1.fffffffffffffp+532, chars_format::fixed, 0,
        "2811821121589497427081731535704515523633837719622067320828157966385283402003983956412226546838369773688090"
        "2627967534434693120541440216477608327540592121967804416"},
    {0x1.fffffffffffffp+533, chars_format::fixed, 0,
        "5623642243178994854163463071409031047267675439244134641656315932770566804007967912824453093676739547376180"
        "5255935068869386241082880432955216655081184243935608832"},
    {0x1.fffffffffffffp+534, chars_format::fixed, 0,
        "1124728448635798970832692614281806209453535087848826928331263186554113360801593582564890618735347909475236"
        "10511870137738772482165760865910433310162368487871217664"},
    {0x1.fffffffffffffp+535, chars_format::fixed, 0,
        "2249456897271597941665385228563612418907070175697653856662526373108226721603187165129781237470695818950472"
        "21023740275477544964331521731820866620324736975742435328"},
    {0x1.fffffffffffffp+536, chars_format::fixed, 0,
        "4498913794543195883330770457127224837814140351395307713325052746216453443206374330259562474941391637900944"
        "42047480550955089928663043463641733240649473951484870656"},
    {0x1.fffffffffffffp+537, chars_format::fixed, 0,
        "8997827589086391766661540914254449675628280702790615426650105492432906886412748660519124949882783275801888"
        "84094961101910179857326086927283466481298947902969741312"},
    {0x1.fffffffffffffp+538, chars_format::fixed, 0,
        "1799565517817278353332308182850889935125656140558123085330021098486581377282549732103824989976556655160377"
        "768189922203820359714652173854566932962597895805939482624"},
    {0x1.fffffffffffffp+539, chars_format::fixed, 0,
        "3599131035634556706664616365701779870251312281116246170660042196973162754565099464207649979953113310320755"
        "536379844407640719429304347709133865925195791611878965248"},
    {0x1.fffffffffffffp+540, chars_format::fixed, 0,
        "7198262071269113413329232731403559740502624562232492341320084393946325509130198928415299959906226620641511"
        "072759688815281438858608695418267731850391583223757930496"},
    {0x1.fffffffffffffp+541, chars_format::fixed, 0,
        "1439652414253822682665846546280711948100524912446498468264016878789265101826039785683059991981245324128302"
        "2145519377630562877717217390836535463700783166447515860992"},
    {0x1.fffffffffffffp+542, chars_format::fixed, 0,
        "2879304828507645365331693092561423896201049824892996936528033757578530203652079571366119983962490648256604"
        "4291038755261125755434434781673070927401566332895031721984"},
    {0x1.fffffffffffffp+543, chars_format::fixed, 0,
        "5758609657015290730663386185122847792402099649785993873056067515157060407304159142732239967924981296513208"
        "8582077510522251510868869563346141854803132665790063443968"},
    {0x1.fffffffffffffp+544, chars_format::fixed, 0,
        "1151721931403058146132677237024569558480419929957198774611213503031412081460831828546447993584996259302641"
        "77164155021044503021737739126692283709606265331580126887936"},
    {0x1.fffffffffffffp+545, chars_format::fixed, 0,
        "2303443862806116292265354474049139116960839859914397549222427006062824162921663657092895987169992518605283"
        "54328310042089006043475478253384567419212530663160253775872"},
    {0x1.fffffffffffffp+546, chars_format::fixed, 0,
        "4606887725612232584530708948098278233921679719828795098444854012125648325843327314185791974339985037210567"
        "08656620084178012086950956506769134838425061326320507551744"},
    {0x1.fffffffffffffp+547, chars_format::fixed, 0,
        "9213775451224465169061417896196556467843359439657590196889708024251296651686654628371583948679970074421134"
        "17313240168356024173901913013538269676850122652641015103488"},
    {0x1.fffffffffffffp+548, chars_format::fixed, 0,
        "1842755090244893033812283579239311293568671887931518039377941604850259330337330925674316789735994014884226"
        "834626480336712048347803826027076539353700245305282030206976"},
    {0x1.fffffffffffffp+549, chars_format::fixed, 0,
        "3685510180489786067624567158478622587137343775863036078755883209700518660674661851348633579471988029768453"
        "669252960673424096695607652054153078707400490610564060413952"},
    {0x1.fffffffffffffp+550, chars_format::fixed, 0,
        "7371020360979572135249134316957245174274687551726072157511766419401037321349323702697267158943976059536907"
        "338505921346848193391215304108306157414800981221128120827904"},
    {0x1.fffffffffffffp+551, chars_format::fixed, 0,
        "1474204072195914427049826863391449034854937510345214431502353283880207464269864740539453431788795211907381"
        "4677011842693696386782430608216612314829601962442256241655808"},
    {0x1.fffffffffffffp+552, chars_format::fixed, 0,
        "2948408144391828854099653726782898069709875020690428863004706567760414928539729481078906863577590423814762"
        "9354023685387392773564861216433224629659203924884512483311616"},
    {0x1.fffffffffffffp+553, chars_format::fixed, 0,
        "5896816288783657708199307453565796139419750041380857726009413135520829857079458962157813727155180847629525"
        "8708047370774785547129722432866449259318407849769024966623232"},
    {0x1.fffffffffffffp+554, chars_format::fixed, 0,
        "1179363257756731541639861490713159227883950008276171545201882627104165971415891792431562745431036169525905"
        "17416094741549571094259444865732898518636815699538049933246464"},
    {0x1.fffffffffffffp+555, chars_format::fixed, 0,
        "2358726515513463083279722981426318455767900016552343090403765254208331942831783584863125490862072339051810"
        "34832189483099142188518889731465797037273631399076099866492928"},
    {0x1.fffffffffffffp+556, chars_format::fixed, 0,
        "4717453031026926166559445962852636911535800033104686180807530508416663885663567169726250981724144678103620"
        "69664378966198284377037779462931594074547262798152199732985856"},
    {0x1.fffffffffffffp+557, chars_format::fixed, 0,
        "9434906062053852333118891925705273823071600066209372361615061016833327771327134339452501963448289356207241"
        "39328757932396568754075558925863188149094525596304399465971712"},
    {0x1.fffffffffffffp+558, chars_format::fixed, 0,
        "1886981212410770466623778385141054764614320013241874472323012203366665554265426867890500392689657871241448"
        "278657515864793137508151117851726376298189051192608798931943424"},
    {0x1.fffffffffffffp+559, chars_format::fixed, 0,
        "3773962424821540933247556770282109529228640026483748944646024406733331108530853735781000785379315742482896"
        "557315031729586275016302235703452752596378102385217597863886848"},
    {0x1.fffffffffffffp+560, chars_format::fixed, 0,
        "7547924849643081866495113540564219058457280052967497889292048813466662217061707471562001570758631484965793"
        "114630063459172550032604471406905505192756204770435195727773696"},
    {0x1.fffffffffffffp+561, chars_format::fixed, 0,
        "1509584969928616373299022708112843811691456010593499577858409762693332443412341494312400314151726296993158"
        "6229260126918345100065208942813811010385512409540870391455547392"},
    {0x1.fffffffffffffp+562, chars_format::fixed, 0,
        "3019169939857232746598045416225687623382912021186999155716819525386664886824682988624800628303452593986317"
        "2458520253836690200130417885627622020771024819081740782911094784"},
    {0x1.fffffffffffffp+563, chars_format::fixed, 0,
        "6038339879714465493196090832451375246765824042373998311433639050773329773649365977249601256606905187972634"
        "4917040507673380400260835771255244041542049638163481565822189568"},
    {0x1.fffffffffffffp+564, chars_format::fixed, 0,
        "1207667975942893098639218166490275049353164808474799662286727810154665954729873195449920251321381037594526"
        "89834081015346760800521671542510488083084099276326963131644379136"},
    {0x1.fffffffffffffp+565, chars_format::fixed, 0,
        "2415335951885786197278436332980550098706329616949599324573455620309331909459746390899840502642762075189053"
        "79668162030693521601043343085020976166168198552653926263288758272"},
    {0x1.fffffffffffffp+566, chars_format::fixed, 0,
        "4830671903771572394556872665961100197412659233899198649146911240618663818919492781799681005285524150378107"
        "59336324061387043202086686170041952332336397105307852526577516544"},
    {0x1.fffffffffffffp+567, chars_format::fixed, 0,
        "9661343807543144789113745331922200394825318467798397298293822481237327637838985563599362010571048300756215"
        "18672648122774086404173372340083904664672794210615705053155033088"},
    {0x1.fffffffffffffp+568, chars_format::fixed, 0,
        "1932268761508628957822749066384440078965063693559679459658764496247465527567797112719872402114209660151243"
        "037345296245548172808346744680167809329345588421231410106310066176"},
    {0x1.fffffffffffffp+569, chars_format::fixed, 0,
        "3864537523017257915645498132768880157930127387119358919317528992494931055135594225439744804228419320302486"
        "074690592491096345616693489360335618658691176842462820212620132352"},
    {0x1.fffffffffffffp+570, chars_format::fixed, 0,
        "7729075046034515831290996265537760315860254774238717838635057984989862110271188450879489608456838640604972"
        "149381184982192691233386978720671237317382353684925640425240264704"},
    {0x1.fffffffffffffp+571, chars_format::fixed, 0,
        "1545815009206903166258199253107552063172050954847743567727011596997972422054237690175897921691367728120994"
        "4298762369964385382466773957441342474634764707369851280850480529408"},
    {0x1.fffffffffffffp+572, chars_format::fixed, 0,
        "3091630018413806332516398506215104126344101909695487135454023193995944844108475380351795843382735456241988"
        "8597524739928770764933547914882684949269529414739702561700961058816"},
    {0x1.fffffffffffffp+573, chars_format::fixed, 0,
        "6183260036827612665032797012430208252688203819390974270908046387991889688216950760703591686765470912483977"
        "7195049479857541529867095829765369898539058829479405123401922117632"},
    {0x1.fffffffffffffp+574, chars_format::fixed, 0,
        "1236652007365522533006559402486041650537640763878194854181609277598377937643390152140718337353094182496795"
        "54390098959715083059734191659530739797078117658958810246803844235264"},
    {0x1.fffffffffffffp+575, chars_format::fixed, 0,
        "2473304014731045066013118804972083301075281527756389708363218555196755875286780304281436674706188364993591"
        "08780197919430166119468383319061479594156235317917620493607688470528"},
    {0x1.fffffffffffffp+576, chars_format::fixed, 0,
        "4946608029462090132026237609944166602150563055512779416726437110393511750573560608562873349412376729987182"
        "17560395838860332238936766638122959188312470635835240987215376941056"},
    {0x1.fffffffffffffp+577, chars_format::fixed, 0,
        "9893216058924180264052475219888333204301126111025558833452874220787023501147121217125746698824753459974364"
        "35120791677720664477873533276245918376624941271670481974430753882112"},
    {0x1.fffffffffffffp+578, chars_format::fixed, 0,
        "1978643211784836052810495043977666640860225222205111766690574844157404700229424243425149339764950691994872"
        "870241583355441328955747066552491836753249882543340963948861507764224"},
    {0x1.fffffffffffffp+579, chars_format::fixed, 0,
        "3957286423569672105620990087955333281720450444410223533381149688314809400458848486850298679529901383989745"
        "740483166710882657911494133104983673506499765086681927897723015528448"},
    {0x1.fffffffffffffp+580, chars_format::fixed, 0,
        "7914572847139344211241980175910666563440900888820447066762299376629618800917696973700597359059802767979491"
        "480966333421765315822988266209967347012999530173363855795446031056896"},
    {0x1.fffffffffffffp+581, chars_format::fixed, 0,
        "1582914569427868842248396035182133312688180177764089413352459875325923760183539394740119471811960553595898"
        "2961932666843530631645976532419934694025999060346727711590892062113792"},
    {0x1.fffffffffffffp+582, chars_format::fixed, 0,
        "3165829138855737684496792070364266625376360355528178826704919750651847520367078789480238943623921107191796"
        "5923865333687061263291953064839869388051998120693455423181784124227584"},
    {0x1.fffffffffffffp+583, chars_format::fixed, 0,
        "6331658277711475368993584140728533250752720711056357653409839501303695040734157578960477887247842214383593"
        "1847730667374122526583906129679738776103996241386910846363568248455168"},
    {0x1.fffffffffffffp+584, chars_format::fixed, 0,
        "1266331655542295073798716828145706650150544142211271530681967900260739008146831515792095577449568442876718"
        "63695461334748245053167812259359477552207992482773821692727136496910336"},
    {0x1.fffffffffffffp+585, chars_format::fixed, 0,
        "2532663311084590147597433656291413300301088284422543061363935800521478016293663031584191154899136885753437"
        "27390922669496490106335624518718955104415984965547643385454272993820672"},
    {0x1.fffffffffffffp+586, chars_format::fixed, 0,
        "5065326622169180295194867312582826600602176568845086122727871601042956032587326063168382309798273771506874"
        "54781845338992980212671249037437910208831969931095286770908545987641344"},
    {0x1.fffffffffffffp+587, chars_format::fixed, 0,
        "1013065324433836059038973462516565320120435313769017224545574320208591206517465212633676461959654754301374"
        "909563690677985960425342498074875820417663939862190573541817091975282688"},
    {0x1.fffffffffffffp+588, chars_format::fixed, 0,
        "2026130648867672118077946925033130640240870627538034449091148640417182413034930425267352923919309508602749"
        "819127381355971920850684996149751640835327879724381147083634183950565376"},
    {0x1.fffffffffffffp+589, chars_format::fixed, 0,
        "4052261297735344236155893850066261280481741255076068898182297280834364826069860850534705847838619017205499"
        "638254762711943841701369992299503281670655759448762294167268367901130752"},
    {0x1.fffffffffffffp+590, chars_format::fixed, 0,
        "8104522595470688472311787700132522560963482510152137796364594561668729652139721701069411695677238034410999"
        "276509525423887683402739984599006563341311518897524588334536735802261504"},
    {0x1.fffffffffffffp+591, chars_format::fixed, 0,
        "1620904519094137694462357540026504512192696502030427559272918912333745930427944340213882339135447606882199"
        "8553019050847775366805479969198013126682623037795049176669073471604523008"},
    {0x1.fffffffffffffp+592, chars_format::fixed, 0,
        "3241809038188275388924715080053009024385393004060855118545837824667491860855888680427764678270895213764399"
        "7106038101695550733610959938396026253365246075590098353338146943209046016"},
    {0x1.fffffffffffffp+593, chars_format::fixed, 0,
        "6483618076376550777849430160106018048770786008121710237091675649334983721711777360855529356541790427528799"
        "4212076203391101467221919876792052506730492151180196706676293886418092032"},
    {0x1.fffffffffffffp+594, chars_format::fixed, 0,
        "1296723615275310155569886032021203609754157201624342047418335129866996744342355472171105871308358085505759"
        "88424152406782202934443839753584105013460984302360393413352587772836184064"},
    {0x1.fffffffffffffp+595, chars_format::fixed, 0,
        "2593447230550620311139772064042407219508314403248684094836670259733993488684710944342211742616716171011519"
        "76848304813564405868887679507168210026921968604720786826705175545672368128"},
    {0x1.fffffffffffffp+596, chars_format::fixed, 0,
        "5186894461101240622279544128084814439016628806497368189673340519467986977369421888684423485233432342023039"
        "53696609627128811737775359014336420053843937209441573653410351091344736256"},
    {0x1.fffffffffffffp+597, chars_format::fixed, 0,
        "1037378892220248124455908825616962887803325761299473637934668103893597395473884377736884697046686468404607"
        "907393219254257623475550718028672840107687874418883147306820702182689472512"},
    {0x1.fffffffffffffp+598, chars_format::fixed, 0,
        "2074757784440496248911817651233925775606651522598947275869336207787194790947768755473769394093372936809215"
        "814786438508515246951101436057345680215375748837766294613641404365378945024"},
    {0x1.fffffffffffffp+599, chars_format::fixed, 0,
        "4149515568880992497823635302467851551213303045197894551738672415574389581895537510947538788186745873618431"
        "629572877017030493902202872114691360430751497675532589227282808730757890048"},
    {0x1.fffffffffffffp+600, chars_format::fixed, 0,
        "8299031137761984995647270604935703102426606090395789103477344831148779163791075021895077576373491747236863"
        "259145754034060987804405744229382720861502995351065178454565617461515780096"},
    {0x1.fffffffffffffp+601, chars_format::fixed, 0,
        "1659806227552396999129454120987140620485321218079157820695468966229755832758215004379015515274698349447372"
        "6518291508068121975608811488458765441723005990702130356909131234923031560192"},
    {0x1.fffffffffffffp+602, chars_format::fixed, 0,
        "3319612455104793998258908241974281240970642436158315641390937932459511665516430008758031030549396698894745"
        "3036583016136243951217622976917530883446011981404260713818262469846063120384"},
    {0x1.fffffffffffffp+603, chars_format::fixed, 0,
        "6639224910209587996517816483948562481941284872316631282781875864919023331032860017516062061098793397789490"
        "6073166032272487902435245953835061766892023962808521427636524939692126240768"},
    {0x1.fffffffffffffp+604, chars_format::fixed, 0,
        "1327844982041917599303563296789712496388256974463326256556375172983804666206572003503212412219758679557898"
        "12146332064544975804870491907670123533784047925617042855273049879384252481536"},
    {0x1.fffffffffffffp+605, chars_format::fixed, 0,
        "2655689964083835198607126593579424992776513948926652513112750345967609332413144007006424824439517359115796"
        "24292664129089951609740983815340247067568095851234085710546099758768504963072"},
    {0x1.fffffffffffffp+606, chars_format::fixed, 0,
        "5311379928167670397214253187158849985553027897853305026225500691935218664826288014012849648879034718231592"
        "48585328258179903219481967630680494135136191702468171421092199517537009926144"},
    {0x1.fffffffffffffp+607, chars_format::fixed, 0,
        "1062275985633534079442850637431769997110605579570661005245100138387043732965257602802569929775806943646318"
        "497170656516359806438963935261360988270272383404936342842184399035074019852288"},
    {0x1.fffffffffffffp+608, chars_format::fixed, 0,
        "2124551971267068158885701274863539994221211159141322010490200276774087465930515205605139859551613887292636"
        "994341313032719612877927870522721976540544766809872685684368798070148039704576"},
    {0x1.fffffffffffffp+609, chars_format::fixed, 0,
        "4249103942534136317771402549727079988442422318282644020980400553548174931861030411210279719103227774585273"
        "988682626065439225755855741045443953081089533619745371368737596140296079409152"},
    {0x1.fffffffffffffp+610, chars_format::fixed, 0,
        "8498207885068272635542805099454159976884844636565288041960801107096349863722060822420559438206455549170547"
        "977365252130878451511711482090887906162179067239490742737475192280592158818304"},
    {0x1.fffffffffffffp+611, chars_format::fixed, 0,
        "1699641577013654527108561019890831995376968927313057608392160221419269972744412164484111887641291109834109"
        "5954730504261756903023422964181775812324358134478981485474950384561184317636608"},
    {0x1.fffffffffffffp+612, chars_format::fixed, 0,
        "3399283154027309054217122039781663990753937854626115216784320442838539945488824328968223775282582219668219"
        "1909461008523513806046845928363551624648716268957962970949900769122368635273216"},
    {0x1.fffffffffffffp+613, chars_format::fixed, 0,
        "6798566308054618108434244079563327981507875709252230433568640885677079890977648657936447550565164439336438"
        "3818922017047027612093691856727103249297432537915925941899801538244737270546432"},
    {0x1.fffffffffffffp+614, chars_format::fixed, 0,
        "1359713261610923621686848815912665596301575141850446086713728177135415978195529731587289510113032887867287"
        "67637844034094055224187383713454206498594865075831851883799603076489474541092864"},
    {0x1.fffffffffffffp+615, chars_format::fixed, 0,
        "2719426523221847243373697631825331192603150283700892173427456354270831956391059463174579020226065775734575"
        "35275688068188110448374767426908412997189730151663703767599206152978949082185728"},
    {0x1.fffffffffffffp+616, chars_format::fixed, 0,
        "5438853046443694486747395263650662385206300567401784346854912708541663912782118926349158040452131551469150"
        "70551376136376220896749534853816825994379460303327407535198412305957898164371456"},
    {0x1.fffffffffffffp+617, chars_format::fixed, 0,
        "1087770609288738897349479052730132477041260113480356869370982541708332782556423785269831608090426310293830"
        "141102752272752441793499069707633651988758920606654815070396824611915796328742912"},
    {0x1.fffffffffffffp+618, chars_format::fixed, 0,
        "2175541218577477794698958105460264954082520226960713738741965083416665565112847570539663216180852620587660"
        "282205504545504883586998139415267303977517841213309630140793649223831592657485824"},
    {0x1.fffffffffffffp+619, chars_format::fixed, 0,
        "4351082437154955589397916210920529908165040453921427477483930166833331130225695141079326432361705241175320"
        "564411009091009767173996278830534607955035682426619260281587298447663185314971648"},
    {0x1.fffffffffffffp+620, chars_format::fixed, 0,
        "8702164874309911178795832421841059816330080907842854954967860333666662260451390282158652864723410482350641"
        "128822018182019534347992557661069215910071364853238520563174596895326370629943296"},
    {0x1.fffffffffffffp+621, chars_format::fixed, 0,
        "1740432974861982235759166484368211963266016181568570990993572066733332452090278056431730572944682096470128"
        "2257644036364039068695985115322138431820142729706477041126349193790652741259886592"},
    {0x1.fffffffffffffp+622, chars_format::fixed, 0,
        "3480865949723964471518332968736423926532032363137141981987144133466664904180556112863461145889364192940256"
        "4515288072728078137391970230644276863640285459412954082252698387581305482519773184"},
    {0x1.fffffffffffffp+623, chars_format::fixed, 0,
        "6961731899447928943036665937472847853064064726274283963974288266933329808361112225726922291778728385880512"
        "9030576145456156274783940461288553727280570918825908164505396775162610965039546368"},
    {0x1.fffffffffffffp+624, chars_format::fixed, 0,
        "1392346379889585788607333187494569570612812945254856792794857653386665961672222445145384458355745677176102"
        "58061152290912312549567880922577107454561141837651816329010793550325221930079092736"},
    {0x1.fffffffffffffp+625, chars_format::fixed, 0,
        "2784692759779171577214666374989139141225625890509713585589715306773331923344444890290768916711491354352205"
        "16122304581824625099135761845154214909122283675303632658021587100650443860158185472"},
    {0x1.fffffffffffffp+626, chars_format::fixed, 0,
        "5569385519558343154429332749978278282451251781019427171179430613546663846688889780581537833422982708704410"
        "32244609163649250198271523690308429818244567350607265316043174201300887720316370944"},
    {0x1.fffffffffffffp+627, chars_format::fixed, 0,
        "1113877103911668630885866549995655656490250356203885434235886122709332769337777956116307566684596541740882"
        "064489218327298500396543047380616859636489134701214530632086348402601775440632741888"},
    {0x1.fffffffffffffp+628, chars_format::fixed, 0,
        "2227754207823337261771733099991311312980500712407770868471772245418665538675555912232615133369193083481764"
        "128978436654597000793086094761233719272978269402429061264172696805203550881265483776"},
    {0x1.fffffffffffffp+629, chars_format::fixed, 0,
        "4455508415646674523543466199982622625961001424815541736943544490837331077351111824465230266738386166963528"
        "257956873309194001586172189522467438545956538804858122528345393610407101762530967552"},
    {0x1.fffffffffffffp+630, chars_format::fixed, 0,
        "8911016831293349047086932399965245251922002849631083473887088981674662154702223648930460533476772333927056"
        "515913746618388003172344379044934877091913077609716245056690787220814203525061935104"},
    {0x1.fffffffffffffp+631, chars_format::fixed, 0,
        "1782203366258669809417386479993049050384400569926216694777417796334932430940444729786092106695354466785411"
        "3031827493236776006344688758089869754183826155219432490113381574441628407050123870208"},
    {0x1.fffffffffffffp+632, chars_format::fixed, 0,
        "3564406732517339618834772959986098100768801139852433389554835592669864861880889459572184213390708933570822"
        "6063654986473552012689377516179739508367652310438864980226763148883256814100247740416"},
    {0x1.fffffffffffffp+633, chars_format::fixed, 0,
        "7128813465034679237669545919972196201537602279704866779109671185339729723761778919144368426781417867141645"
        "2127309972947104025378755032359479016735304620877729960453526297766513628200495480832"},
    {0x1.fffffffffffffp+634, chars_format::fixed, 0,
        "1425762693006935847533909183994439240307520455940973355821934237067945944752355783828873685356283573428329"
        "04254619945894208050757510064718958033470609241755459920907052595533027256400990961664"},
    {0x1.fffffffffffffp+635, chars_format::fixed, 0,
        "2851525386013871695067818367988878480615040911881946711643868474135891889504711567657747370712567146856658"
        "08509239891788416101515020129437916066941218483510919841814105191066054512801981923328"},
    {0x1.fffffffffffffp+636, chars_format::fixed, 0,
        "5703050772027743390135636735977756961230081823763893423287736948271783779009423135315494741425134293713316"
        "17018479783576832203030040258875832133882436967021839683628210382132109025603963846656"},
    {0x1.fffffffffffffp+637, chars_format::fixed, 0,
        "1140610154405548678027127347195551392246016364752778684657547389654356755801884627063098948285026858742663"
        "234036959567153664406060080517751664267764873934043679367256420764264218051207927693312"},
    {0x1.fffffffffffffp+638, chars_format::fixed, 0,
        "2281220308811097356054254694391102784492032729505557369315094779308713511603769254126197896570053717485326"
        "468073919134307328812120161035503328535529747868087358734512841528528436102415855386624"},
    {0x1.fffffffffffffp+639, chars_format::fixed, 0,
        "4562440617622194712108509388782205568984065459011114738630189558617427023207538508252395793140107434970652"
        "936147838268614657624240322071006657071059495736174717469025683057056872204831710773248"},
    {0x1.fffffffffffffp+640, chars_format::fixed, 0,
        "9124881235244389424217018777564411137968130918022229477260379117234854046415077016504791586280214869941305"
        "872295676537229315248480644142013314142118991472349434938051366114113744409663421546496"},
    {0x1.fffffffffffffp+641, chars_format::fixed, 0,
        "1824976247048877884843403755512882227593626183604445895452075823446970809283015403300958317256042973988261"
        "1744591353074458630496961288284026628284237982944698869876102732228227488819326843092992"},
    {0x1.fffffffffffffp+642, chars_format::fixed, 0,
        "3649952494097755769686807511025764455187252367208891790904151646893941618566030806601916634512085947976522"
        "3489182706148917260993922576568053256568475965889397739752205464456454977638653686185984"},
    {0x1.fffffffffffffp+643, chars_format::fixed, 0,
        "7299904988195511539373615022051528910374504734417783581808303293787883237132061613203833269024171895953044"
        "6978365412297834521987845153136106513136951931778795479504410928912909955277307372371968"},
    {0x1.fffffffffffffp+644, chars_format::fixed, 0,
        "1459980997639102307874723004410305782074900946883556716361660658757576647426412322640766653804834379190608"
        "93956730824595669043975690306272213026273903863557590959008821857825819910554614744743936"},
    {0x1.fffffffffffffp+645, chars_format::fixed, 0,
        "2919961995278204615749446008820611564149801893767113432723321317515153294852824645281533307609668758381217"
        "87913461649191338087951380612544426052547807727115181918017643715651639821109229489487872"},
    {0x1.fffffffffffffp+646, chars_format::fixed, 0,
        "5839923990556409231498892017641223128299603787534226865446642635030306589705649290563066615219337516762435"
        "75826923298382676175902761225088852105095615454230363836035287431303279642218458978975744"},
    {0x1.fffffffffffffp+647, chars_format::fixed, 0,
        "1167984798111281846299778403528244625659920757506845373089328527006061317941129858112613323043867503352487"
        "151653846596765352351805522450177704210191230908460727672070574862606559284436917957951488"},
    {0x1.fffffffffffffp+648, chars_format::fixed, 0,
        "2335969596222563692599556807056489251319841515013690746178657054012122635882259716225226646087735006704974"
        "303307693193530704703611044900355408420382461816921455344141149725213118568873835915902976"},
    {0x1.fffffffffffffp+649, chars_format::fixed, 0,
        "4671939192445127385199113614112978502639683030027381492357314108024245271764519432450453292175470013409948"
        "606615386387061409407222089800710816840764923633842910688282299450426237137747671831805952"},
    {0x1.fffffffffffffp+650, chars_format::fixed, 0,
        "9343878384890254770398227228225957005279366060054762984714628216048490543529038864900906584350940026819897"
        "213230772774122818814444179601421633681529847267685821376564598900852474275495343663611904"},
    {0x1.fffffffffffffp+651, chars_format::fixed, 0,
        "1868775676978050954079645445645191401055873212010952596942925643209698108705807772980181316870188005363979"
        "4426461545548245637628888359202843267363059694535371642753129197801704948550990687327223808"},
    {0x1.fffffffffffffp+652, chars_format::fixed, 0,
        "3737551353956101908159290891290382802111746424021905193885851286419396217411615545960362633740376010727958"
        "8852923091096491275257776718405686534726119389070743285506258395603409897101981374654447616"},
    {0x1.fffffffffffffp+653, chars_format::fixed, 0,
        "7475102707912203816318581782580765604223492848043810387771702572838792434823231091920725267480752021455917"
        "7705846182192982550515553436811373069452238778141486571012516791206819794203962749308895232"},
    {0x1.fffffffffffffp+654, chars_format::fixed, 0,
        "1495020541582440763263716356516153120844698569608762077554340514567758486964646218384145053496150404291183"
        "55411692364385965101031106873622746138904477556282973142025033582413639588407925498617790464"},
    {0x1.fffffffffffffp+655, chars_format::fixed, 0,
        "2990041083164881526527432713032306241689397139217524155108681029135516973929292436768290106992300808582367"
        "10823384728771930202062213747245492277808955112565946284050067164827279176815850997235580928"},
    {0x1.fffffffffffffp+656, chars_format::fixed, 0,
        "5980082166329763053054865426064612483378794278435048310217362058271033947858584873536580213984601617164734"
        "21646769457543860404124427494490984555617910225131892568100134329654558353631701994471161856"},
    {0x1.fffffffffffffp+657, chars_format::fixed, 0,
        "1196016433265952610610973085212922496675758855687009662043472411654206789571716974707316042796920323432946"
        "843293538915087720808248854988981969111235820450263785136200268659309116707263403988942323712"},
    {0x1.fffffffffffffp+658, chars_format::fixed, 0,
        "2392032866531905221221946170425844993351517711374019324086944823308413579143433949414632085593840646865893"
        "686587077830175441616497709977963938222471640900527570272400537318618233414526807977884647424"},
    {0x1.fffffffffffffp+659, chars_format::fixed, 0,
        "4784065733063810442443892340851689986703035422748038648173889646616827158286867898829264171187681293731787"
        "373174155660350883232995419955927876444943281801055140544801074637236466829053615955769294848"},
    {0x1.fffffffffffffp+660, chars_format::fixed, 0,
        "9568131466127620884887784681703379973406070845496077296347779293233654316573735797658528342375362587463574"
        "746348311320701766465990839911855752889886563602110281089602149274472933658107231911538589696"},
    {0x1.fffffffffffffp+661, chars_format::fixed, 0,
        "1913626293225524176977556936340675994681214169099215459269555858646730863314747159531705668475072517492714"
        "9492696622641403532931981679823711505779773127204220562179204298548945867316214463823077179392"},
    {0x1.fffffffffffffp+662, chars_format::fixed, 0,
        "3827252586451048353955113872681351989362428338198430918539111717293461726629494319063411336950145034985429"
        "8985393245282807065863963359647423011559546254408441124358408597097891734632428927646154358784"},
    {0x1.fffffffffffffp+663, chars_format::fixed, 0,
        "7654505172902096707910227745362703978724856676396861837078223434586923453258988638126822673900290069970859"
        "7970786490565614131727926719294846023119092508816882248716817194195783469264857855292308717568"},
    {0x1.fffffffffffffp+664, chars_format::fixed, 0,
        "1530901034580419341582045549072540795744971335279372367415644686917384690651797727625364534780058013994171"
        "95941572981131228263455853438589692046238185017633764497433634388391566938529715710584617435136"},
    {0x1.fffffffffffffp+665, chars_format::fixed, 0,
        "3061802069160838683164091098145081591489942670558744734831289373834769381303595455250729069560116027988343"
        "91883145962262456526911706877179384092476370035267528994867268776783133877059431421169234870272"},
    {0x1.fffffffffffffp+666, chars_format::fixed, 0,
        "6123604138321677366328182196290163182979885341117489469662578747669538762607190910501458139120232055976687"
        "83766291924524913053823413754358768184952740070535057989734537553566267754118862842338469740544"},
    {0x1.fffffffffffffp+667, chars_format::fixed, 0,
        "1224720827664335473265636439258032636595977068223497893932515749533907752521438182100291627824046411195337"
        "567532583849049826107646827508717536369905480141070115979469075107132535508237725684676939481088"},
    {0x1.fffffffffffffp+668, chars_format::fixed, 0,
        "2449441655328670946531272878516065273191954136446995787865031499067815505042876364200583255648092822390675"
        "135065167698099652215293655017435072739810960282140231958938150214265071016475451369353878962176"},
    {0x1.fffffffffffffp+669, chars_format::fixed, 0,
        "4898883310657341893062545757032130546383908272893991575730062998135631010085752728401166511296185644781350"
        "270130335396199304430587310034870145479621920564280463917876300428530142032950902738707757924352"},
    {0x1.fffffffffffffp+670, chars_format::fixed, 0,
        "9797766621314683786125091514064261092767816545787983151460125996271262020171505456802333022592371289562700"
        "540260670792398608861174620069740290959243841128560927835752600857060284065901805477415515848704"},
    {0x1.fffffffffffffp+671, chars_format::fixed, 0,
        "1959553324262936757225018302812852218553563309157596630292025199254252404034301091360466604518474257912540"
        "1080521341584797217722349240139480581918487682257121855671505201714120568131803610954831031697408"},
    {0x1.fffffffffffffp+672, chars_format::fixed, 0,
        "3919106648525873514450036605625704437107126618315193260584050398508504808068602182720933209036948515825080"
        "2161042683169594435444698480278961163836975364514243711343010403428241136263607221909662063394816"},
    {0x1.fffffffffffffp+673, chars_format::fixed, 0,
        "7838213297051747028900073211251408874214253236630386521168100797017009616137204365441866418073897031650160"
        "4322085366339188870889396960557922327673950729028487422686020806856482272527214443819324126789632"},
    {0x1.fffffffffffffp+674, chars_format::fixed, 0,
        "1567642659410349405780014642250281774842850647326077304233620159403401923227440873088373283614779406330032"
        "08644170732678377741778793921115844655347901458056974845372041613712964545054428887638648253579264"},
    {0x1.fffffffffffffp+675, chars_format::fixed, 0,
        "3135285318820698811560029284500563549685701294652154608467240318806803846454881746176746567229558812660064"
        "17288341465356755483557587842231689310695802916113949690744083227425929090108857775277296507158528"},
    {0x1.fffffffffffffp+676, chars_format::fixed, 0,
        "6270570637641397623120058569001127099371402589304309216934480637613607692909763492353493134459117625320128"
        "34576682930713510967115175684463378621391605832227899381488166454851858180217715550554593014317056"},
    {0x1.fffffffffffffp+677, chars_format::fixed, 0,
        "1254114127528279524624011713800225419874280517860861843386896127522721538581952698470698626891823525064025"
        "669153365861427021934230351368926757242783211664455798762976332909703716360435431101109186028634112"},
    {0x1.fffffffffffffp+678, chars_format::fixed, 0,
        "2508228255056559049248023427600450839748561035721723686773792255045443077163905396941397253783647050128051"
        "338306731722854043868460702737853514485566423328911597525952665819407432720870862202218372057268224"},
    {0x1.fffffffffffffp+679, chars_format::fixed, 0,
        "5016456510113118098496046855200901679497122071443447373547584510090886154327810793882794507567294100256102"
        "676613463445708087736921405475707028971132846657823195051905331638814865441741724404436744114536448"},
    {0x1.fffffffffffffp+680, chars_format::fixed, 0,
        "1003291302022623619699209371040180335899424414288689474709516902018177230865562158776558901513458820051220"
        "5353226926891416175473842810951414057942265693315646390103810663277629730883483448808873488229072896"},
    {0x1.fffffffffffffp+681, chars_format::fixed, 0,
        "2006582604045247239398418742080360671798848828577378949419033804036354461731124317553117803026917640102441"
        "0706453853782832350947685621902828115884531386631292780207621326555259461766966897617746976458145792"},
    {0x1.fffffffffffffp+682, chars_format::fixed, 0,
        "4013165208090494478796837484160721343597697657154757898838067608072708923462248635106235606053835280204882"
        "1412907707565664701895371243805656231769062773262585560415242653110518923533933795235493952916291584"},
    {0x1.fffffffffffffp+683, chars_format::fixed, 0,
        "8026330416180988957593674968321442687195395314309515797676135216145417846924497270212471212107670560409764"
        "2825815415131329403790742487611312463538125546525171120830485306221037847067867590470987905832583168"},
    {0x1.fffffffffffffp+684, chars_format::fixed, 0,
        "1605266083236197791518734993664288537439079062861903159535227043229083569384899454042494242421534112081952"
        "85651630830262658807581484975222624927076251093050342241660970612442075694135735180941975811665166336"},
    {0x1.fffffffffffffp+685, chars_format::fixed, 0,
        "3210532166472395583037469987328577074878158125723806319070454086458167138769798908084988484843068224163905"
        "71303261660525317615162969950445249854152502186100684483321941224884151388271470361883951623330332672"},
    {0x1.fffffffffffffp+686, chars_format::fixed, 0,
        "6421064332944791166074939974657154149756316251447612638140908172916334277539597816169976969686136448327811"
        "42606523321050635230325939900890499708305004372201368966643882449768302776542940723767903246660665344"},
    {0x1.fffffffffffffp+687, chars_format::fixed, 0,
        "1284212866588958233214987994931430829951263250289522527628181634583266855507919563233995393937227289665562"
        "285213046642101270460651879801780999416610008744402737933287764899536605553085881447535806493321330688"},
    {0x1.fffffffffffffp+688, chars_format::fixed, 0,
        "2568425733177916466429975989862861659902526500579045055256363269166533711015839126467990787874454579331124"
        "570426093284202540921303759603561998833220017488805475866575529799073211106171762895071612986642661376"},
    {0x1.fffffffffffffp+689, chars_format::fixed, 0,
        "5136851466355832932859951979725723319805053001158090110512726538333067422031678252935981575748909158662249"
        "140852186568405081842607519207123997666440034977610951733151059598146422212343525790143225973285322752"},
    {0x1.fffffffffffffp+690, chars_format::fixed, 0,
        "1027370293271166586571990395945144663961010600231618022102545307666613484406335650587196315149781831732449"
        "8281704373136810163685215038414247995332880069955221903466302119196292844424687051580286451946570645504"},
    {0x1.fffffffffffffp+691, chars_format::fixed, 0,
        "2054740586542333173143980791890289327922021200463236044205090615333226968812671301174392630299563663464899"
        "6563408746273620327370430076828495990665760139910443806932604238392585688849374103160572903893141291008"},
    {0x1.fffffffffffffp+692, chars_format::fixed, 0,
        "4109481173084666346287961583780578655844042400926472088410181230666453937625342602348785260599127326929799"
        "3126817492547240654740860153656991981331520279820887613865208476785171377698748206321145807786282582016"},
    {0x1.fffffffffffffp+693, chars_format::fixed, 0,
        "8218962346169332692575923167561157311688084801852944176820362461332907875250685204697570521198254653859598"
        "6253634985094481309481720307313983962663040559641775227730416953570342755397496412642291615572565164032"},
    {0x1.fffffffffffffp+694, chars_format::fixed, 0,
        "1643792469233866538515184633512231462337616960370588835364072492266581575050137040939514104239650930771919"
        "72507269970188962618963440614627967925326081119283550455460833907140685510794992825284583231145130328064"},
    {0x1.fffffffffffffp+695, chars_format::fixed, 0,
        "3287584938467733077030369267024462924675233920741177670728144984533163150100274081879028208479301861543839"
        "45014539940377925237926881229255935850652162238567100910921667814281371021589985650569166462290260656128"},
    {0x1.fffffffffffffp+696, chars_format::fixed, 0,
        "6575169876935466154060738534048925849350467841482355341456289969066326300200548163758056416958603723087678"
        "90029079880755850475853762458511871701304324477134201821843335628562742043179971301138332924580521312256"},
    {0x1.fffffffffffffp+697, chars_format::fixed, 0,
        "1315033975387093230812147706809785169870093568296471068291257993813265260040109632751611283391720744617535"
        "78005815976151170095170752491702374340260864895426840364368667125712548408635994260227666584916104262451"
        "2"},
    {0x1.fffffffffffffp+698, chars_format::fixed, 0,
        "2630067950774186461624295413619570339740187136592942136582515987626530520080219265503222566783441489235071"
        "56011631952302340190341504983404748680521729790853680728737334251425096817271988520455333169832208524902"
        "4"},
    {0x1.fffffffffffffp+699, chars_format::fixed, 0,
        "5260135901548372923248590827239140679480374273185884273165031975253061040160438531006445133566882978470143"
        "12023263904604680380683009966809497361043459581707361457474668502850193634543977040910666339664417049804"
        "8"},
    {0x1.fffffffffffffp+700, chars_format::fixed, 0,
        "1052027180309674584649718165447828135896074854637176854633006395050612208032087706201289026713376595694028"
        "624046527809209360761366019933618994722086919163414722914949337005700387269087954081821332679328834099609"
        "6"},
    {0x1.fffffffffffffp+701, chars_format::fixed, 0,
        "2104054360619349169299436330895656271792149709274353709266012790101224416064175412402578053426753191388057"
        "248093055618418721522732039867237989444173838326829445829898674011400774538175908163642665358657668199219"
        "2"},
    {0x1.fffffffffffffp+702, chars_format::fixed, 0,
        "4208108721238698338598872661791312543584299418548707418532025580202448832128350824805156106853506382776114"
        "496186111236837443045464079734475978888347676653658891659797348022801549076351816327285330717315336398438"
        "4"},
    {0x1.fffffffffffffp+703, chars_format::fixed, 0,
        "8416217442477396677197745323582625087168598837097414837064051160404897664256701649610312213707012765552228"
        "992372222473674886090928159468951957776695353307317783319594696045603098152703632654570661434630672796876"
        "8"},
    {0x1.fffffffffffffp+704, chars_format::fixed, 0,
        "1683243488495479335439549064716525017433719767419482967412810232080979532851340329922062442741402553110445"
        "7984744444947349772181856318937903915553390706614635566639189392091206196305407265309141322869261345593753"
        "6"},
    {0x1.fffffffffffffp+705, chars_format::fixed, 0,
        "3366486976990958670879098129433050034867439534838965934825620464161959065702680659844124885482805106220891"
        "5969488889894699544363712637875807831106781413229271133278378784182412392610814530618282645738522691187507"
        "2"},
    {0x1.fffffffffffffp+706, chars_format::fixed, 0,
        "6732973953981917341758196258866100069734879069677931869651240928323918131405361319688249770965610212441783"
        "1938977779789399088727425275751615662213562826458542266556757568364824785221629061236565291477045382375014"
        "4"},
    {0x1.fffffffffffffp+707, chars_format::fixed, 0,
        "1346594790796383468351639251773220013946975813935586373930248185664783626281072263937649954193122042488356"
        "6387795555957879817745485055150323132442712565291708453311351513672964957044325812247313058295409076475002"
        "88"},
    {0x1.fffffffffffffp+708, chars_format::fixed, 0,
        "2693189581592766936703278503546440027893951627871172747860496371329567252562144527875299908386244084976713"
        "2775591111915759635490970110300646264885425130583416906622703027345929914088651624494626116590818152950005"
        "76"},
    {0x1.fffffffffffffp+709, chars_format::fixed, 0,
        "5386379163185533873406557007092880055787903255742345495720992742659134505124289055750599816772488169953426"
        "5551182223831519270981940220601292529770850261166833813245406054691859828177303248989252233181636305900011"
        "52"},
    {0x1.fffffffffffffp+710, chars_format::fixed, 0,
        "1077275832637106774681311401418576011157580651148469099144198548531826901024857811150119963354497633990685"
        "3110236444766303854196388044120258505954170052233366762649081210938371965635460649797850446636327261180002"
        "304"},
    {0x1.fffffffffffffp+711, chars_format::fixed, 0,
        "2154551665274213549362622802837152022315161302296938198288397097063653802049715622300239926708995267981370"
        "6220472889532607708392776088240517011908340104466733525298162421876743931270921299595700893272654522360004"
        "608"},
    {0x1.fffffffffffffp+712, chars_format::fixed, 0,
        "4309103330548427098725245605674304044630322604593876396576794194127307604099431244600479853417990535962741"
        "2440945779065215416785552176481034023816680208933467050596324843753487862541842599191401786545309044720009"
        "216"},
    {0x1.fffffffffffffp+713, chars_format::fixed, 0,
        "8618206661096854197450491211348608089260645209187752793153588388254615208198862489200959706835981071925482"
        "4881891558130430833571104352962068047633360417866934101192649687506975725083685198382803573090618089440018"
        "432"},
    {0x1.fffffffffffffp+714, chars_format::fixed, 0,
        "1723641332219370839490098242269721617852129041837550558630717677650923041639772497840191941367196214385096"
        "4976378311626086166714220870592413609526672083573386820238529937501395145016737039676560714618123617888003"
        "6864"},
    {0x1.fffffffffffffp+715, chars_format::fixed, 0,
        "3447282664438741678980196484539443235704258083675101117261435355301846083279544995680383882734392428770192"
        "9952756623252172333428441741184827219053344167146773640477059875002790290033474079353121429236247235776007"
        "3728"},
    {0x1.fffffffffffffp+716, chars_format::fixed, 0,
        "6894565328877483357960392969078886471408516167350202234522870710603692166559089991360767765468784857540385"
        "9905513246504344666856883482369654438106688334293547280954119750005580580066948158706242858472494471552014"
        "7456"},
    {0x1.fffffffffffffp+717, chars_format::fixed, 0,
        "1378913065775496671592078593815777294281703233470040446904574142120738433311817998272153553093756971508077"
        "1981102649300868933371376696473930887621337666858709456190823950001116116013389631741248571694498894310402"
        "94912"},
    {0x1.fffffffffffffp+718, chars_format::fixed, 0,
        "2757826131550993343184157187631554588563406466940080893809148284241476866623635996544307106187513943016154"
        "3962205298601737866742753392947861775242675333717418912381647900002232232026779263482497143388997788620805"
        "89824"},
    {0x1.fffffffffffffp+719, chars_format::fixed, 0,
        "5515652263101986686368314375263109177126812933880161787618296568482953733247271993088614212375027886032308"
        "7924410597203475733485506785895723550485350667434837824763295800004464464053558526964994286777995577241611"
        "79648"},
    {0x1.fffffffffffffp+720, chars_format::fixed, 0,
        "1103130452620397337273662875052621835425362586776032357523659313696590746649454398617722842475005577206461"
        "7584882119440695146697101357179144710097070133486967564952659160000892892810711705392998857355599115448322"
        "359296"},
    {0x1.fffffffffffffp+721, chars_format::fixed, 0,
        "2206260905240794674547325750105243670850725173552064715047318627393181493298908797235445684950011154412923"
        "5169764238881390293394202714358289420194140266973935129905318320001785785621423410785997714711198230896644"
        "718592"},
    {0x1.fffffffffffffp+722, chars_format::fixed, 0,
        "4412521810481589349094651500210487341701450347104129430094637254786362986597817594470891369900022308825847"
        "0339528477762780586788405428716578840388280533947870259810636640003571571242846821571995429422396461793289"
        "437184"},
    {0x1.fffffffffffffp+723, chars_format::fixed, 0,
        "8825043620963178698189303000420974683402900694208258860189274509572725973195635188941782739800044617651694"
        "0679056955525561173576810857433157680776561067895740519621273280007143142485693643143990858844792923586578"
        "874368"},
    {0x1.fffffffffffffp+724, chars_format::fixed, 0,
        "1765008724192635739637860600084194936680580138841651772037854901914545194639127037788356547960008923530338"
        "8135811391105112234715362171486631536155312213579148103924254656001428628497138728628798171768958584717315"
        "7748736"},
    {0x1.fffffffffffffp+725, chars_format::fixed, 0,
        "3530017448385271479275721200168389873361160277683303544075709803829090389278254075576713095920017847060677"
        "6271622782210224469430724342973263072310624427158296207848509312002857256994277457257596343537917169434631"
        "5497472"},
    {0x1.fffffffffffffp+726, chars_format::fixed, 0,
        "7060034896770542958551442400336779746722320555366607088151419607658180778556508151153426191840035694121355"
        "2543245564420448938861448685946526144621248854316592415697018624005714513988554914515192687075834338869263"
        "0994944"},
    {0x1.fffffffffffffp+727, chars_format::fixed, 0,
        "1412006979354108591710288480067355949344464111073321417630283921531636155711301630230685238368007138824271"
        "0508649112884089787772289737189305228924249770863318483139403724801142902797710982903038537415166867773852"
        "61989888"},
    {0x1.fffffffffffffp+728, chars_format::fixed, 0,
        "2824013958708217183420576960134711898688928222146642835260567843063272311422603260461370476736014277648542"
        "1017298225768179575544579474378610457848499541726636966278807449602285805595421965806077074830333735547705"
        "23979776"},
    {0x1.fffffffffffffp+729, chars_format::fixed, 0,
        "5648027917416434366841153920269423797377856444293285670521135686126544622845206520922740953472028555297084"
        "2034596451536359151089158948757220915696999083453273932557614899204571611190843931612154149660667471095410"
        "47959552"},
    {0x1.fffffffffffffp+730, chars_format::fixed, 0,
        "1129605583483286873368230784053884759475571288858657134104227137225308924569041304184548190694405711059416"
        "8406919290307271830217831789751444183139399816690654786511522979840914322238168786322430829932133494219082"
        "095919104"},
    {0x1.fffffffffffffp+731, chars_format::fixed, 0,
        "2259211166966573746736461568107769518951142577717314268208454274450617849138082608369096381388811422118833"
        "6813838580614543660435663579502888366278799633381309573023045959681828644476337572644861659864266988438164"
        "191838208"},
    {0x1.fffffffffffffp+732, chars_format::fixed, 0,
        "4518422333933147493472923136215539037902285155434628536416908548901235698276165216738192762777622844237667"
        "3627677161229087320871327159005776732557599266762619146046091919363657288952675145289723319728533976876328"
        "383676416"},
    {0x1.fffffffffffffp+733, chars_format::fixed, 0,
        "9036844667866294986945846272431078075804570310869257072833817097802471396552330433476385525555245688475334"
        "7255354322458174641742654318011553465115198533525238292092183838727314577905350290579446639457067953752656"
        "767352832"},
    {0x1.fffffffffffffp+734, chars_format::fixed, 0,
        "1807368933573258997389169254486215615160914062173851414566763419560494279310466086695277105111049137695066"
        "9451070864491634928348530863602310693023039706705047658418436767745462915581070058115889327891413590750531"
        "3534705664"},
    {0x1.fffffffffffffp+735, chars_format::fixed, 0,
        "3614737867146517994778338508972431230321828124347702829133526839120988558620932173390554210222098275390133"
        "8902141728983269856697061727204621386046079413410095316836873535490925831162140116231778655782827181501062"
        "7069411328"},
    {0x1.fffffffffffffp+736, chars_format::fixed, 0,
        "7229475734293035989556677017944862460643656248695405658267053678241977117241864346781108420444196550780267"
        "7804283457966539713394123454409242772092158826820190633673747070981851662324280232463557311565654363002125"
        "4138822656"},
    {0x1.fffffffffffffp+737, chars_format::fixed, 0,
        "1445895146858607197911335403588972492128731249739081131653410735648395423448372869356221684088839310156053"
        "5560856691593307942678824690881848554418431765364038126734749414196370332464856046492711462313130872600425"
        "08277645312"},
    {0x1.fffffffffffffp+738, chars_format::fixed, 0,
        "2891790293717214395822670807177944984257462499478162263306821471296790846896745738712443368177678620312107"
        "1121713383186615885357649381763697108836863530728076253469498828392740664929712092985422924626261745200850"
        "16555290624"},
    {0x1.fffffffffffffp+739, chars_format::fixed, 0,
        "5783580587434428791645341614355889968514924998956324526613642942593581693793491477424886736355357240624214"
        "2243426766373231770715298763527394217673727061456152506938997656785481329859424185970845849252523490401700"
        "33110581248"},
    {0x1.fffffffffffffp+740, chars_format::fixed, 0,
        "1156716117486885758329068322871177993702984999791264905322728588518716338758698295484977347271071448124842"
        "8448685353274646354143059752705478843534745412291230501387799531357096265971884837194169169850504698080340"
        "066221162496"},
    {0x1.fffffffffffffp+741, chars_format::fixed, 0,
        "2313432234973771516658136645742355987405969999582529810645457177037432677517396590969954694542142896249685"
        "6897370706549292708286119505410957687069490824582461002775599062714192531943769674388338339701009396160680"
        "132442324992"},
    {0x1.fffffffffffffp+742, chars_format::fixed, 0,
        "4626864469947543033316273291484711974811939999165059621290914354074865355034793181939909389084285792499371"
        "3794741413098585416572239010821915374138981649164922005551198125428385063887539348776676679402018792321360"
        "264884649984"},
    {0x1.fffffffffffffp+743, chars_format::fixed, 0,
        "9253728939895086066632546582969423949623879998330119242581828708149730710069586363879818778168571584998742"
        "7589482826197170833144478021643830748277963298329844011102396250856770127775078697553353358804037584642720"
        "529769299968"},
    {0x1.fffffffffffffp+744, chars_format::fixed, 0,
        "1850745787979017213326509316593884789924775999666023848516365741629946142013917272775963755633714316999748"
        "5517896565239434166628895604328766149655592659665968802220479250171354025555015739510670671760807516928544"
        "1059538599936"},
    {0x1.fffffffffffffp+745, chars_format::fixed, 0,
        "3701491575958034426653018633187769579849551999332047697032731483259892284027834545551927511267428633999497"
        "1035793130478868333257791208657532299311185319331937604440958500342708051110031479021341343521615033857088"
        "2119077199872"},
    {0x1.fffffffffffffp+746, chars_format::fixed, 0,
        "7402983151916068853306037266375539159699103998664095394065462966519784568055669091103855022534857267998994"
        "2071586260957736666515582417315064598622370638663875208881917000685416102220062958042682687043230067714176"
        "4238154399744"},
    {0x1.fffffffffffffp+747, chars_format::fixed, 0,
        "1480596630383213770661207453275107831939820799732819078813092593303956913611133818220771004506971453599798"
        "8414317252191547333303116483463012919724474127732775041776383400137083220444012591608536537408646013542835"
        "28476308799488"},
    {0x1.fffffffffffffp+748, chars_format::fixed, 0,
        "2961193260766427541322414906550215663879641599465638157626185186607913827222267636441542009013942907199597"
        "6828634504383094666606232966926025839448948255465550083552766800274166440888025183217073074817292027085670"
        "56952617598976"},
    {0x1.fffffffffffffp+749, chars_format::fixed, 0,
        "5922386521532855082644829813100431327759283198931276315252370373215827654444535272883084018027885814399195"
        "3657269008766189333212465933852051678897896510931100167105533600548332881776050366434146149634584054171341"
        "13905235197952"},
    {0x1.fffffffffffffp+750, chars_format::fixed, 0,
        "1184477304306571016528965962620086265551856639786255263050474074643165530888907054576616803605577162879839"
        "0731453801753237866642493186770410335779579302186220033421106720109666576355210073286829229926916810834268"
        "227810470395904"},
    {0x1.fffffffffffffp+751, chars_format::fixed, 0,
        "2368954608613142033057931925240172531103713279572510526100948149286331061777814109153233607211154325759678"
        "1462907603506475733284986373540820671559158604372440066842213440219333152710420146573658459853833621668536"
        "455620940791808"},
    {0x1.fffffffffffffp+752, chars_format::fixed, 0,
        "4737909217226284066115863850480345062207426559145021052201896298572662123555628218306467214422308651519356"
        "2925815207012951466569972747081641343118317208744880133684426880438666305420840293147316919707667243337072"
        "911241881583616"},
    {0x1.fffffffffffffp+753, chars_format::fixed, 0,
        "9475818434452568132231727700960690124414853118290042104403792597145324247111256436612934428844617303038712"
        "5851630414025902933139945494163282686236634417489760267368853760877332610841680586294633839415334486674145"
        "822483763167232"},
    {0x1.fffffffffffffp+754, chars_format::fixed, 0,
        "1895163686890513626446345540192138024882970623658008420880758519429064849422251287322586885768923460607742"
        "5170326082805180586627989098832656537247326883497952053473770752175466522168336117258926767883066897334829"
        "1644967526334464"},
    {0x1.fffffffffffffp+755, chars_format::fixed, 0,
        "3790327373781027252892691080384276049765941247316016841761517038858129698844502574645173771537846921215485"
        "0340652165610361173255978197665313074494653766995904106947541504350933044336672234517853535766133794669658"
        "3289935052668928"},
    {0x1.fffffffffffffp+756, chars_format::fixed, 0,
        "7580654747562054505785382160768552099531882494632033683523034077716259397689005149290347543075693842430970"
        "0681304331220722346511956395330626148989307533991808213895083008701866088673344469035707071532267589339316"
        "6579870105337856"},
    {0x1.fffffffffffffp+757, chars_format::fixed, 0,
        "1516130949512410901157076432153710419906376498926406736704606815543251879537801029858069508615138768486194"
        "0136260866244144469302391279066125229797861506798361642779016601740373217734668893807141414306453517867863"
        "33159740210675712"},
    {0x1.fffffffffffffp+758, chars_format::fixed, 0,
        "3032261899024821802314152864307420839812752997852813473409213631086503759075602059716139017230277536972388"
        "0272521732488288938604782558132250459595723013596723285558033203480746435469337787614282828612907035735726"
        "66319480421351424"},
    {0x1.fffffffffffffp+759, chars_format::fixed, 0,
        "6064523798049643604628305728614841679625505995705626946818427262173007518151204119432278034460555073944776"
        "0545043464976577877209565116264500919191446027193446571116066406961492870938675575228565657225814071471453"
        "32638960842702848"},
    {0x1.fffffffffffffp+760, chars_format::fixed, 0,
        "1212904759609928720925661145722968335925101199141125389363685452434601503630240823886455606892111014788955"
        "2109008692995315575441913023252900183838289205438689314223213281392298574187735115045713131445162814294290"
        "665277921685405696"},
    {0x1.fffffffffffffp+761, chars_format::fixed, 0,
        "2425809519219857441851322291445936671850202398282250778727370904869203007260481647772911213784222029577910"
        "4218017385990631150883826046505800367676578410877378628446426562784597148375470230091426262890325628588581"
        "330555843370811392"},
    {0x1.fffffffffffffp+762, chars_format::fixed, 0,
        "4851619038439714883702644582891873343700404796564501557454741809738406014520963295545822427568444059155820"
        "8436034771981262301767652093011600735353156821754757256892853125569194296750940460182852525780651257177162"
        "661111686741622784"},
    {0x1.fffffffffffffp+763, chars_format::fixed, 0,
        "9703238076879429767405289165783746687400809593129003114909483619476812029041926591091644855136888118311641"
        "6872069543962524603535304186023201470706313643509514513785706251138388593501880920365705051561302514354325"
        "322223373483245568"},
    {0x1.fffffffffffffp+764, chars_format::fixed, 0,
        "1940647615375885953481057833156749337480161918625800622981896723895362405808385318218328971027377623662328"
        "3374413908792504920707060837204640294141262728701902902757141250227677718700376184073141010312260502870865"
        "0644446746966491136"},
    {0x1.fffffffffffffp+765, chars_format::fixed, 0,
        "3881295230751771906962115666313498674960323837251601245963793447790724811616770636436657942054755247324656"
        "6748827817585009841414121674409280588282525457403805805514282500455355437400752368146282020624521005741730"
        "1288893493932982272"},
    {0x1.fffffffffffffp+766, chars_format::fixed, 0,
        "7762590461503543813924231332626997349920647674503202491927586895581449623233541272873315884109510494649313"
        "3497655635170019682828243348818561176565050914807611611028565000910710874801504736292564041249042011483460"
        "2577786987865964544"},
    {0x1.fffffffffffffp+767, chars_format::fixed, 0,
        "1552518092300708762784846266525399469984129534900640498385517379116289924646708254574663176821902098929862"
        "6699531127034003936565648669763712235313010182961522322205713000182142174960300947258512808249808402296692"
        "05155573975731929088"},
    {0x1.fffffffffffffp+768, chars_format::fixed, 0,
        "3105036184601417525569692533050798939968259069801280996771034758232579849293416509149326353643804197859725"
        "3399062254068007873131297339527424470626020365923044644411426000364284349920601894517025616499616804593384"
        "10311147951463858176"},
    {0x1.fffffffffffffp+769, chars_format::fixed, 0,
        "6210072369202835051139385066101597879936518139602561993542069516465159698586833018298652707287608395719450"
        "6798124508136015746262594679054848941252040731846089288822852000728568699841203789034051232999233609186768"
        "20622295902927716352"},
    {0x1.fffffffffffffp+770, chars_format::fixed, 0,
        "1242014473840567010227877013220319575987303627920512398708413903293031939717366603659730541457521679143890"
        "1359624901627203149252518935810969788250408146369217857764570400145713739968240757806810246599846721837353"
        "641244591805855432704"},
    {0x1.fffffffffffffp+771, chars_format::fixed, 0,
        "2484028947681134020455754026440639151974607255841024797416827806586063879434733207319461082915043358287780"
        "2719249803254406298505037871621939576500816292738435715529140800291427479936481515613620493199693443674707"
        "282489183611710865408"},
    {0x1.fffffffffffffp+772, chars_format::fixed, 0,
        "4968057895362268040911508052881278303949214511682049594833655613172127758869466414638922165830086716575560"
        "5438499606508812597010075743243879153001632585476871431058281600582854959872963031227240986399386887349414"
        "564978367223421730816"},
    {0x1.fffffffffffffp+773, chars_format::fixed, 0,
        "9936115790724536081823016105762556607898429023364099189667311226344255517738932829277844331660173433151121"
        "0876999213017625194020151486487758306003265170953742862116563201165709919745926062454481972798773774698829"
        "129956734446843461632"},
    {0x1.fffffffffffffp+774, chars_format::fixed, 0,
        "1987223158144907216364603221152511321579685804672819837933462245268851103547786565855568866332034686630224"
        "2175399842603525038804030297297551661200653034190748572423312640233141983949185212490896394559754754939765"
        "8259913468893686923264"},
    {0x1.fffffffffffffp+775, chars_format::fixed, 0,
        "3974446316289814432729206442305022643159371609345639675866924490537702207095573131711137732664069373260448"
        "4350799685207050077608060594595103322401306068381497144846625280466283967898370424981792789119509509879531"
        "6519826937787373846528"},
    {0x1.fffffffffffffp+776, chars_format::fixed, 0,
        "7948892632579628865458412884610045286318743218691279351733848981075404414191146263422275465328138746520896"
        "8701599370414100155216121189190206644802612136762994289693250560932567935796740849963585578239019019759063"
        "3039653875574747693056"},
    {0x1.fffffffffffffp+777, chars_format::fixed, 0,
        "1589778526515925773091682576922009057263748643738255870346769796215080882838229252684455093065627749304179"
        "3740319874082820031043224237838041328960522427352598857938650112186513587159348169992717115647803803951812"
        "66079307751149495386112"},
    {0x1.fffffffffffffp+778, chars_format::fixed, 0,
        "3179557053031851546183365153844018114527497287476511740693539592430161765676458505368910186131255498608358"
        "7480639748165640062086448475676082657921044854705197715877300224373027174318696339985434231295607607903625"
        "32158615502298990772224"},
    {0x1.fffffffffffffp+779, chars_format::fixed, 0,
        "6359114106063703092366730307688036229054994574953023481387079184860323531352917010737820372262510997216717"
        "4961279496331280124172896951352165315842089709410395431754600448746054348637392679970868462591215215807250"
        "64317231004597981544448"},
    {0x1.fffffffffffffp+780, chars_format::fixed, 0,
        "1271822821212740618473346061537607245810998914990604696277415836972064706270583402147564074452502199443343"
        "4992255899266256024834579390270433063168417941882079086350920089749210869727478535994173692518243043161450"
        "128634462009195963088896"},
    {0x1.fffffffffffffp+781, chars_format::fixed, 0,
        "2543645642425481236946692123075214491621997829981209392554831673944129412541166804295128148905004398886686"
        "9984511798532512049669158780540866126336835883764158172701840179498421739454957071988347385036486086322900"
        "257268924018391926177792"},
    {0x1.fffffffffffffp+782, chars_format::fixed, 0,
        "5087291284850962473893384246150428983243995659962418785109663347888258825082333608590256297810008797773373"
        "9969023597065024099338317561081732252673671767528316345403680358996843478909914143976694770072972172645800"
        "514537848036783852355584"},
    {0x1.fffffffffffffp+783, chars_format::fixed, 0,
        "1017458256970192494778676849230085796648799131992483757021932669577651765016466721718051259562001759554674"
        "7993804719413004819867663512216346450534734353505663269080736071799368695781982828795338954014594434529160"
        "1029075696073567704711168"},
    {0x1.fffffffffffffp+784, chars_format::fixed, 0,
        "2034916513940384989557353698460171593297598263984967514043865339155303530032933443436102519124003519109349"
        "5987609438826009639735327024432692901069468707011326538161472143598737391563965657590677908029188869058320"
        "2058151392147135409422336"},
    {0x1.fffffffffffffp+785, chars_format::fixed, 0,
        "4069833027880769979114707396920343186595196527969935028087730678310607060065866886872205038248007038218699"
        "1975218877652019279470654048865385802138937414022653076322944287197474783127931315181355816058377738116640"
        "4116302784294270818844672"},
    {0x1.fffffffffffffp+786, chars_format::fixed, 0,
        "8139666055761539958229414793840686373190393055939870056175461356621214120131733773744410076496014076437398"
        "3950437755304038558941308097730771604277874828045306152645888574394949566255862630362711632116755476233280"
        "8232605568588541637689344"},
    {0x1.fffffffffffffp+787, chars_format::fixed, 0,
        "1627933211152307991645882958768137274638078611187974011235092271324242824026346754748882015299202815287479"
        "6790087551060807711788261619546154320855574965609061230529177714878989913251172526072542326423351095246656"
        "16465211137177083275378688"},
    {0x1.fffffffffffffp+788, chars_format::fixed, 0,
        "3255866422304615983291765917536274549276157222375948022470184542648485648052693509497764030598405630574959"
        "3580175102121615423576523239092308641711149931218122461058355429757979826502345052145084652846702190493312"
        "32930422274354166550757376"},
    {0x1.fffffffffffffp+789, chars_format::fixed, 0,
        "6511732844609231966583531835072549098552314444751896044940369085296971296105387018995528061196811261149918"
        "7160350204243230847153046478184617283422299862436244922116710859515959653004690104290169305693404380986624"
        "65860844548708333101514752"},
    {0x1.fffffffffffffp+790, chars_format::fixed, 0,
        "1302346568921846393316706367014509819710462888950379208988073817059394259221077403799105612239362252229983"
        "7432070040848646169430609295636923456684459972487248984423342171903191930600938020858033861138680876197324"
        "931721689097416666203029504"},
    {0x1.fffffffffffffp+791, chars_format::fixed, 0,
        "2604693137843692786633412734029019639420925777900758417976147634118788518442154807598211224478724504459967"
        "4864140081697292338861218591273846913368919944974497968846684343806383861201876041716067722277361752394649"
        "863443378194833332406059008"},
    {0x1.fffffffffffffp+792, chars_format::fixed, 0,
        "5209386275687385573266825468058039278841851555801516835952295268237577036884309615196422448957449008919934"
        "9728280163394584677722437182547693826737839889948995937693368687612767722403752083432135444554723504789299"
        "726886756389666664812118016"},
    {0x1.fffffffffffffp+793, chars_format::fixed, 0,
        "1041877255137477114653365093611607855768370311160303367190459053647515407376861923039284489791489801783986"
        "9945656032678916935544487436509538765347567977989799187538673737522553544480750416686427088910944700957859"
        "9453773512779333329624236032"},
    {0x1.fffffffffffffp+794, chars_format::fixed, 0,
        "2083754510274954229306730187223215711536740622320606734380918107295030814753723846078568979582979603567973"
        "9891312065357833871088974873019077530695135955979598375077347475045107088961500833372854177821889401915719"
        "8907547025558666659248472064"},
    {0x1.fffffffffffffp+795, chars_format::fixed, 0,
        "4167509020549908458613460374446431423073481244641213468761836214590061629507447692157137959165959207135947"
        "9782624130715667742177949746038155061390271911959196750154694950090214177923001666745708355643778803831439"
        "7815094051117333318496944128"},
    {0x1.fffffffffffffp+796, chars_format::fixed, 0,
        "8335018041099816917226920748892862846146962489282426937523672429180123259014895384314275918331918414271895"
        "9565248261431335484355899492076310122780543823918393500309389900180428355846003333491416711287557607662879"
        "5630188102234666636993888256"},
    {0x1.fffffffffffffp+797, chars_format::fixed, 0,
        "1667003608219963383445384149778572569229392497856485387504734485836024651802979076862855183666383682854379"
        "1913049652286267096871179898415262024556108764783678700061877980036085671169200666698283342257511521532575"
        "91260376204469333273987776512"},
    {0x1.fffffffffffffp+798, chars_format::fixed, 0,
        "3334007216439926766890768299557145138458784995712970775009468971672049303605958153725710367332767365708758"
        "3826099304572534193742359796830524049112217529567357400123755960072171342338401333396566684515023043065151"
        "82520752408938666547975553024"},
    {0x1.fffffffffffffp+799, chars_format::fixed, 0,
        "6668014432879853533781536599114290276917569991425941550018937943344098607211916307451420734665534731417516"
        "7652198609145068387484719593661048098224435059134714800247511920144342684676802666793133369030046086130303"
        "65041504817877333095951106048"},
    {0x1.fffffffffffffp+800, chars_format::fixed, 0,
        "1333602886575970706756307319822858055383513998285188310003787588668819721442383261490284146933106946283503"
        "3530439721829013677496943918732209619644887011826942960049502384028868536935360533358626673806009217226060"
        "730083009635754666191902212096"},
    {0x1.fffffffffffffp+801, chars_format::fixed, 0,
        "2667205773151941413512614639645716110767027996570376620007575177337639442884766522980568293866213892567006"
        "7060879443658027354993887837464419239289774023653885920099004768057737073870721066717253347612018434452121"
        "460166019271509332383804424192"},
    {0x1.fffffffffffffp+802, chars_format::fixed, 0,
        "5334411546303882827025229279291432221534055993140753240015150354675278885769533045961136587732427785134013"
        "4121758887316054709987775674928838478579548047307771840198009536115474147741442133434506695224036868904242"
        "920332038543018664767608848384"},
    {0x1.fffffffffffffp+803, chars_format::fixed, 0,
        "1066882309260776565405045855858286444306811198628150648003030070935055777153906609192227317546485557026802"
        "6824351777463210941997555134985767695715909609461554368039601907223094829548288426686901339044807373780848"
        "5840664077086037329535217696768"},
    {0x1.fffffffffffffp+804, chars_format::fixed, 0,
        "2133764618521553130810091711716572888613622397256301296006060141870111554307813218384454635092971114053605"
        "3648703554926421883995110269971535391431819218923108736079203814446189659096576853373802678089614747561697"
        "1681328154172074659070435393536"},
    {0x1.fffffffffffffp+805, chars_format::fixed, 0,
        "4267529237043106261620183423433145777227244794512602592012120283740223108615626436768909270185942228107210"
        "7297407109852843767990220539943070782863638437846217472158407628892379318193153706747605356179229495123394"
        "3362656308344149318140870787072"},
    {0x1.fffffffffffffp+806, chars_format::fixed, 0,
        "8535058474086212523240366846866291554454489589025205184024240567480446217231252873537818540371884456214421"
        "4594814219705687535980441079886141565727276875692434944316815257784758636386307413495210712358458990246788"
        "6725312616688298636281741574144"},
    {0x1.fffffffffffffp+807, chars_format::fixed, 0,
        "1707011694817242504648073369373258310890897917805041036804848113496089243446250574707563708074376891242884"
        "2918962843941137507196088215977228313145455375138486988863363051556951727277261482699042142471691798049357"
        "73450625233376597272563483148288"},
    {0x1.fffffffffffffp+808, chars_format::fixed, 0,
        "3414023389634485009296146738746516621781795835610082073609696226992178486892501149415127416148753782485768"
        "5837925687882275014392176431954456626290910750276973977726726103113903454554522965398084284943383596098715"
        "46901250466753194545126966296576"},
    {0x1.fffffffffffffp+809, chars_format::fixed, 0,
        "6828046779268970018592293477493033243563591671220164147219392453984356973785002298830254832297507564971537"
        "1675851375764550028784352863908913252581821500553947955453452206227806909109045930796168569886767192197430"
        "93802500933506389090253932593152"},
    {0x1.fffffffffffffp+810, chars_format::fixed, 0,
        "1365609355853794003718458695498606648712718334244032829443878490796871394757000459766050966459501512994307"
        "4335170275152910005756870572781782650516364300110789591090690441245561381821809186159233713977353438439486"
        "187605001867012778180507865186304"},
    {0x1.fffffffffffffp+811, chars_format::fixed, 0,
        "2731218711707588007436917390997213297425436668488065658887756981593742789514000919532101932919003025988614"
        "8670340550305820011513741145563565301032728600221579182181380882491122763643618372318467427954706876878972"
        "375210003734025556361015730372608"},
    {0x1.fffffffffffffp+812, chars_format::fixed, 0,
        "5462437423415176014873834781994426594850873336976131317775513963187485579028001839064203865838006051977229"
        "7340681100611640023027482291127130602065457200443158364362761764982245527287236744636934855909413753757944"
        "750420007468051112722031460745216"},
    {0x1.fffffffffffffp+813, chars_format::fixed, 0,
        "1092487484683035202974766956398885318970174667395226263555102792637497115805600367812840773167601210395445"
        "9468136220122328004605496458225426120413091440088631672872552352996449105457447348927386971181882750751588"
        "9500840014936102225444062921490432"},
    {0x1.fffffffffffffp+814, chars_format::fixed, 0,
        "2184974969366070405949533912797770637940349334790452527110205585274994231611200735625681546335202420790891"
        "8936272440244656009210992916450852240826182880177263345745104705992898210914894697854773942363765501503177"
        "9001680029872204450888125842980864"},
    {0x1.fffffffffffffp+815, chars_format::fixed, 0,
        "4369949938732140811899067825595541275880698669580905054220411170549988463222401471251363092670404841581783"
        "7872544880489312018421985832901704481652365760354526691490209411985796421829789395709547884727531003006355"
        "8003360059744408901776251685961728"},
    {0x1.fffffffffffffp+816, chars_format::fixed, 0,
        "8739899877464281623798135651191082551761397339161810108440822341099976926444802942502726185340809683163567"
        "5745089760978624036843971665803408963304731520709053382980418823971592843659578791419095769455062006012711"
        "6006720119488817803552503371923456"},
    {0x1.fffffffffffffp+817, chars_format::fixed, 0,
        "1747979975492856324759627130238216510352279467832362021688164468219995385288960588500545237068161936632713"
        "5149017952195724807368794333160681792660946304141810676596083764794318568731915758283819153891012401202542"
        "32013440238977635607105006743846912"},
    {0x1.fffffffffffffp+818, chars_format::fixed, 0,
        "3495959950985712649519254260476433020704558935664724043376328936439990770577921177001090474136323873265427"
        "0298035904391449614737588666321363585321892608283621353192167529588637137463831516567638307782024802405084"
        "64026880477955271214210013487693824"},
    {0x1.fffffffffffffp+819, chars_format::fixed, 0,
        "6991919901971425299038508520952866041409117871329448086752657872879981541155842354002180948272647746530854"
        "0596071808782899229475177332642727170643785216567242706384335059177274274927663033135276615564049604810169"
        "28053760955910542428420026975387648"},
    {0x1.fffffffffffffp+820, chars_format::fixed, 0,
        "1398383980394285059807701704190573208281823574265889617350531574575996308231168470800436189654529549306170"
        "8119214361756579845895035466528545434128757043313448541276867011835454854985532606627055323112809920962033"
        "856107521911821084856840053950775296"},
    {0x1.fffffffffffffp+821, chars_format::fixed, 0,
        "2796767960788570119615403408381146416563647148531779234701063149151992616462336941600872379309059098612341"
        "6238428723513159691790070933057090868257514086626897082553734023670909709971065213254110646225619841924067"
        "712215043823642169713680107901550592"},
    {0x1.fffffffffffffp+822, chars_format::fixed, 0,
        "5593535921577140239230806816762292833127294297063558469402126298303985232924673883201744758618118197224683"
        "2476857447026319383580141866114181736515028173253794165107468047341819419942130426508221292451239683848135"
        "424430087647284339427360215803101184"},
    {0x1.fffffffffffffp+823, chars_format::fixed, 0,
        "1118707184315428047846161363352458566625458859412711693880425259660797046584934776640348951723623639444936"
        "6495371489405263876716028373222836347303005634650758833021493609468363883988426085301644258490247936769627"
        "0848860175294568678854720431606202368"},
    {0x1.fffffffffffffp+824, chars_format::fixed, 0,
        "2237414368630856095692322726704917133250917718825423387760850519321594093169869553280697903447247278889873"
        "2990742978810527753432056746445672694606011269301517666042987218936727767976852170603288516980495873539254"
        "1697720350589137357709440863212404736"},
    {0x1.fffffffffffffp+825, chars_format::fixed, 0,
        "4474828737261712191384645453409834266501835437650846775521701038643188186339739106561395806894494557779746"
        "5981485957621055506864113492891345389212022538603035332085974437873455535953704341206577033960991747078508"
        "3395440701178274715418881726424809472"},
    {0x1.fffffffffffffp+826, chars_format::fixed, 0,
        "8949657474523424382769290906819668533003670875301693551043402077286376372679478213122791613788989115559493"
        "1962971915242111013728226985782690778424045077206070664171948875746911071907408682413154067921983494157016"
        "6790881402356549430837763452849618944"},
    {0x1.fffffffffffffp+827, chars_format::fixed, 0,
        "1789931494904684876553858181363933706600734175060338710208680415457275274535895642624558322757797823111898"
        "6392594383048422202745645397156538155684809015441214132834389775149382214381481736482630813584396698831403"
        "33581762804713098861675526905699237888"},
    {0x1.fffffffffffffp+828, chars_format::fixed, 0,
        "3579862989809369753107716362727867413201468350120677420417360830914550549071791285249116645515595646223797"
        "2785188766096844405491290794313076311369618030882428265668779550298764428762963472965261627168793397662806"
        "67163525609426197723351053811398475776"},
    {0x1.fffffffffffffp+829, chars_format::fixed, 0,
        "7159725979618739506215432725455734826402936700241354840834721661829101098143582570498233291031191292447594"
        "5570377532193688810982581588626152622739236061764856531337559100597528857525926945930523254337586795325613"
        "34327051218852395446702107622796951552"},
    {0x1.fffffffffffffp+830, chars_format::fixed, 0,
        "1431945195923747901243086545091146965280587340048270968166944332365820219628716514099646658206238258489518"
        "9114075506438737762196516317725230524547847212352971306267511820119505771505185389186104650867517359065122"
        "668654102437704790893404215245593903104"},
    {0x1.fffffffffffffp+831, chars_format::fixed, 0,
        "2863890391847495802486173090182293930561174680096541936333888664731640439257433028199293316412476516979037"
        "8228151012877475524393032635450461049095694424705942612535023640239011543010370778372209301735034718130245"
        "337308204875409581786808430491187806208"},
    {0x1.fffffffffffffp+832, chars_format::fixed, 0,
        "5727780783694991604972346180364587861122349360193083872667777329463280878514866056398586632824953033958075"
        "6456302025754951048786065270900922098191388849411885225070047280478023086020741556744418603470069436260490"
        "674616409750819163573616860982375612416"},
    {0x1.fffffffffffffp+833, chars_format::fixed, 0,
        "1145556156738998320994469236072917572224469872038616774533555465892656175702973211279717326564990606791615"
        "1291260405150990209757213054180184419638277769882377045014009456095604617204148311348883720694013887252098"
        "1349232819501638327147233721964751224832"},
    {0x1.fffffffffffffp+834, chars_format::fixed, 0,
        "2291112313477996641988938472145835144448939744077233549067110931785312351405946422559434653129981213583230"
        "2582520810301980419514426108360368839276555539764754090028018912191209234408296622697767441388027774504196"
        "2698465639003276654294467443929502449664"},
    {0x1.fffffffffffffp+835, chars_format::fixed, 0,
        "4582224626955993283977876944291670288897879488154467098134221863570624702811892845118869306259962427166460"
        "5165041620603960839028852216720737678553111079529508180056037824382418468816593245395534882776055549008392"
        "5396931278006553308588934887859004899328"},
    {0x1.fffffffffffffp+836, chars_format::fixed, 0,
        "9164449253911986567955753888583340577795758976308934196268443727141249405623785690237738612519924854332921"
        "0330083241207921678057704433441475357106222159059016360112075648764836937633186490791069765552111098016785"
        "0793862556013106617177869775718009798656"},
    {0x1.fffffffffffffp+837, chars_format::fixed, 0,
        "1832889850782397313591150777716668115559151795261786839253688745428249881124757138047547722503984970866584"
        "2066016648241584335611540886688295071421244431811803272022415129752967387526637298158213953110422219603357"
        "01587725112026213234355739551436019597312"},
    {0x1.fffffffffffffp+838, chars_format::fixed, 0,
        "3665779701564794627182301555433336231118303590523573678507377490856499762249514276095095445007969941733168"
        "4132033296483168671223081773376590142842488863623606544044830259505934775053274596316427906220844439206714"
        "03175450224052426468711479102872039194624"},
    {0x1.fffffffffffffp+839, chars_format::fixed, 0,
        "7331559403129589254364603110866672462236607181047147357014754981712999524499028552190190890015939883466336"
        "8264066592966337342446163546753180285684977727247213088089660519011869550106549192632855812441688878413428"
        "06350900448104852937422958205744078389248"},
    {0x1.fffffffffffffp+840, chars_format::fixed, 0,
        "1466311880625917850872920622173334492447321436209429471402950996342599904899805710438038178003187976693267"
        "3652813318593267468489232709350636057136995545449442617617932103802373910021309838526571162488337775682685"
        "612701800896209705874845916411488156778496"},
    {0x1.fffffffffffffp+841, chars_format::fixed, 0,
        "2932623761251835701745841244346668984894642872418858942805901992685199809799611420876076356006375953386534"
        "7305626637186534936978465418701272114273991090898885235235864207604747820042619677053142324976675551365371"
        "225403601792419411749691832822976313556992"},
    {0x1.fffffffffffffp+842, chars_format::fixed, 0,
        "5865247522503671403491682488693337969789285744837717885611803985370399619599222841752152712012751906773069"
        "4611253274373069873956930837402544228547982181797770470471728415209495640085239354106284649953351102730742"
        "450807203584838823499383665645952627113984"},
    {0x1.fffffffffffffp+843, chars_format::fixed, 0,
        "1173049504500734280698336497738667593957857148967543577122360797074079923919844568350430542402550381354613"
        "8922250654874613974791386167480508845709596436359554094094345683041899128017047870821256929990670220546148"
        "4901614407169677646998767331291905254227968"},
    {0x1.fffffffffffffp+844, chars_format::fixed, 0,
        "2346099009001468561396672995477335187915714297935087154244721594148159847839689136700861084805100762709227"
        "7844501309749227949582772334961017691419192872719108188188691366083798256034095741642513859981340441092296"
        "9803228814339355293997534662583810508455936"},
    {0x1.fffffffffffffp+845, chars_format::fixed, 0,
        "4692198018002937122793345990954670375831428595870174308489443188296319695679378273401722169610201525418455"
        "5689002619498455899165544669922035382838385745438216376377382732167596512068191483285027719962680882184593"
        "9606457628678710587995069325167621016911872"},
    {0x1.fffffffffffffp+846, chars_format::fixed, 0,
        "9384396036005874245586691981909340751662857191740348616978886376592639391358756546803444339220403050836911"
        "1378005238996911798331089339844070765676771490876432752754765464335193024136382966570055439925361764369187"
        "9212915257357421175990138650335242033823744"},
    {0x1.fffffffffffffp+847, chars_format::fixed, 0,
        "1876879207201174849117338396381868150332571438348069723395777275318527878271751309360688867844080610167382"
        "2275601047799382359666217867968814153135354298175286550550953092867038604827276593314011087985072352873837"
        "58425830514714842351980277300670484067647488"},
    {0x1.fffffffffffffp+848, chars_format::fixed, 0,
        "3753758414402349698234676792763736300665142876696139446791554550637055756543502618721377735688161220334764"
        "4551202095598764719332435735937628306270708596350573101101906185734077209654553186628022175970144705747675"
        "16851661029429684703960554601340968135294976"},
    {0x1.fffffffffffffp+849, chars_format::fixed, 0,
        "7507516828804699396469353585527472601330285753392278893583109101274111513087005237442755471376322440669528"
        "9102404191197529438664871471875256612541417192701146202203812371468154419309106373256044351940289411495350"
        "33703322058859369407921109202681936270589952"},
    {0x1.fffffffffffffp+850, chars_format::fixed, 0,
        "1501503365760939879293870717105494520266057150678455778716621820254822302617401047488551094275264488133905"
        "7820480838239505887732974294375051322508283438540229240440762474293630883861821274651208870388057882299070"
        "067406644117718738815842218405363872541179904"},
    {0x1.fffffffffffffp+851, chars_format::fixed, 0,
        "3003006731521879758587741434210989040532114301356911557433243640509644605234802094977102188550528976267811"
        "5640961676479011775465948588750102645016566877080458480881524948587261767723642549302417740776115764598140"
        "134813288235437477631684436810727745082359808"},
    {0x1.fffffffffffffp+852, chars_format::fixed, 0,
        "6006013463043759517175482868421978081064228602713823114866487281019289210469604189954204377101057952535623"
        "1281923352958023550931897177500205290033133754160916961763049897174523535447285098604835481552231529196280"
        "269626576470874955263368873621455490164719616"},
    {0x1.fffffffffffffp+853, chars_format::fixed, 0,
        "1201202692608751903435096573684395616212845720542764622973297456203857842093920837990840875420211590507124"
        "6256384670591604710186379435500041058006626750832183392352609979434904707089457019720967096310446305839256"
        "0539253152941749910526737747242910980329439232"},
    {0x1.fffffffffffffp+854, chars_format::fixed, 0,
        "2402405385217503806870193147368791232425691441085529245946594912407715684187841675981681750840423181014249"
        "2512769341183209420372758871000082116013253501664366784705219958869809414178914039441934192620892611678512"
        "1078506305883499821053475494485821960658878464"},
    {0x1.fffffffffffffp+855, chars_format::fixed, 0,
        "4804810770435007613740386294737582464851382882171058491893189824815431368375683351963363501680846362028498"
        "5025538682366418840745517742000164232026507003328733569410439917739618828357828078883868385241785223357024"
        "2157012611766999642106950988971643921317756928"},
    {0x1.fffffffffffffp+856, chars_format::fixed, 0,
        "9609621540870015227480772589475164929702765764342116983786379649630862736751366703926727003361692724056997"
        "0051077364732837681491035484000328464053014006657467138820879835479237656715656157767736770483570446714048"
        "4314025223533999284213901977943287842635513856"},
    {0x1.fffffffffffffp+857, chars_format::fixed, 0,
        "1921924308174003045496154517895032985940553152868423396757275929926172547350273340785345400672338544811399"
        "4010215472946567536298207096800065692810602801331493427764175967095847531343131231553547354096714089342809"
        "68628050447067998568427803955886575685271027712"},
    {0x1.fffffffffffffp+858, chars_format::fixed, 0,
        "3843848616348006090992309035790065971881106305736846793514551859852345094700546681570690801344677089622798"
        "8020430945893135072596414193600131385621205602662986855528351934191695062686262463107094708193428178685619"
        "37256100894135997136855607911773151370542055424"},
    {0x1.fffffffffffffp+859, chars_format::fixed, 0,
        "7687697232696012181984618071580131943762212611473693587029103719704690189401093363141381602689354179245597"
        "6040861891786270145192828387200262771242411205325973711056703868383390125372524926214189416386856357371238"
        "74512201788271994273711215823546302741084110848"},
    {0x1.fffffffffffffp+860, chars_format::fixed, 0,
        "1537539446539202436396923614316026388752442522294738717405820743940938037880218672628276320537870835849119"
        "5208172378357254029038565677440052554248482241065194742211340773676678025074504985242837883277371271474247"
        "749024403576543988547422431647092605482168221696"},
    {0x1.fffffffffffffp+861, chars_format::fixed, 0,
        "3075078893078404872793847228632052777504885044589477434811641487881876075760437345256552641075741671698239"
        "0416344756714508058077131354880105108496964482130389484422681547353356050149009970485675766554742542948495"
        "498048807153087977094844863294185210964336443392"},
    {0x1.fffffffffffffp+862, chars_format::fixed, 0,
        "6150157786156809745587694457264105555009770089178954869623282975763752151520874690513105282151483343396478"
        "0832689513429016116154262709760210216993928964260778968845363094706712100298019940971351533109485085896990"
        "996097614306175954189689726588370421928672886784"},
    {0x1.fffffffffffffp+863, chars_format::fixed, 0,
        "1230031557231361949117538891452821111001954017835790973924656595152750430304174938102621056430296668679295"
        "6166537902685803223230852541952042043398785792852155793769072618941342420059603988194270306621897017179398"
        "1992195228612351908379379453176740843857345773568"},
    {0x1.fffffffffffffp+864, chars_format::fixed, 0,
        "2460063114462723898235077782905642222003908035671581947849313190305500860608349876205242112860593337358591"
        "2333075805371606446461705083904084086797571585704311587538145237882684840119207976388540613243794034358796"
        "3984390457224703816758758906353481687714691547136"},
    {0x1.fffffffffffffp+865, chars_format::fixed, 0,
        "4920126228925447796470155565811284444007816071343163895698626380611001721216699752410484225721186674717182"
        "4666151610743212892923410167808168173595143171408623175076290475765369680238415952777081226487588068717592"
        "7968780914449407633517517812706963375429383094272"},
    {0x1.fffffffffffffp+866, chars_format::fixed, 0,
        "9840252457850895592940311131622568888015632142686327791397252761222003442433399504820968451442373349434364"
        "9332303221486425785846820335616336347190286342817246350152580951530739360476831905554162452975176137435185"
        "5937561828898815267035035625413926750858766188544"},
    {0x1.fffffffffffffp+867, chars_format::fixed, 0,
        "1968050491570179118588062226324513777603126428537265558279450552244400688486679900964193690288474669886872"
        "9866460644297285157169364067123267269438057268563449270030516190306147872095366381110832490595035227487037"
        "11875123657797630534070071250827853501717532377088"},
    {0x1.fffffffffffffp+868, chars_format::fixed, 0,
        "3936100983140358237176124452649027555206252857074531116558901104488801376973359801928387380576949339773745"
        "9732921288594570314338728134246534538876114537126898540061032380612295744190732762221664981190070454974074"
        "23750247315595261068140142501655707003435064754176"},
    {0x1.fffffffffffffp+869, chars_format::fixed, 0,
        "7872201966280716474352248905298055110412505714149062233117802208977602753946719603856774761153898679547491"
        "9465842577189140628677456268493069077752229074253797080122064761224591488381465524443329962380140909948148"
        "47500494631190522136280285003311414006870129508352"},
    {0x1.fffffffffffffp+870, chars_format::fixed, 0,
        "1574440393256143294870449781059611022082501142829812446623560441795520550789343920771354952230779735909498"
        "3893168515437828125735491253698613815550445814850759416024412952244918297676293104888665992476028181989629"
        "695000989262381044272560570006622828013740259016704"},
    {0x1.fffffffffffffp+871, chars_format::fixed, 0,
        "3148880786512286589740899562119222044165002285659624893247120883591041101578687841542709904461559471818996"
        "7786337030875656251470982507397227631100891629701518832048825904489836595352586209777331984952056363979259"
        "390001978524762088545121140013245656027480518033408"},
    {0x1.fffffffffffffp+872, chars_format::fixed, 0,
        "6297761573024573179481799124238444088330004571319249786494241767182082203157375683085419808923118943637993"
        "5572674061751312502941965014794455262201783259403037664097651808979673190705172419554663969904112727958518"
        "780003957049524177090242280026491312054961036066816"},
    {0x1.fffffffffffffp+873, chars_format::fixed, 0,
        "1259552314604914635896359824847688817666000914263849957298848353436416440631475136617083961784623788727598"
        "7114534812350262500588393002958891052440356651880607532819530361795934638141034483910932793980822545591703"
        "7560007914099048354180484560052982624109922072133632"},
    {0x1.fffffffffffffp+874, chars_format::fixed, 0,
        "2519104629209829271792719649695377635332001828527699914597696706872832881262950273234167923569247577455197"
        "4229069624700525001176786005917782104880713303761215065639060723591869276282068967821865587961645091183407"
        "5120015828198096708360969120105965248219844144267264"},
    {0x1.fffffffffffffp+875, chars_format::fixed, 0,
        "5038209258419658543585439299390755270664003657055399829195393413745665762525900546468335847138495154910394"
        "8458139249401050002353572011835564209761426607522430131278121447183738552564137935643731175923290182366815"
        "0240031656396193416721938240211930496439688288534528"},
    {0x1.fffffffffffffp+876, chars_format::fixed, 0,
        "1007641851683931708717087859878151054132800731411079965839078682749133152505180109293667169427699030982078"
        "9691627849880210000470714402367112841952285321504486026255624289436747710512827587128746235184658036473363"
        "00480063312792386833443876480423860992879376577069056"},
    {0x1.fffffffffffffp+877, chars_format::fixed, 0,
        "2015283703367863417434175719756302108265601462822159931678157365498266305010360218587334338855398061964157"
        "9383255699760420000941428804734225683904570643008972052511248578873495421025655174257492470369316072946726"
        "00960126625584773666887752960847721985758753154138112"},
    {0x1.fffffffffffffp+878, chars_format::fixed, 0,
        "4030567406735726834868351439512604216531202925644319863356314730996532610020720437174668677710796123928315"
        "8766511399520840001882857609468451367809141286017944105022497157746990842051310348514984940738632145893452"
        "01920253251169547333775505921695443971517506308276224"},
    {0x1.fffffffffffffp+879, chars_format::fixed, 0,
        "8061134813471453669736702879025208433062405851288639726712629461993065220041440874349337355421592247856631"
        "7533022799041680003765715218936902735618282572035888210044994315493981684102620697029969881477264291786904"
        "03840506502339094667551011843390887943035012616552448"},
    {0x1.fffffffffffffp+880, chars_format::fixed, 0,
        "1612226962694290733947340575805041686612481170257727945342525892398613044008288174869867471084318449571326"
        "3506604559808336000753143043787380547123656514407177642008998863098796336820524139405993976295452858357380"
        "807681013004678189335102023686781775886070025233104896"},
    {0x1.fffffffffffffp+881, chars_format::fixed, 0,
        "3224453925388581467894681151610083373224962340515455890685051784797226088016576349739734942168636899142652"
        "7013209119616672001506286087574761094247313028814355284017997726197592673641048278811987952590905716714761"
        "615362026009356378670204047373563551772140050466209792"},
    {0x1.fffffffffffffp+882, chars_format::fixed, 0,
        "6448907850777162935789362303220166746449924681030911781370103569594452176033152699479469884337273798285305"
        "4026418239233344003012572175149522188494626057628710568035995452395185347282096557623975905181811433429523"
        "230724052018712757340408094747127103544280100932419584"},
    {0x1.fffffffffffffp+883, chars_format::fixed, 0,
        "1289781570155432587157872460644033349289984936206182356274020713918890435206630539895893976867454759657061"
        "0805283647846668800602514435029904437698925211525742113607199090479037069456419311524795181036362286685904"
        "6461448104037425514680816189494254207088560201864839168"},
    {0x1.fffffffffffffp+884, chars_format::fixed, 0,
        "2579563140310865174315744921288066698579969872412364712548041427837780870413261079791787953734909519314122"
        "1610567295693337601205028870059808875397850423051484227214398180958074138912838623049590362072724573371809"
        "2922896208074851029361632378988508414177120403729678336"},
    {0x1.fffffffffffffp+885, chars_format::fixed, 0,
        "5159126280621730348631489842576133397159939744824729425096082855675561740826522159583575907469819038628244"
        "3221134591386675202410057740119617750795700846102968454428796361916148277825677246099180724145449146743618"
        "5845792416149702058723264757977016828354240807459356672"},
    {0x1.fffffffffffffp+886, chars_format::fixed, 0,
        "1031825256124346069726297968515226679431987948964945885019216571135112348165304431916715181493963807725648"
        "8644226918277335040482011548023923550159140169220593690885759272383229655565135449219836144829089829348723"
        "71691584832299404117446529515954033656708481614918713344"},
    {0x1.fffffffffffffp+887, chars_format::fixed, 0,
        "2063650512248692139452595937030453358863975897929891770038433142270224696330608863833430362987927615451297"
        "7288453836554670080964023096047847100318280338441187381771518544766459311130270898439672289658179658697447"
        "43383169664598808234893059031908067313416963229837426688"},
    {0x1.fffffffffffffp+888, chars_format::fixed, 0,
        "4127301024497384278905191874060906717727951795859783540076866284540449392661217727666860725975855230902595"
        "4576907673109340161928046192095694200636560676882374763543037089532918622260541796879344579316359317394894"
        "86766339329197616469786118063816134626833926459674853376"},
    {0x1.fffffffffffffp+889, chars_format::fixed, 0,
        "8254602048994768557810383748121813435455903591719567080153732569080898785322435455333721451951710461805190"
        "9153815346218680323856092384191388401273121353764749527086074179065837244521083593758689158632718634789789"
        "73532678658395232939572236127632269253667852919349706752"},
    {0x1.fffffffffffffp+890, chars_format::fixed, 0,
        "1650920409798953711562076749624362687091180718343913416030746513816179757064487091066744290390342092361038"
        "1830763069243736064771218476838277680254624270752949905417214835813167448904216718751737831726543726957957"
        "947065357316790465879144472255264538507335705838699413504"},
    {0x1.fffffffffffffp+891, chars_format::fixed, 0,
        "3301840819597907423124153499248725374182361436687826832061493027632359514128974182133488580780684184722076"
        "3661526138487472129542436953676555360509248541505899810834429671626334897808433437503475663453087453915915"
        "894130714633580931758288944510529077014671411677398827008"},
    {0x1.fffffffffffffp+892, chars_format::fixed, 0,
        "6603681639195814846248306998497450748364722873375653664122986055264719028257948364266977161561368369444152"
        "7323052276974944259084873907353110721018497083011799621668859343252669795616866875006951326906174907831831"
        "788261429267161863516577889021058154029342823354797654016"},
    {0x1.fffffffffffffp+893, chars_format::fixed, 0,
        "1320736327839162969249661399699490149672944574675130732824597211052943805651589672853395432312273673888830"
        "5464610455394988851816974781470622144203699416602359924333771868650533959123373375001390265381234981566366"
        "3576522858534323727033155778042116308058685646709595308032"},
    {0x1.fffffffffffffp+894, chars_format::fixed, 0,
        "2641472655678325938499322799398980299345889149350261465649194422105887611303179345706790864624547347777661"
        "0929220910789977703633949562941244288407398833204719848667543737301067918246746750002780530762469963132732"
        "7153045717068647454066311556084232616117371293419190616064"},
    {0x1.fffffffffffffp+895, chars_format::fixed, 0,
        "5282945311356651876998645598797960598691778298700522931298388844211775222606358691413581729249094695555322"
        "1858441821579955407267899125882488576814797666409439697335087474602135836493493500005561061524939926265465"
        "4306091434137294908132623112168465232234742586838381232128"},
    {0x1.fffffffffffffp+896, chars_format::fixed, 0,
        "1056589062271330375399729119759592119738355659740104586259677768842355044521271738282716345849818939111064"
        "4371688364315991081453579825176497715362959533281887939467017494920427167298698700001112212304987985253093"
        "08612182868274589816265246224336930464469485173676762464256"},
    {0x1.fffffffffffffp+897, chars_format::fixed, 0,
        "2113178124542660750799458239519184239476711319480209172519355537684710089042543476565432691699637878222128"
        "8743376728631982162907159650352995430725919066563775878934034989840854334597397400002224424609975970506186"
        "17224365736549179632530492448673860928938970347353524928512"},
    {0x1.fffffffffffffp+898, chars_format::fixed, 0,
        "4226356249085321501598916479038368478953422638960418345038711075369420178085086953130865383399275756444257"
        "7486753457263964325814319300705990861451838133127551757868069979681708669194794800004448849219951941012372"
        "34448731473098359265060984897347721857877940694707049857024"},
    {0x1.fffffffffffffp+899, chars_format::fixed, 0,
        "8452712498170643003197832958076736957906845277920836690077422150738840356170173906261730766798551512888515"
        "4973506914527928651628638601411981722903676266255103515736139959363417338389589600008897698439903882024744"
        "68897462946196718530121969794695443715755881389414099714048"},
    {0x1.fffffffffffffp+900, chars_format::fixed, 0,
        "1690542499634128600639566591615347391581369055584167338015484430147768071234034781252346153359710302577703"
        "0994701382905585730325727720282396344580735253251020703147227991872683467677917920001779539687980776404948"
        "937794925892393437060243939589390887431511762778828199428096"},
    {0x1.fffffffffffffp+901, chars_format::fixed, 0,
        "3381084999268257201279133183230694783162738111168334676030968860295536142468069562504692306719420605155406"
        "1989402765811171460651455440564792689161470506502041406294455983745366935355835840003559079375961552809897"
        "875589851784786874120487879178781774863023525557656398856192"},
    {0x1.fffffffffffffp+902, chars_format::fixed, 0,
        "6762169998536514402558266366461389566325476222336669352061937720591072284936139125009384613438841210310812"
        "3978805531622342921302910881129585378322941013004082812588911967490733870711671680007118158751923105619795"
        "751179703569573748240975758357563549726047051115312797712384"},
    {0x1.fffffffffffffp+903, chars_format::fixed, 0,
        "1352433999707302880511653273292277913265095244467333870412387544118214456987227825001876922687768242062162"
        "4795761106324468584260582176225917075664588202600816562517782393498146774142334336001423631750384621123959"
        "1502359407139147496481951516715127099452094102230625595424768"},
    {0x1.fffffffffffffp+904, chars_format::fixed, 0,
        "2704867999414605761023306546584555826530190488934667740824775088236428913974455650003753845375536484124324"
        "9591522212648937168521164352451834151329176405201633125035564786996293548284668672002847263500769242247918"
        "3004718814278294992963903033430254198904188204461251190849536"},
    {0x1.fffffffffffffp+905, chars_format::fixed, 0,
        "5409735998829211522046613093169111653060380977869335481649550176472857827948911300007507690751072968248649"
        "9183044425297874337042328704903668302658352810403266250071129573992587096569337344005694527001538484495836"
        "6009437628556589985927806066860508397808376408922502381699072"},
    {0x1.fffffffffffffp+906, chars_format::fixed, 0,
        "1081947199765842304409322618633822330612076195573867096329910035294571565589782260001501538150214593649729"
        "9836608885059574867408465740980733660531670562080653250014225914798517419313867468801138905400307696899167"
        "32018875257113179971855612133721016795616752817845004763398144"},
    {0x1.fffffffffffffp+907, chars_format::fixed, 0,
        "2163894399531684608818645237267644661224152391147734192659820070589143131179564520003003076300429187299459"
        "9673217770119149734816931481961467321063341124161306500028451829597034838627734937602277810800615393798334"
        "64037750514226359943711224267442033591233505635690009526796288"},
    {0x1.fffffffffffffp+908, chars_format::fixed, 0,
        "4327788799063369217637290474535289322448304782295468385319640141178286262359129040006006152600858374598919"
        "9346435540238299469633862963922934642126682248322613000056903659194069677255469875204555621601230787596669"
        "28075501028452719887422448534884067182467011271380019053592576"},
    {0x1.fffffffffffffp+909, chars_format::fixed, 0,
        "8655577598126738435274580949070578644896609564590936770639280282356572524718258080012012305201716749197839"
        "8692871080476598939267725927845869284253364496645226000113807318388139354510939750409111243202461575193338"
        "56151002056905439774844897069768134364934022542760038107185152"},
    {0x1.fffffffffffffp+910, chars_format::fixed, 0,
        "1731115519625347687054916189814115728979321912918187354127856056471314504943651616002402461040343349839567"
        "9738574216095319787853545185569173856850672899329045200022761463677627870902187950081822248640492315038667"
        "712302004113810879549689794139536268729868045085520076214370304"},
    {0x1.fffffffffffffp+911, chars_format::fixed, 0,
        "3462231039250695374109832379628231457958643825836374708255712112942629009887303232004804922080686699679135"
        "9477148432190639575707090371138347713701345798658090400045522927355255741804375900163644497280984630077335"
        "424604008227621759099379588279072537459736090171040152428740608"},
    {0x1.fffffffffffffp+912, chars_format::fixed, 0,
        "6924462078501390748219664759256462915917287651672749416511424225885258019774606464009609844161373399358271"
        "8954296864381279151414180742276695427402691597316180800091045854710511483608751800327288994561969260154670"
        "849208016455243518198759176558145074919472180342080304857481216"},
    {0x1.fffffffffffffp+913, chars_format::fixed, 0,
        "1384892415700278149643932951851292583183457530334549883302284845177051603954921292801921968832274679871654"
        "3790859372876255830282836148455339085480538319463236160018209170942102296721750360065457798912393852030934"
        "1698416032910487036397518353116290149838944360684160609714962432"},
    {0x1.fffffffffffffp+914, chars_format::fixed, 0,
        "2769784831400556299287865903702585166366915060669099766604569690354103207909842585603843937664549359743308"
        "7581718745752511660565672296910678170961076638926472320036418341884204593443500720130915597824787704061868"
        "3396832065820974072795036706232580299677888721368321219429924864"},
    {0x1.fffffffffffffp+915, chars_format::fixed, 0,
        "5539569662801112598575731807405170332733830121338199533209139380708206415819685171207687875329098719486617"
        "5163437491505023321131344593821356341922153277852944640072836683768409186887001440261831195649575408123736"
        "6793664131641948145590073412465160599355777442736642438859849728"},
    {0x1.fffffffffffffp+916, chars_format::fixed, 0,
        "1107913932560222519715146361481034066546766024267639906641827876141641283163937034241537575065819743897323"
        "5032687498301004664226268918764271268384430655570588928014567336753681837377400288052366239129915081624747"
        "33587328263283896291180146824930321198711554885473284877719699456"},
    {0x1.fffffffffffffp+917, chars_format::fixed, 0,
        "2215827865120445039430292722962068133093532048535279813283655752283282566327874068483075150131639487794647"
        "0065374996602009328452537837528542536768861311141177856029134673507363674754800576104732478259830163249494"
        "67174656526567792582360293649860642397423109770946569755439398912"},
    {0x1.fffffffffffffp+918, chars_format::fixed, 0,
        "4431655730240890078860585445924136266187064097070559626567311504566565132655748136966150300263278975589294"
        "0130749993204018656905075675057085073537722622282355712058269347014727349509601152209464956519660326498989"
        "34349313053135585164720587299721284794846219541893139510878797824"},
    {0x1.fffffffffffffp+919, chars_format::fixed, 0,
        "8863311460481780157721170891848272532374128194141119253134623009133130265311496273932300600526557951178588"
        "0261499986408037313810151350114170147075445244564711424116538694029454699019202304418929913039320652997978"
        "68698626106271170329441174599442569589692439083786279021757595648"},
    {0x1.fffffffffffffp+920, chars_format::fixed, 0,
        "1772662292096356031544234178369654506474825638828223850626924601826626053062299254786460120105311590235717"
        "6052299997281607462762030270022834029415089048912942284823307738805890939803840460883785982607864130599595"
        "737397252212542340658882349198885139179384878167572558043515191296"},
    {0x1.fffffffffffffp+921, chars_format::fixed, 0,
        "3545324584192712063088468356739309012949651277656447701253849203653252106124598509572920240210623180471435"
        "2104599994563214925524060540045668058830178097825884569646615477611781879607680921767571965215728261199191"
        "474794504425084681317764698397770278358769756335145116087030382592"},
    {0x1.fffffffffffffp+922, chars_format::fixed, 0,
        "7090649168385424126176936713478618025899302555312895402507698407306504212249197019145840480421246360942870"
        "4209199989126429851048121080091336117660356195651769139293230955223563759215361843535143930431456522398382"
        "949589008850169362635529396795540556717539512670290232174060765184"},
    {0x1.fffffffffffffp+923, chars_format::fixed, 0,
        "1418129833677084825235387342695723605179860511062579080501539681461300842449839403829168096084249272188574"
        "0841839997825285970209624216018267223532071239130353827858646191044712751843072368707028786086291304479676"
        "5899178017700338725271058793591081113435079025340580464348121530368"},
    {0x1.fffffffffffffp+924, chars_format::fixed, 0,
        "2836259667354169650470774685391447210359721022125158161003079362922601684899678807658336192168498544377148"
        "1683679995650571940419248432036534447064142478260707655717292382089425503686144737414057572172582608959353"
        "1798356035400677450542117587182162226870158050681160928696243060736"},
    {0x1.fffffffffffffp+925, chars_format::fixed, 0,
        "5672519334708339300941549370782894420719442044250316322006158725845203369799357615316672384336997088754296"
        "3367359991301143880838496864073068894128284956521415311434584764178851007372289474828115144345165217918706"
        "3596712070801354901084235174364324453740316101362321857392486121472"},
    {0x1.fffffffffffffp+926, chars_format::fixed, 0,
        "1134503866941667860188309874156578884143888408850063264401231745169040673959871523063334476867399417750859"
        "2673471998260228776167699372814613778825656991304283062286916952835770201474457894965623028869033043583741"
        "27193424141602709802168470348728648907480632202724643714784972242944"},
    {0x1.fffffffffffffp+927, chars_format::fixed, 0,
        "2269007733883335720376619748313157768287776817700126528802463490338081347919743046126668953734798835501718"
        "5346943996520457552335398745629227557651313982608566124573833905671540402948915789931246057738066087167482"
        "54386848283205419604336940697457297814961264405449287429569944485888"},
    {0x1.fffffffffffffp+928, chars_format::fixed, 0,
        "4538015467766671440753239496626315536575553635400253057604926980676162695839486092253337907469597671003437"
        "0693887993040915104670797491258455115302627965217132249147667811343080805897831579862492115476132174334965"
        "08773696566410839208673881394914595629922528810898574859139888971776"},
    {0x1.fffffffffffffp+929, chars_format::fixed, 0,
        "9076030935533342881506478993252631073151107270800506115209853961352325391678972184506675814939195342006874"
        "1387775986081830209341594982516910230605255930434264498295335622686161611795663159724984230952264348669930"
        "17547393132821678417347762789829191259845057621797149718279777943552"},
    {0x1.fffffffffffffp+930, chars_format::fixed, 0,
        "1815206187106668576301295798650526214630221454160101223041970792270465078335794436901335162987839068401374"
        "8277555197216366041868318996503382046121051186086852899659067124537232322359132631944996846190452869733986"
        "035094786265643356834695525579658382519690115243594299436559555887104"},
    {0x1.fffffffffffffp+931, chars_format::fixed, 0,
        "3630412374213337152602591597301052429260442908320202446083941584540930156671588873802670325975678136802749"
        "6555110394432732083736637993006764092242102372173705799318134249074464644718265263889993692380905739467972"
        "070189572531286713669391051159316765039380230487188598873119111774208"},
    {0x1.fffffffffffffp+932, chars_format::fixed, 0,
        "7260824748426674305205183194602104858520885816640404892167883169081860313343177747605340651951356273605499"
        "3110220788865464167473275986013528184484204744347411598636268498148929289436530527779987384761811478935944"
        "140379145062573427338782102318633530078760460974377197746238223548416"},
    {0x1.fffffffffffffp+933, chars_format::fixed, 0,
        "1452164949685334861041036638920420971704177163328080978433576633816372062668635549521068130390271254721099"
        "8622044157773092833494655197202705636896840948869482319727253699629785857887306105555997476952362295787188"
        "8280758290125146854677564204637267060157520921948754395492476447096832"},
    {0x1.fffffffffffffp+934, chars_format::fixed, 0,
        "2904329899370669722082073277840841943408354326656161956867153267632744125337271099042136260780542509442199"
        "7244088315546185666989310394405411273793681897738964639454507399259571715774612211111994953904724591574377"
        "6561516580250293709355128409274534120315041843897508790984952894193664"},
    {0x1.fffffffffffffp+935, chars_format::fixed, 0,
        "5808659798741339444164146555681683886816708653312323913734306535265488250674542198084272521561085018884399"
        "4488176631092371333978620788810822547587363795477929278909014798519143431549224422223989907809449183148755"
        "3123033160500587418710256818549068240630083687795017581969905788387328"},
    {0x1.fffffffffffffp+936, chars_format::fixed, 0,
        "1161731959748267888832829311136336777363341730662464782746861307053097650134908439616854504312217003776879"
        "8897635326218474266795724157762164509517472759095585855781802959703828686309844884444797981561889836629751"
        "06246066321001174837420513637098136481260167375590035163939811576774656"},
    {0x1.fffffffffffffp+937, chars_format::fixed, 0,
        "2323463919496535777665658622272673554726683461324929565493722614106195300269816879233709008624434007553759"
        "7795270652436948533591448315524329019034945518191171711563605919407657372619689768889595963123779673259502"
        "12492132642002349674841027274196272962520334751180070327879623153549312"},
    {0x1.fffffffffffffp+938, chars_format::fixed, 0,
        "4646927838993071555331317244545347109453366922649859130987445228212390600539633758467418017248868015107519"
        "5590541304873897067182896631048658038069891036382343423127211838815314745239379537779191926247559346519004"
        "24984265284004699349682054548392545925040669502360140655759246307098624"},
    {0x1.fffffffffffffp+939, chars_format::fixed, 0,
        "9293855677986143110662634489090694218906733845299718261974890456424781201079267516934836034497736030215039"
        "1181082609747794134365793262097316076139782072764686846254423677630629490478759075558383852495118693038008"
        "49968530568009398699364109096785091850081339004720281311518492614197248"},
    {0x1.fffffffffffffp+940, chars_format::fixed, 0,
        "1858771135597228622132526897818138843781346769059943652394978091284956240215853503386967206899547206043007"
        "8236216521949558826873158652419463215227956414552937369250884735526125898095751815111676770499023738607601"
        "699937061136018797398728218193570183700162678009440562623036985228394496"},
    {0x1.fffffffffffffp+941, chars_format::fixed, 0,
        "3717542271194457244265053795636277687562693538119887304789956182569912480431707006773934413799094412086015"
        "6472433043899117653746317304838926430455912829105874738501769471052251796191503630223353540998047477215203"
        "399874122272037594797456436387140367400325356018881125246073970456788992"},
    {0x1.fffffffffffffp+942, chars_format::fixed, 0,
        "7435084542388914488530107591272555375125387076239774609579912365139824960863414013547868827598188824172031"
        "2944866087798235307492634609677852860911825658211749477003538942104503592383007260446707081996094954430406"
        "799748244544075189594912872774280734800650712037762250492147940913577984"},
    {0x1.fffffffffffffp+943, chars_format::fixed, 0,
        "1487016908477782897706021518254511075025077415247954921915982473027964992172682802709573765519637764834406"
        "2588973217559647061498526921935570572182365131642349895400707788420900718476601452089341416399218990886081"
        "3599496489088150379189825745548561469601301424075524500984295881827155968"},
    {0x1.fffffffffffffp+944, chars_format::fixed, 0,
        "2974033816955565795412043036509022150050154830495909843831964946055929984345365605419147531039275529668812"
        "5177946435119294122997053843871141144364730263284699790801415576841801436953202904178682832798437981772162"
        "7198992978176300758379651491097122939202602848151049001968591763654311936"},
    {0x1.fffffffffffffp+945, chars_format::fixed, 0,
        "5948067633911131590824086073018044300100309660991819687663929892111859968690731210838295062078551059337625"
        "0355892870238588245994107687742282288729460526569399581602831153683602873906405808357365665596875963544325"
        "4397985956352601516759302982194245878405205696302098003937183527308623872"},
    {0x1.fffffffffffffp+946, chars_format::fixed, 0,
        "1189613526782226318164817214603608860020061932198363937532785978422371993738146242167659012415710211867525"
        "0071178574047717649198821537548456457745892105313879916320566230736720574781281161671473133119375192708865"
        "08795971912705203033518605964388491756810411392604196007874367054617247744"},
    {0x1.fffffffffffffp+947, chars_format::fixed, 0,
        "2379227053564452636329634429207217720040123864396727875065571956844743987476292484335318024831420423735050"
        "0142357148095435298397643075096912915491784210627759832641132461473441149562562323342946266238750385417730"
        "17591943825410406067037211928776983513620822785208392015748734109234495488"},
    {0x1.fffffffffffffp+948, chars_format::fixed, 0,
        "4758454107128905272659268858414435440080247728793455750131143913689487974952584968670636049662840847470100"
        "0284714296190870596795286150193825830983568421255519665282264922946882299125124646685892532477500770835460"
        "35183887650820812134074423857553967027241645570416784031497468218468990976"},
    {0x1.fffffffffffffp+949, chars_format::fixed, 0,
        "9516908214257810545318537716828870880160495457586911500262287827378975949905169937341272099325681694940200"
        "0569428592381741193590572300387651661967136842511039330564529845893764598250249293371785064955001541670920"
        "70367775301641624268148847715107934054483291140833568062994936436937981952"},
    {0x1.fffffffffffffp+950, chars_format::fixed, 0,
        "1903381642851562109063707543365774176032099091517382300052457565475795189981033987468254419865136338988040"
        "0113885718476348238718114460077530332393427368502207866112905969178752919650049858674357012991000308334184"
        "140735550603283248536297695430215868108966582281667136125989872873875963904"},
    {0x1.fffffffffffffp+951, chars_format::fixed, 0,
        "3806763285703124218127415086731548352064198183034764600104915130951590379962067974936508839730272677976080"
        "0227771436952696477436228920155060664786854737004415732225811938357505839300099717348714025982000616668368"
        "281471101206566497072595390860431736217933164563334272251979745747751927808"},
    {0x1.fffffffffffffp+952, chars_format::fixed, 0,
        "7613526571406248436254830173463096704128396366069529200209830261903180759924135949873017679460545355952160"
        "0455542873905392954872457840310121329573709474008831464451623876715011678600199434697428051964001233336736"
        "562942202413132994145190781720863472435866329126668544503959491495503855616"},
    {0x1.fffffffffffffp+953, chars_format::fixed, 0,
        "1522705314281249687250966034692619340825679273213905840041966052380636151984827189974603535892109071190432"
        "0091108574781078590974491568062024265914741894801766292890324775343002335720039886939485610392800246667347"
        "3125884404826265988290381563441726944871732658253337089007918982991007711232"},
    {0x1.fffffffffffffp+954, chars_format::fixed, 0,
        "3045410628562499374501932069385238681651358546427811680083932104761272303969654379949207071784218142380864"
        "0182217149562157181948983136124048531829483789603532585780649550686004671440079773878971220785600493334694"
        "6251768809652531976580763126883453889743465316506674178015837965982015422464"},
    {0x1.fffffffffffffp+955, chars_format::fixed, 0,
        "6090821257124998749003864138770477363302717092855623360167864209522544607939308759898414143568436284761728"
        "0364434299124314363897966272248097063658967579207065171561299101372009342880159547757942441571200986669389"
        "2503537619305063953161526253766907779486930633013348356031675931964030844928"},
    {0x1.fffffffffffffp+956, chars_format::fixed, 0,
        "1218164251424999749800772827754095472660543418571124672033572841904508921587861751979682828713687256952345"
        "6072886859824862872779593254449619412731793515841413034312259820274401868576031909551588488314240197333877"
        "85007075238610127906323052507533815558973861266026696712063351863928061689856"},
    {0x1.fffffffffffffp+957, chars_format::fixed, 0,
        "2436328502849999499601545655508190945321086837142249344067145683809017843175723503959365657427374513904691"
        "2145773719649725745559186508899238825463587031682826068624519640548803737152063819103176976628480394667755"
        "70014150477220255812646105015067631117947722532053393424126703727856123379712"},
    {0x1.fffffffffffffp+958, chars_format::fixed, 0,
        "4872657005699998999203091311016381890642173674284498688134291367618035686351447007918731314854749027809382"
        "4291547439299451491118373017798477650927174063365652137249039281097607474304127638206353953256960789335511"
        "40028300954440511625292210030135262235895445064106786848253407455712246759424"},
    {0x1.fffffffffffffp+959, chars_format::fixed, 0,
        "9745314011399997998406182622032763781284347348568997376268582735236071372702894015837462629709498055618764"
        "8583094878598902982236746035596955301854348126731304274498078562195214948608255276412707906513921578671022"
        "80056601908881023250584420060270524471790890128213573696506814911424493518848"},
    {0x1.fffffffffffffp+960, chars_format::fixed, 0,
        "1949062802279999599681236524406552756256869469713799475253716547047214274540578803167492525941899611123752"
        "9716618975719780596447349207119391060370869625346260854899615712439042989721651055282541581302784315734204"
        "560113203817762046501168840120541048943581780256427147393013629822848987037696"},
    {0x1.fffffffffffffp+961, chars_format::fixed, 0,
        "3898125604559999199362473048813105512513738939427598950507433094094428549081157606334985051883799222247505"
        "9433237951439561192894698414238782120741739250692521709799231424878085979443302110565083162605568631468409"
        "120226407635524093002337680241082097887163560512854294786027259645697974075392"},
    {0x1.fffffffffffffp+962, chars_format::fixed, 0,
        "7796251209119998398724946097626211025027477878855197901014866188188857098162315212669970103767598444495011"
        "8866475902879122385789396828477564241483478501385043419598462849756171958886604221130166325211137262936818"
        "240452815271048186004675360482164195774327121025708589572054519291395948150784"},
    {0x1.fffffffffffffp+963, chars_format::fixed, 0,
        "1559250241823999679744989219525242205005495575771039580202973237637771419632463042533994020753519688899002"
        "3773295180575824477157879365695512848296695700277008683919692569951234391777320844226033265042227452587363"
        "6480905630542096372009350720964328391548654242051417179144109038582791896301568"},
    {0x1.fffffffffffffp+964, chars_format::fixed, 0,
        "3118500483647999359489978439050484410010991151542079160405946475275542839264926085067988041507039377798004"
        "7546590361151648954315758731391025696593391400554017367839385139902468783554641688452066530084454905174727"
        "2961811261084192744018701441928656783097308484102834358288218077165583792603136"},
    {0x1.fffffffffffffp+965, chars_format::fixed, 0,
        "6237000967295998718979956878100968820021982303084158320811892950551085678529852170135976083014078755596009"
        "5093180722303297908631517462782051393186782801108034735678770279804937567109283376904133060168909810349454"
        "5923622522168385488037402883857313566194616968205668716576436154331167585206272"},
    {0x1.fffffffffffffp+966, chars_format::fixed, 0,
        "1247400193459199743795991375620193764004396460616831664162378590110217135705970434027195216602815751119201"
        "9018636144460659581726303492556410278637356560221606947135754055960987513421856675380826612033781962069890"
        "91847245044336770976074805767714627132389233936411337433152872308662335170412544"},
    {0x1.fffffffffffffp+967, chars_format::fixed, 0,
        "2494800386918399487591982751240387528008792921233663328324757180220434271411940868054390433205631502238403"
        "8037272288921319163452606985112820557274713120443213894271508111921975026843713350761653224067563924139781"
        "83694490088673541952149611535429254264778467872822674866305744617324670340825088"},
    {0x1.fffffffffffffp+968, chars_format::fixed, 0,
        "4989600773836798975183965502480775056017585842467326656649514360440868542823881736108780866411263004476807"
        "6074544577842638326905213970225641114549426240886427788543016223843950053687426701523306448135127848279563"
        "67388980177347083904299223070858508529556935745645349732611489234649340681650176"},
    {0x1.fffffffffffffp+969, chars_format::fixed, 0,
        "9979201547673597950367931004961550112035171684934653313299028720881737085647763472217561732822526008953615"
        "2149089155685276653810427940451282229098852481772855577086032447687900107374853403046612896270255696559127"
        "34777960354694167808598446141717017059113871491290699465222978469298681363300352"},
    {0x1.fffffffffffffp+970, chars_format::fixed, 0,
        "1995840309534719590073586200992310022407034336986930662659805744176347417129552694443512346564505201790723"
        "0429817831137055330762085588090256445819770496354571115417206489537580021474970680609322579254051139311825"
        "469555920709388335617196892283434034118227742982581398930445956938597362726600704"},
    {0x1.fffffffffffffp+971, chars_format::fixed, 0,
        "3991680619069439180147172401984620044814068673973861325319611488352694834259105388887024693129010403581446"
        "0859635662274110661524171176180512891639540992709142230834412979075160042949941361218645158508102278623650"
        "939111841418776671234393784566868068236455485965162797860891913877194725453201408"},
    {0x1.fffffffffffffp+972, chars_format::fixed, 0,
        "7983361238138878360294344803969240089628137347947722650639222976705389668518210777774049386258020807162892"
        "1719271324548221323048342352361025783279081985418284461668825958150320085899882722437290317016204557247301"
        "878223682837553342468787569133736136472910971930325595721783827754389450906402816"},
    {0x1.fffffffffffffp+973, chars_format::fixed, 0,
        "1596672247627775672058868960793848017925627469589544530127844595341077933703642155554809877251604161432578"
        "4343854264909644264609668470472205156655816397083656892333765191630064017179976544487458063403240911449460"
        "3756447365675106684937575138267472272945821943860651191443567655508778901812805632"},
    {0x1.fffffffffffffp+974, chars_format::fixed, 0,
        "3193344495255551344117737921587696035851254939179089060255689190682155867407284311109619754503208322865156"
        "8687708529819288529219336940944410313311632794167313784667530383260128034359953088974916126806481822898920"
        "7512894731350213369875150276534944545891643887721302382887135311017557803625611264"},
    {0x1.fffffffffffffp+975, chars_format::fixed, 0,
        "6386688990511102688235475843175392071702509878358178120511378381364311734814568622219239509006416645730313"
        "7375417059638577058438673881888820626623265588334627569335060766520256068719906177949832253612963645797841"
        "5025789462700426739750300553069889091783287775442604765774270622035115607251222528"},
    {0x1.fffffffffffffp+976, chars_format::fixed, 0,
        "1277337798102220537647095168635078414340501975671635624102275676272862346962913724443847901801283329146062"
        "7475083411927715411687734776377764125324653117666925513867012153304051213743981235589966450722592729159568"
        "30051578925400853479500601106139778183566575550885209531548541244070231214502445056"},
    {0x1.fffffffffffffp+977, chars_format::fixed, 0,
        "2554675596204441075294190337270156828681003951343271248204551352545724693925827448887695803602566658292125"
        "4950166823855430823375469552755528250649306235333851027734024306608102427487962471179932901445185458319136"
        "60103157850801706959001202212279556367133151101770419063097082488140462429004890112"},
    {0x1.fffffffffffffp+978, chars_format::fixed, 0,
        "5109351192408882150588380674540313657362007902686542496409102705091449387851654897775391607205133316584250"
        "9900333647710861646750939105511056501298612470667702055468048613216204854975924942359865802890370916638273"
        "20206315701603413918002404424559112734266302203540838126194164976280924858009780224"},
    {0x1.fffffffffffffp+979, chars_format::fixed, 0,
        "1021870238481776430117676134908062731472401580537308499281820541018289877570330979555078321441026663316850"
        "1980066729542172329350187821102211300259722494133540411093609722643240970995184988471973160578074183327654"
        "640412631403206827836004808849118225468532604407081676252388329952561849716019560448"},
    {0x1.fffffffffffffp+980, chars_format::fixed, 0,
        "2043740476963552860235352269816125462944803161074616998563641082036579755140661959110156642882053326633700"
        "3960133459084344658700375642204422600519444988267080822187219445286481941990369976943946321156148366655309"
        "280825262806413655672009617698236450937065208814163352504776659905123699432039120896"},
    {0x1.fffffffffffffp+981, chars_format::fixed, 0,
        "4087480953927105720470704539632250925889606322149233997127282164073159510281323918220313285764106653267400"
        "7920266918168689317400751284408845201038889976534161644374438890572963883980739953887892642312296733310618"
        "561650525612827311344019235396472901874130417628326705009553319810247398864078241792"},
    {0x1.fffffffffffffp+982, chars_format::fixed, 0,
        "8174961907854211440941409079264501851779212644298467994254564328146319020562647836440626571528213306534801"
        "5840533836337378634801502568817690402077779953068323288748877781145927767961479907775785284624593466621237"
        "123301051225654622688038470792945803748260835256653410019106639620494797728156483584"},
    {0x1.fffffffffffffp+983, chars_format::fixed, 0,
        "1634992381570842288188281815852900370355842528859693598850912865629263804112529567288125314305642661306960"
        "3168106767267475726960300513763538080415555990613664657749775556229185553592295981555157056924918693324247"
        "4246602102451309245376076941585891607496521670513306820038213279240989595456312967168"},
    {0x1.fffffffffffffp+984, chars_format::fixed, 0,
        "3269984763141684576376563631705800740711685057719387197701825731258527608225059134576250628611285322613920"
        "6336213534534951453920601027527076160831111981227329315499551112458371107184591963110314113849837386648494"
        "8493204204902618490752153883171783214993043341026613640076426558481979190912625934336"},
    {0x1.fffffffffffffp+985, chars_format::fixed, 0,
        "6539969526283369152753127263411601481423370115438774395403651462517055216450118269152501257222570645227841"
        "2672427069069902907841202055054152321662223962454658630999102224916742214369183926220628227699674773296989"
        "6986408409805236981504307766343566429986086682053227280152853116963958381825251868672"},
    {0x1.fffffffffffffp+986, chars_format::fixed, 0,
        "1307993905256673830550625452682320296284674023087754879080730292503411043290023653830500251444514129045568"
        "2534485413813980581568240411010830464332444792490931726199820444983348442873836785244125645539934954659397"
        "93972816819610473963008615532687132859972173364106454560305706233927916763650503737344"},
    {0x1.fffffffffffffp+987, chars_format::fixed, 0,
        "2615987810513347661101250905364640592569348046175509758161460585006822086580047307661000502889028258091136"
        "5068970827627961163136480822021660928664889584981863452399640889966696885747673570488251291079869909318795"
        "87945633639220947926017231065374265719944346728212909120611412467855833527301007474688"},
    {0x1.fffffffffffffp+988, chars_format::fixed, 0,
        "5231975621026695322202501810729281185138696092351019516322921170013644173160094615322001005778056516182273"
        "0137941655255922326272961644043321857329779169963726904799281779933393771495347140976502582159739818637591"
        "75891267278441895852034462130748531439888693456425818241222824935711667054602014949376"},
    {0x1.fffffffffffffp+989, chars_format::fixed, 0,
        "1046395124205339064440500362145856237027739218470203903264584234002728834632018923064400201155611303236454"
        "6027588331051184465254592328808664371465955833992745380959856355986678754299069428195300516431947963727518"
        "351782534556883791704068924261497062879777386912851636482445649871423334109204029898752"},
    {0x1.fffffffffffffp+990, chars_format::fixed, 0,
        "2092790248410678128881000724291712474055478436940407806529168468005457669264037846128800402311222606472909"
        "2055176662102368930509184657617328742931911667985490761919712711973357508598138856390601032863895927455036"
        "703565069113767583408137848522994125759554773825703272964891299742846668218408059797504"},
    {0x1.fffffffffffffp+991, chars_format::fixed, 0,
        "4185580496821356257762001448583424948110956873880815613058336936010915338528075692257600804622445212945818"
        "4110353324204737861018369315234657485863823335970981523839425423946715017196277712781202065727791854910073"
        "407130138227535166816275697045988251519109547651406545929782599485693336436816119595008"},
    {0x1.fffffffffffffp+992, chars_format::fixed, 0,
        "8371160993642712515524002897166849896221913747761631226116673872021830677056151384515201609244890425891636"
        "8220706648409475722036738630469314971727646671941963047678850847893430034392555425562404131455583709820146"
        "814260276455070333632551394091976503038219095302813091859565198971386672873632239190016"},
    {0x1.fffffffffffffp+993, chars_format::fixed, 0,
        "1674232198728542503104800579433369979244382749552326245223334774404366135411230276903040321848978085178327"
        "3644141329681895144407347726093862994345529334388392609535770169578686006878511085112480826291116741964029"
        "3628520552910140667265102788183953006076438190605626183719130397942773345747264478380032"},
    {0x1.fffffffffffffp+994, chars_format::fixed, 0,
        "3348464397457085006209601158866739958488765499104652490446669548808732270822460553806080643697956170356654"
        "7288282659363790288814695452187725988691058668776785219071540339157372013757022170224961652582233483928058"
        "7257041105820281334530205576367906012152876381211252367438260795885546691494528956760064"},
    {0x1.fffffffffffffp+995, chars_format::fixed, 0,
        "6696928794914170012419202317733479916977530998209304980893339097617464541644921107612161287395912340713309"
        "4576565318727580577629390904375451977382117337553570438143080678314744027514044340449923305164466967856117"
        "4514082211640562669060411152735812024305752762422504734876521591771093382989057913520128"},
    {0x1.fffffffffffffp+996, chars_format::fixed, 0,
        "1339385758982834002483840463546695983395506199641860996178667819523492908328984221522432257479182468142661"
        "8915313063745516115525878180875090395476423467510714087628616135662948805502808868089984661032893393571223"
        "49028164423281125338120822305471624048611505524845009469753043183542186765978115827040256"},
    {0x1.fffffffffffffp+997, chars_format::fixed, 0,
        "2678771517965668004967680927093391966791012399283721992357335639046985816657968443044864514958364936285323"
        "7830626127491032231051756361750180790952846935021428175257232271325897611005617736179969322065786787142446"
        "98056328846562250676241644610943248097223011049690018939506086367084373531956231654080512"},
    {0x1.fffffffffffffp+998, chars_format::fixed, 0,
        "5357543035931336009935361854186783933582024798567443984714671278093971633315936886089729029916729872570647"
        "5661252254982064462103512723500361581905693870042856350514464542651795222011235472359938644131573574284893"
        "96112657693124501352483289221886496194446022099380037879012172734168747063912463308161024"},
    {0x1.fffffffffffffp+999, chars_format::fixed, 0,
        "1071508607186267201987072370837356786716404959713488796942934255618794326663187377217945805983345974514129"
        "5132250450996412892420702544700072316381138774008571270102892908530359044402247094471987728826314714856978"
        "792225315386249002704966578443772992388892044198760075758024345468337494127824926616322048"},
    {0x1.fffffffffffffp+1000, chars_format::fixed, 0,
        "2143017214372534403974144741674713573432809919426977593885868511237588653326374754435891611966691949028259"
        "0264500901992825784841405089400144632762277548017142540205785817060718088804494188943975457652629429713957"
        "584450630772498005409933156887545984777784088397520151516048690936674988255649853232644096"},
    {0x1.fffffffffffffp+1001, chars_format::fixed, 0,
        "4286034428745068807948289483349427146865619838853955187771737022475177306652749508871783223933383898056518"
        "0529001803985651569682810178800289265524555096034285080411571634121436177608988377887950915305258859427915"
        "168901261544996010819866313775091969555568176795040303032097381873349976511299706465288192"},
    {0x1.fffffffffffffp+1002, chars_format::fixed, 0,
        "8572068857490137615896578966698854293731239677707910375543474044950354613305499017743566447866767796113036"
        "1058003607971303139365620357600578531049110192068570160823143268242872355217976755775901830610517718855830"
        "337802523089992021639732627550183939111136353590080606064194763746699953022599412930576384"},
    {0x1.fffffffffffffp+1003, chars_format::fixed, 0,
        "1714413771498027523179315793339770858746247935541582075108694808990070922661099803548713289573353559222607"
        "2211600721594260627873124071520115706209822038413714032164628653648574471043595351155180366122103543771166"
        "0675605046179984043279465255100367878222272707180161212128389527493399906045198825861152768"},
    {0x1.fffffffffffffp+1004, chars_format::fixed, 0,
        "3428827542996055046358631586679541717492495871083164150217389617980141845322199607097426579146707118445214"
        "4423201443188521255746248143040231412419644076827428064329257307297148942087190702310360732244207087542332"
        "1351210092359968086558930510200735756444545414360322424256779054986799812090397651722305536"},
    {0x1.fffffffffffffp+1005, chars_format::fixed, 0,
        "6857655085992110092717263173359083434984991742166328300434779235960283690644399214194853158293414236890428"
        "8846402886377042511492496286080462824839288153654856128658514614594297884174381404620721464488414175084664"
        "2702420184719936173117861020401471512889090828720644848513558109973599624180795303444611072"},
    {0x1.fffffffffffffp+1006, chars_format::fixed, 0,
        "1371531017198422018543452634671816686996998348433265660086955847192056738128879842838970631658682847378085"
        "7769280577275408502298499257216092564967857630730971225731702922918859576834876280924144292897682835016932"
        "85404840369439872346235722040802943025778181657441289697027116219947199248361590606889222144"},
    {0x1.fffffffffffffp+1007, chars_format::fixed, 0,
        "2743062034396844037086905269343633373993996696866531320173911694384113476257759685677941263317365694756171"
        "5538561154550817004596998514432185129935715261461942451463405845837719153669752561848288585795365670033865"
        "70809680738879744692471444081605886051556363314882579394054232439894398496723181213778444288"},
    {0x1.fffffffffffffp+1008, chars_format::fixed, 0,
        "5486124068793688074173810538687266747987993393733062640347823388768226952515519371355882526634731389512343"
        "1077122309101634009193997028864370259871430522923884902926811691675438307339505123696577171590731340067731"
        "41619361477759489384942888163211772103112726629765158788108464879788796993446362427556888576"},
    {0x1.fffffffffffffp+1009, chars_format::fixed, 0,
        "1097224813758737614834762107737453349597598678746612528069564677753645390503103874271176505326946277902468"
        "6215424461820326801838799405772874051974286104584776980585362338335087661467901024739315434318146268013546"
        "283238722955518978769885776326423544206225453259530317576216929759577593986892724855113777152"},
    {0x1.fffffffffffffp+1010, chars_format::fixed, 0,
        "2194449627517475229669524215474906699195197357493225056139129355507290781006207748542353010653892555804937"
        "2430848923640653603677598811545748103948572209169553961170724676670175322935802049478630868636292536027092"
        "566477445911037957539771552652847088412450906519060635152433859519155187973785449710227554304"},
    {0x1.fffffffffffffp+1011, chars_format::fixed, 0,
        "4388899255034950459339048430949813398390394714986450112278258711014581562012415497084706021307785111609874"
        "4861697847281307207355197623091496207897144418339107922341449353340350645871604098957261737272585072054185"
        "132954891822075915079543105305694176824901813038121270304867719038310375947570899420455108608"},
    {0x1.fffffffffffffp+1012, chars_format::fixed, 0,
        "8777798510069900918678096861899626796780789429972900224556517422029163124024830994169412042615570223219748"
        "9723395694562614414710395246182992415794288836678215844682898706680701291743208197914523474545170144108370"
        "265909783644151830159086210611388353649803626076242540609735438076620751895141798840910217216"},
    {0x1.fffffffffffffp+1013, chars_format::fixed, 0,
        "1755559702013980183735619372379925359356157885994580044911303484405832624804966198833882408523114044643949"
        "7944679138912522882942079049236598483158857767335643168936579741336140258348641639582904694909034028821674"
        "0531819567288303660318172421222776707299607252152485081219470876153241503790283597681820434432"},
    {0x1.fffffffffffffp+1014, chars_format::fixed, 0,
        "3511119404027960367471238744759850718712315771989160089822606968811665249609932397667764817046228089287899"
        "5889358277825045765884158098473196966317715534671286337873159482672280516697283279165809389818068057643348"
        "1063639134576607320636344842445553414599214504304970162438941752306483007580567195363640868864"},
    {0x1.fffffffffffffp+1015, chars_format::fixed, 0,
        "7022238808055920734942477489519701437424631543978320179645213937623330499219864795335529634092456178575799"
        "1778716555650091531768316196946393932635431069342572675746318965344561033394566558331618779636136115286696"
        "2127278269153214641272689684891106829198429008609940324877883504612966015161134390727281737728"},
    {0x1.fffffffffffffp+1016, chars_format::fixed, 0,
        "1404447761611184146988495497903940287484926308795664035929042787524666099843972959067105926818491235715159"
        "8355743311130018306353663239389278786527086213868514535149263793068912206678913311666323755927227223057339"
        "24254556538306429282545379369782213658396858017219880649755767009225932030322268781454563475456"},
    {0x1.fffffffffffffp+1017, chars_format::fixed, 0,
        "2808895523222368293976990995807880574969852617591328071858085575049332199687945918134211853636982471430319"
        "6711486622260036612707326478778557573054172427737029070298527586137824413357826623332647511854454446114678"
        "48509113076612858565090758739564427316793716034439761299511534018451864060644537562909126950912"},
    {0x1.fffffffffffffp+1018, chars_format::fixed, 0,
        "5617791046444736587953981991615761149939705235182656143716171150098664399375891836268423707273964942860639"
        "3422973244520073225414652957557115146108344855474058140597055172275648826715653246665295023708908892229356"
        "97018226153225717130181517479128854633587432068879522599023068036903728121289075125818253901824"},
    {0x1.fffffffffffffp+1019, chars_format::fixed, 0,
        "1123558209288947317590796398323152229987941047036531228743234230019732879875178367253684741454792988572127"
        "8684594648904014645082930591511423029221668971094811628119411034455129765343130649333059004741781778445871"
        "394036452306451434260363034958257709267174864137759045198046136073807456242578150251636507803648"},
    {0x1.fffffffffffffp+1020, chars_format::fixed, 0,
        "2247116418577894635181592796646304459975882094073062457486468460039465759750356734507369482909585977144255"
        "7369189297808029290165861183022846058443337942189623256238822068910259530686261298666118009483563556891742"
        "788072904612902868520726069916515418534349728275518090396092272147614912485156300503273015607296"},
    {0x1.fffffffffffffp+1021, chars_format::fixed, 0,
        "4494232837155789270363185593292608919951764188146124914972936920078931519500713469014738965819171954288511"
        "4738378595616058580331722366045692116886675884379246512477644137820519061372522597332236018967127113783485"
        "576145809225805737041452139833030837068699456551036180792184544295229824970312601006546031214592"},
    {0x1.fffffffffffffp+1022, chars_format::fixed, 0,
        "8988465674311578540726371186585217839903528376292249829945873840157863039001426938029477931638343908577022"
        "9476757191232117160663444732091384233773351768758493024955288275641038122745045194664472037934254227566971"
        "152291618451611474082904279666061674137398913102072361584369088590459649940625202013092062429184"},
    {0x1.fffffffffffffp+1023, chars_format::fixed, 0,
        "1797693134862315708145274237317043567980705675258449965989174768031572607800285387605895586327668781715404"
        "5895351438246423432132688946418276846754670353751698604991057655128207624549009038932894407586850845513394"
        "2304583236903222948165808559332123348274797826204144723168738177180919299881250404026184124858368"},
};

#endif // DOUBLE_FIXED_PRECISION_TO_CHARS_TEST_CASES_4_HPP
